/*
    File:       GaugeView.m

    Contains:   Subclass of view to implement a simple round analog gauge.
		You can set the minimum, maximum value, start angle,
		angle range, title, font, and more. It is a pretty generic
		round gauge view, if you ever have need for one.

    Written By:	Bruce Blumberg

    Created:    June 19, 1997

    Copyright:  (c) 1997 by Apple Computer, Inc., all rights reserved.

    Change History: Originally written for 0.6 mid 1988 by Bruce Blumberg.
                     Modified for 1.0 by Ali Ozer.
                     Redesigned for 2.0 by Julie Zelenski.
                     Redesigned for Rhapsody by Stephen Chick.
		     Changed to use NSString API to draw text (7/15/97, SC).

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "Gauge.h"
#import "GaugeView.h"

//  Constants.

#define HANDRATIO 0.65		// Ratio of the hand length to the face size.

@implementation GaugeView

//-----------------------------------------------------------------------------
//
//  initWithFrame:
//
//  Overrides NSView's initWithFrame method to initialize this custom view.
//
//  Basic initialization method which just calls our designated initializer
//  with generally useful default values.
//

- (id) initWithFrame:(NSRect) frameRect
{
	self = [self initWithFrame:frameRect min:0.0 max:100.0 startAngle:215.0
		angleRange:250.0 tickInterval:10]; 
	return self;
}

//-----------------------------------------------------------------------------
//
//  initWithFrame: min: max: startAngle: angleRange: tickInterval:
//
//  Our designated initializer. Creates an offscreen image where the face of
//  the gauge is drawn. Later the face image is composited into the view and
//  gauge's hand is drawn on top.
//

- (id) initWithFrame:(NSRect) frameRect min:(float) initMin max:(float) initMax
	startAngle:(float) initStartAngle angleRange:(float) initAngleRange
	tickInterval:(int) initTickInterval
{
	NSRect bounds;

	self = [super initWithFrame:frameRect];

	//  Get our bounds rect.

	bounds = [self bounds];

	//  Create an offscreen image for the face.

	face = [[NSImage allocWithZone:[self zone]] initWithSize:bounds.size];

	//  Set the parameters.

	startAngle = initStartAngle;
	angleRange = initAngleRange;
	tickInterval = initTickInterval;
	value = initMin;
	[self setMin:initMin];
	[self setMax:initMax];

	//  Set our view's font, size, and title.

	font = [NSFont fontWithName:@"Helvetica" size:10.0];
	[self setTitle:@"Stress"];

	//  Calculate geometrical stuff.

	radius = (bounds.size.height / 2.0) - 8.0;
	center.x = bounds.size.width / 2.0;
	center.y = bounds.size.height / 2.0;

	//  Call a PSWrap function to create the PostScript function mhand.
	//  mhand draws the gauge's hand.

	PSWmakeHand (radius * HANDRATIO);

	//  Draw the gauge face.

	[self drawFace];
	return self;
}

//-----------------------------------------------------------------------------
//
//  dealloc
//
//  Overrides NSView's dealloc method to release objects we own.
//
//  Releases the off-screen image, font, and gauge title string.
//  Then calls super.
//

- (void) dealloc
{
	[face release];
	[font release];
	[title release];
	[super dealloc];
}

//-----------------------------------------------------------------------------
//
//  setTitle:
//
//  Sets the gauge's title text. needRedraw is set to indicate the face needs
//  to be redrawn.
//

- (void) setTitle:(NSString *) newTitle
{
	[title release];
	title = [NSString stringWithString:newTitle];
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setFont
//
//  Sets the gauge's title font. needRedraw is set to indicate the face needs
//  to be redrawn.
//

- (void) setFont:(NSFont *) newFont
{
	if (font != newFont) {
		[font release];
		font = [newFont retain];
	}
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setMin: newMin
//
//  Sets the gauge's minimum value and recalculates degreesPerUnit which is
//  used to determine tick and label intervals. needRedraw is set to indicate
//  the face needs to be redrawn.
//

- (void) setMin:(float) newMin
{
	min = newMin;
	degreesPerUnit = angleRange / (max - min);
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setMax: newMax
//
//  Sets the gauge's maximum value and recalculates degreesPerUnit which is
//  used to determine tick and label intervals. needRedraw is set to indicate
//  the face needs to be redrawn.
//

- (void) setMax:(float) newMax
{
	max = newMax;
	degreesPerUnit = angleRange/ (max - min);
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setStartAngle:
//
//  Sets the gauge's start angle (the angle of the arm when at the minimum
//  value). needRedraw is set to indicate the face needs to be redrawn.
//

- (void) setStartAngle:(float) newStartAngle
{
	startAngle = newStartAngle;
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setAngleRange:
//
//  Sets the gauge's angle range (the sweep of the arm from its minimum to
//  maximum value). The value cannot exceed 360 degrees (one full revolution).
//  degreesPerUnit is recalculated to determine tick and label intervals.
//  needRedraw is set to indicate the face needs to be redrawn.
//

- (void) setAngleRange:(float) newAngleRange
{
	if (newAngleRange > 360)
		newAngleRange = 360.0;
	angleRange = newAngleRange;
	degreesPerUnit = angleRange / (max - min);
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setTickInterval:
//
//  Sets the gauge's tick mark interval (the number of units between tick marks
//  on the gauge's face).  needRedraw is set to indicate the face needs to be
//  redrawn.
//

- (void) setTickInterval:(int) newTickInterval
{
	tickInterval = newTickInterval;
	needRedraw = YES;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  changeValue:
//
//  Target action for an Interface Builder control. Takes the floatValue from
//  the sender control (either the slider or the text field), makes sure the
//  slider and the text field are in sync, and displays the gauge's arm at the
//  new value.
//

- (void) changeValue:(id) sender
{
	float newValue;

	newValue = [sender floatValue];
	if (newValue != value) {	// if value changed
		value = newValue;
		if (sender == valueSlider)
			[valueFormCell setIntValue:newValue];
		else
			[valueSlider setFloatValue:newValue];
		[self setNeedsDisplay:YES];
    }
}
	
//-----------------------------------------------------------------------------
//
//  drawFace
//
//  Draws the gauge's face image in the off-screen image by erasing the
//  background and drawing the circular border, title text, tick marks, and
//  labels. The off-screen cache is composited on-screen with the gauge hand in
//  drawRect.
//

- (void) drawFace
{	
	float angle, angleIncrement;
	int number;
        NSPoint point;
        NSRect bounds;
	NSSize size;
	NSDictionary *attributes;

	//  Get our boundsRect.

	bounds = [self bounds];

	//  Lock the PostScript focus on the content view of our off-screen
	//  image so that the following drawing operations take effect there.

	[face lockFocus];

	//  Erase the background with the default background color for
        //  controls.

        [[NSColor whiteColor] set];
        NSRectFill (bounds);
        [[NSColor blackColor] set];

	//  Call our PS wrap function to draw the gauge border.

	PSWdrawBorder (center.x, center.y, radius);

	//  Create the attributes used for drawing the text.

	attributes = [NSDictionary dictionaryWithObject:font
		      forKey:NSFontAttributeName];

	//  Draw the gauge title.

	size = [title sizeWithAttributes:attributes];
	point.x = bounds.size.width / 2.0 - size.width / 2.0;
	point.y = center.y - 1.0;
	[title drawAtPoint:point withAttributes:attributes];

	//  Draw the gauge ticks.

	angleIncrement = angleRange / ((max - min) / tickInterval); 
	PSWdrawTicks (center.x, center.y, radius * HANDRATIO,
		      angleIncrement / 2, startAngle, startAngle+angleRange);
	number = min;
	for (angle = startAngle;
	     angle >= startAngle - angleRange;
	     angle -= angleIncrement) {
		NSString *numString;

		//  Convert this tick label into a string.

		numString = [NSString stringWithFormat:@"%d", number];

		//  Calculate its position.

		size = [numString sizeWithAttributes:attributes];
                point.x = cos (angle / 57.3) * (radius - 1.0 - size.width / 2.0) +
			  center.x - (size.width / 2);
                point.y = sin (angle / 57.3) * (radius - 1.0 - size.height / 2.0) +
			  center.y - (size.height / 2.5) - 8.0;

		//  Draw it.

                [numString drawAtPoint:point withAttributes:attributes];
		number += tickInterval;
	}

	// We're done drawing off-screen, so unlock the PostScript focus.

	[face unlockFocus];
	needRedraw = NO;
}

//-----------------------------------------------------------------------------
//
//  drawHand
//
//  Calculates the angle for the current gauge value and draws the hand there.
//

- (void) drawHand
{
	float valueAngle;

	valueAngle = startAngle - degreesPerUnit * (value - min);
	PSWdrawHand (center.x, center.y, valueAngle);
}

//-----------------------------------------------------------------------------
//
//  drawRect:
//
//  Overrides NSView's drawRect method to draw this custom view.
//
//  Draws the face and hand of the gauge. If needRedraw is YES, the face needs
//  to be redrawn in the offscreen image. Otherwise, the image is composited
//  into the bounds of the view, and a PSWrap is called which draws the hand in
//  the correct position.
//

- (void) drawRect:(NSRect) rect
{
	if (needRedraw)
		[self drawFace];
	[face compositeToPoint:[self bounds].origin operation:NSCompositePlusDarker];
	[self drawHand];
}

@end
