/*
    File:       AnimalView.m

    Contains:   Subclass of view to implement barnyard animals (and other
		sundry images).

    Written By:	Julie Zelenski

    Created:    June 19, 1997

    Copyright:  (c) 1997 by Apple Computer, Inc., all rights reserved.

    Change History: Originally written by Julie Zelenski.
		     Redesigned for Rhapsody by Stephen Chick.

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "AnimalView.h"

@implementation AnimalView

//-----------------------------------------------------------------------------
//
//  initWithFrame:
//
//  Overrides NSView's initWithFrame method to initialize this custom view.
//
//  Calls super. Then loads the default duck image.
//

- (id) initWithFrame:(NSRect) frameRect
{
	[super initWithFrame:frameRect];
	[self setImageToAnimal:@"Duck"];	//  Default is the duck.
	return self;
}

//-----------------------------------------------------------------------------
//
//  dealloc
//
//  Overrides NSView's dealloc method to release objects we own.
//
//  Releases the image. Then calls super.
//

- (void) dealloc
{
	[imageRep release];	//  Release the animal image.
	[super dealloc];
}

//-----------------------------------------------------------------------------
//
//  changeAnimal:
//
//  Target action for an Interface Builder control. Takes the selected cell's
//  title from the sender control and attempts to load an animal image of the
//  same name.
//

- (void) changeAnimal:(id) sender
{
	[self setImageToAnimal:[[sender selectedCell] title]];
}

//-----------------------------------------------------------------------------
//
//  shrinkEnlarge:
//
//  Target action for an Interface Builder control. Toggles this view between
//  shrunk and enlarged. Also updates the menu item title appropriately.
//

- (void) shrinkEnlarge:(id) sender
{
	NSString *title;

	//  Retrieve the localized string for the specified key.
	//  Strings are stored in "Localizable.strings".

	if (shrunk) {

		//  Key is "SHRINK".

		title = NSLocalizedString (@"SHRINK", nil);
		shrunk = NO;
	}
	else {

		//  Key is "ENLARGE".

		title = NSLocalizedString (@"ENLARGE", nil);
		shrunk = YES;
	}
	[sender setTitle:title];
        [self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  rotateLeft:
//
//  Target action for an Interface Builder control. Rotates this view's bounds
//  by 90 degrees.
//

- (void) rotateLeft:(id) sender
{
	[self rotateByAngle:90.0];
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  rotateRight:
//
//  Target action for an Interface Builder control. Rotates this view's bounds
//  by -90 degrees.
//

- (void) rotateRight:(id) sender
{
	[self rotateByAngle:-90.0];
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setImageToAnimal:
//
//  Loads the specified animal image.
//

- (void) setImageToAnimal:(NSString *) imageName
{
	[imageRep release];
	imageRep = [[NSImage imageNamed:imageName]
		    bestRepresentationForDevice:nil];
	[imageRep retain];				//  Keep it around.
        [self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  drawRect:
//
//  Overrides NSView's drawRect method to draw this custom view.
//
//  Erases the entire view and draws the animal image.
//

- (void) drawRect:(NSRect) rect
{
	NSRect drawRect;

	//  Calculate the rect within which to draw the animal (if it's shrunk
	//  adjust the draw rect appropriately).

	drawRect = [self bounds];
	if (shrunk) {
		drawRect.size.width /= 2;
		drawRect.size.height /= 2;
		drawRect.origin.x += (drawRect.size.width / 2);
		drawRect.origin.y += (drawRect.size.height / 2);
	}

	//  Erase the view (white).

	[[NSColor whiteColor] set];
	NSRectFill ([self bounds]);
	[[NSColor blackColor] set];

	//  Draw the animal.

	[imageRep drawInRect:drawRect];
}

@end
