/* cpdfMemBuf.c -- Memory Stream
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

	Write to memory stream as if writing to file via fprintf(), fwrite() etc.
	without worrying about how long the memory buffer is (no buffer over-runs).
	The buffer will be expanded as needed to accomodate more data.
	At the end, the entire buffer content will be available with the address
	of the buffer, and the length of the content.
	Modelled after NeXT's streams, but I implemented write related functions only,
	i.e., equivalents of fscanf(), fread() are not available.
	See near the end of file for usage example.

1998-08-27 [IO]

For testing this module stand-alone (with main() for testing):
cc -Wall -g -DMAINDEF -o cpdfMemBuf cpdfMemBuf.c

---
*/

#include "version.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "cpdflib.h"		/* This must be included before all other local include files */

#ifdef MacOS8
#include <stat.mac.h>
#include <unix.mac.h>
#include <StandardFile.h>
#include <Memory.h>
#include <Files.h>
#include <Strings.h>	/* c2pstr() */
extern size_t strlen(const char *s);
extern int strcmp(const char *s1, const char *s2);
extern void *memcpy(void *s1, const void *s2, size_t n);
#endif


#define MEMSTREAM_MAGIC		0xa5b5cafe
#define MINBUFSIZE 		256
/* don't increase buffer size by more than this much at a time */
#define MAXBUFBUMP 		65536

static char *memErrorFmt = "ClibPDF: corrupted  memory stream: 0x%lx (%lu)\n";

/* For write operation, address must be NULL, and size = 0 */
CPDFmemStream *cpdf_openMemoryStream(void)
{
CPDFmemStream *aMstrm;
	aMstrm = (CPDFmemStream *) malloc((size_t)sizeof(CPDFmemStream));
	_cpdf_malloc_check((void *)aMstrm);
	aMstrm->magic_number = (unsigned long)MEMSTREAM_MAGIC;
	aMstrm->buffer = (char *)malloc((size_t)MINBUFSIZE);
	_cpdf_malloc_check((void *)aMstrm->buffer);
 	aMstrm->bufSize = MINBUFSIZE;
	aMstrm->count = 0;
	return aMstrm;
}

/* clears content of memory stream while keeping the buffer as is */
void cpdf_clearMemoryStream(CPDFmemStream *aMstrm)
{
	if(aMstrm) {
	    if(aMstrm->magic_number != MEMSTREAM_MAGIC)
	        fprintf(stderr, memErrorFmt, (long unsigned)aMstrm, (long unsigned)aMstrm);
	    aMstrm->count = 0;
	    *aMstrm->buffer = 0;	/* null terminate */
	}
}

void cpdf_closeMemoryStream(CPDFmemStream *aMstrm)
{
	if(aMstrm->magic_number != MEMSTREAM_MAGIC)
	    fprintf(stderr, memErrorFmt, (long unsigned)aMstrm, (long unsigned)aMstrm);
 	if(aMstrm->buffer) free(aMstrm->buffer);
	if(aMstrm) free(aMstrm);
	aMstrm = NULL;
}

/* Write (append) more data of length 'len' to the memory stream */

int cpdf_writeMemoryStream(CPDFmemStream *memStream, char *data, int len)
{
int errcode = 0;
    /* expand buffer as needed */
    if(memStream->magic_number != MEMSTREAM_MAGIC)
	fprintf(stderr, memErrorFmt, (long unsigned)memStream, (long unsigned)memStream);

    if((len + memStream->count + 1) > memStream->bufSize) {    /* Must expand buffer */
        if (memStream->buffer == NULL) {
            memStream->buffer = malloc(MINBUFSIZE);
	    _cpdf_malloc_check((void *)memStream->buffer);
            memStream->bufSize = MINBUFSIZE;
        }
        else {
	    while((len + memStream->count + 1) > memStream->bufSize) {
		if(memStream->bufSize < MAXBUFBUMP)
                    memStream->bufSize *= 2;		/* geometrical growth upto a limit */
		else
               	    memStream->bufSize += MAXBUFBUMP;	/* switch to linear increment */
	    }
            memStream->buffer = realloc(memStream->buffer, memStream->bufSize);
	    _cpdf_malloc_check((void *)memStream->buffer);
        }
    }
    if(errcode) return(errcode);

    /* Put the new data in the buffer and update the count */
    memcpy(memStream->buffer + memStream->count, data, len);
    memStream->count += len;

    /* null-terminate buffer (for string applications) */
    *(memStream->buffer + memStream->count) = 0;
     return(0);
}


int cpdf_memPutc(int ch, CPDFmemStream *memStream)
{
char c1[2];
	c1[0] = ch; c1[1] = 0;
	cpdf_writeMemoryStream(memStream, c1, 1);
	return (1);
}

int cpdf_memPuts(char *str, CPDFmemStream *memStream)
{
int numchar = strlen(str);
	cpdf_writeMemoryStream(memStream, str, numchar);
	return numchar;
}

void cpdf_getMemoryBuffer(CPDFmemStream *memStream, char **streambuf, int *len, int *maxlen)
{
	if(memStream->magic_number != MEMSTREAM_MAGIC)
	    fprintf(stderr, memErrorFmt, (long unsigned)memStream, (long unsigned)memStream);
	*streambuf = memStream->buffer;
	*len = memStream->count;
	*maxlen = memStream->bufSize;
}

/* Using a special filename "-" will send the buffer to stdout */

int cpdf_saveMemoryStreamToFile(CPDFmemStream *memStream, const char *name)
{
FILE *fpout;
char *memBuffer;
int memLen, bufSize;
    if( strcmp(name, "-") == 0)
	fpout = fdopen(1, BINARY_WRITE);	/* fd of 1 == stdout */
    else {
	if((fpout = fopen(name, BINARY_WRITE)) == NULL) {
	    fprintf(stderr, "ClibPDF: Cannot open output file: %s\n", name);
	    return(-1);
	}
    }
    cpdf_getMemoryBuffer(memStream, &memBuffer, &memLen, &bufSize);
    /* fprintf(stderr, "file=%s\nlength=%d, bufferSize=%d\n", name, memLen, bufSize); */
    fwrite(memBuffer, 1, (size_t)memLen, fpout);
    fclose(fpout);

    return(0);
}


/* memory stream debug dump function */

void _checkMemMagic(char *idstr, CPDFmemStream *memStream)
{
    if(memStream->magic_number != MEMSTREAM_MAGIC) {
	fprintf(stderr, "%s: stream=%lu, magic= %lu\n",
	    idstr, (long unsigned)memStream, (long unsigned)memStream->magic_number);
        fprintf(stderr, "buffer= %lu\n", (long unsigned)memStream->buffer);
        fprintf(stderr, "count= %d, bufSize=%d\n", memStream->count, memStream->bufSize);
    }
}


#ifdef MAINDEF
static char *string1 = "0123456789 This is a test of memory stream0123456789";

void main(void)
{
char sbuf[256];
int i;
CPDFmemStream *memStream = { NULL };
char *memBuffer;
int memLen, bufSize;
	memStream = cpdf_openMemoryStream();
	for(i=0; i<50; i++) {
	    sprintf(sbuf, "%05d - %s\n", i, string1);
	    cpdf_writeMemoryStream(memStream, sbuf, strlen(sbuf));
	}
	cpdf_getMemoryBuffer(memStream, &memBuffer, &memLen, &bufSize);
	cpdf_saveMemoryStreamToFile(memStream, "junkmem.out");
	fprintf(stderr,"memLen=%d,  bufSize=%d\n", memLen, bufSize);
	cpdf_closeMemoryStream(memStream);
}


#endif

