/* cpdfMarker.c  -- plot markers 
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

1998-07-09 [IO]
*/

#include "version.h"

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"



void cpdf_marker(float x, float y, int markertype, float size)
{
	cpdf_rawMarker(x_Domain2Points(x), y_Domain2Points(y), markertype, size);
}

void cpdf_pointer(float x, float y, int direction, float size)
{
	cpdf_rawPointer(x_Domain2Points(x), y_Domain2Points(y), direction, size);
}

/* draws errorbar with caps, capsize in points */
void cpdf_errorbar(float x, float y1, float y2, float capsize)
{
	cpdf_rawErrorbar(x_Domain2Points(x), y_Domain2Points(y1), y_Domain2Points(y2), capsize);
}

/* primarily for daily stock prices: high, low and close */
void cpdf_highLowClose(float x, float vhigh, float vlow, float vclose, float ticklen)
{
	cpdf_rawHighLowClose(x_Domain2Points(x),
	     y_Domain2Points(vhigh), y_Domain2Points(vlow), y_Domain2Points(vclose), ticklen);
}

/* =============================================================================== */
/* Raw (point-based corrdinate) data point markers and other symbols */

/* draw a marker at a raw point-based coordinate */
void cpdf_rawMarker(float x, float y, int markertype, float size)
{
float halfsize = size*0.5;

    cpdf_gsave();
    switch(markertype) {
	case 0:		/* circle */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawQuickCircle(x, y, halfsize);
    		cpdf_fillAndStroke();
		break;
	case 1:		/* up triangle */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawMoveto(x, y+size*0.6667);
		cpdf_rawRlineto(-size/1.7321, -size);
		cpdf_rawRlineto(1.1546*size, 0.0);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 2:		/* diamond */
    		cpdf_setlinewidth(size*0.15);
		size *= 0.9;
		cpdf_rawMoveto(x, y+size/1.38);
		cpdf_rawRlineto(0.546*size, -size/1.38);
		cpdf_rawRlineto(-0.546*size, -size/1.38);
		cpdf_rawRlineto(-0.546*size, size/1.38);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 3:		/* square */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawRect(x - halfsize, y - halfsize, size, size);
    		cpdf_fillAndStroke();
		break;
	case 4:		/* star */
		size *= 1.2;
		halfsize = 0.5*size;
    		cpdf_setlinewidth(size*0.09);
		cpdf_rawMoveto(x, y+size*0.5);
		cpdf_rawLineto(x+0.112255*size, y+0.15451*size);
		cpdf_rawLineto(x+0.47552*size, y+0.15451*size);
		cpdf_rawLineto(x+0.181635*size, y-0.05902*size);
		cpdf_rawLineto(x+0.29389*size, y-0.40451*size);
		cpdf_rawLineto(x, y-0.19098*size);
		cpdf_rawLineto(x-0.29389*size, y-0.40451*size);
		cpdf_rawLineto(x-0.181635*size, y-0.05902*size);
		cpdf_rawLineto(x-0.47552*size, y+0.15451*size);
		cpdf_rawLineto(x-0.112255*size, y+0.15451*size);
		cpdf_closepath();
		cpdf_fillAndStroke();
		break;
	case 5:		/* pentagon */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawMoveto(x-0.5257*size, y+size*0.1708);
		cpdf_rawRlineto(0.5257*size, 0.382*size);
		cpdf_rawRlineto(0.5257*size, -0.382*size);
		cpdf_rawRlineto(-0.2008*size, -0.6181*size);
		cpdf_rawRlineto(-0.6499*size, 0.0);
		cpdf_closepath();
		cpdf_fillAndStroke();
		break;
	case 6:		/* down triangle */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawMoveto(x, y-size*0.6667);
		cpdf_rawRlineto(-size/1.7321, size);
		cpdf_rawRlineto(1.1546*size, 0.0);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 7:		/* horiz bow-tie */
    		cpdf_setlinewidth(size*0.13);
		cpdf_rawMoveto(x-0.5*size, y-0.5*size);
		cpdf_rawLineto(x+0.5*size, y+0.5*size);
		cpdf_rawLineto(x+0.5*size, y-0.5*size);
		cpdf_rawLineto(x-0.5*size, y+0.5*size);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 8:		/* vert bow-tie */
    		cpdf_setlinewidth(size*0.13);
		cpdf_rawMoveto(x-0.5*size, y-0.5*size);
		cpdf_rawLineto(x+0.5*size, y+0.5*size);
		cpdf_rawLineto(x-0.5*size, y+0.5*size);
		cpdf_rawLineto(x+0.5*size, y-0.5*size);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
		break;
	case 9:		/* asterisk */
		size *= 1.05;
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawMoveto(x, y+size*0.5);
		cpdf_rawRlineto(0.0, -size);
		cpdf_rawMoveto(x+0.433*size, y+0.25*size);
		cpdf_rawLineto(x-0.433*size, y-0.25*size);
		cpdf_rawMoveto(x+0.433*size, y-0.25*size);
		cpdf_rawLineto(x-0.433*size, y+0.25*size);
    		cpdf_stroke();
		break;
	case 10:	/* sun */
    		cpdf_setlinewidth(size*0.15);
		cpdf_rawQuickCircle(x, y, size*0.25);
		cpdf_rawMoveto(x+size*0.5, y);
		cpdf_rawLineto(x+size*0.25, y);
		cpdf_rawMoveto(x-size*0.5, y);
		cpdf_rawLineto(x-size*0.25, y);
		cpdf_rawMoveto(x, y-size*0.5);
		cpdf_rawLineto(x, y-size*0.25);
		cpdf_rawMoveto(x, y+size*0.5);
		cpdf_rawLineto(x, y+size*0.25);
    		cpdf_fillAndStroke();
		break;

	default:
		break;
    } /* end of switch() */
    cpdf_grestore();
}


/* draw a marker at a raw point-based coordinate */
void cpdf_rawPointer(float x, float y, int direction, float size)
{
    cpdf_gsave();
    switch(direction & 3) {
	case 0:		/* right */
    		cpdf_setlinewidth(size*0.14);
		cpdf_rawMoveto(x, y);
		cpdf_rawLineto(x-size, y+size*0.3333);
		cpdf_rawLineto(x-size, y-size*0.3333);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 1:		/* down */
    		cpdf_setlinewidth(size*0.14);
		cpdf_rawMoveto(x, y);
		cpdf_rawLineto(x+size*0.3333, y+size);
		cpdf_rawLineto(x-size*0.3333, y+size);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 2:		/* left */
    		cpdf_setlinewidth(size*0.14);
		cpdf_rawMoveto(x, y);
		cpdf_rawLineto(x+size, y+size*0.3333);
		cpdf_rawLineto(x+size, y-size*0.3333);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	case 3:		/* up */
    		cpdf_setlinewidth(size*0.14);
		cpdf_rawMoveto(x, y);
		cpdf_rawLineto(x+size*0.3333, y-size);
		cpdf_rawLineto(x-size*0.3333, y-size);
		cpdf_closepath();
    		cpdf_fillAndStroke();
		break;
	default:
		break;
    } /* end of switch() */
    cpdf_grestore();
}

void cpdf_rawErrorbar(float x, float y1, float y2, float capsize)
{
	cpdf_rawMoveto(x, y1);
	cpdf_rawLineto(x, y2);
	cpdf_rawMoveto(x-0.5*capsize, y1);
	cpdf_rawLineto(x+0.5*capsize, y1);
	cpdf_rawMoveto(x-0.5*capsize, y2);
	cpdf_rawLineto(x+0.5*capsize, y2);
	cpdf_stroke();
}

void cpdf_rawHighLowClose(float x, float vhigh, float vlow, float vclose, float ticklen)
{
	cpdf_rawMoveto(x, vlow);
	cpdf_rawLineto(x, vhigh);
	cpdf_rawMoveto(x, vclose);
	cpdf_rawLineto(x+ticklen, vclose);
	cpdf_stroke();

}


