/* cpdfInit.c  -- Initialization and Primary Processing Module ----------------------------
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

I know this module has gotten a bit too large.
I am not going to break it up just yet.

1998-06-30 [IO]
	-- First version, NEXTSTEP3.x
*/

#include "version.h"


#include <string.h>
#include <math.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef UNIX
#include <unistd.h>		/* for access() etc */
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#endif

#if defined(_WIN32) || defined(WIN32)
/* #include <sys/file.h> */
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#endif


#ifdef MacOS8
#include <stat.mac.h>
#include <unix.mac.h>
#include <StandardFile.h>
#include <Memory.h>
#include <Files.h>
#include <Strings.h>	/* c2pstr() */
#endif

#ifndef EXT_ZLIBCOMP
#include "zlib.h"
#endif

#define MAINDEF 1

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"

#define CHECK_ERR(err, msg) { \
    if (err != Z_OK) { \
        fprintf(stderr, "ClibPDF: %s error: %d\n", msg, err); \
        exit(1); \
    } \
}

/* VC++ malloc debug: run the executable from VC++ workspace: Build->Debug->Go */

#if defined(_WIN32) || defined(WIN32)
#include <crtdbg.h>

void my_init_debug() {
#ifdef _DEBUG
        int tmpFlag = _CrtSetDbgFlag( _CRTDBG_REPORT_FLAG);
        tmpFlag |= _CRTDBG_CHECK_ALWAYS_DF; /* check memory at every alloc & dealloc */
        tmpFlag |= _CRTDBG_LEAK_CHECK_DF; /* checks for leaks at end of program */
        _CrtSetDbgFlag(tmpFlag);
#endif
}
#endif

/* ---------------------------------------------------------------------------------------- */
/* Key puclic API functions for ClibPDF library */
/* ---------------------------------------------------------------------------------------- */

static init_check = 0;

void cpdf_open(int pspdf)
{
int i;
#if defined(_WIN32) || defined(WIN32)
	my_init_debug();
#endif
	_cpdf_initDocumentGolbals();		/* initialize global data for document */
	ps_pdf_mode = pspdf;
	cpdf_setCreator("A ClibPDF program");
	cpdf_setTitle("No Title");
	cpdf_setSubject("None");
	cpdf_setKeywords("ClibPDF");
	/* set default compression command: zlibcomp */
	cpdf_setCompressionFilter(ZLIB_COMPRESS_PATH, "/FlateDecode");
	monthName = (char **)malloc((size_t)(12*sizeof(char *)));
	for(i=0; i<12; i++)
	    monthName[i] = NULL;
	cpdf_setMonthNames(monthNameEnglish);
	/* provide non-overflowing memory stream for scratch pad use */
	scratchMem = cpdf_openMemoryStream();
#if defined(_WIN32) || defined(WIN32)
	_fmode = _O_BINARY;	    /* Why is this so hard in _WIN32 ?? */
#endif
#ifdef EXT_ZLIBCOMP
	useContentMemStream = 0;    /* can't use external command if memory stream is used */
#endif
	inlineImages = 0;
	init_check = 1;
}

void cpdf_enableCompression(int cmpON)
{
    if(cmpON == 0) {
	compressionON = 0;
	return;
    }
/* Check only if we are using external compression command */
#ifdef EXT_ZLIBCOMP
#ifndef MacOS8			/* why Metrowerks CW IDE 1.7.4 doesn't have this */
    else if( access(compress_command, F_OK) != 0) {
	/* Check if the compression program specifiled in compress_command exists.
	   If the compress_command does not exist, we will fall back on NO compression.
	   Things should still work.  This check provides some safety.
	*/
	compressionON = 0;	/* compression program does not exist, so don't use it */
	return;
    }
#endif
#endif
    /* OK, we want compression, and the compression program exists */
    /* Let's see what kind of decode filter we specify */
    if(streamFilterList != NULL) {
	compressionON = cmpON;		/* Only if filter command is set */
	if(cmpON && (strstr(streamFilterList, "FlateDecode") != NULL))
	    cpdf_setPDFLevel(1, 2);	/* FlateDecode requires PDF-1.2 */
    }
    else {
	compressionON = 0;		/* compression filter has not been set */
    }
}


/* "Command" must be a FULL pathname of the executable that implements Acrobat Reader-
   compatible compression, of the type decodeded by "filterlist" decode-filter chain.
   Existence of the compression command file is checked.  If it does not exist,
   no compression will be used, but things should still work.
   Default is: cpdf_setCompressionFilter(ZLIB_COMPRESS_PATH, "/FlateDecode");
   where ZLIB_COMPRESS_PATH is specified in config.h.
*/
void cpdf_setCompressionFilter(char *command, char *filterlist)
{
	if(compress_command) free(compress_command);
	compress_command = (char *)malloc((size_t)(strlen(command) + 1));
	_cpdf_malloc_check((void *)compress_command);
	strcpy(compress_command, command);
	/* Decode filter list as specified in PDF /Filter [] list */
	if(streamFilterList) free(streamFilterList);
	streamFilterList = (char *)malloc((size_t)(strlen(filterlist) +1));
	_cpdf_malloc_check((void *)streamFilterList);
	strcpy(streamFilterList, filterlist);
}

void cpdf_init(void)
{
int i;
CPDFpageInfo *pT;
#ifdef UNIX
uid_t myuid;
struct passwd *pw;
extern uid_t getuid(void);
	myuid = getuid();
	pw = getpwuid(myuid);
	strncpy(username, pw->pw_gecos, 32);
	strcat(username, " [");
	strncat(username, pw->pw_name, 16);
	strcat(username, "]");
#endif
#ifdef MacOS8
	strncpy(username, "MacOS 8.* or less User", 60);
#endif
#if defined(_WIN32) || defined(WIN32)
	strncpy(username, "Windows NT/95/98 User", 60);
#endif

	if(init_check != 1) {
	    fprintf(stderr, "ClibPDF: cpdf_open() has not been called.\n");
	    exit(1);
	}
	init_check = 2;

	/* For later sanity check to detect undefined pages, e.g.,
	   cpdf_pageInit() may be called with page number skipped.  If that
	   happens, we skip undefined pages when writing the final file.
	   pageInfos[0] is unused.  Initialize it here, but don't free.
	*/
	for(i=0; i<= NMAXPAGES; i++) {
	    pT = &pageInfos[i];
	    pT->pagenum = -1;
	    pT->npFont = 0;
	    pT->npImage = 0;
	    pT->npAnnot = 0;
	    pT->pageMemStream = NULL;	/* initialize all malloc items to NULL */
	    pT->defDomain = NULL;
	    pT->fontIdx = NULL;
	    pT->imageIdx = NULL;
	    pT->annotIdx = NULL;
	    pT->mediaBox = NULL;
	    pT->cropBox = NULL;
	    pT->fppage = NULL;
	    pT->contentfile = NULL;
	    pT->duration = -1.0;
	    pT->transition = NULL;
	}

	/* If we use memory stream for the entire PDF (file), then creat that too. */
	if(usePDFMemStream)
	    pdfMemStream = cpdf_openMemoryStream();	/* write PDF to memory stream */
	else
	    _cpdf_file_open();				/* open a stream to file for PDF output */
}


/* cpdf_pageInit() must be called for each page.
   This creats a memory stream and sets it as currentStream.
*/
int cpdf_pageInit(int pagenum, int rot, char *mediaboxstr, char *cropboxstr)
{	
float llxt, llyt, urxt, uryt;
CPDFpageInfo  *nP;		/* pointer to page info struct */
/* CPDFmemStream *contentStream; */	/* memory stream */
#if defined(UNIX) && !defined(SunOS5x)
extern int getpid(void);
int mypid = getpid();
#else
int mypid = 123;
#endif
	if(pagenum < 1) {
	    fprintf(stderr, "ClibPDF: Page number must be 1 or greater.\n");
	    return(-1);
	}
	else if(pagenum >= NMAXPAGES) {
	    fprintf(stderr, "ClibPDF: Too many pages. Increase NMAXPAGES in cpdflib.h and recompile library.\n");
	    return(-1);
	}
	nP = &pageInfos[pagenum];
	if(nP->pagenum != -1) {
	    fprintf(stderr, "ClibPDF: Page %d is already initialized.\n", pagenum);
	    cpdf_setCurrentPage(pagenum);
	    return(0);
	}
	nP->pagenum = pagenum;
	currentPage = pagenum;
	if(numPages < pagenum) numPages = pagenum;

	if(!rot) display_rotation = 0;		/* rot=0: portrait means no rotation for display */
	else	 display_rotation = 270;	/* rot=1: landscape means rotation for display */
	nP->orientation = display_rotation;

	/* nP->mediaBox, and nP->cropBox are given malloced storage for strings in cpdf_setPageSize() */
	cpdf_setPageSize(mediaboxstr, cropboxstr);
	/* create and set default domain in inch coordinate system */
	/* get default domain size from cropBox string (defined in %%BoundingBox format) in cglobals.h */
	sscanf(cropBox, "%f %f %f %f", &llxt, &llyt, &urxt, &uryt);

	/* allocate storage for font, image, annotation index arrays */
	nP->fontIdx = (int *)malloc((size_t)NMAXFONTS);
	nP->imageIdx = (int *)malloc((size_t)NMAXIMAGES);
	nP->annotIdx = (int *)malloc((size_t)NMAXANNOTS);
	_cpdf_malloc_check((void *)nP->fontIdx);
	_cpdf_malloc_check((void *)nP->imageIdx);
	_cpdf_malloc_check((void *)nP->annotIdx);

	/* Each page has its own domains */
	/* (CPDFplotDomain *)cpdf_createPlotDomain(float x, float y, float w, float h,
				float xL, float xH, float yL, float yH,
				int xtype, int ytype, int rsvd) */
	defaultDomain = cpdf_createPlotDomain(llxt, llyt, urxt-llxt, uryt-llyt,
				llxt/defdomain_unit, (urxt-llxt)/defdomain_unit,
				llyt/defdomain_unit, (uryt-llyt)/defdomain_unit, LINEAR, LINEAR, 0);
	nP->defDomain = defaultDomain;

	/* These default settings probably should be moved into createDomain itself */
	cpdf_setLinearMeshParams(defaultDomain, Y_MESH, 0.0, 1.0, 0.0, 0.5);
	cpdf_setLinearMeshParams(defaultDomain, X_MESH, 0.0, 1.0, 0.0, 0.5);
	cpdf_setPlotDomain(defaultDomain);

	/* If we use memory stream for Content, then create stream and set is as current stream. */
	if(useContentMemStream) {
	    nP->pageMemStream = cpdf_openMemoryStream();
	    cpdf_setCurrentMemoryStream(nP->pageMemStream);
	}
	/* === Now, set filename for temporary file or memory stream to store Content === */
	else {	/* use temporary file for Content */
	    strcpy(contentfile, TEMP_DIRECTORY);
	    strcat(contentfile, "_cpdf");
	    str_append_int(contentfile, mypid);
	    strcat(contentfile, "-");
	    str_append_int(contentfile, fncounter);
	    strcat(contentfile, "-");
	    str_append_int(contentfile, pagenum);
	    fpcontent = fopen(contentfile, BINARY_WRITE);	/* open for actual drawing code */
	    if(fpcontent == NULL) {
		    fprintf(stderr, "ClibPDF: Cannot open %s for temporary content.\n", contentfile);
		    return(-2);
	    }

	    nP->fppage = fpcontent;
	    nP->contentfile = (char *)malloc((size_t)(strlen(contentfile) + 8));  /* ".zlib" */
	    _cpdf_malloc_check((void *)nP->contentfile);
	    strcpy(nP->contentfile, contentfile);
	}
	return(0);
}

void cpdf_setPageDuration(float seconds)
{
	pageInfos[currentPage].duration = seconds;
}


int cpdf_setPageTransition(int type, float duration, float direction, int HV, int IO)
{
char trbuf[128];
char *hv, *io;
    if(HV) hv = "/H";		/* /Dm value */
    else   hv = "/V";

    if(IO) io = "/I";		/* /M value */
    else   io = "/O";

    switch(type) {
	default:
	case TRANS_NONE:
			sprintf(trbuf, "/S /R");
			break;
	case TRANS_SPLIT:
			sprintf(trbuf, "/S /Split /D %.3f /Dm %s /M %s", duration, hv, io);
			break;
	case TRANS_BLINDS:
			sprintf(trbuf, "/S /Blinds /D %.3f /Dm %s", duration, hv);
			break;
	case TRANS_BOX:
			sprintf(trbuf, "/S /Box /D %.3f /M %s", duration, io);
			break;
	case TRANS_WIPE:
			sprintf(trbuf, "/S /Wipe /D %.3f /Di %.2f", duration, direction);
			break;
	case TRANS_DISSOLVE:
			sprintf(trbuf, "/S /Dissolve /D %.3f", duration);
			break;
	case TRANS_GLITTER:
			sprintf(trbuf, "/S /Glitter /D %.3f /Di %.2f", duration, direction);
			break;
    }
    /* now store transition spec string in pageInfos[] */
    if(pageInfos[currentPage].transition)
	free(pageInfos[currentPage].transition);
    pageInfos[currentPage].transition = (char *)malloc((size_t)(strlen(trbuf) +1));
    _cpdf_malloc_check((void *)(pageInfos[currentPage].transition));
    strcpy(pageInfos[currentPage].transition, trbuf);
    return(0);
}


/* Switch current page to another previously initialized page.
   With this function, you can draw into multiple pages in an interleaved manner.
   Switching pages will reset the current domain to the default domain for that page.
*/
int cpdf_setCurrentPage(int page)
{
CPDFpageInfo *cP;

	if(page == currentPage) return(0);	/* no change */
	cP = &pageInfos[page];
	if( cP->pagenum == -1) {
		    fprintf(stderr, "ClibPDF: Cannot switch to page %d because it has not been initialized.\n", page);
		    return(-1);
	}
	/* Now, we should be able to switch. */
	currentPage = page;
	defaultDomain = cP->defDomain;			/* switch default domain */
	cpdf_setPlotDomain(defaultDomain);

	if(useContentMemStream) {
	    cpdf_setCurrentMemoryStream(cP->pageMemStream);
	}
	else {
	    fpcontent = cP->fppage;			/* switch content file descriptor */
	    strcpy(contentfile, cP->contentfile);	/* sync current content filename */
	}

	/* Also copy mediaBox, cropBox */
	strncpy(mediaBox, cP->mediaBox, 62);
	strncpy(cropBox, cP->cropBox, 62);

	return(0);
}

/* This function closes the content stream file for a given page.
   It has an effect only when temporary files are used to store each page.
   It has no effect for the default case where memory streams are used.
   The purpose of this function is to allow files for completed pages to be
   closed on systems that can open only a limited number of files simultaneously.
   If this is not a problem, there is no need to call this function.
*/
void cpdf_finalizePage(int page)
{
	if(!useContentMemStream && (pageInfos[page].pagenum != -1) && pageInfos[page].fppage)
	    fclose(pageInfos[page].fppage);
}



int cpdf_savePDFmemoryStreamToFile(char *file)
{
int retcode= -1;
#ifdef MacOS8
    OSType filetype='PDF ', creator='CARO';	/* Acrobat/PDF file */
    /* The convention of storing four characters in a long, and of specifying
    them with single quotes is not part of standard C, but is a standard
    Macintosh extension to C. */
#endif

    strncpy(filenamepath, file, 1022);
    filename_set  = 1;
    if(usePDFMemStream && pdfMemStream)
	retcode = cpdf_saveMemoryStreamToFile(pdfMemStream, file);

#ifdef MacOS8
    SetFileInfo(file, filetype, creator);	/* set PDF file type */
#endif
    return(retcode);
}

char *cpdf_getBufferForPDF(int *length)
{
int bufsize;
char *mbuff;
    if(usePDFMemStream && pdfMemStream) {
        cpdf_getMemoryBuffer(pdfMemStream, &mbuff, length, &bufsize);
    }
    else {
	mbuff = NULL;
	*length = 0;
    }
    return mbuff;
}

void _cpdf_pdfWrite(char *s)
{
int bcount = strlen(s);
    if(usePDFMemStream)
	cpdf_writeMemoryStream(pdfMemStream, s, bcount);
    else
	fputs(s, fpcg);
    currentByteCount += bcount;
}


/* Shut down the whole library */
void cpdf_close(void)
{
	cpdf_closeMemoryStream(scratchMem);	/* release scratch pad buffer */
	if(usePDFMemStream && pdfMemStream)
	    cpdf_closeMemoryStream(pdfMemStream);	/* release memory stream for PDF output */
	_cpdf_freeMonthNames();
	free(monthName);
}

/* cpdf_finalizeAll() is where the PDF file is actually written.
   The drawing code (Contents) is first written to a temp file
   or memory stream before this function is called.
   The size of the temp file is determined here and copied to the PDF file.
   If the PDF stream is to a file, the file is closed in this function.
   Put some binary data near the beginning so binary/text autodetection
   in some software will treat PDF files as binary.
   Do not modify the bmagic string below.  */
static char *bmagic ="zG_\325\371\337J\244\030\267\260#s6\037\246dR L\204s\037";

void cpdf_finalizeAll(void)
{
int i, objectCount;
CPDFfontInfo *tfont;
#ifdef MacOS8
    OSType filetype='PDF ', creator='CARO';	/* Acrobat/PDF file */
    /* The convention of storing four characters in a long, and of specifying
    them with single quotes is not part of standard C, but is a standard
    Macintosh extension to C. */
#endif
	currentByteCount = 0;
	sprintf(spbuf, "%%PDF-%d.%d\n%s\n", pdfLevelMaj, pdfLevelMin, bmagic); _cpdf_pdfWrite(spbuf);
	objByteOffset[0] = 0;
	objByteOffset[1] = currentByteCount;

	/* Scan through all objects (except for xref and trailer) in order, and
	   assign serialized object number for xref entries. */
	objectCount = 1;
	objIndex[CPDF_Catalog] = objectCount++;
	objIndex[CPDF_Outlines] = objectCount++;
	objIndex[CPDF_Pages] = objectCount++;
	objIndex[CPDF_ProcSet] = objectCount++;
/*
	objIndex[CPDF_Page] = objectCount++;
	objIndex[CPDF_Contents] = objectCount++;
*/
	for(i=1; i<= numPages; i++) {
	    /* only those pages with pagenum != -1 are valid */
	    if(pageInfos[i].pagenum != -1) {
		pageInfos[i].objIndex = objectCount++;		/* Page object for this page */
		kidsIndex[numKids++] = pageInfos[i].objIndex;
		pageInfos[i].parent = objIndex[CPDF_Pages];	/* Only one Pages object (see above) */
		/* Contents object for this page is (pageInfos[i].objIndex + 1) */
		pageInfos[i].contents = objectCount++;		/* advance count for Contents for this page */
		if(!useContentMemStream && pageInfos[i].fppage != NULL)
		    fclose(pageInfos[i].fppage);		/* close temp content file */
	    }
	}
	for(i=0; i< numFonts; i++)
	    fontInfos[i].objIndex = objectCount++;		/* for Page object Resources entry */
	for(i=0; i< numImages; i++)
	    imageInfos[i].objIndex = objectCount++;
	for(i=0; i< numAnnots; i++)
	    annotInfos[i].objIndex = objectCount++;

	objIndex[CPDF_Info] = objectCount++;

	/* =========== Now, object indexed, so write objects to output file ======================= */
	_cpdf_WriteCatalogObject(objIndex[CPDF_Catalog]);
	_cpdf_WriteOutlinesObject(objIndex[CPDF_Outlines]);
	_cpdf_WritePagesObject(objIndex[CPDF_Pages]);
	_cpdf_WriteProcSetArray(objIndex[CPDF_ProcSet]);

	/* Write out all pages */
	/* Now, multiple pages -- pageInfos[0] is unused. */
	for(i=1; i<= numPages; i++) {
	    /* only those pages with pagenum != -1 are valid */
	    if(pageInfos[i].pagenum != -1) {
		_cpdf_WritePageObject(&pageInfos[i]);
		if(useContentMemStream)
		    _cpdf_WriteContentsFromMemory(&pageInfos[i]);
		else
		    _cpdf_WriteContentsFromFile(&pageInfos[i]);
	    }
	}

	for(i=0; i< numFonts; i++) {
	    tfont = &fontInfos[i];
	    /* _cpdf_WriteFont(2, "Times-RomanMac", "Times-Roman", "MacRomanEncoding"); */
	    _cpdf_WriteFont(tfont->objIndex, tfont->name, tfont->baseFont, tfont->encoding);
	}
	for(i=0; i< numImages; i++)
	    _cpdf_WriteImage(&imageInfos[i]);

	for(i=0; i< numAnnots; i++)
	    _cpdf_WriteAnnotation(&annotInfos[i]);

	_cpdf_WriteProducerDate(objIndex[CPDF_Info]);
	_cpdf_WriteXrefTrailer(objectCount);
	/* PDF file objects all written */

	if(!usePDFMemStream)
	    _cpdf_file_close();		/* close the PDF file */
					/* but keep PDF memory stream open if that is used */
	_cpdf_freeAllFontInfos();	/* release memory for malloc'ed CPDFfontInfo's */
	_cpdf_freeAllImageInfos();	/* release memory for malloc'ed CPDFimageInfo's */
	_cpdf_freeAllAnnotInfos();
	_cpdf_freeAllPageInfos();

#ifdef MacOS8
	SetFileInfo(filenamepath, filetype, creator);	/* set PDF file type */
#endif
	if(compress_command) free(compress_command);
	compress_command = NULL;
	if(streamFilterList) free(streamFilterList);
	streamFilterList = NULL;
	numAnnots = 0;		/* reset annotation for next time */
	/* cpdf_freePlotDomain(defaultDomain); */	/* now done in _cpdf_freeAllPageInfos() */
	defaultDomain = NULL;
	currentDomain = NULL;
	init_check = 0;
}

/* set page size as a string in %%BoundingBox: format in PS */
void cpdf_setPageSize(char *mboxstr, char *cboxstr)
{
CPDFpageInfo *pI;
	strncpy(mediaBox, mboxstr, 62);
	strncpy(cropBox, cboxstr, 62);
	/* Iffy, but allow changing page size after cpdf_pageInit() */
        pI = &pageInfos[currentPage];
	if(pI->mediaBox)
	    free(pI->mediaBox);
	pI->mediaBox = (char *)malloc((size_t)(strlen(mediaBox) + 1));
	_cpdf_malloc_check((void *)pI->mediaBox);
	strcpy(pI->mediaBox, mediaBox);
	if(pI->cropBox)
	    free(pI->cropBox);
	pI->cropBox = (char *)malloc((size_t)(strlen(cropBox) + 1));
	_cpdf_malloc_check((void *)pI->cropBox);
	strcpy(pI->cropBox, cropBox);
	return;
}

void cpdf_setBoundingBox(int LLx, int LLy, int URx, int URy)
{
	cpdf_setMediaBox(LLx, LLy, URx, URy);
	cpdf_setCropBox(LLx, LLy, URx, URy);
}

void cpdf_setMediaBox(int LLx, int LLy, int URx, int URy)
{
char tstr[64];
	sprintf(tstr, "%d %d %d %d", LLx, LLy, URx, URy);
	strncpy(mediaBox, tstr, 62);
}

void cpdf_setCropBox(int LLx, int LLy, int URx, int URy)
{
char tstr[64];
	sprintf(tstr, "%d %d %d %d", LLx, LLy, URx, URy);
	strncpy(cropBox, tstr, 62);
}


CPDFmemStream *cpdf_setCurrentMemoryStream(CPDFmemStream *memStream)
{
CPDFmemStream *oldMS;
	oldMS = currentMemStream;
	currentMemStream = memStream;
	return oldMS;
}

/* Returns the size of file in bytes */

long getFileSize(char *file)
{
long filesize = 0;
#if defined(_WIN32) || defined(WIN32)
struct _stat filestat;		/* For Windows VC++ */
#else
struct stat filestat;
#endif

#if defined(_WIN32) || defined(WIN32)
	if(_stat(file, &filestat))	/* Windows */
#else
	if(stat(file, &filestat))
#endif
	    filesize = 0;		/* stat failed -- no such file */
	else {
	    if( !(filestat.st_mode & S_IFREG) )
		filesize = 0;	/* not a regular file */
#ifdef UNIX
	    else if( !(filestat.st_mode & S_IREAD) )
		filesize = 0;	/* not readable */
#endif
	    else
		filesize = filestat.st_size;
	}
	return(filesize);
}



/* ---------------------------------------------------------------------------------------- */
/*   Private functions -- Do not call functions below directly. */
/* ---------------------------------------------------------------------------------------- */

int  _cpdf_freeAllPageInfos(void)
{
CPDFpageInfo *tpage;
int i;
    for(i=1; i<= numPages; i++) {
	tpage = &pageInfos[i];
	if(tpage->pagenum == -1)
	    continue;		    /* skip unused pages */
	if(tpage->pageMemStream) {
	    cpdf_closeMemoryStream(tpage->pageMemStream);
	    tpage->pageMemStream = NULL;
	}
	if(tpage->defDomain) {
	    cpdf_freePlotDomain(tpage->defDomain);
	    tpage->defDomain = NULL;
	}
	if(tpage->contentfile) {
	    free(tpage->contentfile);
	    tpage->contentfile = NULL;
	}
	if(tpage->fontIdx) {
	    free(tpage->fontIdx);
	    tpage->fontIdx = NULL;
	}
	if(tpage->imageIdx) {
	    free(tpage->imageIdx);
	    tpage->imageIdx = NULL;
	}
	if(tpage->annotIdx) {
	    free(tpage->annotIdx);
	    tpage->annotIdx = NULL;
	}
	if(tpage->annotIdx) {
	    free(tpage->annotIdx);
	    tpage->annotIdx = NULL;
	}
	if(tpage->mediaBox) {
	    free(tpage->mediaBox);
	    tpage->mediaBox = NULL;
	}
	if(tpage->cropBox) {
	    free(tpage->cropBox);
	    tpage->cropBox = NULL;
	}
	if(tpage->transition) {
	    free(tpage->transition);
	    tpage->transition = NULL;
	}
    } /* end for(i...) */
    return(0);
}


int _cpdf_file_open(void)
{
char pps[]=".pdf";
#if defined(UNIX) && !defined(SunOS5x)
extern int getpid(void);
int mypid = getpid();
#else
int mypid = 123;
#endif

	if(!filename_set) {
	    strcpy(filenamepath, TEMP_DIRECTORY);
	    strcat(filenamepath, "_cpdf");
	    str_append_int(filenamepath, mypid);
	    strcat(filenamepath, "-");
	    str_append_int(filenamepath, fncounter);
	    strcat(filenamepath, pps);
	    filename_set = 1;
	}

	if(strcmp(filenamepath, "-") == 0)
	    useStandardOutput = 1;
	if(useStandardOutput)
	    fpcg = fdopen(1, BINARY_WRITE);	/* fd of 1 == stdout */
	else
	    fpcg = fopen(filenamepath, BINARY_WRITE);	/* open for write */
	    /*  Open PDF file in binary mode to turn off LF -> CRLF translation.
		This is necessary because we are doing byte offset counting by strlen()
		function which assumes '\n' to be the end-of-line char.  Otherwise, the
		offsets in PDF xref table will be incorrect, and Acrobat Reader will give
		a warning about the damaged file (though it can read it fine).
	    */

	if(fpcg == NULL) {
		fprintf(stderr, "ClibPDF: Cannot open %s for PDF output\n", filenamepath);
		return(1);
	}

	fncounter++;
	return(0);			/* successful file/device open */
}

/* used in showpage() to flush/close network i/o */
void _cpdf_file_close(void)
{
	fclose(fpcg);
}



/* ---------------------------------------------------------------------------------------- */
/*   Private functions that write PDF objects */
/* ---------------------------------------------------------------------------------------- */

long _cpdf_WriteCatalogObject(int objNumber)
{
	sprintf(spbuf, "%d 0 obj\n", objNumber); 	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n"); 			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Catalog\n");		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Pages %d 0 R\n", objIndex[CPDF_Pages]);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Outlines %d 0 R\n", objIndex[CPDF_Outlines]);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");			_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WriteOutlinesObject(int objNumber)
{
	sprintf(spbuf, "%d 0 obj\n", objNumber);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Outlines\n");		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Count 0\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");			_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WritePagesObject(int objNumber)
{
int i;
	sprintf(spbuf, "%d 0 obj\n", objNumber);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Pages\n");		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Count %d\n", numKids);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Kids [ ");  			_cpdf_pdfWrite(spbuf);
	for(i=0; i< numKids; i++) {
	    sprintf(spbuf, "%d 0 R ", kidsIndex[i]);
	    _cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, "]\n>>\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");			_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WritePageObject(CPDFpageInfo *pInf)
{
int i;
CPDFfontInfo *tfont;
CPDFimageInfo *timg;
CPDFannotInfo *tann;
	sprintf(spbuf, "%d 0 obj\n", pInf->objIndex);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Page\n");		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Parent %d 0 R\n", pInf->parent);  _cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Resources <<\n/Font <<\n");	_cpdf_pdfWrite(spbuf);
	for(i=0; i< (pInf->npFont); i++) {
	    tfont = &fontInfos[pInf->fontIdx[i]];
	    sprintf(spbuf, "/%s %d 0 R\n", tfont->name, tfont->objIndex);
	    _cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, ">>\n"); _cpdf_pdfWrite(spbuf);
	if(pInf->npImage) {
	    sprintf(spbuf, "/XObject <<\n");	_cpdf_pdfWrite(spbuf);
	    for(i=0; i < (pInf->npImage); i++) {
		timg = &imageInfos[pInf->imageIdx[i]];
		sprintf(spbuf, "/%s %d 0 R\n", timg->name, timg->objIndex);
	        _cpdf_pdfWrite(spbuf);
	    }
	    sprintf(spbuf, ">>\n"); _cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, "/ProcSet %d 0 R >>\n", objIndex[CPDF_ProcSet]); _cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/MediaBox [%s]\n", pInf->mediaBox);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/CropBox [%s]\n", pInf->cropBox);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Rotate %d\n", pInf->orientation);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Contents %d 0 R\n", pInf->contents);		_cpdf_pdfWrite(spbuf);
	if(pInf->npAnnot) {
	    sprintf(spbuf, "/Annots [ "); 				_cpdf_pdfWrite(spbuf);
	    for(i=0; i < (pInf->npAnnot); i++) {
		tann = &annotInfos[pInf->annotIdx[i]];
		sprintf(spbuf, "%d 0 R ", tann->objIndex);
	    _cpdf_pdfWrite(spbuf);
	    }
	    sprintf(spbuf, "]\n"); 					_cpdf_pdfWrite(spbuf);
	}
	if(pInf->duration > 0.0) {
	    sprintf(spbuf, "/Dur %.3f\n", pInf->duration);		_cpdf_pdfWrite(spbuf);
	}
	if(pInf->transition) {
	    sprintf(spbuf, "/Trans << %s >>\n", pInf->transition);	_cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, ">>\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");			_cpdf_pdfWrite(spbuf);
	objByteOffset[pInf->objIndex+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WriteContentsFromFile(CPDFpageInfo *pInf)
{
long	filesize;
#ifdef EXT_ZLIBCOMP
int c;
char syscomdbuf[1024];
#else
Byte *inputBuf, *outputBuf;
uLong comprLen = 0;
int err;
#endif

	strcpy(contentfile, pInf->contentfile);		/* filaname for this page */

#ifdef EXT_ZLIBCOMP
	if(compressionON) {
	    sprintf(syscomdbuf, "%s %s", compress_command, contentfile);
	    system(syscomdbuf);
	    strcat(contentfile, ".zlib");
	}
#endif

	filesize = getFileSize(contentfile);
	sprintf(spbuf, "%d 0 obj\n", pInf->contents);		_cpdf_pdfWrite(spbuf);

#ifdef EXT_ZLIBCOMP
	if(compressionON) {
	    /* Here, we use external zlibcomp command for compression, so use filesize */
	    sprintf(spbuf, "<<\n/Length %ld\n/Filter [%s]\n>>\nstream\n", filesize, streamFilterList);
	    _cpdf_pdfWrite(spbuf);
	}
	else {
	    sprintf(spbuf, "<</Length %ld>>\nstream\n", filesize);  _cpdf_pdfWrite(spbuf);
	}

	/* copy file from fpcontent to fpcg */
	if((fpcontent = fopen(contentfile, BINARY_READ)) != NULL) {
	    while((c = fgetc(fpcontent)) != EOF)
		    fputc(c, fpcg);
	    fclose(fpcontent);
	}
	currentByteCount += filesize;				/* add size of Contents file */
#else
/* else of #ifdef EXT_ZLIBCOMP -- So, here, we do zlib compression internally. */
	inputBuf = (Byte *)malloc((size_t)(filesize+16));
	_cpdf_malloc_check((void *)inputBuf);
	if((fpcontent = fopen(contentfile, BINARY_READ)) != NULL) {
	    fread(inputBuf, 1, (size_t)filesize, fpcontent);
	    fclose(fpcontent);
	}
	if(compressionON) {
	    comprLen = filesize+1024;		/* must set this for input */
	    outputBuf = (Byte *)malloc((size_t)comprLen);
	    _cpdf_malloc_check((void *)outputBuf);
	    err = compress(outputBuf, &comprLen, inputBuf, filesize);
	    CHECK_ERR(err, "zlib compress");
	    /* fprintf(stderr, "input: %ld,  output: %ld\n", filesize, comprLen); */
	    sprintf(spbuf, "<<\n/Length %ld\n/Filter [%s]\n>>\nstream\n", comprLen, streamFilterList);
	    _cpdf_pdfWrite(spbuf);
	    if(usePDFMemStream)
	        cpdf_writeMemoryStream( pdfMemStream, (char *)outputBuf, comprLen);
	    else
	        fwrite(outputBuf, 1, (size_t)comprLen, fpcg);
	    currentByteCount += comprLen;				/* add size of Contents file */
	    free(outputBuf);
	}
	else {
	    /* No compression */
	    sprintf(spbuf, "<</Length %ld>>\nstream\n", filesize); _cpdf_pdfWrite(spbuf);
	    if(usePDFMemStream)
	        cpdf_writeMemoryStream( pdfMemStream, (char *)inputBuf, filesize);
	    else
	        fwrite(inputBuf, 1, (size_t)filesize, fpcg);
	    currentByteCount += filesize;				/* add size of Contents file */
	}
	free(inputBuf);
#endif
/* end of #ifdef EXT_ZLIBCOMP */

#ifndef	DEBUG
	remove(contentfile);				/* delete temporary content file */
#endif

	sprintf(spbuf, "\nendstream\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[pInf->contents+1] = currentByteCount;
	return(currentByteCount);
}


long _cpdf_WriteContentsFromMemory(CPDFpageInfo *pInf)
{
Byte *inputBuf, *outputBuf;
uLong comprLen = 0;
int filesize;
float headroom;
int iheadroom;
int bufSize;
int err;

	sprintf(spbuf, "%d 0 obj\n", pInf->contents);		_cpdf_pdfWrite(spbuf);
	cpdf_getMemoryBuffer(pInf->pageMemStream, (char **)&inputBuf, &filesize, &bufSize);

	if(compressionON) {
	    /* zlib compress needs a little head room for output buffer size */
	    headroom = (float)filesize * 0.001 + 16.0;	/* 0.1% plus 12 bytes */
	    iheadroom = (int)headroom;
	    /* fprintf(stderr, "filesize=%d, bufSize=%d, headroom=%d\n", filesize, bufSize, iheadroom); */
	    comprLen = filesize+iheadroom;		/* must set this for input */
	    outputBuf = (Byte *)malloc((size_t)comprLen);
	    _cpdf_malloc_check((void *)outputBuf);
	    err = compress(outputBuf, &comprLen, inputBuf, filesize);
	    CHECK_ERR(err, "zlib compress");
	    /* fprintf(stderr, "input: %ld,  output: %ld\n", filesize, comprLen); */
	    sprintf(spbuf, "<<\n/Length %ld\n/Filter [%s]\n>>\nstream\n", comprLen, streamFilterList);
	    _cpdf_pdfWrite(spbuf);
	    if(usePDFMemStream)
	        cpdf_writeMemoryStream( pdfMemStream, (char *)outputBuf, comprLen);
	    else
	        fwrite(outputBuf, 1, (size_t)comprLen, fpcg);
	    currentByteCount += comprLen;				/* add size of Contents file */
	    free(outputBuf);
	}
	else {
	    /* No compression */
	    sprintf(spbuf, "<</Length %d>>\nstream\n", filesize); _cpdf_pdfWrite(spbuf);
	    if(usePDFMemStream)
	        cpdf_writeMemoryStream( pdfMemStream, (char *)inputBuf, filesize);
	    else
	        fwrite(inputBuf, 1, (size_t)filesize, fpcg);
	    currentByteCount += filesize;				/* add size of Contents file */
	}

	sprintf(spbuf, "\nendstream\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[pInf->contents+1] = currentByteCount;
	return(currentByteCount);
}



long _cpdf_WriteProcSetArray(int objNumber)
{
	sprintf(spbuf, "%d 0 obj\n", objNumber);		_cpdf_pdfWrite(spbuf);
	if(numImages || inlineImages) {
	    sprintf(spbuf, "[/PDF /Text ");			_cpdf_pdfWrite(spbuf);
	    if(imageFlagBCI & 1) {
		sprintf(spbuf, "/ImageB ");		/* gray scale images */
		_cpdf_pdfWrite(spbuf);
	    }
	    if(imageFlagBCI & 2) {
		sprintf(spbuf, "/ImageC ");		/* color images */
		_cpdf_pdfWrite(spbuf);
	    }
	    if(imageFlagBCI & 4) {
		sprintf(spbuf, "/ImageI ");		/* indexed color images */
		_cpdf_pdfWrite(spbuf);
	    }
	    sprintf(spbuf, "]\n");				_cpdf_pdfWrite(spbuf);
	}
	else {
		sprintf(spbuf, "[/PDF /Text]\n");
		_cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WriteFont(int objNumber, char *fontName, char *baseFont, char *encoding)
{
	sprintf(spbuf, "%d 0 obj\n", objNumber);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Font\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Subtype /Type1\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Name /%s\n", fontName);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/BaseFont /%s\n", baseFont);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Encoding /%s\n", encoding);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}

/*                                 0               1            2           3              4 */
static char *colorspaces[] = { "DeviceGray", "DeviceGray", "DeviceRGB", "DeviceRGB", "DeviceRGB" };

long _cpdf_WriteImage(CPDFimageInfo *imgInf)
{
FILE *fpimg;
char *inputBuf;
	sprintf(spbuf, "%d 0 obj\n", imgInf->objIndex);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /XObject\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Subtype /Image\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Name /%s\n", imgInf->name);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Width %d\n", imgInf->width);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Height %d\n", imgInf->height);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/BitsPerComponent %d\n", imgInf->bitspersample);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/ColorSpace /%s\n", colorspaces[imgInf->ncomponents]);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Filter /DCTDecode\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Length %ld\n", imgInf->filesize);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "stream\n");				_cpdf_pdfWrite(spbuf);

	/* Now, copy the JPEG file to output, and adjust currentByteCount */
	inputBuf = (char *)malloc((size_t)(imgInf->filesize+16));
	_cpdf_malloc_check((void *)inputBuf);
	if((fpimg = fopen(imgInf->filepath, BINARY_READ)) != NULL) {
	    fread(inputBuf, 1, (size_t)(imgInf->filesize), fpimg);
	    fclose(fpimg);
	}
/*
	fprintf(stderr, "jpeg read open mode %s\n", BINARY_READ);
*/
	if(usePDFMemStream)
	    cpdf_writeMemoryStream( pdfMemStream, inputBuf, imgInf->filesize);
	else
	    fwrite(inputBuf, 1, (size_t)(imgInf->filesize), fpcg);
	currentByteCount += (imgInf->filesize);		/* add size of Contents file */
	free(inputBuf);

	sprintf(spbuf, "\nendstream\n");			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[imgInf->objIndex +1] = currentByteCount;
	return(currentByteCount);
}


long _cpdf_WriteAnnotation(CPDFannotInfo *aInf)
{
	sprintf(spbuf, "%d 0 obj\n", aInf->objIndex);			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");						_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Type /Annot\n");				_cpdf_pdfWrite(spbuf);
	if(aInf->type == ANNOT_TEXT) {
	    sprintf(spbuf, "/Subtype /Text\n");				_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/Rect [%.0f %.0f %.0f %.0f]\n",
			aInf->xLL, aInf->yLL, aInf->xUR, aInf->yUR);	_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/T (%s)\n", aInf->annot_title);		_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/Contents (%s)\n", aInf->content_link);	_cpdf_pdfWrite(spbuf);
	}
	else if(aInf->type == ANNOT_LINK) {
	    sprintf(spbuf, "/Subtype /Link\n");				_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/Rect [%.0f %.0f %.0f %.0f]\n",
			aInf->xLL, aInf->yLL, aInf->xUR, aInf->yUR);	_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/Border [0 0 0]\n");			_cpdf_pdfWrite(spbuf);
	    sprintf(spbuf, "/A << /S /URI\n/URI (%s)\n>>\n", aInf->content_link);
									_cpdf_pdfWrite(spbuf);
	}
	
	sprintf(spbuf, ">>\n");						_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");					_cpdf_pdfWrite(spbuf);
	objByteOffset[aInf->objIndex+1] = currentByteCount;
	return(currentByteCount);
}

long _cpdf_WriteProducerDate(int objNumber)
{
	sprintf(spbuf, "%d 0 obj\n", objNumber);			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");						_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Creator (%s)\n", creator_name);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/CreationDate (D:%s)\n", timestring(0));  	_cpdf_pdfWrite(spbuf);
/*
   As part of the license, the following conditions must be observed for the
   /Producer field: You may modify the /Producer field provided that the original
   LIBRARY_NAME and CPDF_VERSION are included intact formatted as [%s %s] in the
   modified field.  You may add any additional strings before or after the above.
   There are no restrictions on other fields.
*/
	sprintf(spbuf, "/Producer ([%s %s] %s)\n", LIBRARY_NAME, CPDF_VERSION, PLATFORM_NAME);
								_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Author (User: %s)\n", username);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Title (%s)\n", file_title);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Subject (%s)\n", file_subject);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Keywords (%s)\n", file_keywords);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "endobj\n");				_cpdf_pdfWrite(spbuf);
	objByteOffset[objNumber+1] = currentByteCount;
	return(currentByteCount);
}


long _cpdf_WriteXrefTrailer(int objNumber)
{
int i;
	startXref = currentByteCount;
	sprintf(spbuf, "xref\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "0 %d\n", objNumber);			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "0000000000 65535 f \n");		_cpdf_pdfWrite(spbuf);
	for(i=1; i<objNumber; i++) {
	     sprintf(spbuf, "%010ld 00000 n \n", objByteOffset[i]);	_cpdf_pdfWrite(spbuf);
	}
	sprintf(spbuf, "trailer\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "<<\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Size %d\n", objNumber);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Root %d 0 R\n", objIndex[CPDF_Catalog]);	_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "/Info %d 0 R\n", objIndex[CPDF_Info]);		_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, ">>\n");					_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "startxref\n");				_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "%ld\n", startXref);			_cpdf_pdfWrite(spbuf);
	sprintf(spbuf, "%%%%EOF\n");				_cpdf_pdfWrite(spbuf);

	return(currentByteCount);
}

void _cpdf_initDocumentGolbals(void)
{
    ps_pdf_mode = 0;			/* PDF=0, EPDF=1, PS=2, EPS=3, FDF=4 (not used) */
    pdfLevelMaj = 1;			/* PDF level, do not use operators beyond these */
    pdfLevelMin = 1;
    defdomain_unit = POINTSPERINCH;	/* unit for default domain */
    display_rotation = 270;
    useStandardOutput = 0;		/* send output to stdout if non-zero */
    compressionON = 0;			/* compress stream */
    compress_command = NULL;		/* command for LZW compression */
    streamFilterList = NULL;		/* for PDF stream /Filter spec */
    launchPreview = 1;			/* launch viewer application on the output file */
    filename_set = 0;			/* flag indicating if output filename is set explicitly */
    fncounter = 0;			/* filename counter for a given process */
    inTextObj = 0;			/* flag indicating within Text block between BT ET */
    defaultDomain = NULL;		/* default plot domain */
    currentDomain = NULL;		/* current plot domain */
    x2points=1.0, y2points=1.0;		/* scaling factor for current domain */
    /* double xLlog, xHlog, yLlog, yHlog; */		/* scaling factor for current domain (logarithmic) */
    numFonts = 0;					/* number of fonts used */
    /* CPDFfontInfo fontInfos[NMAXFONTS]; */		/* array of font infos */
    currentFont = 0;			/* current font index (into fontInfos[]) */
    inlineImages = 0;			/* in-line image count */
    numImages = 0;
    /* CPDFimageInfo imageInfos[NMAXIMAGES]; */
    imageFlagBCI = 0;			/* bit-0 (/ImageB), bit-1 (/ImageC), bit-2 (/ImageI) */
    font_size = 12.0;			/* current font size and info below */
    word_spacing = 0.0;
    char_spacing = 0.0;
    text_rise = 0.0;
    horiz_scaling = 100.0;		/* text horizontal scaling in percent */
    text_leading = 0.0;

    usePDFMemStream = 1;		/* if non-zero use memory stream for PDF generation */
    pdfMemStream = NULL;		/* memory stream for PDF file that is currently active */
    useContentMemStream = 1;		/* if non-zero use memory stream for Content */
    currentMemStream = NULL;		/* memory stream for Content that is currently active */
    currentPage =1;			/* current page number that is being drawn */
    numPages =1;			/* number of pages - may be greater than actual # of pages */
    /* CPDFpageInfo pageInfos[NMAXPAGES+1]; */	/* array of pageInfo structure for all pages */
    numKids = 0;			/* actual # of pages counted for Pages object */
    /* int kidsIndex[NMAXPAGES]; */	/* object index list for kids to be written to Pages object */
    scratchMem = NULL;		/* use this as non-overflowing scratch pad */
    fpcg = NULL; 			/* Output file */
    fpcontent = NULL;			/* Content stream (need length) */
    numAnnots = 0;			/* count of annotations */
    /* CPDFannotInfo annotInfos[NMAXANNOTS]; */	/* array of annotInfo structure for all annotations */
    /* char mediaBox[64]; */		/* MediaBox for current page*/
    /* cropBox[64]; */			/* CropBox for current page */
    currentByteCount = 0;		/* # of bytes written, or offset of next object */
    /* char creator_name[64]; */	/* Info: set it by cpdf_setCreator() */
    /* char file_title[64]; */		/* Info: title of PDF file */
    /* char file_subject[64]; */	/* Info: subject of PDF file */
    /* char file_keywords[128]; */	/* Info: keywords */
    /* char username[64]; */		/* user name */
    /* char filenamepath[1024]; */
    /* char contentfile[1024]; */
    /* long objByteOffset[NMAXOBJECTS]; */	/* offset into object number N */
    /* int  objIndex[NMAXOBJECTS]; */	/* object index for selected objects */
    startXref = 0;			/* offset of xref */
    /* Don't change these, use cpdf_setMonthNames(char *mnArray[]) for other languages. */
    /* char *monthNameEnglish[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			     	 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"}; */
    /* char spbuf[2048]; */		/* scratch buffer for sprintf */

}


/* ---------------------------------------------------------------------------------------- */
#ifdef MacOS8

/* MacOS function to set info fork items */
void SetFileInfo(char *fileName, OSType fileType, OSType fileCreator)
{
    FInfo outFileInfo;
	c2pstr(fileName);
	GetFInfo((StringPtr)fileName, 0, &outFileInfo);
	outFileInfo.fdType = fileType;
	outFileInfo.fdCreator = fileCreator;
	SetFInfo((StringPtr)fileName, 0, &outFileInfo);
	p2cstr((void *)fileName);
}
#endif

