/* cpdfDomain.c  -- creation and management of plot domains 
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

1998-07-09 [IO]
*/

#include "version.h"

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"


static char *defaultMeshDash = "[2 3] 0";

CPDFplotDomain *cpdf_createPlotDomain(float x, float y, float w, float h,
			float xL, float xH, float yL, float yH,
			int xtype, int ytype, int polar)
{
CPDFplotDomain *newDomain;
int i;
float sL, sH;
	newDomain = (CPDFplotDomain *) malloc((size_t)sizeof(CPDFplotDomain));
	_cpdf_malloc_check((void *)newDomain);
        newDomain->magic = (unsigned int)DOMAIN_MAGIC_NUMBER;
	newDomain->xloc = x;
	newDomain->yloc = y;
	newDomain->width = w;
	newDomain->height = h;
	if(xtype == LINEAR) {
	    cpdf_suggestLinearDomainParams(xL, xH, &sL, &sH,
		&newDomain->xvalFirstMeshLinMajor, &newDomain->xmeshIntervalLinMajor,
		&newDomain->xvalFirstMeshLinMinor, &newDomain->xmeshIntervalLinMinor);
	}
	newDomain->xvalL = xL;
	newDomain->xvalH = xH;

	if(ytype == LINEAR) {
	    cpdf_suggestLinearDomainParams(yL, yH, &sL, &sH,
		&newDomain->yvalFirstMeshLinMajor, &newDomain->ymeshIntervalLinMajor,
		&newDomain->yvalFirstMeshLinMinor, &newDomain->ymeshIntervalLinMinor);
	}
	newDomain->yvalL = yL;
	newDomain->yvalH = yH;

	newDomain->xtype = xtype;
	newDomain->ytype = ytype;
	newDomain->polar = polar;	/* reserved: not implemented */
	newDomain->enableMeshMajor = 1;
	newDomain->enableMeshMinor = 1;
	newDomain->meshDashMajor = defaultMeshDash;
	newDomain->meshLineWidthMajor = 0.15;
	newDomain->meshDashMinor = defaultMeshDash;
	newDomain->meshLineWidthMinor = 0.15;
	for(i=0; i<3; i++) {
	    newDomain->meshLineColorMajor[i] = 0.0;
	    newDomain->meshLineColorMinor[i] = 0.0;
	}

	return newDomain;
}


CPDFplotDomain *cpdf_createTimePlotDomain(float x, float y, float w, float h,
			struct tm *xTL, struct tm *xTH, float yL, float yH,
			int xtype, int ytype, int polar)
{
CPDFplotDomain *newDomain;
int i;
float sL, sH;
	newDomain = (CPDFplotDomain *) malloc((size_t)sizeof(CPDFplotDomain));
	_cpdf_malloc_check((void *)newDomain);
        newDomain->magic = (unsigned int)DOMAIN_MAGIC_NUMBER;
	newDomain->xloc = x;
	newDomain->yloc = y;
	newDomain->width = w;
	newDomain->height = h;
	mktime(xTL); mktime(xTH);	/* This sets all fields of struct tm correctly */
	memcpy(&newDomain->xvTL, xTL, sizeof(struct tm));
	memcpy(&newDomain->xvTH, xTH, sizeof(struct tm));
	newDomain->xvalL = 0.0;			/* keep also float representation for date */
	newDomain->xvalH = tm_to_NumDays(xTL, xTH);	 /* in float number of days */

	if(ytype == LINEAR) {
	    cpdf_suggestLinearDomainParams(yL, yH, &sL, &sH,
		&newDomain->yvalFirstMeshLinMajor, &newDomain->ymeshIntervalLinMajor,
		&newDomain->yvalFirstMeshLinMinor, &newDomain->ymeshIntervalLinMinor);
	}
	newDomain->yvalL = yL;
	newDomain->yvalH = yH;
	newDomain->xtype = xtype;
	newDomain->ytype = ytype;
	newDomain->polar = polar;	/* reserved: not implemented */
	newDomain->enableMeshMajor = 1;
	newDomain->enableMeshMinor = 1;
	newDomain->meshDashMajor = defaultMeshDash;
	newDomain->meshLineWidthMajor = 0.16;
	newDomain->meshDashMinor = defaultMeshDash;
	newDomain->meshLineWidthMinor = 0.1;
	for(i=0; i<3; i++) {
	    newDomain->meshLineColorMajor[i] = 0.0;
	    newDomain->meshLineColorMinor[i] = 0.0;
	}

	return newDomain;
}


void cpdf_freePlotDomain(CPDFplotDomain *aDomain)
{
	if(aDomain) free(aDomain);
}

/* fill plot domain with a gray */
void cpdf_fillDomainWithGray(CPDFplotDomain *aDomain, float gray)
{
	cpdf_gsave();
	cpdf_rawRect(aDomain->xloc, aDomain->yloc, aDomain->width, aDomain->height);
	cpdf_setgrayFill(gray);
	cpdf_fill();
	cpdf_grestore();
}

/* fill plot domain with a gray */
void cpdf_fillDomainWithRGBcolor(CPDFplotDomain *aDomain, float r, float g, float b)
{
	cpdf_gsave();
	cpdf_rawRect(aDomain->xloc, aDomain->yloc, aDomain->width, aDomain->height);
	cpdf_setrgbcolorFill(r, g, b);
	cpdf_fill();
	cpdf_grestore();
}


/* clip drawing at domain border. Bracket this call with gsave and grestore */
void cpdf_clipDomain(CPDFplotDomain *aDomain)
{
	cpdf_newpath();
	cpdf_rawRect(aDomain->xloc, aDomain->yloc, aDomain->width, aDomain->height);
	cpdf_clip();
	cpdf_newpath();
}


CPDFplotDomain *cpdf_setPlotDomain(CPDFplotDomain *aDomain)
{
CPDFplotDomain *oldDomain;
	oldDomain = currentDomain;
	currentDomain = aDomain;
	x2points = (aDomain->width)/((aDomain->xvalH) - (aDomain->xvalL));
	y2points = (aDomain->height)/((aDomain->yvalH) - (aDomain->yvalL));
	if(aDomain->xtype == LOGARITHMIC) {
	    /* logarithmic */
	    xLlog = log10((double)(aDomain->xvalL));
	    xHlog = log10((double)(aDomain->xvalH));
	}
	if(aDomain->ytype == LOGARITHMIC) {
	    /* logarithmic */
	    yLlog = log10((double)(aDomain->yvalL));
	    yHlog = log10((double)(aDomain->yvalH));
	}
	return(oldDomain);
}

void cpdf_setMeshColor(CPDFplotDomain *aDomain, float meshMajorR, float meshMajorG, float meshMajorB,
						float meshMinorR, float meshMinorG, float meshMinorB)
{
	aDomain->meshLineColorMajor[0] = meshMajorR;
	aDomain->meshLineColorMajor[1] = meshMajorG;
	aDomain->meshLineColorMajor[2] = meshMajorB;
	aDomain->meshLineColorMinor[0] = meshMinorR;
	aDomain->meshLineColorMinor[1] = meshMinorG;
	aDomain->meshLineColorMinor[2] = meshMinorB;
}

void cpdf_drawMeshForDomain(CPDFplotDomain *aDomain)
{
	cpdf_gsave();
	cpdf_setstrokeadjust(1);	/* valid only in PDF-1.2 and later */
	_do_meshLines_X(aDomain);
	_do_meshLines_Y(aDomain);
	cpdf_grestore();
}

/* ToDo: consolidate X and Y versions of mesh functions below into one */

void _do_meshLines_X(CPDFplotDomain *aDomain)
{
float vL, vH, v, xp;
int m, ep;
int mL, expL, mH, expH;	/* mantissa MSD and exponents */
float fndays;
int minorBump = 1, majorBump = 1;
int minorBumpVar = MINUTE, majorBumpVar = HOUR;
struct tm vtm;

    vL = aDomain->xvalL;
    vH = aDomain->xvalH;

    if(aDomain->xtype == LOGARITHMIC) {
	/* Log X domain */
	mL = (int)getMantissaExp(vL*1.0001, &expL);
	mH = (int)getMantissaExp(vH*1.0001, &expH);
	/* fprintf(stderr, "%dE%d -- %dE%d\n", mL, expL, mH, expH); */
	/* Do major and minor mesh lines as we go. */
	for(m=mL, ep=expL; (v = (float)m * pow(10.0, (double)ep)) <= vH*1.0001; ) {
	    /* fprintf(stderr, "tick %g\n", v); */
	    if(m == 1) {
		/* Major mesh lines at m == 1 */
	        cpdf_setlinewidth(aDomain->meshLineWidthMajor);
		cpdf_setdash(aDomain->meshDashMajor);
		cpdf_setrgbcolorStroke(aDomain->meshLineColorMajor[0],
			aDomain->meshLineColorMajor[1], aDomain->meshLineColorMajor[2]);
	    }
	    else {
		/* Minor mesh lines */
	        cpdf_setlinewidth(aDomain->meshLineWidthMinor);
		cpdf_setdash(aDomain->meshDashMinor);
		cpdf_setrgbcolorStroke(aDomain->meshLineColorMinor[0],
			aDomain->meshLineColorMinor[1], aDomain->meshLineColorMinor[2]);
	    }
	    xp = x_Domain2Points(v);
	    cpdf_rawMoveto(xp, aDomain->yloc);
	    cpdf_rawLineto(xp, aDomain->yloc + aDomain->height);
	    cpdf_stroke();
	    m++;
	    if(m >= 10) {	/* cycle m through 1 .. 9 */
		m = 1;
		ep++;
	    }
	} /* end for(m=mL,... ) */
    }
    else if(aDomain->xtype == LINEAR) {
	/* Linear X domain */
	/* do minor mesh lines */
	cpdf_setlinewidth(aDomain->meshLineWidthMinor);
	cpdf_setdash(aDomain->meshDashMinor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMinor[0],
			aDomain->meshLineColorMinor[1], aDomain->meshLineColorMinor[2]);
	for(v = aDomain->xvalFirstMeshLinMinor; v <= vH*1.0001; v += aDomain->xmeshIntervalLinMinor) {
	    xp = x_Domain2Points(v);
	    cpdf_rawMoveto(xp, aDomain->yloc);
	    cpdf_rawLineto(xp, aDomain->yloc + aDomain->height);
	    cpdf_stroke();
	}
	/* do minor mesh lines */
	cpdf_setlinewidth(aDomain->meshLineWidthMajor);
	cpdf_setdash(aDomain->meshDashMajor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMajor[0],
			aDomain->meshLineColorMajor[1], aDomain->meshLineColorMajor[2]);
	for(v = aDomain->xvalFirstMeshLinMajor; v <= vH*1.0001; v += aDomain->xmeshIntervalLinMajor) {
	    xp = x_Domain2Points(v);
	    cpdf_rawMoveto(xp, aDomain->yloc);
	    cpdf_rawLineto(xp, aDomain->yloc + aDomain->height);
	    cpdf_stroke();
	}
    }
    else if(aDomain->xtype == TIME) {
	/* Linear TIME X domain */
	/* do minor mesh lines */
	fndays = tm_to_NumDays(&aDomain->xvTL, &aDomain->xvTH);
	_setDefaultTimeBumpVar(fndays, &minorBumpVar, &majorBumpVar, &minorBump, &majorBump);
	/* do minor ticks */
	cpdf_setlinewidth(aDomain->meshLineWidthMinor);
	cpdf_setdash(aDomain->meshDashMinor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMinor[0],
			aDomain->meshLineColorMinor[1], aDomain->meshLineColorMinor[2]);
	memcpy(&vtm, &aDomain->xvTL, sizeof(struct tm));
	for(v = tm_to_NumDays(&aDomain->xvTL, &vtm) ; v <= fndays*1.0001 ;
			v = _bump_tm_Time(&aDomain->xvTL, &vtm, minorBumpVar, minorBump)) {
	    xp = x_Domain2Points(v);
	    cpdf_rawMoveto(xp, aDomain->yloc);
	    cpdf_rawLineto(xp, aDomain->yloc + aDomain->height);
	    cpdf_stroke();
	}
	/* do minor mesh lines */
	cpdf_setlinewidth(aDomain->meshLineWidthMajor);
	cpdf_setdash(aDomain->meshDashMajor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMajor[0],
			aDomain->meshLineColorMajor[1], aDomain->meshLineColorMajor[2]);
	memcpy(&vtm, &aDomain->xvTL, sizeof(struct tm));
	for(v = tm_to_NumDays(&aDomain->xvTL, &vtm) ; v <= fndays*1.0001 ;
			v = _bump_tm_Time(&aDomain->xvTL, &vtm, majorBumpVar, majorBump)) {
	    xp = x_Domain2Points(v);
	    cpdf_rawMoveto(xp, aDomain->yloc);
	    cpdf_rawLineto(xp, aDomain->yloc + aDomain->height);
	    cpdf_stroke();
	}
    } /* end the else part of: if(aDomain-> xtype) {} else {} */
}

void _do_meshLines_Y(CPDFplotDomain *aDomain)
{
float vL, vH, v, yp;
int m, ep;
int mL, expL, mH, expH;	/* mantissa MSD and exponents */

    vL = aDomain->yvalL;
    vH = aDomain->yvalH;

    if(aDomain->ytype == LOGARITHMIC) {
	/* Log Y domain */
	mL = (int)getMantissaExp(vL*1.0001, &expL);
	mH = (int)getMantissaExp(vH*1.0001, &expH);
	/* fprintf(stderr, "%dE%d -- %dE%d\n", mL, expL, mH, expH); */
	/* Do major and minor mesh lines as we go. */
	for(m=mL, ep=expL; (v = (float)m * pow(10.0, (double)ep)) <= vH*1.0001; ) {
	    /* fprintf(stderr, "tick %g\n", v); */
	    if(m == 1) {
		/* Major mesh lines at m == 1 */
	        cpdf_setlinewidth(aDomain->meshLineWidthMajor);
		cpdf_setdash(aDomain->meshDashMajor);
		cpdf_setrgbcolorStroke(aDomain->meshLineColorMajor[0],
			aDomain->meshLineColorMajor[1], aDomain->meshLineColorMajor[2]);
	    }
	    else {
		/* Minor mesh lines */
	        cpdf_setlinewidth(aDomain->meshLineWidthMinor);
		cpdf_setdash(aDomain->meshDashMinor);
		cpdf_setrgbcolorStroke(aDomain->meshLineColorMinor[0],
			aDomain->meshLineColorMinor[1], aDomain->meshLineColorMinor[2]);
	    }
	    yp = y_Domain2Points(v);
	    cpdf_rawMoveto(aDomain->xloc, yp);
	    cpdf_rawLineto(aDomain->xloc + aDomain->width, yp);
	    cpdf_stroke();
	    m++;
	    if(m >= 10) {	/* cycle m through 1 .. 9 */
		m = 1;
		ep++;
	    }
	} /* end for(m=mL,... ) */
    }
    else {
	/* Linear Y domain */
	/* do minor mesh lines */
	cpdf_setlinewidth(aDomain->meshLineWidthMinor);
	cpdf_setdash(aDomain->meshDashMinor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMinor[0],
			aDomain->meshLineColorMinor[1], aDomain->meshLineColorMinor[2]);
	for(v = aDomain->yvalFirstMeshLinMinor; v <= vH*1.0001; v += aDomain->ymeshIntervalLinMinor) {
	    yp = y_Domain2Points(v);
	    cpdf_rawMoveto(aDomain->xloc, yp);
	    cpdf_rawLineto(aDomain->xloc + aDomain->width, yp);
	    cpdf_stroke();
	}
	/* do minor mesh lines */
	cpdf_setlinewidth(aDomain->meshLineWidthMajor);
	cpdf_setdash(aDomain->meshDashMajor);
	cpdf_setrgbcolorStroke(aDomain->meshLineColorMajor[0],
			aDomain->meshLineColorMajor[1], aDomain->meshLineColorMajor[2]);
	for(v = aDomain->yvalFirstMeshLinMajor; v <= vH*1.0001; v += aDomain->ymeshIntervalLinMajor) {
	    yp = y_Domain2Points(v);
	    cpdf_rawMoveto(aDomain->xloc, yp);
	    cpdf_rawLineto(aDomain->xloc + aDomain->width, yp);
	    cpdf_stroke();
	}
    } /* end the else part of: if(aDomain-> xtype) {} else {} */
}


void cpdf_setLinearMeshParams(CPDFplotDomain *aDomain, int xy, float mesh1ValMajor, float intervalMajor,
					      float mesh1ValMinor, float intervalMinor)
{
    if(xy == X_MESH) {
	aDomain->xvalFirstMeshLinMajor = mesh1ValMajor;	/* value of first major mesh line */
	aDomain->xvalFirstMeshLinMinor = mesh1ValMinor;	/* value of first minor mesh line */
	aDomain->xmeshIntervalLinMajor = intervalMajor;	/* mesh interval for linear axis */
	aDomain->xmeshIntervalLinMinor = intervalMinor;
    }
    else {
	aDomain->yvalFirstMeshLinMajor = mesh1ValMajor;	/* value of first major mesh line */
	aDomain->yvalFirstMeshLinMinor = mesh1ValMinor;	/* value of first minor mesh line */
	aDomain->ymeshIntervalLinMajor = intervalMajor;	/* mesh interval for linear axis */
	aDomain->ymeshIntervalLinMinor = intervalMinor;
    }
}


/* Thse must be called after cpdf_setPlotDomain() has been called */
float x_Domain2Points(float x)
{
float xrval = 0.0;
double xvlog = 0.0, fraction=0.0;
    if(!currentDomain) {
	fprintf(stderr, "ClibPDF: Plot domain has not been set. See cpdf_setPlotDomain()\n");
	return(0.0);
    }

    xrval = currentDomain->xloc;	/* min point value for X */
    if(currentDomain->xtype == LOGARITHMIC) {
	/* logarithmic */
	if(x > 0.0) {
	    xvlog = log10((double)x);
	    fraction = (xvlog - xLlog)/(xHlog - xLlog); /* xHlog, xLlog precomputed in cpdf_setPlotDomain() */
	    xrval += (currentDomain->width) * fraction;
	}
    }
    else {
	/* linear */
	xrval += (x - (currentDomain->xvalL)) * x2points;
    }
    return xrval;
}

float y_Domain2Points(float y)
{
float yrval = 0.0;
double yvlog = 0.0, fraction=0.0;
    if(!currentDomain) {
	fprintf(stderr, "ClibPDF: Plot domain has not been set. See cpdf_setPlotDomain()\n");
	return(0.0);
    }

    yrval = currentDomain->yloc;	/* min point value for Y */
    if(currentDomain->ytype == LOGARITHMIC){
	/* logarithmic */
	if(y > 0.0) {
	    yvlog = log10((double)y);
	    fraction = (yvlog - yLlog)/(yHlog - yLlog);	 /* yHlog, yLlog precomputed in cpdf_setPlotDomain() */
	    yrval += (currentDomain->height) * fraction;
	}
    }
    else {
	/* linear */
	yrval += (y - (currentDomain->yvalL)) * y2points;
    }
    return yrval;
}

