/* cover.c -- Program to generate a cover for the ClibPDF API Reference Manual
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

OPENSTEP
cc -Wall -o cover cover.c -I/usr/local/include -lcpdf
NEXTSTEP
cc -Wall -s -object -o cover cover.c -lcpdf

BSDI/FreeBSD/gcc
gcc -Wall -o cover cover.c -L/usr/local/lib -lcpdf -lm

1998-10-18 [IO]
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include <cpdflib.h>

#define WIDTH 		6.0
#define HEIGHT		4.0
#define XBASE		1.5
#define YBASE		4.0
#define NCYCLES		10.0
#define WEBSITE_STR	"http://www.fastio.com/"
#define EMAIL_STR	"clibpdf@fastio.com"
#define PAGE2TEXT	"page2.txt"

float random01(void);
void do_page_2(void);


float random01(void)
{
float v;
    v = (float)rand();
    v /= (float)RAND_MAX;
    return(v);
}

void main(int argc, char *argv[])
{
int i;
float radius;
float xorig, yorig;
float xr, yr;
float sangle, eangle, angle;
float r, g, b;
float strwidth1, strwidth2;
unsigned char sinewave[1024];

    /* == Initialization == */
    srand(23456);
    cpdf_open(0);
    cpdf_enableCompression(YES);		/* use Flate/Zlib compression */
    cpdf_init();

    /* PDF Info object */
    cpdf_setCreator("FastIO Systems: cover.c");
    cpdf_setTitle("ClibPDF Reference Manual");
    cpdf_setSubject("ANSI C Library for Direct PDF Generation");
    cpdf_setKeywords("ClibPDF, ANSI C Library, Acrobat, PDF, Dynamic Web, Graph, Plot");

    cpdf_pageInit(1, PORTRAIT, LETTER, LETTER);		/* page orientation */

    /* == Simple graphics drawing example == */
    cpdf_setgrayStroke(0.0);	/* black as stroke color */
    cpdf_setlinewidth(0.2);

    /* Install clipping path for the rectangle */
    cpdf_gsave();			/* needed to remove clip path later */
    cpdf_newpath();
    cpdf_rect(XBASE, YBASE, WIDTH, HEIGHT);
    cpdf_clip();
    cpdf_newpath();			/* clip does not consume path */

    /* Just a bunch of random pie/pacmans */
    cpdf_comments("\n%% random pie/pacmans.\n");
    for(i=0; i<160; i++) {
	radius = 0.2 + random01();
	eangle = 360.0*random01();	/* end angle */
	sangle = 360.0*random01();	/* start angle */
	xr = WIDTH * random01();	/* position within the box */
	yr = HEIGHT * random01();
	r = 0.7*random01() + 0.3;	/* random RGB but desaturate a bit */
	g = 0.7*random01() + 0.3;
	b = 0.7*random01() + 0.3;
	cpdf_setrgbcolorFill(r, g, b);
	xorig = XBASE + xr;		/* center of arc */
	yorig = YBASE + yr;
	cpdf_moveto(xorig, yorig);
	cpdf_arc(xorig, yorig, radius, sangle, eangle, 0);	/* lineto to starting point of arc */
	cpdf_closepath();
        cpdf_fillAndStroke();		/* fill and stroke */
    }
    cpdf_grestore();			/* remove clipping path */

    cpdf_rect(XBASE, YBASE, WIDTH, HEIGHT);	/* stroke the domain border of pie/pacman field */
    cpdf_setlinewidth(0.8);
    cpdf_stroke();

    /* == Text examples == */
    cpdf_setrgbcolorFill(0.0, 0.0, 1.0);				/* Blue */
    cpdf_beginText(0);
    cpdf_setFont("Times-BoldItalic", "MacRomanEncoding", 38.0);
    cpdf_textAligned(4.5, 9.0, 0.0, TEXTPOS_LM, "ClibPDF Reference Manual");
    cpdf_endText();

    cpdf_setrgbcolorFill(0.4, 0.3, 0.2);
    cpdf_beginText(0);
    cpdf_setFont("Times-Roman", "MacRomanEncoding", 20.0);
    cpdf_textAligned(4.5, 8.4, 0.0, TEXTPOS_LM, "ANSI C Library for Direct PDF Generation");
    cpdf_endText();

    cpdf_setgray(0.0);
    cpdf_beginText(0);
    cpdf_setFont("Helvetica-Bold", "MacRomanEncoding", 18.0);
    cpdf_setHorizontalScaling(140.0);
    cpdf_setCharacterSpacing(1.0);
    cpdf_textAligned(4.5, 2.8, 0.0, TEXTPOS_LM, "FastIO Systems");
    cpdf_endText();

    cpdf_setgray(0.0);
    cpdf_beginText(0);
    cpdf_setFont("Helvetica", "MacRomanEncoding", 14.0);
    cpdf_setHorizontalScaling(120.0);
    cpdf_setCharacterSpacing(0.5);
    cpdf_textAligned(4.5, 2.4, 0.0, TEXTPOS_LM, WEBSITE_STR);
    cpdf_textAligned(4.5, 2.1, 0.0, TEXTPOS_LM, EMAIL_STR);
    strwidth1 = cpdf_stringWidth(WEBSITE_STR);
    strwidth2 = cpdf_stringWidth(EMAIL_STR);
    cpdf_endText();

    /* Add Web and email links over the address text above. */
    cpdf_rawSetActionURL(4.5*inch - strwidth1/2.0, 2.4*inch - 3.0,
			 4.5*inch + strwidth1/2.0, 2.4*inch + 12.0,
			 WEBSITE_STR);
    cpdf_rawSetActionURL(4.5*inch - strwidth2/2.0, 2.1*inch - 3.0,
			 4.5*inch + strwidth2/2.0, 2.1*inch + 12.0,
			 "mailto:clibpdf@fastio.com");

    /* sine wave density plot */
    for(i=0; i<1024; i++) {
	angle = (float)i/512.0 * PI * 2.0 * NCYCLES;
	if(i < 512)
	    sinewave[i] = (unsigned char)(127.0 + 60.0*sin(angle));
	else
	    sinewave[i] = (unsigned char)(127.0 - 60.0*sin(angle));
    }

    cpdf_placeInLineImage((void *)sinewave, 1024,
		XBASE, 1.0, 0.0, WIDTH*72.0, 0.2*72.0,
		512, 2, 8, CS_GRAY, 1);
    cpdf_rect(XBASE, 1.0, WIDTH, 0.2);	/* put a box around it */
    cpdf_setgray(0.0);
    cpdf_setlinewidth(0.2);
    cpdf_stroke();

    /* === Do page 2, mostly typical fine print stuff === */
    do_page_2();

    /* === All drawing done.  Finalize the content === */
    cpdf_finalizeAll();			/* PDF file/memstream is actually written here */
    cpdf_savePDFmemoryStreamToFile("cover.pdf");

    /* == Clean up == */
    cpdf_close();			/* shut down */
    cpdf_launchPreview();		/* launch Acrobat/PDF viewer on the output file */
}


void do_page_2(void)
{
FILE *fp;
char linebuf[1024];
    cpdf_pageInit(2, PORTRAIT, LETTER, LETTER);		/* page orientation */
    cpdf_beginText(0);
    cpdf_setFont("Helvetica", "MacRomanEncoding", 7.0);
    cpdf_setHorizontalScaling(130.0);
    cpdf_setTextLeading(11.0);
    cpdf_setTextPosition(1.0, 10.0);
    cpdf_textShow("ClibPDF Library Reference Manual");

    if((fp = fopen(PAGE2TEXT, "r")) != NULL) {
	while( ( fgets(linebuf, 1022, fp) ) != NULL )
	    cpdf_textCRLFshow(linebuf);
	fclose(fp);
    }
    cpdf_endText();
}

