// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OFBitField.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>

#import "NSData-OFExtensions.h"
#import "NSMutableData-OFExtensions.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniFoundation/DataStructures.subproj/OFBitField.m,v 1.6 1998/12/08 04:07:58 kc Exp $")

@implementation OFBitField

- initWithLength:(unsigned int)newLength;
{
    unsigned int realLength;

    if (![super init])
        return nil;

    realLength = newLength / CHAR_BIT;
    if (newLength % CHAR_BIT)
	realLength++;
    data = [[NSMutableData allocWithZone:[self zone]] initWithLength:realLength];
    return self;
}

- initWithData:(NSData *)someData type:(NSString *)string;
{
    if (![super init])
        return nil;
    data = [someData mutableCopyWithZone:[self zone]];
    return self;
}

- (void)dealloc
{
    [data release];
    [super dealloc];
}

- (NSData *)dataForType:(NSString *)typeString;
{
    return data;
}

- (NSNumber *)valueAtIndex:(unsigned int)anIndex;
{
    return [self boolValueAtIndex:anIndex] ? [NSNumber numberWithBool:YES] : [NSNumber numberWithBool:NO];
}


- (void)setValue:(NSNumber *)aBooleanNumber atIndex:(unsigned int)anIndex;
{
    [self setBoolValue:[aBooleanNumber boolValue] atIndex:anIndex];
}

- (BOOL)boolValueAtIndex:(unsigned int)anIndex;
{
    unsigned int theByteIndex = anIndex / CHAR_BIT;
    unsigned int theBitIndex = anIndex % CHAR_BIT;
    unsigned char value;

    if (theByteIndex >= [data length])
	[NSException raise:NSRangeException format:@""];

    value = ((unsigned char *)[data bytes])[theByteIndex];
    value &= (1 << theBitIndex);

    return (BOOL)value;
}

- (void)setBoolValue:(BOOL)aBool atIndex:(unsigned int)anIndex;
{
    unsigned int theByteIndex = anIndex / CHAR_BIT;
    unsigned int theBitIndex = anIndex % CHAR_BIT;
    unsigned char *byte;

    if (theByteIndex >= [data length])
	[NSException raise:NSRangeException format:@""];

    byte = &((unsigned char *)[data mutableBytes])[theByteIndex];
    if (aBool)
	*byte |= (1 << theBitIndex);
    else
	*byte &= ~(1 << theBitIndex);
}

- (unsigned int)length;
{
    return [data length] * CHAR_BIT;
}

- (void)setLength:(unsigned)newLength;
{
    unsigned int realLength;

    realLength = newLength / CHAR_BIT;
    if (newLength % CHAR_BIT)
	realLength++;

    if ([data length] != realLength)
	[data setLength:realLength];
}

- objectAtIndex:(unsigned int)anIndex;
{
    return [self valueAtIndex:anIndex];
}

- copy; /* TODO: Shouldn't we actually be implementing -copyWithZone: ? */
{
    return [[[self class] alloc] initWithData:data type:nil];
}

- (NSString *)description;
{
    return [data description];
}

- (BOOL)isEqual:(id)anObject;
{
    return [anObject isKindOfClass:[OFBitField class]] && [self isEqualToBitField:anObject];
}

- (BOOL)isEqualToBitField:(OFBitField *)aBitField;
{
    return [data isEqualToData:[aBitField dataForType:nil]];
}

- (void)resetBitsTo:(BOOL)aBool;
{
    memset((char *)[data mutableBytes], aBool ? 0xff : 0, (int)[data length]);
}

- (NSData *)deltaValue:(OFBitField *)aBitField;
{
    NSMutableData *deltaData;
    unsigned char *bytes;
    const unsigned char *otherBytes;
    unsigned long int length;

    OBPRECONDITION(aBitField != nil);
    OBPRECONDITION([aBitField isKindOfClass:[OFBitField class]]);
    OBPRECONDITION([aBitField length] == [self length]);

    deltaData = [[data mutableCopy] autorelease];
    length = [data length];

    bytes = (unsigned char *)[deltaData mutableBytes];
    otherBytes = (const unsigned char *)[[aBitField dataForType:nil] bytes];

    // This is inefficient.  Should xor words until we run out and then xor bytes.
    while (length--)
	*bytes++ ^= *otherBytes++;

    if ([deltaData indexOfFirstNonZeroByte] == NSNotFound)
	deltaData = nil;

    OBPOSTCONDITION(deltaData || [self isEqualToBitField:aBitField]);

    return deltaData;
}


- (void)andWithData:(NSData *)aData;
{
    [data andWithData:aData];
}

- (void)orWithData:(NSData *)aData;
{
    [data orWithData: aData];
}

- (void)xorWithData:(NSData *)aData;
{
    [data xorWithData:aData];
}

@end
