/*
    File:       QResourceObject.h

    Contains:   An object to encapsulate a Mac OS resource derived from a QResourceFile.

    Written by: Quinn "The Eskimo!"

    Created:    Mon 19-May-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import <AppKit/AppKit.h>

typedef long QResourceType;

// Special Note
//
// QResourceObject objects can store two different types of data.  The standard
// type is a bona fide Mac OS resource, with type, ID, (optional) name, and resource
// data.  The second type is the type returned by the QResourceTypesEnumerator.
// This *only* contains the resource type, with no ID, name, or data.
//
// I could have separated this out into two distinct classes.  I didn't because:
//
// a) this project already has enough classes
// b) the functionality overlaps somewhat
// c) IMHO you're unlikely to mix the two up
// d) the only reason for having the second type is that NSEnumerator requires
//    you to return an object from QResourceTypesEnumerator.
// e) I'm lazy.
//
// [Pick one or more of the above excuses.]

// Constants for the various bits in the resource attributes.

enum {
    QResourceSystemHeapMask = 64,
    QResourcePurgeableMask = 32,
    QResourceLockedMask = 16,
    QResourceProtectedMask = 8,
    QResourcePreloadMask = 4,
    QResourceChangedMask = 2
};

@interface QResourceObject : NSObject
{
    QResourceType   resType;			// The resource type.
    long            resID;			// The resource ID.
    NSString        *resName;			// The name of the resource, or nil if none.
    NSData          *resData;			// The data for the resource, or nil if none.
    long            resAttributes;		// The attributes for the resource.
}

- (id)initWithResourceType:(QResourceType)aResType;
    // Initialises a newly allocated QResourceObject to represent just
    // a resource type.  A cover method for initWithResourceType:aResType andID:0 andName:nil andData:nil andAttr:0.

- (id)initWithResourceType:(QResourceType)aResType resourceID:(long)aResID resourceName:(NSString *)aResName
                   resourceData:(NSData *)aResData resourceAttributes:(long)aResAttributes;
    // The designated initialisor for this class.  Initialises a newly allocated
    // QResourceObject to contain a resource of the given type, ID, name and data.
    // QResourceObject does not intepret any of this information, so any of the
    // object parameters can be nil.  Objects parameters will be retained by the
    // newly inited object.

- (QResourceType)resourceType;
    // Returns the resource type for the object.

- (long)resourceID;
    // Returns the resource ID for the object.

- (NSString *)resourceName;
    // Returns the resource name for the object, or nil if there is none.

- (NSData *)resourceData;
    // Returns the resource data for the object, or nil if there is none.

- (long)resourceAttributes;
    // Returns the resource attributes for the object.

- (NSString *)description;
    // Overridden for debugging fun and convenience.

- (void)dealloc;
    // Overridden to allow us to release the resources retained at init time.

@end
