/*
 File:       NSFileManager+unique.m
 Written By: Scott Anguish
 Created:    Dec 9, 1997
 Copyright:  Copyright 1997 by Scott Anguish, all rights reserved.
*/

#import "NSFileManager+unique.h"
#include <sys/types.h>
#include <sys/stat.h>

@implementation NSFileManager(unique)

- (NSString *)existingPortionOfPath:(NSString *)path
{
    NSArray *pathComponents;
    NSString *currentPartialPath=@"";
    BOOL fileExists=NO;
    int lastComponent;
    int i;

    pathComponents = [path pathComponents];
    lastComponent = [pathComponents count]-1;
    for (i=lastComponent; i >= 0; i--)
      {
        BOOL isDirectory;

        currentPartialPath=[NSString pathWithComponents:[pathComponents subarrayWithRange:NSMakeRange(0, i+1)]];
        fileExists=[self fileExistsAtPath:currentPartialPath isDirectory:&isDirectory];
        if (fileExists)
            break;
      }
    if (!fileExists)
        return @"";
    return currentPartialPath;
}


- (NSDictionary *)permissionsForFileAtPath:(NSString *)thePath traverseLink:(BOOL)traverseLink
{
    NSMutableDictionary *theDict;
    NSDictionary *tempDict;
    unsigned long permissions;

    permissions=[[self fileAttributesAtPath:thePath traverseLink:traverseLink] filePosixPermissions];
    theDict=[[NSMutableDictionary alloc] init];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IRUSR) != 0] forKey:@"OwnerReadPermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IWUSR) != 0] forKey:@"OwnerWritePermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IXUSR) != 0] forKey:@"OwnerExecutePermission"];

    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IRGRP) != 0] forKey:@"GroupReadPermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IWGRP) != 0] forKey:@"GroupWritePermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IXGRP) != 0] forKey:@"GroupExecutePermission"];

    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IROTH) != 0] forKey:@"OtherReadPermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IWOTH) != 0] forKey:@"OtherWritePermission"];
    [theDict setObject:[NSNumber numberWithBool:(permissions & S_IXOTH) != 0] forKey:@"OtherExecutePermission"];
    tempDict=[NSDictionary dictionaryWithDictionary:theDict];
    [theDict release];
    return tempDict;
}

- (BOOL)createDirectories:(NSString *)path attributes:(NSDictionary *)attributes
{
    NSArray *pathComponents;
    NSString *partialPath;
    NSString *preexistingPath;
    int i;
    int startingComponent;

    preexistingPath=[self existingPortionOfPath:path];
    startingComponent=([[preexistingPath pathComponents] count]);

    pathComponents = [path pathComponents];
    partialPath=preexistingPath;
    for (i=startingComponent; i < [pathComponents count]; i++)
      {
        BOOL fileExists, isDirectory;

        partialPath = [partialPath stringByAppendingPathComponent:[pathComponents objectAtIndex:i]];

        fileExists = [self fileExistsAtPath:partialPath isDirectory:&isDirectory];

        if (!fileExists)
          {
            if (![self createDirectoryAtPath:partialPath attributes:attributes])
                return NO;
          }
        else
            if (!isDirectory)
                return NO;
      }
    return YES;
}


- (NSString *)createUniqueDirectoryAtPath:(NSString *)path
                         withBaseFilename:(NSString *)baseName
                               attributes:(NSDictionary *)attributes
                           maximumRetries:(int)maximumRetries;
{
    // We'll need to attempt to create a uniquely named directory based on the baseComponent.
    // which should be in the form of "SomeName%@"
    //
    // We should specify how many times we'll allow this to try to avoid getting caught in a loop.

    int each_try=0;
    BOOL keep_trying=YES;

    if (![[NSFileManager defaultManager] createDirectories:path attributes:nil])
        return nil;
    
    if (![[NSFileManager defaultManager] isWritableFileAtPath:path])
        return nil;

    while (keep_trying)
      {
        NSString *filename;
        NSString *attemptPath;

        filename=[NSString stringWithFormat:baseName,[NSNumber numberWithInt:each_try]];
        attemptPath=[NSString pathWithComponents:[NSArray arrayWithObjects:path,filename,nil]];
        if ([[NSFileManager defaultManager] createDirectoryAtPath:attemptPath attributes:attributes])
            return attemptPath;
        each_try++;
        keep_trying = (each_try < maximumRetries);
      }
    return nil;
}

- (NSString *)createUniqueDirectoryAtPath:(NSString *)path
                         withBaseFilename:(NSString *)baseName
                               attributes:(NSDictionary *)attributes;
{
    return [self createUniqueDirectoryAtPath:path
                            withBaseFilename:baseName
                                  attributes:attributes
                              maximumRetries:500];
}

@end
