/* -*-ObjC-*-
*******************************************************************************
*
* File:         appnmail.m
* RCS:          /usr/local/sources/CVS/mailapp-utilities/appnmail.m,v 1.42 1999/04/26 15:46:41 tom Exp
* Description:  Append stdin to Mail.app mailbox
* Author:       Carl Edman
* Created:      Fri Mar 12 18:21:23 1993
* Modified:     Sat Nov 21 16:28:21 1998 Tom Hageman <tom@basil.icce.rug.nl>
* Language:     Objective C
* Package:      mailapp-utilities
* Status:       Exp.
*
* (C) Copyright 1993, but otherwise this file is perfect freeware.
*
*******************************************************************************
*/

#import <libc.h>
#import <errno.h>
//#import <stdlib.h>
//#import <stdio.h>
//#import <string.h>
//#import <stdarg.h>
#import <ctype.h>
//#import <sys/file.h>
//#import <sys/param.h>
//#import <sys/types.h>
//#import <sys/stat.h>
#import "compat.h"
#import "MailProxy.h"
#import "mailutil.h"
#define _MAILTOC_DEFINES 1
#import "mailtoc.h"
#import "optutil.h"
#import "iso_convert.h"
#import "timestamp.h"
#import "re.h"

#if RHAPSODY
#  define NEXT_MODE_DEF_STR
#  define MAC_MODE_DEF_STR	" (default)"
#  define DEF_MAC_MODE		YES
#else
#  define NEXT_MODE_DEF_STR	" (default)"
#  define MAC_MODE_DEF_STR
#  define DEF_MAC_MODE		NO
#endif

#define USAGE "\
Usage: %s [-nvi] [-rfd] [-p pri] [-TD sec] [-NM] [-I enc|JL] [-HV] mbox...\n"

#define HELP "\
 -n            skip incorporation in mailbox if locked\n\
 -v            talkative mode\n\
 -i            force incorporation of pending messages into mailbox\n\
 -r            mark message as read\n\
 -f            mark message as flagged\n\
 -d            mark message as deleted\n\
 -p pri        set message's priority level to `pri'\n\
 -T sec        timeout: give up after `sec' seconds if mailbox is locked\n\
 -D sec        wait `sec' seconds before delivering into open mailbox\n\
\n\
 -N            NeXT mode, for NEXTSTEP / OPENSTEP" NEXT_MODE_DEF_STR "\n\
 -M            Mac mode, for MacOS X Server / Rhapsody" MAC_MODE_DEF_STR "\n\
\n\
 -I enc        use `enc' as default input encoding (default `"DEF_INPUT_ENCODING"')\n\
 -J              special case: Japanese (iso-2022-jp)\n\
 -L              special case: Latin 1 (iso-8859-1)\n\
\n\
 -V,--version  show version\n\
 -H,--help     this help\n\
"

#define APPNMAIL_LOCK	".appnmail.lock"
#define APPNMAIL_MBOX	"appnmail_"MBOX_CONTENTS_FILE
#define APPNMAIL_TOC	"appnmail_"MBOX_TOC_FILE
#define APPNMAIL_DEQUEUE_LOCK ".appnmail_dequeue.lock"

#define PGP_SIGNED "-----BEGIN PGP SIGNED MESSAGE-----"
#define PGP_ENCRYPTED "-----BEGIN PGP MESSAGE-----"

char *header=0,*content=0;
int headerlen=0,contentlen=0;
int headermaxlen=0,contentmaxlen=0;
char line[LINELEN];

int verboseflg=0;

struct regex *mailutilre=0;
const char *decodeCmd=0;
const char*uncompressCmd=0;
const char *tarCmd=0;

const char *inputEncoding = DEF_INPUT_ENCODING;
const char *outputEncoding = DEF_OUTPUT_ENCODING;

BOOL macMode = DEF_MAC_MODE;

int incorporateDelay = -1;


#ifndef DUMP_CORE
#  define DUMP_CORE  0
#endif

#if DUMP_CORE
#  import <sys/vlimit.h>
#  define dumpcore()  (vlimit(LIM_CORE, INFINITY), abort())
#else
#  define dumpcore()
#endif

static BOOL checked_data_integrity = NO;

#define CHECK_DATA_INTEGRITY(where) do { \
	 if (memchr(header, '\0', headerlen) || \
	     memchr(content, '\0', contentlen))	\
	 { \
	    if (!checked_data_integrity) \
	    { \
	       logprintf("[%s] possibly corrupt data (file \"%s\", line %d)", \
		   where, __FILE__, __LINE__); \
	       checked_data_integrity = YES; \
	    } \
	    dumpcore(); \
	 } \
	} while (0)


// Read header line, taking RFC822 line continuation into account.
char *fgets_header(char *buf, int size, FILE *f)
{
   char *line = buf;

   if (fgets(line, size, f) == NULL) return NULL;

   if (*line == '\n') return buf;

   do
   {   
      int len;
      int c = getc(f);

      if (c != EOF) ungetc(c, f);

      if (c != ' ' && c != '\t') break;

      // Line continuation.
      len = strlen(line);
      line += len;
      size -= len;
      if (size <= 1) break; // overflow.
   }
   while (fgets(line, size, f));

   return buf;
}

void unfold(char *line)
{
   char *s = strchr(line, '\n');
   char *d;

   if (s == NULL || s[1] == '\0') return;
   d = s;
   do
   {
      *d++ = ' ';
      while (*++s == ' ' || *s == '\t') ;

      while (*s != '\n') *d++ = *s++;
   }
   while (*s == '\n' && s[1] != '\0');
   *d++ = '\n';
   *d = '\0';
}

struct message_index *readmail(void)
{
   struct message_index *mi;
   int i,pri=0;
   char from[LINELEN]="",subject[LINELEN]="",reference[LINELEN]="";
   struct regex *fre=re_compile("^From:  *\\(.*\\)\n",0);
   struct regex *sre=re_compile("^Subject:  *\\(.*\\)\n",0);
   // Y2000 fix applied here.
   struct regex *dre=re_compile("^Date:.* \\([0123]*[0-9]\\)  *\\([A-Z][a-z][a-z]\\)[a-z]*  *\\([12]*[0-9]*[0-9][0-9]\\)  *\\([0-9]*[0-9]\\):\\([0-9][0-9]\\):*\\([0-9]*\\)  *\\([^ \n]*\\)",0);
   struct regex *rre=re_compile("^Next-Attachment:  *\\.tar\\.\\([0-9]*\\)\\.\\(.*\\)\\.attach, \\(E*,* *\\)\\([0-9]*\\), \\([0-9]*/[0-9]*\\), \\([0-9]*\\), \\([0-9]*\\)",0);
   struct regex *mre=re_compile("^Content-Type:  *\\([-_a-zA-Z0-9]*/[-_a-zA-Z0-9]*\\)",0);
   struct regex *pre=0;
   const char *pridef=0;
   time_t date;


   if ((pridef=NXGetDefaultValue(MAIL_APP,"PriorityHeader")) != NULL)
   {
      static char buf[MAXPATHLEN];
      sprintf(buf,"^%s: [ \t]*\\([^ \t]*\\)[ \t]*\n",pridef);
      pre=re_compile(buf,0);
      pridef=NXGetDefaultValue(MAIL_APP,"PriorityValues");
   }
   
   mi = malloc(sizeof(*mi));
   
   mi->record_length = sizeof(*mi);
   mi->mes_offset = -1;
   mi->status = MT_STATUS_NEW;
   mi->msgtype = MT_UNSET;
   mi->encrypted = MT_UNSET;
   mi->sync = MT_UNSET;
   mi->mes_date = message_current_date();
   date = time(NULL);	// XXX Race condition with previous.

   headerlen=0;
   
   while (fgets_header(line, LINELEN, stdin))
   {
      if (*line=='\n') break;

      if (re_match(line,rre)==1)
      {
	 char *c;
	 mi->msgtype = MT_TYPE_NEXT;
	 strpcpy(reference,rre->braslist[1],rre->braelist[1]);
	 c=reference+strlen(reference);
	 *c++='_';
	 while(c-reference<22) *c++='_';
	 *c='\0';
	 sprintf(c,"%lu",(unsigned long)time(0));
	 strcat(reference,".attach");
	 c=reference+strlen(reference);
	 strcat(reference,", ");
	 if (*(rre->braslist[2])=='E')
	 {
	    strcat(reference,"E, ");
	    mi->encrypted = MT_ENCRYPTED;
	 }
	 strpcat(reference,rre->braslist[4],rre->braelist[4]);
	 strcat(reference,"\n");
	 appstring(&header,&headerlen,&headermaxlen,"Next-Reference: ",
		   strlen("Next-Reference: "));
	 appstring(&header,&headerlen,&headermaxlen,reference,strlen(reference));
	 *c='\0';
      }
      else
      {
	 appstring(&header,&headerlen,&headermaxlen,line,strlen(line));

	 unfold(line);

	 if (re_match(line,fre)==1)
	 {
	    strpcpy(from,fre->braslist[0],fre->braelist[0]);
#if 0	    // moved to message_index_for_mode()
	    iso_convert(from, inputEncoding, outputEncoding);
#endif
	 }
	 else if (re_match(line,sre)==1)
	 {
	    strpcpy(subject,sre->braslist[0],sre->braelist[0]);
#if 0	    // moved to message_index_for_mode()
	    iso_convert(subject, inputEncoding, outputEncoding);
#endif
	 }
	 else if (re_match(line,dre)==1)
	 {
	    // Y2000 fix applied here.
	    unsigned year = atoi(dre->braslist[2]);

	    if (year < 70) year += 2000;
	    else if (year < 200) year += 1900;
	    // else assume year includes century...
	    mi->mes_date=message_date(year,dre->braslist[1],atoi(dre->braslist[0]));
	    /* Determine timestamp.  We cheat here to determine the month */
	    {
	       int y, m, d, hh, mm, ss;
	       message_get_date(mi, &y, &m, &d);
	       hh = atoi(dre->braslist[3]);
	       mm = atoi(dre->braslist[4]);
	       ss = dre->braslist[5] < dre->braelist[5] ? atoi(dre->braslist[5]) : 0;
	       *(char *)dre->braelist[6] = '\0';  // kludge: zero-terminate timezone.
	       date = timestamp(y, m, d, hh, mm, ss, dre->braslist[6]);
	    }
	 }
	 else if (pre && pridef && (re_match(line,pre)==1))
	 {
	    const char *beg,*end;

	    for(beg=pridef,pri=1;*beg;(beg=*end ? end+1 : end),(pri++))
	    {
	       if (!(end=index(beg,' '))) end=beg+strlen(beg);
	       if (((pre->braelist[0]-pre->braslist[0])==(end-beg))
		   &&(strncmp(pre->braslist[0],beg,end-beg)==0))
		  break;
	    }
	    if (!*beg) pri=0;
	 }
	 else if (re_match(line, mre) == 1)
	 {
	    if ((mi->msgtype == MT_UNSET) &&
		!strpcaseequ("text/plain",mre->braslist[0],mre->braelist[0]))
	       mi->msgtype = MT_TYPE_MIME;
	    // Experimental RFC2015 encrypt/signature detection.
	    if (strpcaseequ("multipart/encrypted",mre->braslist[0],mre->braelist[0]))
	       mi->encrypted = MT_ENCRYPTED;
	    else if (strpcaseequ("multipart/signed",mre->braslist[0],mre->braelist[0]))
	       mi->encrypted = 's'; // Naughty, naughty...
	    else if (strpcaseequ("text/html",mre->braslist[0],mre->braelist[0]))
	       mi->msgtype = MT_TYPE_HTML;
	    /* XXX TODO: should check multipart/alternative content for inline text/html,
	        but it's too painful right now... */
	 }
      }
   }
   if (headerlen > 0) appstring(&header,&headerlen,&headermaxlen,"\n",1);

   contentlen=0;
   
   while((i=fread(growstring(&content,&contentlen,&contentmaxlen,LINELEN*16),
		  1,16*LINELEN,stdin))>0)
      contentlen+=i;
   if (contentlen > 0 && content[contentlen-1] != '\n')
      appstring(&content,&contentlen,&contentmaxlen,"\n",1);

   mi->mes_length=headerlen;
   mi->mes_length += (mi->msgtype != MT_TYPE_NEXT) ? contentlen : 1;
   mi->record_length+=strlen(from)+1+strlen(subject)+1+strlen(reference)+1;
   mi->record_length+=16;  // sizeof(int)+sizeof(time_t)+sizeof(int)+sizeof(time_t);
#if 0 // Diff now handled in message_index_for_mode(), was:
   mi->record_length+=12;  // sizeof(int)+sizeof(time_t)+sizeof(int);
   if (macMode) mi->record_length+=4;   // sizeof(time_t);
#endif

   mi=realloc(mi,mi->record_length);
   // NB. order matters in the following!!!
   strcpy(message_from(mi),from);
   strcpy(message_subject(mi),subject); 
   strcpy(message_reference(mi),reference);

   message_set_attachsize(mi,0);
   message_set_attachtime(mi,0);
   message_set_priority(mi,pri);
   message_set_time(mi, date);

   // Experimental PGP encrypt/signature detection.
   if (mi->encrypted == MT_UNSET)
   {
      if (contentlen >= sizeof(PGP_ENCRYPTED)-1 &&
	  strncmp(content, PGP_ENCRYPTED, sizeof(PGP_ENCRYPTED)-1)==0)
	 mi->encrypted = MT_ENCRYPTED;
      else if (contentlen >= sizeof(PGP_SIGNED)-1 &&
	       strncmp(content, PGP_SIGNED, sizeof(PGP_SIGNED)-1)==0)
	 mi->encrypted = 's'; // XXX Naughty, naughty...
   }

   re_free(fre); re_free(sre); re_free(rre); re_free(dre);
   if (pre) re_free(pre);
   CHECK_DATA_INTEGRITY("readmail");
   return mi;
}

/* Make a copy of message index, converted to the proper mode and encoding.
   It is assumed that source message index contains original (non-iso-converted)
   From: and Subject: lines. */
struct message_index *message_index_for_mode(const struct message_index *mi, int mac_mode, const char *encoding)
{
   char *from, *subject, *reference;
   struct message_index *result = malloc(mi->record_length);

   memcpy(result, mi, sizeof(*mi));

   from = strcpy(message_from(result), message_from(mi));
   iso_convert(from, inputEncoding, encoding);

   subject = strcpy(message_subject(result), message_subject(mi));
   iso_convert(subject, inputEncoding, encoding);

   reference = strcpy(message_reference(result), message_reference(mi));

   // adjust record size
   result->record_length = sizeof(*mi) + strlen(from)+1 + strlen(subject)+1 + strlen(reference)+1 + 12;
   // Always add detailed timestamp on RDR2/MOSXS.
   if (DEF_MAC_MODE || mac_mode) result->record_length += 4;
   result = realloc(result, result->record_length);

   message_set_attachsize(result, message_attachsize(mi));
   message_set_attachtime(result, message_attachtime(mi));
   message_set_priority(result, message_priority(mi));
   message_set_time(result, message_time(mi));

   if (mac_mode)
   {
      switch (result->msgtype)
      {
       case MT_TYPE_NEXT_1:
	 result->msgtype = MT_TYPE_NEXT_2;
	 break;
      }
   }
   else
   {
      switch (result->msgtype)
      {
       case MT_TYPE_HTML:
	 result->msgtype = MT_TYPE_MIME;
	 break;
      }
   }
   return result;
}

int determine_mode_from_table_of_contents_header(struct table_of_contents_header *toch, int *mac_mode_p)
{
   int result = 0;

   if (toch)
   {
      result = 1;

      if (toch->num_msgs == 0)
      {
	 // Assume we just created the mailbox -- adjust magic number.
	 result = 2;
	 toch->magic = (*mac_mode_p) ? MBOX_TOC_MAGIC_MACOS_X : MBOX_TOC_MAGIC;
      }

      switch (toch->magic)
      {
       default:

	 result = -1;
	 // fall through...

       case MBOX_TOC_MAGIC_MACOS_X:

	 (*mac_mode_p) = YES;
	 break;

       case MBOX_TOC_MAGIC:

	 (*mac_mode_p) = NO;
	 break;
      }
   }
   return result;
}

void determine_mode_from_table_of_contents_file(const char *mboxname, const char *table_of_contents, int *mac_mode_p)
{
   FILE *tocf;

   if ((tocf = fopen(table_of_contents, "r")) != NULL)
   {
      struct table_of_contents_header *toch;
      int st;

      toch = get_table_of_contents_header(tocf, 0);
      st = determine_mode_from_table_of_contents_header(toch, mac_mode_p);
      if (st == -1)
      {
	 logprintf("%s: [%s] unknown magic number (%u); using \"%s\" encoding", mboxname, table_of_contents, toch->magic, MAC_OUTPUT_ENCODING);
      }
      free(toch);
      fclose(tocf);
   }
}


const char *get_command(const char *defaultKey, ...)
{
   /* Find command named `defaultKey' in Mail's defaults database, and check
      if it is an executable.  If not, try each of the defaults given on
      the (NULL-terminated) argument list in turn. */
   const char *command = NXGetDefaultValue(MAIL_APP, defaultKey);
   va_list ap;

   va_start(ap, defaultKey);
   do
   {
      struct stat st;
      if (command == NULL) continue;
      if (command[0] != '/') break; /* XXX should check PATH? */
      if (access(command, X_OK) < 0) continue;
      if (stat(command, &st) < 0) continue;
      if ((st.st_mode & S_IFMT) == S_IFREG) break;
   }
   while ((command = va_arg(ap, const char *)) != NULL);
   va_end(ap);
   if (command == NULL)
   {
      logprintf("cannot find executable for `%s'.", defaultKey);
   }
   return command;
}

#define INCOMING_MBOX	"Incoming_Mail"
#define INCOMING_TOC	"Incoming_Table_of_Contents"

int incorporate_mail(MailProxy *mailProxy, const char *mboxname, const char *mbox, const char *table_of_contents)
{
   int interval=1,lastinterval=0;

   /* Signal Mail.app to incorporate mail.  Assumes Mail has the mailbox open,
      and appropriate locks are in place. */
   if (mailProxy == nil) return -1;
   if (link(mbox, INCOMING_MBOX)<0)
   {
      logprintf("%s: cannot link %s to %s (%s)",mboxname,mbox,INCOMING_MBOX,strerror(errno));
      return -1;
   }
   if (link(table_of_contents, INCOMING_TOC)<0)
   {
      logprintf("%s: cannot link %s to %s (%s)",mboxname,mbox,INCOMING_MBOX,strerror(errno));
      unlink(INCOMING_MBOX);
      return -1;
   }
   if (![mailProxy incorporateNewMail])
   {
      logprintf("%s: -incorporateNewMail failed",mboxname);
      unlink(INCOMING_MBOX);
      unlink(INCOMING_TOC);
      return 1;
   }

   /* Unfortunately this seems to be asynchronous, ie. Mail.app does not
      immediately process (and remove) the Incoming mbox when it receives
      a remote kick in the butt.  So we have to wait a little while... */
   while (access(INCOMING_MBOX, F_OK)==0 || access(INCOMING_TOC, F_OK)==0)
   {
      /* Nice Fibonacci series. */
      int newinterval=interval+lastinterval;

      lastinterval=interval;
      interval=newinterval;
      if (interval > 34)
      {
	 /* so we waited for 1+1+2+3+5+8+13+21+34 = 88 seconds.  Is this enough? */
	 logprintf("%s: Mail.app failed to incorporate Incoming mail???",mboxname);
	 unlink(INCOMING_MBOX);
	 unlink(INCOMING_TOC);
	 return 1;
      }
      sleep(interval);
   }
   unlink(mbox);
   unlink(table_of_contents);
   return 0;
}

/* Transfer messages from one {mbox, table_of_contents} combo to another. */
int transfer_mbox_contents(const char *mboxname, const char *srcmbox, const char *srctoc, const char *dstmbox, const char *dsttoc)
{
   int result=-1;
   int srcboxfd=-1, dstboxfd=-1;
   FILE *srctocf=NULL, *dsttocf=NULL;
   struct table_of_contents_header *srctoch=NULL, *dsttoch=NULL;
   struct message_index *mi;
   time_t mboxtime;
   int n;

   mboxtime = mtime(srcmbox);
   if ((srcboxfd = open(srcmbox, O_RDONLY)) < 0)
   {
      logprintf("%s: opening %s (%s)", mboxname, srcmbox, strerror(errno));
      goto _close;
   }
   if ((srctocf = fopen(srctoc, "r+b")) == NULL)
   {
      logprintf("%s: opening %s (%s)", mboxname, srctoc, strerror(errno));
      goto _close;
   }
   if ((srctoch = get_table_of_contents_header(srctocf, 0)) == NULL)
   {
      logprintf("%s: [%s] invalid header", mboxname, srctoc);
      goto _close;
   }
   if (srctoch->mbox_time != mboxtime)
   {
      logprintf("%s: [%s] out of sync", mboxname, srctoc);
      goto _close;
   }

   mboxtime = mtime(dstmbox);
   if ((dstboxfd = open(dstmbox,O_WRONLY)) < 0)
   {
      logprintf("%s: opening %s (%s)", mboxname, dstmbox, strerror(errno));
      goto _close;
   }
   if ((dsttocf = fopen(dsttoc,"r+b")) == NULL)
   {
      logprintf("%s: opening %s (%s)", mboxname, dsttoc, strerror(errno));
      goto _close;
   }
   if ((dsttoch = get_table_of_contents_header(dsttocf,1)) == NULL)
   {
      logprintf("%s: [%s] invalid header", mboxname, dsttoc);
      goto _close;
   }
   if (dsttoch->mbox_time != mboxtime)
   {
      logprintf("%s: [%s] out of sync", mboxname, dsttoc);
      goto _close;
   }

   for (n = 0;  n < srctoch->num_msgs;  free(mi), n++)
   {
      long src_offset, src_length;
      void *buf=NULL;

      errno = 0;
      if ((mi = get_message_index(srctocf)) == NULL)
      {
         logprintf("%s: [%s] inconsistency (%s)", mboxname, srctoc, (errno ? strerror(errno) : "invalid"));
         goto _close;
      }
      if (mi->status == 'D') continue; // Deleted in a previous pass.

      src_offset = mi->mes_offset;
      src_length = mi->mes_length;
      if (lseek(srcboxfd, src_offset, SEEK_SET) != src_offset ||
	  (buf = malloc(src_length)) == NULL ||
	  (src_length = read(srcboxfd, buf, src_length)) != mi->mes_length)
      {
	 logprintf("%s: [%s] inconsistency (%s)", mboxname, srcmbox, (errno ? strerror(errno) : "invalid"));
	 free(buf);
	 free(mi);
         goto _close;
      }
      mi->mes_offset = lseek(dstboxfd, 0, SEEK_END);
      if ((mi->mes_length = write(dstboxfd, buf, src_length)) != src_length)
      {
         logprintf("%s: [%s] inconsistency (%s)", mboxname, dstmbox, (errno ? strerror(errno) : "invalid"));
      }
      free(buf);

      /* Append message index to destination table of contents. */
      fseek(dsttocf, 0, SEEK_END);
      put_message_index(dsttocf, mi);

      /* Update header. */
      dsttoch->num_msgs++;
      dsttoch->mbox_time = mtime(dstmbox);
      fseek(dsttocf, 0, SEEK_SET);
      put_table_of_contents_header(dsttocf, dsttoch);

      /* Mark message as deleted (and restore original offset/length) in source mbox. */
      mi->status = 'D';
      mi->mes_offset = src_offset;
      mi->mes_length = src_length;
      fseek(srctocf, -mi->record_length, SEEK_CUR);
      put_message_index(srctocf, mi);
      fseek(srctocf, 0, SEEK_CUR);
   }

   /* Update header once again, just to be sure... */
   close(dstboxfd), dstboxfd = -1;
   dsttoch->mbox_time = mtime(dstmbox);
   fseek(dsttocf, 0, SEEK_SET);
   put_table_of_contents_header(dsttocf, dsttoch);

   result = 0;
 _close:
   free(dsttoch);
   if (dstboxfd >= 0) close(dstboxfd);
   if (dsttocf != NULL) fclose(dsttocf);
   free(srctoch);
   if (srcboxfd >= 0) close(srcboxfd);
   if (srctocf != NULL) fclose(srctocf);

   return result;
}

/* Dequeue mail by copying it from appnmail's private mbox to the real mbox.
   assumes proper locking has been done. */
void dequeue_mail(const char *mboxname)
{
   if (access(APPNMAIL_MBOX, F_OK) != 0) return;
   if (transfer_mbox_contents(mboxname, APPNMAIL_MBOX, APPNMAIL_TOC, MBOX_CONTENTS_FILE, MBOX_TOC_FILE) != 0) return;

   /* If successful, remove our private inbox. */
   unlink(APPNMAIL_MBOX);
   unlink(APPNMAIL_TOC);
}

/* Extract NeXTmail attachment from message into `path'. */
int extract_attachment(const char *mboxname, const char *path)
{
   FILE *f;
   int st;

   if (verboseflg>1) logprintf("Extracting NeXTmail attachment...");

   /* Locate executables, if not already done so. */
   if (!decodeCmd && !(decodeCmd=get_command("appnmailDecodeCommand","/NextApps/Mail.app/decode","/usr/local/lib/uudecode-filter","uudecode-filter",NULL)))
   {
      /* ...not really a regular Mail default. */
      return EXIT_FAILURE;
   }
   if (!uncompressCmd && !(uncompressCmd=get_command("UncompressCommand","/usr/bin/gunzip","/usr/ucb/uncompress",NULL)))
   {
      return EXIT_FAILURE;
   }
   if (!tarCmd && !(tarCmd=get_command("TarCommand","/NextApps/Mail.app/safetar","/usr/bin/gnutar",NULL)))
   {
      return EXIT_FAILURE;
   }
   if ((mkdir(path,0755))==-1)
   {
      logprintf("%s: %s: %s",mboxname,path,strerror(errno));
      return EXIT_FAILURE;
   }
   sprintf(line,"cd %s && %s | %s | %s xf -",path,decodeCmd,uncompressCmd,tarCmd);
   if (!(f=popen(line,"w")))
   {
      logprintf("%s: %s: %s",mboxname,path,strerror(errno));
      return EXIT_FAILURE;
   }
   fwrite(content,contentlen,1,f);
   if ((st=pclose(f))!=0)
   {
      logprintf("%s: %s failed (status %d)",mboxname,line,st);
      return EXIT_FAILURE;
   }
   return EXIT_SUCCESS;
}

/* Save message in {mbox, table_of_contents}.
   Assumes proper locking has been done. */
int save_message(const char *mboxname,
		 const char *mbox, const char *table_of_contents,
		 struct message_index *a_mi, int force, int *mac_mode_p)
{
   int mfd = -1, tocfd = -1;
   FILE *tocf = NULL;
   struct table_of_contents_header *toch = NULL;
   struct message_index *mi = NULL;
   const char *attachment;
   int result = EXIT_SUCCESS;
   int bodylen = contentlen;
   char *body = content;
   time_t mboxtime = mtime(mbox);
   const char *encoding = outputEncoding;

   if ((mfd = open(mbox, O_WRONLY|O_APPEND|O_CREAT, 0644)) == -1)
   {
      logprintf("%s: [%s] open (%s)", mboxname, mbox, strerror(errno));
      result = EXIT_FAILURE;
      goto _close;
   }

   /* Even if we cannot update the table of contents,
      we'll try our damnedest to deliver the message anyway (if force'd). */

   if ((tocfd = open(table_of_contents, O_CREAT|O_RDWR,0644)) == -1 ||
       (tocf = fdopen(tocfd, "r+b")) == NULL)
   {
      logprintf("%s: [%s] open (%s)", mboxname, table_of_contents, strerror(errno));
      result = EXIT_FAILURE;
      if (!force) goto _close;
   }
   else
   {
      errno = 0;
      toch = get_table_of_contents_header(tocf, 1);
      if (!toch)
      {
	 logprintf("%s: [%s] get header (%s)", mboxname, table_of_contents, errno ? strerror(errno) : "invalid");
	 result = EXIT_FAILURE;
	 if (!force) goto _close;
      }
      else if (toch->mbox_time != mboxtime)
      {
	 logprintf("%s: [%s] out of sync", mboxname, table_of_contents);
	 result = EXIT_FAILURE;
	 if (!force) goto _close;
      }
      if (determine_mode_from_table_of_contents_header(toch, mac_mode_p) == -1)
      {
	 logprintf("%s: [%s] unknown magic number (%u); using \"%s\" encoding", mboxname, table_of_contents, toch->magic, MAC_OUTPUT_ENCODING);
      }
      encoding = (*mac_mode_p) ? MAC_OUTPUT_ENCODING : NEXT_OUTPUT_ENCODING;
   }
   if (verboseflg > 2)
   {
      logprintf("%s: [%s] magic number (%u); using \"%s\" encoding...", mboxname, table_of_contents, (toch ? toch->magic : 0), encoding);
   }
   mi = message_index_for_mode(a_mi, (*mac_mode_p), encoding);

   errno = 0;
   if ((mi->mes_offset = lseek(mfd,0,SEEK_END)) < 0)
   {
      logprintf("%s: [%s] lseek (%s)", mboxname, mbox, strerror(errno));
      result = EXIT_FAILURE;
      goto _close;      
   }

   if ((attachment = message_reference(mi)) && *attachment)
   {
      char path[MAXPATHLEN+1];

      if (extract_attachment(mboxname, attachment) != EXIT_SUCCESS)
      {
	 result = EXIT_FAILURE;
	 goto _close;
      }
      message_set_attachsize(mi, dirsize(strcpy(path, attachment)));
      message_set_attachtime(mi, mtime(attachment));

      /* Add fake message body *after* all failure points, to avoid TOC corruption. */
      body = "\n";
      bodylen = 1;
   }

   CHECK_DATA_INTEGRITY("save_message: before write");

   errno = 0;
   if (write(mfd, header, headerlen) != headerlen ||
       write(mfd, body, bodylen) != bodylen)
   {
      logprintf("%s: [%s] truncated message (%s)", mboxname, mbox, strerror(errno));
      result = EXIT_FAILURE;

      // Fix up length of truncated message.
      mi->mes_length = lseek(mfd, 0, SEEK_END) - mi->mes_offset;
      if (mi->mes_length <= 0 || !force)
      {
	 extern int utime();
	 time_t times[2];

	 lseek(mfd, mi->mes_offset, SEEK_SET);
	 ftruncate(mfd, mi->mes_offset);

	 // Restore mbox timestamp so as not to cause out-of-sync error.
	 close(mfd), mfd = -1;
	 times[0] = times[1] = mboxtime;
	 utime((char *)mbox, times);   // [Ugh! bad prototype...]

	 goto _close;
      }
   }

   if (toch)
   {
      long pos;

      fseek(tocf, 0, SEEK_END);
      pos = ftell(tocf);
      errno = 0;
      if (put_message_index(tocf, mi) != 0)
      {
	 logprintf("%s: [%s] truncated message index (%s)", mboxname, table_of_contents, strerror(errno));
	 fflush(tocf);
	 fseek(tocf, 0, SEEK_SET); // trying to defeat FILE <-> fd interference...
	 ftruncate(fileno(tocf), pos);
	 result = EXIT_FAILURE;
	 goto _close;
      }
      toch->num_msgs++;

      // Close mbox file handle, so its modification time can be determined more reliably.
      close(mfd), mfd=-1;
      if (toch->mbox_time == mboxtime) toch->mbox_time = mtime(mbox);
      fseek(tocf, 0, SEEK_SET);
      errno = 0;
      if (put_table_of_contents_header(tocf, toch) != 0)
      {
	 logprintf("%s: [%s] saving header (%s)", mboxname, table_of_contents, strerror(errno));
      }
   }
 _close:
   if (tocf != NULL) fclose(tocf);
   else if (tocfd >= 0) close(tocfd);
   if (mfd >= 0) close(mfd);
   if (toch) free(toch);
   if (mi) free(mi);

   return result;
}

/* Save message in the mailbox, or, if it is locked, in a (temporary) private
   mailbox inside the .mbox folder. */
int lock_mailbox_and_save_message(const char *mboxname, struct message_index *mi)
{
   char host[MAXHOSTNAMELEN];
   char *mbox=MBOX_CONTENTS_FILE, *table_of_contents=MBOX_TOC_FILE;
   int ownslock=1,ownsappnmaillock=0;
   int result=EXIT_SUCCESS;
   int st;
   int mac_mode = macMode;

   if (verboseflg) logprintf("Entering %s to deliver...",mboxname);

   if (cd_mbox(mboxname,1) != 0) return EXIT_FAILURE;

   if ((st = try_lock_mbox(host, line, NULL)) != 0)
   {
      if (st < 0) return EXIT_FAILURE;

      CHECK_DATA_INTEGRITY("save_message: .lock");
      /* If the mbox is locked (by Mail.app), we'll write the message to
	 a temporary mbox, and try to incorporate it into the mbox later.
	 (so the message is at least stored to disk in case anything
	  goes wrong).
	 XXX The way to determine whether the lock is owned by Mail.app
	 or one of our own mailapp-utilities is completely bogus
	 and depends on what is written in the user field of the lock
	 by try_lock_mbox(). */
      if (!mailutilre) mailutilre=re_compile(LOCK_USER_RE,0);
      if (*line && !re_match(line,mailutilre))
      {
	 if (verboseflg>1) logprintf("Delivering into temporary mbox...");
	 ownslock=0;
	 mbox=APPNMAIL_MBOX;
	 table_of_contents=APPNMAIL_TOC;
	 /* Operations on our temp. mbox are protected with our own lock. */
	 /* XXX What about stale locks? */
	 if (lock_mbox_file(APPNMAIL_LOCK,0)!=0)
	 {
	    result=EXIT_FAILURE;
	    goto unlock;
	 }
	 ownsappnmaillock=1;
	 CHECK_DATA_INTEGRITY("save_message: .appnmail.lock");

	 // Try to determine target mailbox format.
	 determine_mode_from_table_of_contents_file(mboxname, MBOX_TOC_FILE, &mac_mode);
      }
      else if (lock_mbox(0)!=0)
      {
	 ownslock=0;
	 result=EXIT_FAILURE;
	 goto unlock;
      }
      CHECK_DATA_INTEGRITY("save_message: locked");
   }
   if (ownslock)
   {
      /* try to dequeue mail first, but don't insist on it if already
	 locked by another appnmail. */
      if (lock_mbox_file(APPNMAIL_LOCK,1)==0)
      {
	 dequeue_mail(mboxname);
	 unlock_mbox_file(APPNMAIL_LOCK);
      }
   }

   result = save_message(mboxname, mbox, table_of_contents, mi, !ownslock, &mac_mode);

   if (result != EXIT_SUCCESS && ownslock)
   {
      // Retry saving in APPNMAIL_MBOX if failed to save in regular mbox.
      logprintf("Retrying to deliver into temporary mbox...");
      result = save_message(mboxname, APPNMAIL_MBOX, APPNMAIL_TOC, mi, YES, &mac_mode);
   }

 unlock:
   if (ownslock)
   {
      unlock_mbox();
   }
   else if (ownsappnmaillock)
   {
      unlock_mbox_file(APPNMAIL_LOCK);
   }
   if (result==EXIT_SUCCESS && verboseflg) logprintf("...deliver %s done",mboxname);
   uncd_mbox();
   return result;
}

/* Incorporate mail from appnmail's private mbox into the real mbox.
   Handles synchronization with other instances of appnmail, proper locking,
   etc. */
void incorporate(const char *mboxname, int force, int timeout)
{
   int st;
   char host[MAXHOSTNAMELEN];

   if (cd_mbox(mboxname, 0) != 0) return;
   if (access(APPNMAIL_MBOX, F_OK) == 0)
   {
      /* Make sure that only a single appnmail is waiting to dequeue. */
      if (lock_mbox_file(APPNMAIL_DEQUEUE_LOCK, 1) == 0)
      {
	 int unlocked = 0;

	 if (verboseflg) logprintf("Entering %s to dequeue...",mboxname);
	 if ((st = try_lock_mbox(host, line, NULL)) > 0)
	 {
	    if (!mailutilre) mailutilre = re_compile(LOCK_USER_RE,0);
	    if (*line && !re_match(line, mailutilre))
	    {
	       Class mailProxyClass = [MailProxy classForMailer:NULL
					  directDelivery:NO incorporation:YES];

	       if (incorporateDelay < 0)
	       {
		  const char *delaydef = NXGetDefaultValue(MAIL_APP, "appnmailIncorporateDelay");

		  if (delaydef && isdigit(*delaydef)) incorporateDelay = atoi(delaydef);
	       }

	       /* Wait some time (&& recheck lock) */
	       if (incorporateDelay > 0)
	       {
		  if (verboseflg>2) logprintf("IncorporateDelay: sleeping for %d seconds...",incorporateDelay);
		  sleep(incorporateDelay);
		  st = try_lock_mbox(host, line, NULL);
	       }

	       if (st > 0 && *line && !re_match(line, mailutilre) &&
		   [mailProxyClass canIncorporateNewMail])
	       {
		  if (verboseflg>2) logprintf("Locking %s...",APPNMAIL_LOCK);
		  if ((st = lock_mbox_file(APPNMAIL_LOCK, 0)) == 0)
		  {
		     /* XXX Should re-check .lock host? */
		     st = -1;
		     if (access(APPNMAIL_MBOX, F_OK) == 0)
		     {
			MailProxy *mailProxy;

			if (verboseflg>2) logprintf("Locating Mail port on host %s...", host);
			mailProxy = [[mailProxyClass alloc] initForMailer:NULL host:host
				       forceLaunch:NO directDelivery:NO];
			if (mailProxy && [mailProxy isConnected])
			{
			   /* Unlock dequeue_lock now, to avoid race conditions.
			      XXX should analyze this more thoroughly... */
			   unlock_mbox_file(APPNMAIL_DEQUEUE_LOCK);
			   unlocked++;
			   if (verboseflg>1) logprintf("Incorporating mail in open mailbox...");
			   if (incorporate_mail(mailProxy, mboxname, APPNMAIL_MBOX, APPNMAIL_TOC) != 0)
			   {
			      /* Retry using conventional mbox lock, but only
			         if no other appnmails are dequeueing. */
			      if (lock_mbox_file(APPNMAIL_DEQUEUE_LOCK, 1) == 0)
			      {
				 unlocked = 0;
				 st = 1;
			      }
			      else
			      {
				 st = -1;
			      }
			   }
			}
			[mailProxy release];
		     }
		     unlock_mbox_file(APPNMAIL_LOCK);
		  }
	       }
	    }
	    if (st > 0) st = lock_mbox((timeout > 0) ? -timeout : (!timeout));
	 }

	 if (st == 0)
	 {
	    if (verboseflg>2) logprintf("Locking %s...",APPNMAIL_LOCK);
	    if ((st = lock_mbox_file(APPNMAIL_LOCK, 0)) == 0)
	    {
	       /* Unlock dequeue_lock now, to avoid race conditions. */
	       unlock_mbox_file(APPNMAIL_DEQUEUE_LOCK);
	       unlocked++;

	       if (access(APPNMAIL_MBOX, F_OK) == 0)
	       {
		  if (verboseflg>1) logprintf("Appending mail to mailbox...");
		  dequeue_mail(mboxname);
	       }
	       unlock_mbox_file(APPNMAIL_LOCK);
	    }
	    unlock_mbox();
	 }
	 if (!unlocked)
	 {
	    unlock_mbox_file(APPNMAIL_DEQUEUE_LOCK);
	 }
	 if (verboseflg) logprintf("...dequeue %s done",mboxname);
      }
   }
   uncd_mbox();
}


int main(int ac,char *av[])
{
   struct message_index *mi;
   int c,ai;
   int readflg=0,nowaitflg=0,multiflg=0,flagflg=0,deleteflg=0,incorporateflg=0;
   int timeoutflg=-1;
   int pri=-1;
   int status=EXIT_SUCCESS;
   POOL_INIT

   set_progname(av[0]);
   log_set_show_pid(YES);
   log_set_show_time(YES);

   for (;;)
   {
      switch ((c=getopt(ac,av,"rfdvnT:D:p:iI:JLMNVH")))
      {
       case 'r':
	 readflg++;
	 if (flagflg||deleteflg) { status=EXIT_USAGE; break; }
	 continue;
       case 'f':
	 flagflg++;
	 if (readflg||deleteflg) { status=EXIT_USAGE; break; }
	 continue;
       case 'd':
	 deleteflg++;
	 if (readflg||flagflg) { status=EXIT_USAGE; break; }
	 continue;
       case 'n':
	 nowaitflg++;
	 continue;
       case 'T':
	 timeoutflg = atoi(optarg);
	 continue;
       case 'D':
	 incorporateDelay = atoi(optarg);
	 continue;
       case 'v':
	 verboseflg++;
	 continue;
       case 'm': // XXX currently unused.
	 multiflg++;
	 continue;
       case 'p':
	 if (pri!=-1) { status=EXIT_USAGE; break; }
	 pri=atoi(optarg);
	 if (pri<0) { status=EXIT_USAGE; break; }
	 continue;
       case 'i':
	 incorporateflg++;
	 continue;
       case 'I':   // input encoding.
	 inputEncoding = optarg;
	 continue;
       case 'J':   // input encoding specil case: Japanese.
	 inputEncoding = "iso-2022-jp";
	 continue;
       case 'L':   // input encoding special case: Latin.
	 inputEncoding = "iso-8859-1";
	 continue;
       case 'N':   // NeXT mode.
	 macMode = NO;
	 outputEncoding = NEXT_OUTPUT_ENCODING;
	 continue;
       case 'M':   // MacOS X mode.
	 macMode = YES;
	 outputEncoding = MAC_OUTPUT_ENCODING;
	 continue;
       case 'V':
	 status=EXIT_VERSION;
	 break;
       case 'H':
	 status=EXIT_HELP;
	 break;
       case EOF:
	 status=check_arg_for_long_help_version(av[optind-1]);
	 break;
       case '?':
       default:
	 status=EXIT_USAGE;
      }
      break;
   }
   if (status==EXIT_SUCCESS && optind>=ac) status=EXIT_USAGE;
   handle_usage_help_version(status, USAGE, HELP);

   /* This is to avoid problems with effective vs. real user ids. */
   setuid(geteuid());

   /* Mail message is optional if -i flag is given, so avoid reading stdin
      in interactive use, and allow empty message if redirected. */
   if (!incorporateflg || !isatty(fileno(stdin)))
   {
      mi=readmail();

      if (!(incorporateflg && headerlen == 0 && contentlen == 0))
      {
	 if (!header || (headerlen<5) || strncmp(header,"From ",5))
	 {
	    logprintf("message is not in UNIX mailbox format");
	    exit(EXIT_FAILURE);
	 }

	 if (readflg) mi->status = MT_STATUS_READ;
	 else if (flagflg) mi->status = MT_STATUS_FLAGGED;
	 else if (deleteflg) mi->status = MT_STATUS_DELETED;

	 if (pri>=0) message_set_priority(mi,pri);

	 if (verboseflg)
	 {
	    time_t date = message_time(mi);
	    logprintf("Delivering \"%s\" from %s (%.24s).", message_subject(mi), message_from(mi), ctime(&date));
	 }

	 for (ai=optind; ai<ac; ai++)
	 {
	    c = lock_mailbox_and_save_message(av[ai], mi);
	    if (c != EXIT_SUCCESS) status = c;
	 }
      }
   }

   if (incorporateflg || (status == EXIT_SUCCESS && !nowaitflg))
   {
      /* Dequeue pass. */
      for (ai = optind;  ai < ac;  ai++)
      {
	 incorporate(av[ai], incorporateflg, timeoutflg);
      }
   }
   POOL_RELEASE
   return status;
}
