// MOCompletionManager.m
// MOKit - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

// ABOUT MOKit
// 
// MOKit is a collection of useful and general objects.  Permission is 
// granted by the author to use MOKit in your own programs in any way 
// you see fit.  All other rights to the kit are reserved by the author 
// including the right to sell these objects as part of a LIBRARY or as 
// SOURCE CODE.  In plain English, I wish to retain rights to these 
// objects as objects, but allow the use of the objects as pieces in a 
// fully functional program.  NO WARRANTY is expressed or implied.  The author 
// will under no circumstances be held responsible for ANY consequences to 
// you from the use of these objects.  Since you don't have to pay for 
// them, and full source is provided, I think this is perfectly fair.

#import <MOKit/MOKit.h>

typedef enum {
    MOInitialVersion = 1,
} MOClassVersion;

static const MOClassVersion MOCurrentClassVersion = MOInitialVersion;

@implementation MOCompletionManager

+ (void)initialize {
    // Set the version.  Load classes, and init class variables.
    if (self == [MOCompletionManager class])  {
        [self setVersion:MOCurrentClassVersion];
    }
}

- (id)init {
    self = [super init];

    if (self != nil) {
        _cachedTextView = nil;
        _cachedSelectedRange = NSMakeRange(NSNotFound, 0);
        _cachedBasePath = nil;
        _completionStrategy = nil;
        _completionMatches = nil;
        _lastMatchIndex = NSNotFound;
        _completionRange = NSMakeRange(NSNotFound, 0);
        _completionPrefixString = nil;
        _dumpCompletionsEnabled = YES;

        _completionStrategies = nil;
    }
    
    return self;
}

- (void)dealloc {
    _dumpCompletionsEnabled = YES;
    [self dumpCompletionState];
    [_completionStrategies release];
    [super dealloc];
}

- (void)setCompletionStrategies:(NSArray *)strategies {
    [_completionStrategies autorelease];
    _completionStrategies = [strategies retain];
}

- (NSArray *)completionStrategies {
    return _completionStrategies;
}

- (void)dumpCompletionState {
    if (_dumpCompletionsEnabled) {
        //NSLog(@"Dumping completion state");
        _cachedTextView = nil;
        _cachedSelectedRange = NSMakeRange(NSNotFound, 0);
        [_cachedBasePath release];
        _cachedBasePath = nil;
        _completionStrategy = nil;
        [_completionMatches release];
        _completionMatches = nil;
        _lastMatchIndex = NSNotFound;
        _completionRange = NSMakeRange(NSNotFound, 0);
        [_completionPrefixString release];
        _completionPrefixString = nil;
    }
}

- (void)doCompletionInTextView:(NSTextView *)textView startLimit:(unsigned)startLimit basePath:(NSString *)basePath {
    static NSMutableCharacterSet *completableSet = nil;
    static NSCharacterSet *nonCompletableSet = nil;
    NSRange selectedRange = [textView selectedRange];
    unsigned i, c;
    
    // Set up the statics
    if (!completableSet) {
        nonCompletableSet = [[NSCharacterSet whitespaceCharacterSet] retain];
        completableSet = [[nonCompletableSet invertedSet] retain];
    }

    if ((textView != _cachedTextView) || (selectedRange.location != _cachedSelectedRange.location) || (selectedRange.length != _cachedSelectedRange.length) || (((basePath != nil) || (_cachedBasePath != nil)) && (![basePath isEqualToString:_cachedBasePath]))) {
        [self dumpCompletionState];
        _cachedTextView = textView;
        _cachedBasePath = [basePath copyWithZone:[self zone]];
    }

    if (_lastMatchIndex == NSNotFound) {
        NSString *string = [textView string];
        NSRange searchRange, foundRange;
        NSString *path;

        // Find the search path and partial name.
        searchRange.location = startLimit;
        if (selectedRange.location <= searchRange.location) {
            NSBeep();
            return;
        }
        searchRange.length = selectedRange.location - searchRange.location;
        foundRange = [string rangeOfCharacterFromSet:nonCompletableSet options:NSBackwardsSearch range:searchRange];
        if (foundRange.length > 0) {
            _completionRange = NSMakeRange(NSMaxRange(foundRange), NSMaxRange(selectedRange) - NSMaxRange(foundRange));
            path = [string substringWithRange:NSMakeRange(NSMaxRange(foundRange), NSMaxRange(searchRange) - NSMaxRange(foundRange))];
        } else {
            _completionRange = NSMakeRange(searchRange.location, NSMaxRange(selectedRange) - searchRange.location);
            path = [string substringWithRange:searchRange];
        }

        // Get the matches.
        c = [_completionStrategies count];
        for (i=0; i<c; i++) {
            _completionStrategy = [_completionStrategies objectAtIndex:i];
            //NSLog(@"Trying completion strategy '%@' with prefix '%@' and basePath '%@'", NSStringFromClass([_completionStrategy class]), path, basePath);
            _completionPrefixString = nil;
            _completionMatches = [_completionStrategy matchesForPrefixString:path newPrefixString:&_completionPrefixString basePath:basePath];
            if ([_completionMatches count] > 0) {
                break;
            }
        }
        [_completionMatches retain];
        [_completionPrefixString retain];
    }

    c = [_completionMatches count];
    if (c == 0) {
        NSBeep();
    } else {
        NSString *match;
        BOOL initialPrefixMatch;
        
        // Prevent the text and selection mucking happening here from dumping completion state.
        _dumpCompletionsEnabled = NO;
        
        if (_lastMatchIndex == NSNotFound) {
            _lastMatchIndex = c - 1;
            match = [_completionMatches MO_longestCommonPrefixForStrings];
            initialPrefixMatch = ((c > 1) ? YES : NO);
        } else {
            _lastMatchIndex = ((_lastMatchIndex + 1) % c);
            match = [_completionMatches objectAtIndex:_lastMatchIndex];
            initialPrefixMatch = NO;
        }

        // Get the (possibly) processed full string.
        match = [_completionStrategy fullStringForPrefixString:_completionPrefixString completionString:match isInitialPrefixMatch:initialPrefixMatch basePath:basePath];

        [textView replaceCharactersInRange:_completionRange withString:match];
        _completionRange.length = [match length];
        _cachedSelectedRange = NSMakeRange(NSMaxRange(_completionRange), 0);
        [textView setSelectedRange:_cachedSelectedRange];
        [textView scrollRangeToVisible:_cachedSelectedRange];

        _dumpCompletionsEnabled = YES;
        if (initialPrefixMatch) {
            NSBeep();
        }
        if (c == 1) {
            // Only one match, just get ready to complete the next component.
            [self dumpCompletionState];
        }
    }
}

- (void)textDidChange:(NSNotification *)notification {
    [self dumpCompletionState];
}

- (void)textDidEndEditing:(NSNotification *)notification {
    [self dumpCompletionState];
}

- (BOOL)textView:(NSTextView *)textView doCommandBySelector:(SEL)commandSelector {
    if (commandSelector == @selector(complete:)) {
        // MF:??? Should this use the textView's window's document path (if it has one) for the basePath?
        [self doCompletionInTextView:textView startLimit:0 basePath:nil];
        return YES;
    } else {
        return NO;
    }
}

- (void)controlTextDidChange:(NSNotification *)notification {
    [self dumpCompletionState];
}

- (void)controlTextDidEndEditing:(NSNotification *)notification {
    [self dumpCompletionState];
}

- (BOOL)control:(NSControl *)control textView:(NSTextView *)textView doCommandBySelector:(SEL)commandSelector {
    if (commandSelector == @selector(complete:)) {
        // MF:??? Should this use the textView's window's document path (if it has one) for the basePath?
        [self doCompletionInTextView:textView startLimit:0 basePath:nil];
        return YES;
    } else {
        return NO;
    }
}


@end
