/**
 * Zephyr message string parser
 **
 * Wilfredo Sanchez | wsanchez@apple.com, tritan@mit.edu
 *
 * Copyright 1998 Apple Computer, Inc.
 *
 * Copyright (c) 1989 by the Massachusetts Institute of Technology.
 * For copying and distribution information, see the file
 * "mit-copyright.h".
 *
 * All rights reserved.
 **/

#include <zephyr/mit-copyright.h>

#include <sysdep.h>

#include <new_string.h>
#include <formatter.h>
#include <variables.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import "Extensions.h"

#include "message_parser.h"

/**
 * Convert scribe-style text to an attributed string
 **/

#define envmatch(string,length)							\
        ( (aToken->len == (length))                         &&			\
          (strncasecmp(aToken->str,(string),(length)) == 0) )

NSAttributedString* attributedStringFromScribeText (string anInput)
{
    ////
    // Because it's easier to convert to HTML than to create an NSAttributedString, convert
    //   to HTML and let the AppKit/HTML framework deal with generating the attributed string.
    ////
    NSMutableString* anHTMLString     = [NSMutableString string];
    NSMutableArray*  anAttributeStack = [NSMutableArray  array ];

    string anInputCopy;
    desctype *aToken;
    int aStringCount, aLineCount;

    anInputCopy = string_Copy(anInput);
    aToken = disp_get_cmds(anInputCopy, &aStringCount, &aLineCount);

    [anHTMLString appendString: @"<pre>"];

    while (aToken->code != DT_EOF)
      {
        switch (aToken->code)
          {
            case DT_ENV: /* Begin attribute */
               {
                    NSString* anOpenTag;
                    NSString* aCloseTag;

                    if (envmatch("center",6))
                      {
                        anOpenTag = @"<div align=\"center\">"; aCloseTag = @"</div>";
                      }
                    if (envmatch("bold",4) || envmatch("b",1))
                      {
                        anOpenTag = @"<b>"; aCloseTag = @"</b>";
                      }
                    else if (envmatch("italic",6) || envmatch("i",1))
                      {
                        anOpenTag = @"<i>"; aCloseTag = @"</i>";
                      }
                    else if (envmatch("underline",9) || envmatch("u",1))
                      {
                        anOpenTag = @"<u>"; aCloseTag = @"</u>";
                      }
                    else if (envmatch("superscript",11) || envmatch("sup",3))
                      {
                        anOpenTag = @"<sup>"; aCloseTag = @"</sup>";
                      }
                    else if (envmatch("subscript",9) || envmatch("sub",3))
                      {
                        anOpenTag = @"<sub>"; aCloseTag = @"</sub>";
                      }
                    else if (envmatch("tiny",5))
                      {
                        anOpenTag = @"<small>"; aCloseTag = @"</small>"; // Should be smaller
                      }
                    else if (envmatch("small",5))
                      {
                        anOpenTag = @"<small>"; aCloseTag = @"</small>";
                      }
                    else if (envmatch("large",5))
                      {
                        anOpenTag = @"<big>"; aCloseTag = @"</big>";
                      }
                    else if (envmatch("huge",5))
                      {
                        anOpenTag = @"<big>"; aCloseTag = @"</big>"; // Should be bigger
                      }
                    else
                      {
                        anOpenTag = @""; aCloseTag = @"";
                      }

                    [anHTMLString appendString: anOpenTag];
                    [anAttributeStack addObject: aCloseTag];
                }
                break;

            case DT_STR: /* Text */
                if (aToken->len > 0)
                  {
                    NSMutableString* aText = [[NSString stringWithCString: aToken->str
                                                                       length: aToken->len] mutableCopy];

                    [aText replaceSubstring:@"&" withString:@"&amp;" globally:YES];
                    [aText replaceSubstring:@"<" withString:@"&lt;"  globally:YES];
                    [aText replaceSubstring:@">" withString:@"&gt;"  globally:YES];

                    [anHTMLString appendString: aText];

                    [aText release];
                  }
                break;

            case DT_END: /* End attribute */
                [anHTMLString appendString: [anAttributeStack lastObject]];
                [anAttributeStack removeLastObject];
                break;

            case DT_NL: /* Newline */
                [anHTMLString appendString: @"\n"];
                break;
          }
        aToken = aToken->next;
      }

    [anHTMLString appendString: @"</pre>"];

    free(anInputCopy);
    free_desc(aToken);

    return
        [[[NSAttributedString alloc]
                  initWithHTML: [NSData dataWithBytes: [anHTMLString cString      ]
                                               length: [anHTMLString cStringLength]]
            documentAttributes: NULL] autorelease];
}

/**
 * Convert HTML text to an attributed string
 **/

NSAttributedString* attributedStringFromHTMLText (string anInput)
{
    return [[[NSAttributedString alloc] initWithHTML: [NSData dataWithBytes: anInput
                                                                     length: strlen(anInput)]
                                  documentAttributes: NULL] autorelease];
}
