/* StockDataSource.m created by mlaster on Thu 15-Jul-1999 */

#import "WindowController.h"
#import "StockDataSource.h"
#import "NSBundle+BundleSearching.h"
#import "NSUserDefaults+ColorExtensions.h"
#import "PropertyListCategories.h"
#import "NSNumber+FractionExtensions.h"

static NSMutableArray *dataSourceClassNames = nil;

@implementation StockDataSource

+ (StockDataSource *) dataSource
{
    StockDataSource *newInstance = nil;

    if ([dataSourceClassNames count] > 0)
    {
        newInstance = [[[NSClassFromString([dataSourceClassNames objectAtIndex:0]) alloc] init] autorelease];
    }
    return newInstance;
}

+ (void) loadDataSourceBundles
{
    NSArray *bundleArray = nil;
    unsigned i=0;
    
    bundleArray = [NSBundle availableBundlesOfType:@"StockDataSource"];
    for (i=0;i<[bundleArray count];i++)
    {
        if ([[bundleArray objectAtIndex:i] smartLoad] == NO)
        {
            NSLog(@"Unable to load [%@]", [bundleArray objectAtIndex:i]);
        }
    }
}

+ (void) registerDataSource:(NSString *)className
{
    if (dataSourceClassNames == nil)
    {
        dataSourceClassNames = [[NSMutableArray alloc] init];
    }
    [dataSourceClassNames addObject:className];
}

- (id) init
{
    NSMutableDictionary *newDictionary = nil;
    
    self = [super init];
    if (self != nil)
    {
        NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
        newDictionary = [[NSMutableDictionary alloc] init];
        [self setStatDictionary:newDictionary];

        displayInMenuBar = [userDefaults boolForKey: @"DisplayInMenuBar"];

        if (displayInMenuBar == YES)
            colorCodes = [[NSDictionary alloc] initWithObjectsAndKeys:
                [userDefaults colorForKey: @"TickerMenuNeutralColor"], @"Neutral",
                [userDefaults colorForKey: @"TickerMenuPositiveColor"], @"Positive",
                [userDefaults colorForKey: @"TickerMenuNegativeColor"], @"Negative",
                nil,nil];
        else
            colorCodes = [[NSDictionary alloc] initWithObjectsAndKeys:
                [userDefaults colorForKey: @"TickerNeutralColor"], @"Neutral",
                [userDefaults colorForKey: @"TickerPositiveColor"], @"Positive",
                [userDefaults colorForKey: @"TickerNegativeColor"], @"Negative",
                nil,nil];
        [newDictionary release];

        [self setPositiveThreshold: [userDefaults floatForKey: @"PositiveThreshold"]];
        [self setNegativeThreshold: [userDefaults floatForKey: @"NegativeThreshold"]];

        useFractionalRepresentation = [userDefaults boolForKey: @"DisplayAsFractions"];
        indexNumberFormatter = [[NSNumberFormatter alloc] init];
        [indexNumberFormatter setFormat: [userDefaults stringForKey: @"IndexNumberFormat"]];
        stockNumberFormatter = [[NSNumberFormatter alloc] init];
        [stockNumberFormatter setFormat: [userDefaults stringForKey: @"StockNumberFormat"]];

        showVolume = [userDefaults boolForKey: @"ShowVolume"];
        showPercentageChange  = [userDefaults boolForKey: @"ShowPercentageChange"];

        baseFont = [[WindowController baseFont] retain];
    }
    return self;
}

- (void) dealloc
{
    [colorCodes release];
    [self setStatDictionary:nil];
    [indexNumberFormatter release];
    [stockNumberFormatter release];
    
    [super dealloc];
}
- (void) update
{
    NSDictionary *tempDictionary = nil;
    
    // synchronize preferences, and notify everyone that the data has been updated
    tempDictionary = [[statDictionary plist] copy];
    [[NSUserDefaults standardUserDefaults] setObject:tempDictionary forKey:@"Symbols"];
    [tempDictionary release];
    [[NSUserDefaults standardUserDefaults] synchronize];
    [[NSNotificationCenter defaultCenter] postNotificationName:StockDataUpdatedNotification
                                                        object:self
                                                      userInfo:nil];
}

// symbol list management
- (void) addSymbol:(NSMutableDictionary *)aSymbol
{
    NSString *key = nil;

    key = [aSymbol objectForKey:@"Symbol"];
    NSAssert(key != nil, @"addSymbol: dictionary must contain 'Symbol' key!");
    if ([[self statDictionary] objectForKey:key] == nil)
    {
        [[self statDictionary] setObject:aSymbol forKey:key];
    }
}

- (void) removeSymbol:(NSString *)aSymbol
{
    [[self statDictionary] removeObjectForKey:aSymbol];
}

- (void) removeAllSymbols
{
    [[self statDictionary] removeAllObjects];
}

- (NSAttributedString *) attributedString
{
#ifdef DEBUG
    static int updateCount = 0;
#endif
    unsigned i=0;
    NSMutableAttributedString *workString = nil;
    NSArray *keys = nil;
    NSMutableDictionary *stock = nil;
    NSMutableString *appendString;
    
    keys = [[[self statDictionary] allKeys] sortedArrayUsingSelector:@selector(compare:)];
    
#ifdef DEBUG
    workString = [[NSMutableAttributedString alloc] initWithString: [NSString stringWithFormat: @"[UPDATE #%d]", updateCount]];
    NSLog(@"DEBUG:  Update pass #%d", updateCount);
    updateCount = updateCount + 1;
#else 
    workString = [[NSMutableAttributedString alloc] initWithString:@""];
#endif

    appendString = [workString mutableString];
    [appendString appendString: @"[Q "];
    [appendString appendString: [[NSCalendarDate calendarDate] descriptionWithCalendarFormat: @"%m/%d %I:%M %p"]];
    [appendString appendString: @"] "];

    for (i=0;i<[keys count];i++)
    {
        stock = [[self statDictionary] objectForKey:[keys objectAtIndex:i]];
/*        if (i != 0)
        {
            [[workString mutableString] appendString:@" "];
        }*/
        [workString appendAttributedString:[self attributedStringForStockDictionary:stock]];
    }
    return workString;
}

- (NSData *)hyperlinkForSymbol:(NSString *)aSymbol
{
    return [aSymbol dataUsingEncoding:NSWindowsCP1252StringEncoding];
}

- (NSAttributedString *) attributedStringForStockDictionary:(NSDictionary *)aDictionary
{
    NSMutableAttributedString *workString = nil;
    NSMutableAttributedString *tempString = nil;
    BOOL isIndex;
    NSAttributedString *formattedNumber = nil;
    NSNumber *workNumber = nil;
    NSNumberFormatter *formatter = nil;
    NSColor *highlightColor = nil;
    double percentChange = 0.0;
    NSData *tempSymbolHyperlink = nil;
    NSDictionary *tempAttributeDictionary = nil;
    NSString *theSymbol;
    BOOL displayAsFraction;

    workString = [[NSMutableAttributedString alloc] initWithString:@""];

    theSymbol = [aDictionary objectForKey:@"Symbol"];
    isIndex = [theSymbol hasPrefix: @"^"];
    displayAsFraction = ((isIndex == NO) && (useFractionalRepresentation == YES));

    // Append symbol
    tempSymbolHyperlink = [self hyperlinkForSymbol:[aDictionary objectForKey:@"Symbol"]];
    tempString = [[NSMutableAttributedString alloc] initWithHTML: tempSymbolHyperlink
                                              documentAttributes: &tempAttributeDictionary]; // stringBy is faster than Format: -- not that it matters here
    [tempString applyFontTraits: NSBoldFontMask range: NSMakeRange(0, ([tempString length] - 1) )];
    [tempString applyFontTraits: NSUnboldFontMask range: NSMakeRange(([tempString length] - 1), 1 )];

    [workString appendAttributedString:tempString];
    [[workString mutableString] appendString:@" "];

    // append price
    workNumber = [aDictionary objectForKey:@"Price"];
    NSAssert([workNumber isKindOfClass:[NSNumber class]], @"workNumber is not an NSNumber!");
    if (displayAsFraction == YES)
        formattedNumber = [workNumber fractionalRepresentation: YES withFont: baseFont];
    else
        formattedNumber = [workNumber attributedDescriptionWithFormat: (isIndex ? indexNumberFormatter : stockNumberFormatter) withFont: baseFont];
    [workString appendAttributedString:formattedNumber];

    // append change
//    [[workString mutableString] appendString:@" "];
    workNumber = [aDictionary objectForKey:@"Change"];
    NSAssert([workNumber isKindOfClass:[NSNumber class]], @"Change is not an NSNumber!");
    if (displayAsFraction == YES)
    {
        formattedNumber = [workNumber signedFractionalRepresentation: YES withFont: baseFont];
    }
    else
    {
        formattedNumber = [workNumber attributedDescriptionWithFormat: (isIndex ? indexNumberFormatter : stockNumberFormatter) withFont: baseFont];
    }
    [workString appendAttributedString:formattedNumber];

    // append percent change
    if (showPercentageChange == YES) {
        percentChange = 100.0 * [[aDictionary objectForKey:@"Change"] doubleValue] /
        [[aDictionary objectForKey:@"Price"] doubleValue] ;
        [[workString mutableString] appendString:[NSString stringWithFormat:@"(%+.2f%%) ", percentChange]];
    }

    if (showVolume == YES)
    {
        NSString *volumeString = nil;
        // append volume
        [workString unscriptRange:NSMakeRange([workString length]-1,1)];
        formatter = [[NSNumberFormatter alloc] init];
        [formatter setFormat:@",0"];
        volumeString = [NSString stringWithFormat:@"Vol: %@ ",
            [formatter stringForObjectValue:[aDictionary objectForKey:@"Volume"]]];
        [[workString mutableString] appendString:volumeString];
        [formatter release];
    }
    
    // colorize entire string
    // TO-DO: make the thresholds configurable
    if ( (positiveThreshold != 0.0) && (percentChange >= positiveThreshold))
    {
        highlightColor = [colorCodes objectForKey:@"Positive"];
    }
    else if ( (negativeThreshold != 0.0) && (percentChange <= negativeThreshold))
    {
        highlightColor = [colorCodes objectForKey:@"Negative"];
    }
    else
    {
        highlightColor = [colorCodes objectForKey:@"Neutral"];
    }
    
    [workString addAttribute:NSForegroundColorAttributeName
                       value:highlightColor
                       range:NSMakeRange(0,[workString length])];

    if (isIndex == YES)
        [workString applyFontTraits: NSItalicFontMask range: NSMakeRange(0, [workString length])];

    return [workString autorelease];
}

- (NSMutableDictionary *) statDictionary
{
    return statDictionary;
}

- (void) setStatDictionary:(NSMutableDictionary *)newDictionary
{
    [newDictionary retain];
    [statDictionary release];
    statDictionary = newDictionary;
}

// this method is ugly.  There has to be a more elegant way to account
// for the fact that our NSNumbers turn back into NSStrings when we
// read them back from defaults

- (void) reloadSymbols
{
    int i=0;
    NSArray *keys = nil;
    NSString *key = nil;
    NSMutableArray *keysToRemove = nil;
    NSDictionary *symbols = nil;
    NSMutableDictionary *entity = nil;
    id workObject = nil;
    NSNumber *workNumber = nil;

    symbols = [[NSUserDefaults standardUserDefaults] dictionaryForKey:@"Symbols"];
    keys= [symbols allKeys];
    keysToRemove = [[[self statDictionary] allKeys] mutableCopy];
    for (i=0;i<[keys count];i++)
    {
        key = [keys objectAtIndex:i];
        entity = [[symbols objectForKey:key] mutableCopy];

        // objectForKey tends to return NSStrings, even when they should probably be NSNumbers...

        // convert price to an NSNumber
        workObject = [entity objectForKey:@"Price"];
        if ([workObject isKindOfClass:[NSString class]])
        {
            workNumber = [NSNumber numberWithFloat:[workObject floatValue]];
            [entity setObject:workNumber forKey:@"Price"];
        }

        // convert volume to an NSNumber
        workObject = [entity objectForKey:@"Volume"];
        if ([workObject isKindOfClass:[NSString class]])
        {
            workNumber = [NSNumber numberWithInt:[workObject intValue]];
            [entity setObject:workNumber forKey:@"Volume"];
        }

        [self addSymbol:entity];

        if ([[self statDictionary] objectForKey:key] != nil)
        {
            [keysToRemove removeObject:key];
        }
        [entity release];
    }
    for (i=0;i<[keysToRemove count];i++)
    {
        [self removeSymbol:[keysToRemove objectAtIndex:i]];
    }
    [keysToRemove release];
    [self update];
}


- (void) setPositiveThreshold: (float) aNum;
{
    positiveThreshold = aNum;
}

- (float) positiveThreshold { return positiveThreshold; }

- (void) setNegativeThreshold: (float) aNum;
{
    if (aNum > 0)
        aNum = -aNum;

    negativeThreshold = aNum;
}

- (float) negativeThreshold { return positiveThreshold; }
@end
