/*
 * Copyright 1998 by Kenneth C. Dyke
 *       All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN ACTION OF CONTRACT,
 * NEGLIGENCE, OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#import "G3Control.h"
#import "InfoPanelController.h"

#import <sys/types.h>
#import <unistd.h>

#import <driverkit/IODeviceMaster.h>
#import <driverkit/IOConfigTable.h>

static NSString *sortKey = @"Title";

static int sortByTitle(id a, id b, void *context)
{
    return [(NSString *)[a objectForKey:sortKey] compare:(NSString *)[b objectForKey:sortKey]];
}

// These are C strings for some obscure reason that I no longer remember.
static char CacheEnableKey[] = "CacheEnable";
static char CacheParityEnableKey[] = "CacheParityEnable";
static char CacheClockRatioKey[] = "CacheClockRatio";
static char CacheRAMSizeKey[] = "CacheRAMSize";
static char CacheRAMTypeKey[] = "CacheRAMType";
static char CacheControlKey[] = "CacheControl";
static char CacheWriteThroughKey[] = "CacheWriteThrough";
static char CacheOutputHoldKey[] = "CacheOutputHold";
static char CacheDLLSlowKey[] = "CacheDLLSlow";
static char CacheDifferentialClockKey[] = "CacheDifferentialClock";
static char CacheDLLBypassKey[] = "CacheDLLBypass";
static char CPUJunctionTemperatureKey[] = "CPUJunctionTemperature";
static char CPUHzKey[] = "CPUHz";
static char BusHzKey[] = "BusHz";
static char InstructionForwardingIntervalKey[] = "InstructionForwardingInterval";

static unsigned long cacheDivider[8] =
{
    1,
    2,
    3,
    1,
    4,
    5,
    6,
    1
};

static BOOL isYES(const char *str)
{
    return (strcmp(str,"YES") == 0) ? YES : NO;
}

static void setBOOL(char *str, BOOL value)
{
    if(value == YES)
        strcpy(str,"YES");
    else
        strcpy(str,"NO");
}

@implementation G3Control

- (id)init
{
    self = [super init];
    if(self)
    {
        _modelArray = [[NSMutableArray alloc] init];
        _isRoot = (geteuid() == 0) ? YES : NO;
    }
    return self;
}

- (BOOL)windowShouldClose:(NSNotification *)theNotification
{
    [NSApp terminate:nil];
    return YES;
}

- (void)applicationWillFinishLaunching:(NSNotification *)notification
{
    NXBundle *driverBundle;
    NSArray *modelPaths;
    NSString *modelPath, *tablePath;
    NSDictionary *modelDict;
    NSMenu *menu;
    NSMenuItem *menuItem;
    IOString kind;
    IOReturn r;
    NSEnumerator *modelEnumerator;

    // Fix up our easter egg contextual menu.
    if(_isRoot)
    {
        menu = [[NSMenu alloc] initWithTitle:@"CPU Speed"];
        menuItem = [menu addItemWithTitle:@"Arthur" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:0];
        [menuItem setEnabled:_isRoot];

        menuItem = [menu addItemWithTitle:@"Merced" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:8];
        [menuItem setEnabled:_isRoot];

        menuItem = [menu addItemWithTitle:@"Katami" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:32];
        [menuItem setEnabled:_isRoot];
        menuItem = [menu addItemWithTitle:@"Xenon" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:64];
        [menuItem setEnabled:_isRoot];
        menuItem = [menu addItemWithTitle:@"Pentagon" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:128];
        [menuItem setEnabled:_isRoot];
        menuItem = [menu addItemWithTitle:@"Celery" action:@selector(setForwardingIntervalAction:)
                            keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:255];
        [menuItem setEnabled:NO];
        [[cpuImageView cell] setMenu:menu];
        [menu release];
    }
    [cpuSpeedHintField setStringValue:@""];
    _g3Image = [[NSImage imageNamed:@"750_thumb"] retain];
    _snailImage = [[NSImage imageNamed:@"snail_micro"] retain];
    
    // First, try to get default configuration file.  If it's there, the driver
    // is at least installed.
    _cacheConfigTable = [IOConfigTable newDefaultTableForDriver:"G3CacheEnabler"];
    if(!_cacheConfigTable)
    {
        int result;
        result = NSRunCriticalAlertPanel(@"G3Control",
                                         @"G3CacheEnabler driver not installed.",
                                         @"Quit",nil,nil);
        //NSLog(@"Whoops.  Driver not installed. Can't do anything.");
        [NSApp terminate:nil];
    }
    
    // Okay, build an NSBundle for our driver bundle.
    driverBundle = [_cacheConfigTable driverBundle];

    // Get path and create an NSBundle, which is more useful when looking for resources.
    _driverBundle = [[NSBundle alloc] initWithPath:[NSString stringWithCString:[driverBundle directory]]];

    // Okay... now look for an Instance0.config resource and load it up if it's there.
    tablePath = [_driverBundle pathForResource:@"Instance0" ofType:@"table"];

    if(!tablePath)
    {
        // Try Defaut.table.  This shouldn't fail if we reach this point.
        tablePath = [_driverBundle pathForResource:@"Default" ofType:@"table"];
    }

    // Load current config table and build property list.
    _currentConfigurationDictionary = [[[NSString stringWithContentsOfFile:tablePath] propertyListFromStringsFileFormat] mutableCopy];

    // See if the driver is running in the kernel.
    _deviceMaster = [IODeviceMaster new];
    r = [_deviceMaster lookUpByDeviceName:"G3CacheEnabler"
                             objectNumber:&_cacheDeviceNumber
                               deviceKind:&kind];
    if(r == IO_R_SUCCESS)
    {
        _driverRunning = YES;
    }
    else
    {
        _driverRunning = NO;
    }
    
    // Search for known accelerator models.  I use the .model extension so that
    // I don't have to filter out Instance0.table or Default.table.  This may be
    // changed at some point to be more consistant with other DriverKit drivers.
    modelPaths = [_driverBundle pathsForResourcesOfType:@"model" inDirectory:@""];

    // Load up models, build property lists.
    modelEnumerator = [modelPaths objectEnumerator];
    while(modelPath = [modelEnumerator nextObject])
    {
        NSString *modelDescription = [NSString stringWithContentsOfFile:modelPath];
        modelDict = [modelDescription propertyListFromStringsFileFormat];
        [_modelArray addObject:modelDict];
    }

    // Sort model array so it looks nice.
    [_modelArray sortUsingFunction:sortByTitle context:NULL];

    // Set up popup.
    [modelPopup removeAllItems];
    {
        int i, c;
        c = [_modelArray count];
        for(i = 0; i < c; i++)
        {
            [modelPopup addItemWithTitle:[[_modelArray objectAtIndex:i] objectForKey:@"Title"]];
            [[modelPopup lastItem] setTag:i];
            [[modelPopup lastItem] setRepresentedObject:[_modelArray objectAtIndex:i]];
        }
    }

    if(_driverRunning)
    {
        [self syncCurrentModelSettingsWithSystem];
        _timer = [[NSTimer scheduledTimerWithTimeInterval:5.0f
                                                   target:self
                                                 selector:@selector(updateFromTimer:)
                                                 userInfo:nil
                                                  repeats:YES] retain];
    }
    [self updateUI];
    [window setFrameAutosaveName:@"G3Control"];
    [window setFrameUsingName:@"G3Control"];
    [window makeKeyAndOrderFront:nil];
}

- (void)syncCurrentModelSettingsWithSystem
{
    // Get cache size.
    unsigned int count;
    unsigned int intValue;
    unsigned char charValue[256];
    
    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:CacheRAMSizeKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];

    [self setCacheSize:intValue];

    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:CacheClockRatioKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];
    [self setClockRatio:intValue];
    
    count = sizeof(charValue);
    [_deviceMaster getCharValues:charValue
                    forParameter:CacheEnableKey
                    objectNumber:_cacheDeviceNumber
                           count:&count];
    [self setCacheEnabled:isYES(charValue)];

    count = sizeof(charValue);
    [_deviceMaster getCharValues:charValue
                    forParameter:CacheWriteThroughKey
                    objectNumber:_cacheDeviceNumber
                           count:&count];
    [self setWriteThrough:isYES(charValue)];

    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:CPUJunctionTemperatureKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];
    _temperature = intValue;

    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:CPUHzKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];
    _cpuHz = intValue;

    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:BusHzKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];
    _busHz = intValue;

    count = 1;
    [_deviceMaster getIntValues:&intValue
                   forParameter:InstructionForwardingIntervalKey
                   objectNumber:_cacheDeviceNumber
                          count:&count];
    _interval = intValue;
    
}

- (void)updateDynamicUI
{
    if(_driverRunning)
    {
        [cpuSpeedField setStringValue:[NSString stringWithFormat:@"%4.1f Mhz",((float)_cpuHz / 1000000.0f)]];
        [busSpeedField setStringValue:[NSString stringWithFormat:@"%4.1f Mhz",((float)_busHz / 1000000.0f)]];
        if([self cacheEnabled])
        {
            [l2SpeedField setStringValue:[NSString stringWithFormat:@"%4.1f Mhz",((float)((_cpuHz * 2) / cacheDivider[[self clockRatio]]) / 1000000.0f)]];
        }
        else
        {
            [l2SpeedField setStringValue:@"--"];
        }
        [cpuTemperatureField setStringValue:[NSString stringWithFormat:@"%dC (%dF)",_temperature,_temperature * 9 / 5 + 32]];
    }
    else
    {
        [cpuSpeedField setStringValue:@"--"];
        [busSpeedField setStringValue:@"--"];
        [l2SpeedField setStringValue:@"--"];
        [cpuTemperatureField setStringValue:@"--"];
    }
}

- (void)updateUI
{
    [modelPopup selectItemWithTitle:[_currentConfigurationDictionary objectForKey:@"Title"]];
    [modelPopup setEnabled:_isRoot];
    
    [cacheSizePopup selectItem:[cacheSizePopup itemAtIndex:[cacheSizePopup indexOfItemWithTag:[self cacheSize]]]];
    [cacheSizePopup setEnabled:_isRoot];
    
    [clockRatioPopup selectItem:[clockRatioPopup itemAtIndex:[clockRatioPopup indexOfItemWithTag:[self clockRatio]]]];
    [clockRatioPopup setEnabled:_isRoot];
    
    [writeThroughMatrix selectCellWithTag:[self writeThrough] ? 1 : 0];
    [writeThroughMatrix setEnabled:_isRoot];
    
    if([self cacheEnabled])
    {
        [enableText setStringValue:@"L2 Cache is currently enabled."];
        [enableButton setTitle:@"Disable Now"];
        [enableButton setTag:0];
    }
    else
    {
        [enableText setStringValue:@"L2 Cache is currently disabled."];
        [enableButton setTitle:@"Enable Now"];
        [enableButton setTag:1];
    }
    [enableButton setEnabled:_isRoot];
    
    [enableOnRestartSwitch setState:[self enabledOnRestart]];
    [enableOnRestartSwitch setEnabled:_isRoot];

    if(_interval != _lastInterval)
    {
        _lastInterval = _interval;
        if(_interval > 0)
        {
            [[cpuImageView cell] setImage:_snailImage];
        }
        else
        {
            [[cpuImageView cell] setImage:_g3Image];
        }
        [cpuImageView setNeedsDisplay:YES];
    }
    if(_interval > 0)
    {
        [cpuSpeedHintField setStringValue:[NSString stringWithFormat:@"ictc %d",_interval]];
    }
    else
    {
        [cpuSpeedHintField setStringValue:@""];
    }
    [self updateDynamicUI];
}

- (void)updateFromTimer:(id)timer
{
    [self syncCurrentModelSettingsWithSystem];
    [self updateDynamicUI];
}

- (void)saveConfigTable
{
    NSEnumerator *keyEnumerator;
    NSString *key, *value, *instancePath;
    NSMutableString *configTable = [[NSMutableString alloc] init];
    
    keyEnumerator = [_currentConfigurationDictionary keyEnumerator];

    while(key = [keyEnumerator nextObject])
    {
        value = [_currentConfigurationDictionary objectForKey:key];
        [configTable appendFormat:@"\"%@\" = \"%@\";\n",key,value];
    }

    // This seems to wind up missing for some reason.
    [configTable appendString:@"\"Server Name\" = \"G3CacheEnabler\";\n"];
    
    instancePath = [[_driverBundle bundlePath] stringByAppendingPathComponent:@"Instance0.table"];
    [configTable writeToFile:instancePath atomically:YES];
    
    [configTable release];
}

- (void)updateCacheSettings
{
    unsigned int count;
    unsigned int intValue;
    unsigned char charValue[256];

    if(!_driverRunning)
        return;

    // First, disable cache.
    setBOOL(charValue,NO);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                   forParameter:CacheEnableKey
                   objectNumber:_cacheDeviceNumber
                          count:count];

    // Set Cache Size
    intValue = [self cacheSize];
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:CacheRAMSizeKey
                   objectNumber:_cacheDeviceNumber
                          count:count];

    // Set Clock Ratio
    intValue = [self clockRatio];
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:CacheClockRatioKey
                   objectNumber:_cacheDeviceNumber
                          count:count];

    // Set RAM type
    intValue = [self cacheRAMType];
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:CacheRAMTypeKey
                   objectNumber:_cacheDeviceNumber
                          count:count];

    // Set Output Hold
    intValue = [self outputHold];
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:CacheOutputHoldKey
                   objectNumber:_cacheDeviceNumber
                          count:count];

    // Set DLL Bypass Key
    intValue = [self dllBypass];
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:CacheDLLBypassKey
                   objectNumber:_cacheDeviceNumber
                          count:count];
    
    // Set Write Through
    setBOOL(charValue,[self writeThrough]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheWriteThroughKey
                    objectNumber:_cacheDeviceNumber
                           count:count];

    // Set differential clock
    setBOOL(charValue,[self differentialClock]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheDifferentialClockKey
                    objectNumber:_cacheDeviceNumber
                           count:count];

    // Set DLL slow
    setBOOL(charValue,[self dllSlow]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheDLLSlowKey
                    objectNumber:_cacheDeviceNumber
                           count:count];

    // Set parity
    setBOOL(charValue,[self parityEnabled]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheParityEnableKey
                    objectNumber:_cacheDeviceNumber
                           count:count];

    // Set control
    setBOOL(charValue,[self cacheControlZZ]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheControlKey
                    objectNumber:_cacheDeviceNumber
                           count:count];

    // Set Cache Enabled state.
    setBOOL(charValue,[self cacheEnabled]);
    count = strlen(charValue)+1;
    [_deviceMaster setCharValues:charValue
                    forParameter:CacheEnableKey
                    objectNumber:_cacheDeviceNumber
                           count:count];
    
    // Set Instruction Forwarding interval
    intValue = _interval;
    count = 1;
    [_deviceMaster setIntValues:&intValue
                   forParameter:InstructionForwardingIntervalKey
                   objectNumber:_cacheDeviceNumber
                          count:count];
}

// Primitives
- (void)setConfigBOOL:(BOOL)value forKey:(NSString *)key
{
    [_currentConfigurationDictionary setObject:(value ? @"YES" : @"NO")
                                        forKey:key];
}

- (BOOL)configBOOLForKey:(NSString *)key
{
    return [[_currentConfigurationDictionary objectForKey:key] isEqual:@"YES"];
}

- (void)setConfigInt:(unsigned int)value forKey:(NSString *)key
{
    [_currentConfigurationDictionary setObject:[NSString stringWithFormat:@"%d",value]
                                        forKey:key];
}

- (unsigned int)configIntForKey:(NSString *)key
{
    return [[_currentConfigurationDictionary objectForKey:key] intValue];
}

// Our "Model".
// All of this stuff really belongs in another class.

- (void)setCacheEnabled:(BOOL)enabled
{
    _cacheEnable = enabled;
}

- (BOOL)cacheEnabled
{
    return _cacheEnable;
}

- (void)setEnabledOnRestart:(BOOL)enabled
{
    [self setConfigBOOL:enabled forKey:@"CacheEnable"];
}

- (BOOL)enabledOnRestart
{
    return [self configBOOLForKey:@"CacheEnable"];
}

- (void)setCacheSize:(unsigned int)size
{
    [self setConfigInt:size forKey:@"CacheRAMSize"];
}

- (unsigned int)cacheSize
{
    return [self configIntForKey:@"CacheRAMSize"];
}

- (void)setClockRatio:(unsigned int)ratio
{
    [self setConfigInt:ratio forKey:@"CacheClockRatio"];
}

- (unsigned int)clockRatio
{
    return [self configIntForKey:@"CacheClockRatio"];
}

- (void)setWriteThrough:(BOOL)writeThrough
{
    [self setConfigBOOL:writeThrough forKey:@"CacheWriteThrough"];
}

- (BOOL)writeThrough
{
    return [self configBOOLForKey:@"CacheWriteThrough"];
}

- (unsigned int)cacheRAMType
{
    return [self configIntForKey:@"CacheRAMType"];
}

- (unsigned int)outputHold
{
    return [self configIntForKey:@"CacheOutputHold"];
}

- (unsigned int)dllBypass
{
    return [self configIntForKey:@"CacheDLLBypass"];
}

- (BOOL)dllSlow
{
    return [self configBOOLForKey:@"CacheDLLSlow"];
}

- (BOOL)parityEnabled
{
    return [self configBOOLForKey:@"CacheParityEnable"];
}

- (BOOL)differentialClock
{
    return [self configBOOLForKey:@"CacheDifferentialClock"];
}

- (BOOL)cacheControlZZ
{
    return [self configBOOLForKey:@"CacheControl"];
}

// Action methods
- (void)enableCacheAction:(id)sender
{
    sync();
    sync();
    sync();
    [self setCacheEnabled:[sender tag]];
    [self updateCacheSettings];
    [self updateUI];
}

- (void)enableCacheOnRestartAction:(id)sender
{
    [self setEnabledOnRestart:[sender state] ? YES : NO];
    [self saveConfigTable];
    [self updateUI];
}

- (void)setCacheModeAction:(id)sender
{
    [self setWriteThrough:[sender selectedTag] == 0 ? NO : YES];
    [self saveConfigTable];
    [self updateCacheSettings];
    [self updateUI];
}

- (void)setCacheSizeAction:(id)sender
{
    [self setCacheSize:[sender selectedTag]];
    [self saveConfigTable];
    [self updateCacheSettings];
    [self updateUI];
}

- (void)setClockRatioAction:(id)sender
{
    [self setClockRatio:[sender selectedTag]];
    [self saveConfigTable];
    [self updateCacheSettings];
    [self updateUI];
}

- (void)setModelAction:(id)sender
{
    // This is a bit more complicated.   Release the current configuration, then
    // grab a mutable copy of the new one, update it with system configuration info if
    // the driver is running, then update the UI and save the config file.
    [_currentConfigurationDictionary release];
    _currentConfigurationDictionary = [[_modelArray objectAtIndex:[sender selectedTag]] mutableCopy];
    if(_driverRunning)
    {
        [self syncCurrentModelSettingsWithSystem];
    }
    [self saveConfigTable];
    [self updateCacheSettings];
    [self updateUI];
}

- (void)setForwardingIntervalAction:(id)sender
{
    _interval = [sender tag];

    // Paranoia.  Make sure the tags aren't set to something stupid.
    if(_interval > 255)
        _interval = 255;
    if(_interval < 0)
        _interval = 0;

    [self updateCacheSettings];
    [self updateUI];
}

- (void)showInfoPanelAction:(id)sender
{
    [[InfoPanelController sharedInfoPanelController] showWindow:sender];
}

@end
