// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWURL.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <ctype.h>

#import "OWContentType.h"
#import "OWNetLocation.h"
#import "OWHTMLToSGMLObjects.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OWF/Content.subproj/Address.subproj/OWURL.m,v 1.22 1998/12/08 04:05:48 kc Exp $")

@interface OWURL (Private)
+ (void)controllerDidInit:(NSNotification *)notification;

+ (OWURL *)urlWithLowercaseScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
+ (OWURL *)urlWithLowercaseScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;

- initWithLowercaseScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
- initWithLowercaseScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;
@end

@implementation OWURL

// These are carefully derived from RFC1808.
// (http://www.w3.org/hypertext/WWW/Addressing/rfc1808.txt)

static CSBitmap SchemeDelimiterCSBitmap;
static CSBitmap NetLocationDelimiterCSBitmap;
static CSBitmap PathDelimiterCSBitmap;
static CSBitmap ParamDelimiterCSBitmap;
static CSBitmap QueryDelimiterCSBitmap;
static CSBitmap FragmentDelimiterCSBitmap;
static CSBitmap SchemeSpecificPartDelimiterCSBitmap;
static CSBitmap NonWhitespaceCSBitmap;
static NSMutableCharacterSet *EscapeCharacterSet;
static NSMutableDictionary *ContentTypeDictionary;
static OFSimpleLockType ContentTypeDictionarySimpleLock;
static BOOL NetscapeCompatibleRelativeAddresses;

#define NonBuggyCharacterSet NSMutableCharacterSet

+ (void)initialize;
{
    static BOOL initialized = NO;
    NSCharacterSet *AlphaSet, *DigitSet, *ReservedSet;
    NSMutableCharacterSet *UnreservedSet, *UCharSet, *PCharSet;
    NSMutableCharacterSet *SchemeSet, *NetLocationSet, *PathSet;
    NSMutableCharacterSet *ParamSet, *QuerySet, *FragmentSet;
    NSMutableCharacterSet *SchemeSpecificPartSet;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    AlphaSet = [NonBuggyCharacterSet letterCharacterSet];
    DigitSet = [NonBuggyCharacterSet characterSetWithCharactersInString:
		@"0123456789"];
    ReservedSet = [NonBuggyCharacterSet characterSetWithCharactersInString:
		   @";/?:@&="];

    // This is a bit richer than the standard allows
    UnreservedSet = [[ReservedSet invertedSet] mutableCopy];
    [UnreservedSet removeCharactersInString:@"%#"];

    UCharSet = [[NSMutableCharacterSet alloc] init];
    [UCharSet formUnionWithCharacterSet:UnreservedSet];
    [UCharSet addCharactersInString:@"%"]; // escapes

    PCharSet = [[NSMutableCharacterSet alloc] init];
    [PCharSet formUnionWithCharacterSet:UCharSet];
    [PCharSet addCharactersInString:@":@&="];

    SchemeSet = [[NSMutableCharacterSet alloc] init];
    [SchemeSet formUnionWithCharacterSet:AlphaSet];
    [SchemeSet formUnionWithCharacterSet:DigitSet];
    [SchemeSet addCharactersInString:@"+-."];

    NetLocationSet = [[NSMutableCharacterSet alloc] init];
    [NetLocationSet formUnionWithCharacterSet:PCharSet];
    [NetLocationSet addCharactersInString:@";?"];

    PathSet = [[NSMutableCharacterSet alloc] init];
    [PathSet formUnionWithCharacterSet:PCharSet];
    [PathSet addCharactersInString:@"/"];

    ParamSet = [[NSMutableCharacterSet alloc] init];
    [ParamSet formUnionWithCharacterSet:PCharSet];
    [ParamSet addCharactersInString:@"/"];
    [ParamSet addCharactersInString:@";"];

    QuerySet = [[NSMutableCharacterSet alloc] init];
    [QuerySet formUnionWithCharacterSet:UCharSet];
    [QuerySet formUnionWithCharacterSet:ReservedSet];

    FragmentSet = [QuerySet retain];
    SchemeSpecificPartSet = [QuerySet retain];

    // Now, get the CSBitmap representations of all those character sets
#define delimiterBitmapForSet(aSet) bitmapForCharacterSetDoRetain([aSet invertedSet], YES)
    SchemeDelimiterCSBitmap = delimiterBitmapForSet(SchemeSet);
    NetLocationDelimiterCSBitmap = delimiterBitmapForSet(NetLocationSet);
    PathDelimiterCSBitmap = delimiterBitmapForSet(PathSet);
    ParamDelimiterCSBitmap = delimiterBitmapForSet(ParamSet);
    QueryDelimiterCSBitmap = delimiterBitmapForSet(QuerySet);
    FragmentDelimiterCSBitmap = delimiterBitmapForSet(FragmentSet);
    SchemeSpecificPartDelimiterCSBitmap = delimiterBitmapForSet(SchemeSpecificPartSet);
    NonWhitespaceCSBitmap = delimiterBitmapForSet([NSCharacterSet whitespaceAndNewlineCharacterSet]);
#undef delimiterBitmapForSet

    [SchemeSet release];
    [NetLocationSet release];
    [PathSet release];
    [ParamSet release];
    [QuerySet release];
    [FragmentSet release];
    [SchemeSpecificPartSet release];

    EscapeCharacterSet = [[NSMutableCharacterSet alloc] init];
    [EscapeCharacterSet addCharactersInString:
     @"*-.0123456789@ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz"];
    [EscapeCharacterSet invert];

    OFSimpleLockInit(&ContentTypeDictionarySimpleLock);
    ContentTypeDictionary = [[NSMutableDictionary alloc] init];
}

+ (void)didLoad;
{
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(controllerDidInit:) name:OFControllerDidInitNotification object:nil];
}

+ (void)readDefaults;
{
    OFUserDefaults *userDefaults;
    NSStringEncoding searchEncoding;

    userDefaults = [OFUserDefaults sharedUserDefaults];
    NetscapeCompatibleRelativeAddresses = [userDefaults boolForKey:@"OWURLNetscapeCompatibleRelativeAddresses"];

    searchEncoding = [userDefaults integerForKey:@"OWOutgoingStringEncoding"];
    if (searchEncoding == 0) {
        // Note that 0 is guaranteed never to be a valid encoding by the semantics of +[NSString availableStringEncodings], so we use it to indicate a default encoding setting.
        searchEncoding = [OWHTMLToSGMLObjects stringEncoding];
    }
    [self setURLEncoding:searchEncoding];
}

+ (OWURL *)urlWithScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
{
    return [self urlWithLowercaseScheme:[aScheme lowercaseString] netLocation:aNetLocation path:aPath params:someParams query:aQuery fragment:aFragment];
}

+ (OWURL *)urlWithScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;
{
    return [self urlWithLowercaseScheme:[aScheme lowercaseString] schemeSpecificPart:aSchemeSpecificPart fragment:aFragment];
}

+ (OWURL *)urlFromString:(NSString *)aString;
{
    NSString *aScheme, *aNetLocation;
    NSString *aPath, *someParams;
    NSString *aQuery, *aFragment;
    NSString *aSchemeSpecificPart;
    OFStringScanner *scanner;

    if (!aString || [aString length] == 0)
	return nil;

    scanner = [[OFStringScanner alloc] initWithString:aString];
    scannerScanUpToCharacterInCSBitmap(scanner, NonWhitespaceCSBitmap);
    aScheme = [scanner readFullTokenWithDelimiterCSBitmap:SchemeDelimiterCSBitmap forceLowercase:YES];
    if (!aScheme || [aScheme length] == 0 ||
        scannerReadCharacter(scanner) != ':') {
        [scanner release];
        return nil;
    }
    if (scannerPeekCharacter(scanner) == '/') {
        // Scan net location or path
        BOOL pathPresent;

        scannerSkipPeekedCharacter(scanner);
        if (scannerPeekCharacter(scanner) == '/') {
            // Scan net location
            scannerSkipPeekedCharacter(scanner);
            aNetLocation = [scanner readFullTokenWithDelimiterCSBitmap:NetLocationDelimiterCSBitmap forceLowercase:NO];
            if (aNetLocation && [aNetLocation length] == 0)
                aNetLocation = @"localhost";
            pathPresent = scannerPeekCharacter(scanner) == '/';
            if (pathPresent) {
                // To be consistent with the non-netLocation case, skip the '/' here, too
                scannerSkipPeekedCharacter(scanner);
            }
        } else {
            aNetLocation = nil;
            pathPresent = YES;
        }
        if (pathPresent) {
            // Scan path
            aPath = [scanner readFullTokenWithDelimiterCSBitmap:PathDelimiterCSBitmap forceLowercase:NO];
        } else {
            aPath = nil;
        }
    } else {
        // No net location
        aNetLocation = nil;
        if (scannerPeekCharacter(scanner) == '~') {
            // Scan path that starts with '~'
            //
            // I'm not sure this is actually a path URL, maybe URLs with this
            // form should just drop through to schemeSpecificParams
            aPath = [scanner readFullTokenWithDelimiterCSBitmap:PathDelimiterCSBitmap forceLowercase:NO];
        } else {
            // No path
            aPath = nil;
        }
    }

    if (scannerPeekCharacter(scanner) == ';') {
        // Scan params
        scannerSkipPeekedCharacter(scanner);
        someParams = [scanner readFullTokenWithDelimiterCSBitmap:ParamDelimiterCSBitmap forceLowercase:NO];
    } else {
        someParams = nil;
    }

    if (scannerPeekCharacter(scanner) == '?') {
        // Scan query
        scannerSkipPeekedCharacter(scanner);
        aQuery = [scanner readFullTokenWithDelimiterCSBitmap:QueryDelimiterCSBitmap forceLowercase:NO];
        if (!aQuery)
            aQuery = @"";
    } else {
        aQuery = nil;
    }

    if (!aNetLocation && !aPath && !someParams && !aQuery) {
        // Scan scheme-specific part
        aSchemeSpecificPart = [scanner readFullTokenWithDelimiterCSBitmap:SchemeSpecificPartDelimiterCSBitmap forceLowercase:NO];
    } else {
        aSchemeSpecificPart = nil;
    }

    if (scannerPeekCharacter(scanner) == '#') {
        // Scan fragment
        scannerSkipPeekedCharacter(scanner);
        aFragment = [scanner readFullTokenWithDelimiterCSBitmap:FragmentDelimiterCSBitmap forceLowercase:NO];
    } else {
        aFragment = nil;
    }

    [scanner release];

    if (aSchemeSpecificPart)
	return [self urlWithLowercaseScheme:aScheme schemeSpecificPart:aSchemeSpecificPart fragment:aFragment];
    return [self urlWithLowercaseScheme:aScheme netLocation:aNetLocation path:aPath params:someParams query:aQuery fragment:aFragment];
}

+ (OWURL *)urlFromDirtyString:(NSString *)aString;
{
    return [self urlFromString:[self cleanURLString:aString]];
}

+ (NSString *)cleanURLString:(NSString *)aString;
{
    if (!aString || [aString length] == 0)
	return nil;
    if ([aString hasPrefix:@"<"])
	aString = [aString substringFromIndex:1];
    if ([aString hasSuffix:@">"])
	aString = [aString substringToIndex:[aString length] - 1];
    if ([aString hasPrefix:@"URL:"])
	aString = [aString substringFromIndex:4];
    if ([aString containsString:@"\n"]) {
	NSArray *lines;
	NSMutableString *newString;
	unsigned int lineIndex, lineCount;

	newString = [[[NSMutableString alloc] initWithCapacity:[aString length]] autorelease];
	lines = [aString componentsSeparatedByString:@"\n"];
	lineCount = [lines count];
	for (lineIndex = 0; lineIndex < lineCount; lineIndex++)
	    [newString appendString:[[lines objectAtIndex:lineIndex] stringByRemovingSurroundingWhitespace]];
	aString = newString;
    }
    return aString;
}

static inline unichar hexDigit(unichar digit)
{
    if (isdigit(digit))
	return digit - '0';
    else if (isupper(digit))
	return 10 + digit - 'A';
    else 
	return 10 + digit - 'a';
}

+ (NSString *)decodeURLString:(NSString *)encodedString;
{
    unsigned int length;
    unichar *characters, *inPtr, *outPtr;
    unsigned int characterCount;

    length = [encodedString length];
    characters = (unichar *)NSZoneMalloc(NULL, length * sizeof(unichar));
    [encodedString getCharacters:characters];
    inPtr = characters;
    outPtr = characters;
    characterCount = length;
    while (characterCount--) {
	unichar                     character;

	character = *inPtr++;
	if (character == '%' && characterCount >= 2) {
	    character = hexDigit(*inPtr++) << 4 | hexDigit(*inPtr++);
	    characterCount -= 2;
	}
	*outPtr++ = character;
    }
    return [[[NSString alloc] initWithCharactersNoCopy:characters length:outPtr - characters freeWhenDone:YES] autorelease];
}

static inline char hex(int i)
{
    static const char *hexchars = "0123456789ABCDEF";

    return hexchars[i];
}

static NSStringEncoding urlEncoding = NSISOLatin1StringEncoding;

+ (void)setURLEncoding:(NSStringEncoding)newURLEncoding;
{
    urlEncoding = newURLEncoding;
    if (urlEncoding <= 0)
        urlEncoding = NSISOLatin1StringEncoding;
}

+ (NSStringEncoding)urlEncoding
{
    return urlEncoding;
}

+ (NSString *)encodeURLString:(NSString *)unencodedString asQuery:(BOOL)asQuery leaveSlashes:(BOOL)leaveSlashes leaveColons:(BOOL)leaveColons;
{
    return [self encodeURLString:unencodedString encoding:urlEncoding asQuery:asQuery leaveSlashes:leaveSlashes leaveColons:leaveColons];
}

+ (NSString *)encodeURLString:(NSString *)unencodedString encoding:(NSStringEncoding)thisUrlEncoding asQuery:(BOOL)asQuery leaveSlashes:(BOOL)leaveSlashes leaveColons:(BOOL)leaveColons;
{
    NSString *escapedString;
    NSData *sourceData;
    unsigned const char *sourceBuffer;
    int sourceLength;
    int sourceIndex;
    unichar *destinationBuffer;
    int destinationBufferSize;
    int destinationIndex;
    static const BOOL isAcceptable[96] =
    //   0 1 2 3 4 5 6 7 8 9 A B C D E F
    {    0,0,0,0,0,0,0,0,0,0,1,0,0,1,1,0,	// 2x   !"#$%&'()*+,-./
         1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,	// 3x  0123456789:;<=>?
	 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,	// 4x  @ABCDEFGHIJKLMNO
	 1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,1,	// 5X  PQRSTUVWXYZ[\]^_
	 0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,	// 6x  `abcdefghijklmno
	 1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0 };	// 7X  pqrstuvwxyz{\}~	DEL

    if (!unencodedString)
	return @"";

    // This is actually a pretty common occurrence
    if (![unencodedString rangeOfCharacterFromSet:EscapeCharacterSet].length)
        return unencodedString;

    if (thisUrlEncoding <= 0) thisUrlEncoding = urlEncoding;
    sourceData = [unencodedString dataUsingEncoding:thisUrlEncoding
                  allowLossyConversion:YES];
    sourceBuffer = [sourceData bytes];
    sourceLength = [sourceData length];
    
    destinationBufferSize = sourceLength + (sourceLength >> 2) + 12;
    destinationBuffer = NSZoneMalloc(NULL, (destinationBufferSize) * sizeof(unichar));
    destinationIndex = 0;
    
    for (sourceIndex = 0; sourceIndex < sourceLength; sourceIndex++) {
	unsigned char ch;
	
	ch = sourceBuffer[sourceIndex];
	
	if (destinationIndex >= destinationBufferSize - 3) {
	    destinationBufferSize += destinationBufferSize >> 2;
	    destinationBuffer = NSZoneRealloc(NULL, destinationBuffer, (destinationBufferSize) * sizeof(unichar));
	}
	
        if (ch >= 32 && ch <= 127 && isAcceptable[ch-32]) {
	    destinationBuffer[destinationIndex++] = ch;
	} else if (asQuery && ch == ' ') {
	    destinationBuffer[destinationIndex++] = '+';
	} else if (leaveSlashes && ch == '/') {
	    destinationBuffer[destinationIndex++] = '/';
	} else if (leaveColons && ch == ':') {
	    destinationBuffer[destinationIndex++] = ':';
	} else {
	    destinationBuffer[destinationIndex++] = '%';
	    destinationBuffer[destinationIndex++] = hex((ch & 0xF0) >> 4);
	    destinationBuffer[destinationIndex++] = hex(ch & 0x0F);
	}
    }
    
    escapedString = [[[NSString alloc] initWithCharactersNoCopy:destinationBuffer length:destinationIndex freeWhenDone:YES] autorelease];
    
    return escapedString;
}

+ (OWContentType *)contentTypeForScheme:(NSString *)aScheme;
{
    OWContentType *aContentType;

    OFSimpleLock(&ContentTypeDictionarySimpleLock);
    aContentType = [ContentTypeDictionary objectForKey:aScheme];
    if (!aContentType) {
	aContentType = [OWContentType contentTypeForString:[@"url/" stringByAppendingString:aScheme]];
	[ContentTypeDictionary setObject:aContentType forKey:aScheme];
    }
    OFSimpleUnlock(&ContentTypeDictionarySimpleLock);
    return aContentType;
}

+ (NSArray *)pathComponentsForPath:(NSString *)aPath;
{
    if (!aPath)
        return nil;

    return [aPath componentsSeparatedByString:@"/"];
}

+ (NSString *)lastPathComponentForPath:(NSString *)aPath;
{
    NSRange lastSlashRange;
    unsigned int originalLength, lengthMinusTrailingSlash;
    
    if (!aPath)
        return nil;

    originalLength = [aPath length];

    // If the last character is a slash, ignore it.
    if (originalLength > 0 && [aPath characterAtIndex:originalLength - 1] == '/')
        lengthMinusTrailingSlash = originalLength - 1;
    else
        lengthMinusTrailingSlash = originalLength;

    // If the path (minus any trailing slash) is empty, return an empty string
    if (lengthMinusTrailingSlash == 0)
        return @"";

    // Find the last slash in the path
    lastSlashRange = [aPath rangeOfString:@"/" options:NSLiteralSearch | NSBackwardsSearch range:NSMakeRange(0, lengthMinusTrailingSlash - 1)];

    // If there is none, return the existing path (minus trailing slash).
    if (lastSlashRange.length == 0)
        return originalLength == lengthMinusTrailingSlash ? aPath : [aPath substringToIndex:lengthMinusTrailingSlash];

    // Return the substring between the last slash and the end of the string (ignoring any trailing slash)
    return [aPath substringWithRange:NSMakeRange(NSMaxRange(lastSlashRange), lengthMinusTrailingSlash - NSMaxRange(lastSlashRange))];
}

+ (NSString *)stringByDeletingLastPathComponentFromPath:(NSString *)aPath;
{
    NSRange lastSlashRange;

    if (!aPath)
        return nil;

    lastSlashRange = [aPath rangeOfString:@"/" options:NSLiteralSearch | NSBackwardsSearch];
    if (lastSlashRange.length == 0)
        return @"";
    if (lastSlashRange.location == 0 && [aPath length] > 1)
        return @"/";
    return [aPath substringToIndex:lastSlashRange.location];
}

- (void)dealloc;
{
    [scheme release];
    [netLocation release];
    [path release];
    [params release];
    [query release];
    [fragment release];
    [schemeSpecificPart release];
    [cachedCompositeString release];
    [cachedShortDisplayString release];
    [cachedParsedNetLocation release];
    [cacheKey release];
    [super dealloc];
}

- (NSString *)scheme;
{
    return scheme;
}

- (NSString *)netLocation;
{
    return netLocation;
}

- (NSString *)path;
{
    return path;
}

- (NSString *)params;
{
    return params;
}

- (NSString *)query;
{
    return query;
}

- (NSString *)fragment;
{
    return fragment;
}

- (NSString *)schemeSpecificPart;
{
    return schemeSpecificPart;
}

- (NSString *)compositeString;
{
    NSMutableString *compositeString;

    if (cachedCompositeString)
	return cachedCompositeString;

    compositeString = [[NSMutableString alloc] initWithString:scheme];
    [compositeString appendString:@":"];

    if (schemeSpecificPart) {
	[compositeString appendString:schemeSpecificPart];
    } else {
	if (netLocation) {
	    [compositeString appendString:@"//"];
	    [compositeString appendString:netLocation];
	}
	[compositeString appendString:@"/"];
	if (path)
	    [compositeString appendString:path];
	if (params) {
	    [compositeString appendString:@";"];
	    [compositeString appendString:params];
	}
	if (query) {
	    [compositeString appendString:@"?"];
	    [compositeString appendString:query];
	}
    }
    if (fragment) {
	[compositeString appendString:@"#"];
	[compositeString appendString:fragment];
    }

    // Make the cachedCompositeString immutable so that others will be able to just retain
    // it rather than making their own immutable copy
    cachedCompositeString = [compositeString copy];
    [compositeString release];
    
    return cachedCompositeString;
}

- (NSString *)cacheKey;
{
    NSMutableString *key;

    if (cacheKey)
	return cacheKey;

    key = [[NSMutableString alloc] initWithString:scheme];
    [key appendString:@":"];

    if (schemeSpecificPart) {
	[key appendString:schemeSpecificPart];
    } else {
	if (netLocation) {
	    [key appendString:@"//"];
	    [key appendString:netLocation];
	}
	[key appendString:@"/"];
	if (path)
	    [key appendString:path];
	if (params) {
	    [key appendString:@";"];
	    [key appendString:params];
	}
	if (query) {
	    [key appendString:@"?"];
	    [key appendString:query];
	}
    }

    // Make the cacheKey immutable so that others will be able to just retain
    // it rather than making their own immutable copy.
    cacheKey = [key copyWithZone:[self zone]];
    [key release];
    
    return cacheKey;
}

- (NSString *)fetchPath;
{
    NSMutableString *fetchPath;

    fetchPath = [NSMutableString stringWithCapacity:[path length] + 1];

    if (schemeSpecificPart) {
	[fetchPath appendString:schemeSpecificPart];
    } else {
	[fetchPath appendString:@"/"];
	if (path) {
	    if (NetscapeCompatibleRelativeAddresses && [path containsString:@".."]) {
                OWURL *siteURL, *resolvedPathURL;

                // Not the most efficient process, but I think it should work, and hopefully this happens rarely.  I didn't want to go to the trouble of abstracting out all that relative path code from -urlFromRelativeString:.
                siteURL = [self urlFromRelativeString:@"/"];
                resolvedPathURL = [siteURL urlFromRelativeString:path];
                [fetchPath appendString:[resolvedPathURL path]];
            } else
		[fetchPath appendString:path];
	}
	if (params) {
	    [fetchPath appendString:@";"];
	    [fetchPath appendString:params];
	}
	if (query) {
	    [fetchPath appendString:@"?"];
	    [fetchPath appendString:query];
	}
    }
    return fetchPath;
}

- (NSString *)proxyFetchPath;
{
    NSMutableString *proxyFetchPath;

    // Yes, this ends up looking a lot like our -cacheKey, except we're calling -fetchPath so the NetscapeCompatibleRelativeAddresses preference will kick in (and we don't want it to kick in for our -cacheKey because it's relatively expensive and -cacheKey gets called a lot more).

    proxyFetchPath = [[NSMutableString alloc] initWithString:scheme];
    [proxyFetchPath appendString:@":"];
    if (netLocation) {
        [proxyFetchPath appendString:@"//"];
        [proxyFetchPath appendString:netLocation];
    }
    [proxyFetchPath appendString:[self fetchPath]];
    return proxyFetchPath;
}

- (NSArray *)pathComponents;
{
    return [OWURL pathComponentsForPath:path];
}

- (NSString *)lastPathComponent;
{
    return [OWURL lastPathComponentForPath:path];
}

- (NSString *)stringByDeletingLastPathComponent;
{
    return [OWURL stringByDeletingLastPathComponentFromPath:path];
}

- (OWNetLocation *)parsedNetLocation;
{
    if (!cachedParsedNetLocation)
        cachedParsedNetLocation = [[OWNetLocation netLocationWithString: netLocation ? netLocation : schemeSpecificPart] retain];

    return cachedParsedNetLocation;
}


- (NSString *)shortDisplayString;
{
    NSMutableString *shortDisplayString;

    if (cachedShortDisplayString)
        return cachedShortDisplayString;

    shortDisplayString = [[NSMutableString alloc] init];
    if (netLocation) {
        [shortDisplayString appendString:[[self parsedNetLocation] shortDisplayString]];
        [shortDisplayString appendString:[NSString horizontalEllipsisString]];
    } else {
        [shortDisplayString appendString:scheme];
        [shortDisplayString appendString:@":"];
    }
    
    if (path) {
        [shortDisplayString appendString:[self lastPathComponent]];
        if ([path hasSuffix:@"/"])
            [shortDisplayString appendString:@"/"];
    }
    if (params) {
        [shortDisplayString appendString:@";"];
        [shortDisplayString appendString:params];
    }
    if (query) {
        [shortDisplayString appendString:@"?"];
        [shortDisplayString appendString:query];
    }
    if (fragment) {
        [shortDisplayString appendString:@"#"];
        [shortDisplayString appendString:fragment];
    }
    cachedShortDisplayString = shortDisplayString;
    return cachedShortDisplayString;
}

//

- (BOOL)isEqual:(id)anObject;
{
    OWURL *otherURL;

    if (self == anObject)
	return YES;
    if (anObject == nil)
        return NO;
    otherURL = anObject;
    if (otherURL->isa != isa)
	return NO;
    if (!cachedCompositeString)
	[self compositeString];
    if (!otherURL->cachedCompositeString)
	[otherURL compositeString];
    return [cachedCompositeString isEqualToString:otherURL->cachedCompositeString];
}

- (OWContentType *)contentType;
{
    if (!contentType)
	contentType = [OWURL contentTypeForScheme:scheme];
    return contentType;
}

//

- (OWURL *)urlFromRelativeString:(NSString *)aString;
{
    OWURL *absoluteURL;
    NSString *aNetLocation;
    NSString *aPath, *someParams, *aQuery, *aFragment;
    OFStringScanner *scanner;

    absoluteURL = [OWURL urlFromString:aString];
    if (absoluteURL) {
        if (schemeSpecificPart) {
            // If our scheme uses a non-uniform URL syntax, relative URLs are illegal
            return absoluteURL;
        }

        if (NetscapeCompatibleRelativeAddresses && [scheme isEqualToString:[absoluteURL scheme]] && ![absoluteURL netLocation]) {
            NSString *otherFetchPath, *otherFragment;

            // For Netscape compatibility, treat "http:whatever" as a relative link to "whatever".

            otherFetchPath = [absoluteURL fetchPath];
            otherFragment = [absoluteURL fragment];
            if (otherFragment)
                aString = [NSString stringWithFormat:@"%@#%@", otherFetchPath, otherFragment];
            else
                aString = otherFetchPath;
            absoluteURL = nil;
        } else {
            return absoluteURL;
        }
    }

    if (!aString || [aString length] == 0)
	return self;

    // Relative URLs default to the current location
    aNetLocation = netLocation;
    aPath = path;
    someParams = params;
    aQuery = query;
    aFragment = fragment;

    scanner = [[OFStringScanner alloc] initWithString:aString];
    scannerScanUpToCharacterInCSBitmap(scanner, NonWhitespaceCSBitmap);
    if (scannerPeekCharacter(scanner) == '/') {
        // Scan net location or absolute path
        BOOL absolutePathPresent;

        scannerSkipPeekedCharacter(scanner);
        if (scannerPeekCharacter(scanner) == '/') {
            // Scan net location
            scannerSkipPeekedCharacter(scanner);
            aNetLocation = [scanner readFullTokenWithDelimiterCSBitmap:NetLocationDelimiterCSBitmap forceLowercase:NO];
            if (aNetLocation && [aNetLocation length] == 0)
                aNetLocation = @"localhost";
            absolutePathPresent = scannerPeekCharacter(scanner) == '/';
            if (absolutePathPresent) {
                // To be consistent with the non-netLocation case, skip the '/' here, too
                scannerSkipPeekedCharacter(scanner);
            }
        } else {
            // That slash started a path, not a net location
            absolutePathPresent = YES;
        }
        if (absolutePathPresent) {
            // Scan path
            aPath = [scanner readFullTokenWithDelimiterCSBitmap:PathDelimiterCSBitmap forceLowercase:NO];
        } else {
            // Reset path
            aPath = nil;
        }
        // Reset remaining parameters
        someParams = nil;
        aQuery = nil;
        aFragment = nil;
    } else if (scannerHasData(scanner) && !characterIsMemberOfCSBitmap(PathDelimiterCSBitmap, scannerPeekCharacter(scanner))) {
        // Scan relative path
	NSMutableArray *pathElements;
	unsigned int preserveCount = 0, pathElementCount;
	NSArray *relativePathArray;
	unsigned int relativePathIndex, relativePathCount;
	BOOL lastElementWasDirectory = NO;

        aPath = [scanner readFullTokenWithDelimiterCSBitmap:PathDelimiterCSBitmap forceLowercase:NO];

        if (!path || [path length] == 0)
	    pathElements = [NSMutableArray arrayWithCapacity:1];
	else
            pathElements = [[[OWURL pathComponentsForPath:path] mutableCopy] autorelease];
	if ((pathElementCount = [pathElements count]) > 0) {
	    if ([[pathElements objectAtIndex:0] length] == 0)
		preserveCount = 1;
	    if (pathElementCount > preserveCount)
		[pathElements removeLastObject];
	}
        relativePathArray = [OWURL pathComponentsForPath:aPath];
	relativePathCount = [relativePathArray count];
	for (relativePathIndex = 0; relativePathIndex < relativePathCount; relativePathIndex++) {
	    NSString *pathElement;

	    pathElement = [relativePathArray objectAtIndex:relativePathIndex];
	    if ([pathElement isEqualToString:@".."]) {
		lastElementWasDirectory = YES;
		if ([pathElements count] > preserveCount)
		    [pathElements removeLastObject];
		else {
		    if (NetscapeCompatibleRelativeAddresses) {
			// Netscape doesn't preserve leading ..'s
		    } else {
			[pathElements addObject:pathElement];
			preserveCount++;
		    }
		}
	    } else if ([pathElement isEqualToString:@"."]) {
		lastElementWasDirectory = YES;
	    } else {
		lastElementWasDirectory = NO;
		[pathElements addObject:pathElement];
	    }
	}
	if (lastElementWasDirectory && [[pathElements lastObject] length] != 0) {
	    [pathElements addObject:@""];
	}
	aPath = [pathElements componentsJoinedByString:@"/"];

        // Reset remaining parameters
        someParams = nil;
        aQuery = nil;
        aFragment = nil;
    }
    if (scannerPeekCharacter(scanner) == ';') {
        // Scan params
        scannerSkipPeekedCharacter(scanner);
        someParams = [scanner readFullTokenWithDelimiterCSBitmap:ParamDelimiterCSBitmap forceLowercase:NO];

        // Reset remaining parameters
        aQuery = nil;
        aFragment = nil;
    }
    if (scannerPeekCharacter(scanner) == '?') {
        // Scan query
        scannerSkipPeekedCharacter(scanner);
        aQuery = [scanner readFullTokenWithDelimiterCSBitmap:QueryDelimiterCSBitmap forceLowercase:NO];
        if (!aQuery)
            aQuery = @"";

        // Reset remaining parameters
        aFragment = nil;
    }
    if (scannerPeekCharacter(scanner) == '#') {
        // Scan fragment
        scannerSkipPeekedCharacter(scanner);
        aFragment = [scanner readFullTokenWithDelimiterCSBitmap:FragmentDelimiterCSBitmap forceLowercase:NO];
    }

    [scanner release];

    return [OWURL urlWithLowercaseScheme:scheme netLocation:aNetLocation path:aPath params:someParams query:aQuery fragment:aFragment];
}

- (OWURL *)urlForPath:(NSString *)newPath;
{
    return [OWURL urlWithLowercaseScheme:scheme netLocation:netLocation path:newPath params:nil query:nil fragment:nil];
}

- (OWURL *)urlForQuery:(NSString *)newQuery;
{
    return [OWURL urlWithLowercaseScheme:scheme netLocation:netLocation path:path params:params query:newQuery fragment:nil];
}

- (OWURL *)urlWithoutFragment;
{
    if (!fragment)
	return self;
    return [OWURL urlWithLowercaseScheme:scheme netLocation:netLocation path:path params:params query:query fragment:nil];
}

- (OWURL *)urlWithFragment:(NSString *)newFragment
{
    if (newFragment == fragment ||
        [fragment isEqualToString:newFragment])
        return self;

    return [OWURL urlWithLowercaseScheme:scheme netLocation:netLocation path:path params:params query:query fragment:newFragment];
}

// NSCopying protocol

- (id)copyWithZone:(NSZone *)zone
{
    OWURL *newURL;
    
    if (NSShouldRetainWithZone(self, zone))
        return [self retain];

    newURL = [[isa allocWithZone:zone] init];
    
    newURL->scheme = [scheme copyWithZone:zone];
    newURL->netLocation = [netLocation copyWithZone:zone];
    newURL->path = [path copyWithZone:zone];
    newURL->params = [params copyWithZone:zone];
    newURL->query = [query copyWithZone:zone];
    newURL->fragment = [fragment copyWithZone:zone];
    newURL->schemeSpecificPart = [schemeSpecificPart copyWithZone:zone];
        
    return newURL;
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];

    [debugDictionary setObject:scheme forKey:@"scheme"];
    if (netLocation)
	[debugDictionary setObject:netLocation forKey:@"netLocation"];
    if (path)
	[debugDictionary setObject:path forKey:@"path"];
    if (params)
	[debugDictionary setObject:params forKey:@"params"];
    if (query)
	[debugDictionary setObject:query forKey:@"query"];
    if (fragment)
	[debugDictionary setObject:fragment forKey:@"fragment"];
    if (schemeSpecificPart)
	[debugDictionary setObject:schemeSpecificPart forKey:@"schemeSpecificPart"];

    [debugDictionary setObject:[self compositeString] forKey:@"compositeString"];

    return debugDictionary;
}

- (NSString *)shortDescription;
{
    return [NSString stringWithFormat:@"<URL:%@>", [self compositeString]];
}

@end

@implementation OWURL (Private)

+ (void)controllerDidInit:(NSNotification *)notification;
{
    [self readDefaults];
}

+ (OWURL *)urlWithLowercaseScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
{
    if (!aScheme)
	return nil;
    return [[[self alloc] initWithLowercaseScheme:aScheme netLocation:aNetLocation path:aPath params:someParams query:aQuery fragment:aFragment] autorelease];
}

+ (OWURL *)urlWithLowercaseScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;
{
    if (!aScheme)
	return nil;

    return [[[self alloc] initWithLowercaseScheme:aScheme schemeSpecificPart:aSchemeSpecificPart fragment:aFragment] autorelease];
}

- initWithLowercaseScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
{
    if (![super init])
	return nil;

    if (!aScheme) {
	[self release];
	return nil;
    }
    
    scheme = [aScheme retain];
    netLocation = [aNetLocation retain];
    path = [aPath retain];
    params = [someParams retain];
    query = [aQuery retain];
    fragment = [aFragment retain];
    contentType = nil;

    return self;
}

- initWithScheme:(NSString *)aScheme netLocation:(NSString *)aNetLocation path:(NSString *)aPath params:(NSString *)someParams query:(NSString *)aQuery fragment:(NSString *)aFragment;
{
    return [self initWithLowercaseScheme:[aScheme lowercaseString] netLocation:aNetLocation path:aPath params:someParams query:aQuery fragment:aFragment];
}

- initWithLowercaseScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;
{
    if (![self init])
	return nil;

    if (!aScheme) {
	[self release];
	return nil;
    }
    
    scheme = [aScheme retain];
    schemeSpecificPart = [aSchemeSpecificPart retain];
    fragment = [aFragment retain];
    contentType = nil;
    
    return self;
}

- initWithScheme:(NSString *)aScheme schemeSpecificPart:(NSString *)aSchemeSpecificPart fragment:(NSString *)aFragment;
{
    return [self initWithLowercaseScheme:[aScheme lowercaseString] schemeSpecificPart:aSchemeSpecificPart fragment:aFragment];
}

@end

// Enabling this causes the app to do a run a bunch of URL tests and then immediately exit.

#if 0
@implementation OWURL (Test)

static inline void testURL(NSString *urlString)
{
    NSLog(@"%@ -> %@", urlString, [[OWURL urlFromDirtyString:urlString] shortDescription]);
}

static inline void testRelativeURL(NSString *urlString)
{
    static OWURL *baseURL = nil;
    
    if (!baseURL) {
	baseURL = [OWURL urlFromDirtyString:@"<URL:http://a/b/c/d;p?q#f>"];
	NSLog(@"Base: %@", [baseURL shortDescription]);
    }

    NSLog(@"%@ = %@", [urlString stringByPaddingToLength:13], [[baseURL urlFromRelativeString:urlString] shortDescription]);
}


+ (void)didLoad;
{
    testURL(@"http://www.omnigroup.com/Test/path.html");
    testURL(@"file:/LocalLibrary/Web/");
    testURL(@"http://www.omnigroup.com/blegga.cgi?blah");
    testURL(@"<URL:ftp://ds.internic.net/rfc/rfc1436.txt;type=a>");
    testURL(@"<URL:ftp://info.cern.ch/pub/www/doc;\n      type=d>");
    testURL(@"<URL:ftp://info.cern.ch/pub/www/doc;\n      type=d>");
    testURL(@"<URL:ftp://ds.in\n      ternic.net/rfc>");
    testURL(@"<URL:http://ds.internic.\n      net/instructions/overview.html#WARNING>");
    testURL(@"index.html");
    testURL(@"../index.html");

    testRelativeURL(@"g:h");
    testRelativeURL(@"g");
    testRelativeURL(@"./g");
    testRelativeURL(@"g/");
    testRelativeURL(@"/g");
    testRelativeURL(@"//g");
    testRelativeURL(@"?y");
    testRelativeURL(@"g?y");
    testRelativeURL(@"g?y/./x");
    testRelativeURL(@"#s");
    testRelativeURL(@"g#s");
    testRelativeURL(@"g#s/./x");
    testRelativeURL(@"g?y#s");
    testRelativeURL(@";x");
    testRelativeURL(@"g;x");
    testRelativeURL(@"g;x?y#s");
    testRelativeURL(@".");
    testRelativeURL(@"./");
    testRelativeURL(@"..");
    testRelativeURL(@"../");
    testRelativeURL(@"../g");
    testRelativeURL(@"../..");
    testRelativeURL(@"../../");
    testRelativeURL(@"../../g");

    testRelativeURL(@"");
    testRelativeURL(@"../../../g");
    testRelativeURL(@"../../../../g");
    testRelativeURL(@"/./g");
    testRelativeURL(@"/../g");
    testRelativeURL(@"g.");
    testRelativeURL(@".g");
    testRelativeURL(@"g..");
    testRelativeURL(@"..g");
    testRelativeURL(@"./../g");
    testRelativeURL(@"./g/.");
    testRelativeURL(@"g/./h");
    testRelativeURL(@"g/../h");
    testRelativeURL(@"http:g");
    testRelativeURL(@"http:");

    exit(1);
}

@end
#endif
