/**
 * OpenAmp Console Controller
 *
 * Wilfredo Sanchez | wsanchez@apple.com
 * September 26, 1998
 **/

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#include "common.h"
#include "decoder.h"

#import "ConsoleController.h"
#import "AudioFile.h"
#import "SoundStream.h"
#import "SoundDecoder.h"

#ifndef DEBUG
#define DEBUG 0
#endif

#if 1 || DEBUG
#define MYLOCK_LOCK   [myDecodingLock lock  ]; NSLog(@"%s Locked decoder"  , __PRETTY_FUNCTION__);
#define MYLOCK_UNLOCK [myDecodingLock unlock]; NSLog(@"%s Unlocked decoder", __PRETTY_FUNCTION__);
#else
#define MYLOCK_LOCK   [myDecodingLock lock  ];
#define MYLOCK_UNLOCK [myDecodingLock unlock];
#endif

@implementation SoundDecoder

/**
 * Inits
 **/

- (id) initWithOwner: (ConsoleController*) anOwner
{
    if ((self = [super init]))
      {
        myOwner        = [anOwner retain];
        myDecodingLock = [[NSLock alloc] init];

        play    = 0;
        decoder = self;
      }
    return self;
}

- (void) dealloc
{
    [ myDecodingLock release ];
    [ myOwner        release ];

    [super dealloc];
}

/**
 * Actions
 **/

- (void) play: (AudioFile*) anAudioFile
{
    if (anAudioFile)
      {
        NSAutoreleasePool* aPool = [[NSAutoreleasePool alloc] init];

        play = 1;

        MYLOCK_LOCK;

        while (1)
          {
            Arguments.topSb                = 0;
            Arguments.forkoff              = 0;
            Arguments.need_aiff            = FALSE;
            Arguments.need_esps            = FALSE;
            Arguments.write_to_file        = FALSE;
#if DEBUG
            Arguments.verbose              = TRUE;
#else
            Arguments.verbose              = FALSE;
#endif
            Arguments.encoded_file_name[0] = '\0';
            Arguments.decoded_file_name[0] = '\0';

            strcpy(Arguments.encoded_file_name, [[anAudioFile filename] cString]);

            mpeg3play(&Arguments);

            [myOwner setProgressIsIndeterminate: YES];

            if (play)
              {
                AudioFile* aTrack = [myOwner currentTrack];

                if (aTrack == anAudioFile)
                    anAudioFile = [myOwner nextTrack];
                else
                    anAudioFile = aTrack;

                if (!anAudioFile) break;
              }
            else
                break;
          }

        MYLOCK_UNLOCK;

        play = 0;

        [myOwner donePlaying];

        [aPool release];
      }
}

- (void) stop
{
    play = 0;
    [audiofd abort];
}

- (void) pause
{
    [audiofd pause];
}

- (void) resume
{
    [audiofd resume];
}

- (void) wait
{
    MYLOCK_LOCK;
    MYLOCK_UNLOCK;
}

/**
 * Stream messages
 **/

- (void) done
{
    [myOwner setProgressIsIndeterminate: NO];
}

- (void) soundStream: (SoundStream*) aStream
      finishedBuffer: (int         ) aTag
             ofTotal: (int         ) aTotal
{
    [myOwner updateProgressWithValue: (double)aTag
                                 max: (double)aTotal];
}

@end
