/*
 * Copyright 2009 Vasilkin Andrew <digi@os2.snc.ru>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "resource.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

//#define RES_HARD_LOCKS		1
//#define PLAIN_VECTORS		1

#define TEST_CONFIG_INTERVAL		60000 // test mount points list each minute
#define MOUNTPOINT_FLUSH_TIMEOUT	10000 // flush & clear "dirty" flag each 10 sec.
#define MOUNT_POINT_NAME_MAX_LEN	64
#define LQTREE_PIPE_NAME		"\\PIPE\\LQTREE"
#define EA_UNIT_ALLOC			"LQ.BYTEALLOC"
#define MUTEX_TIMEOUT			1500
#define FIND_BUF_MAX_ITEMS		1 // 24

#define ID_STOP_SEM			0
#define ID_SCAN_SEM			1
#define ID_CONF_SEM			2
#define ID_TESTCONFFILE_SEM		3

#define MPFL_DIRTY			1
#define MPFL_CHANGED			2
#define MPFL_NOT_ACTUAL			4

typedef struct _MOUNTPOINT {
  PSZ			pszName;
  ULONG			ulNameLen;
  HMTX			hmtxLock;
  ULONG			ulFlags;
  ULONG			ulWriteRequests;
  LLONG			llLimit;
  PSZ			pszPath;
  ULONG			ulPathLen;

  ULONG			ulSectorUnit;
  ULONG			ulByteSector;
  LLONG		        llByteAlloc;
  ULONG			ulChangeTimeStamp;
} MOUNTPOINT, *PMOUNTPOINT;

typedef PMOUNTPOINT *PPMOUNTPOINT;

typedef struct _RESFILE {
  HFILE		hFile;
  PMOUNTPOINT	psMountPoint;
} RESFILE, *PRESFILE;

typedef struct _RES {
  struct _RES		*psNext;
  struct _RES		**ppsSelf;
  PSZ			pszCfgFile;

  HMTX			hmtxMPList;
  HEV			hevMPList;
  ULONG			ulMPListReadLocks;
  PPMOUNTPOINT		ppsMPList;
  ULONG			ulMPListCount;

  HMTX			hmtxFileList;
  PRESFILE		psFileList;
  ULONG			ulFileListCount;

  ULONG			ulMountCount;
  TID			tidFlush;
  HEV			hevStop;
  HEV			hevScan;
  HEV			hevConf;
  HEV			hevTestConfFile;
  HMUX			hmuxThreadEvents;
  ULONG			ulFlags;
  FILESTATUS3L		sCfgFileStat;
  HTIMER		htimerTestConfFile;

  LONG			lConnectionsCount;
} RES;

typedef struct _CALCDIRSIZE {
  HMUX		hmuxCancelEvents;
  LLONG		llBytes;
  BOOL		fCancel;
  CHAR		acPath[CCHMAXPATH];
} CALCDIRSIZE, *PCALCDIRSIZE;

typedef struct _SCANDIR {
  HDIR			hDir;
  ULONG			ulCount;
  FILEFINDBUF3L		sFindBuf[FIND_BUF_MAX_ITEMS];
} SCANDIR, *PSCANDIR;

typedef LONG FNBINCOMP(PVOID pKey, PVOID pItem);
typedef FNBINCOMP *PFNBINCOMP;

static HMTX		hmtxResList;
static PRES		psResList = NULL;
static PRES		*ppsResLast = &psResList;
static LQPIPE		sLQPipe;


// Utils
// -----
//
// static PCHAR _CopyQuoteString(PCHAR pcString, PCHAR pcBuf, ULONG ulBufLen)
//
//   Description:
//	Copy quoted or not quoted string (up to space) from pcString to pcBuf
//	up to ulBufLen byes. A null character is placed at the end of the
//	generated character string.
//   Returns:
//      a pointer to the start of the next element
//
// static BOOL _BinSearch(PVOID pArray, ULONG ulCount, ULONG ulItemSize,
//                        PFNBINCOMP fnBinComp, PVOID pKey, PULONG pulIndex)
//
//   Description:
//	The _BinSearch function performs a search of a sorted array of ulCount
//	elements, which is pointed to by pArray, for an item which matches the
//	object pointed to by pKey. Each element in the array is ulItemSize
//	bytes in size. The comparison function pointed to by fnBinComp is
//	called with two arguments. One of the arguments to the fnBinComp
//	function will be pKey, and the other will be an array element.
//	The comparison function shall return an integer less than, equal to, or
//	greater than zero if the pKey object is less than, equal to, or greater
//	than the element in the array. 
//   Returns:
//      The return value is FALSE if there no element found, pulIndex -
//	index where element (corresponding pKey) must be placed. The return
//	value is TRUE if element found and pulIndex will be index of that
//	element.
//
// static BOOL _BinInsert(PVOID *ppArray, PULONG pulCount, ULONG ulItemSize,
//                        ULONG ulIndex, PVOID pItem)
//
// static BOOL _BinDelete(PVOID *ppArray, PULONG pulCount, ULONG ulItemSize,
//                        ULONG ulIndex)
//
// static VOID _CalcDirSize(PCALCDIRSIZE psDir)
//
//   Description:
//	The _CalcDirSize function performs a calculate directory size (recurse).
//	psDir input:
//	  psDir->hmuxCancelEvents - must be initialized multiple wait semaphore,
//	  psDir->llBytes - must be zero,
//	  psDir->fCancel - must be FALSE,
//	  psDir->acPath - path of direcotory.
//	psDir output:
//	  psDir->llBytes - directory size or -1 if error,
//	  psDir->fCancel - TRUE if error or canceled by semaphore.

static PCHAR _CopyQuoteString(PCHAR pcString, PCHAR pcBuf, ULONG ulBufLen)
{
  BOOL		fEscape = FALSE;

  if ( pcString == NULL )
    return NULL;

  if ( ulBufLen == 0 )
    return pcString;

  if ( *pcString != '"' )
  {
    PCHAR	pcEnd = pcString + strcspn( pcString, " \t\n" );
    ULONG	ulLength;

    if ( pcEnd == NULL )
    {
      pcEnd = strchr( pcString, '\0' );
    }

    ulLength = min( pcEnd - pcString, ulBufLen - 1 );
    memcpy( pcBuf, pcString, ulLength );
    pcBuf += ulLength;
    pcString = pcEnd;
  }
  else
  {
    for( pcString++; ( *pcString != '\n' ) && ( *pcString != '\0' ); )
    {
      if ( !fEscape )
      {
        if ( *pcString == '\\' )
        {
          pcString++;
          fEscape = TRUE;
          continue;
        }

        if ( *pcString == '"' )
        {
          pcString++;
          break;
        }
      }
      else
        fEscape = FALSE;

      if ( ulBufLen > 1 )
      {
        *(pcBuf++) = ( *pcString == '/' ) ? '\\' : *pcString;
        ulBufLen--;
      }

      pcString++;
    }
  }

  *pcBuf = '\0';

  while( isspace( *pcString ) )
    pcString++;

  return pcString;
}


static BOOL _BinSearch(PVOID pArray, ULONG ulCount, ULONG ulItemSize,
                       PFNBINCOMP fnBinComp, PVOID pKey, PULONG pulIndex)
{
  BOOL		fResult;
  LONG		lIdx, lCmp;
  PCHAR		pcArray = (PCHAR)pArray;
#ifndef PLAIN_VECTORS
  LONG		lLeft, lRight;

  lLeft = 0;
  if ( ulCount != 0 )
  {
    lRight = ulCount - 1;
    lCmp = fnBinComp( pKey, &pcArray[lRight * ulItemSize] );
    if ( lCmp > 0 )
    {
      lLeft = ulCount;
    }
    if ( lCmp == 0 )
    {
      lLeft = lRight;
    }
    else
    {
      lRight--;
      while( lLeft <= lRight )
      {
        lIdx = ( lLeft + lRight ) / 2;
        lCmp = fnBinComp( pKey, &pcArray[lIdx * ulItemSize] );
        if ( lCmp < 0 )
        {
          lRight = lIdx - 1;
        }
        else if ( lCmp > 0 )
        {
          lLeft = lIdx + 1;
        }
        else
        {
          lLeft = lIdx;
          break;
        }
      }
    }

    *pulIndex = lLeft;
    fResult = ( lCmp == 0 );
  }
  else
  {
    fResult = FALSE;
    *pulIndex = 0;
  }
#else
  fResult = FALSE;
  *pulIndex = ulCount;

  for( lIdx = 0; lIdx < ulCount; lIdx++ )
  {
    lCmp = fnBinComp( pKey, &pcArray[lIdx * ulItemSize] );
    if ( lCmp <= 0 )
    {
      fResult = ( lCmp == 0 );
      *pulIndex = lIdx;
      break;
    }
  }
#endif

  return fResult;
}

static BOOL _BinInsert(PVOID *ppArray, PULONG pulCount, ULONG ulItemSize,
                       ULONG ulIndex, PVOID pItem)
{
  volatile PCHAR	pcArray = *(PCHAR *)ppArray;
  volatile ULONG	ulCount = *pulCount;

  if ( ulIndex > ulCount )
  {
    debug( "Invalid index value (%u, must be less or equal %u)", ulIndex, ulCount );
    return FALSE;
  }

#ifndef PLAIN_VECTORS
  if ( ( pcArray == NULL ) || ( (ulCount & 0x0FF) == 0x0FF ) )
  {
    pcArray = realloc( pcArray,
                       ( ((ulCount + 1) & 0xFFFFFF00) + 0x0FF ) * ulItemSize );
    if ( pcArray == NULL )
    {
      debug( "Not enought memory" );
      return FALSE;
    }

    *ppArray = pcArray;
  }
#else
  if ( pcArray == NULL )
  {
    pcArray = malloc( 255 * ulItemSize );
    if ( pcArray == NULL )
    {
      debug( "Not enought memory" );
      return FALSE;
    }

    *ppArray = pcArray;
  }
  else if ( ulCount == 255 )
  {
    debug( "Too many items" );
    return FALSE;
  }
#endif

  memmove( &pcArray[(ulIndex + 1) * ulItemSize], &pcArray[ulIndex * ulItemSize],
           (ulCount - ulIndex) * ulItemSize );
  memcpy( &pcArray[ulIndex * ulItemSize], pItem, ulItemSize );

  *pulCount = ulCount + 1;

  return TRUE;
}

static BOOL _BinDelete(PVOID *ppArray, PULONG pulCount, ULONG ulItemSize,
                       ULONG ulIndex)
{
  volatile PCHAR	pcArray = *(PCHAR *)ppArray;
  volatile ULONG	ulCount = *pulCount;

  if ( ulIndex >= ulCount )
  {
    debug( "Invalid index value (%u, must be less %u)", ulIndex, ulCount );
    return FALSE;
  }

#ifndef PLAIN_VECTORS
  if ( (ulCount & 0x0FF) == 0x00 )
  {
    PCHAR	pcArrayNew;

    pcArrayNew = malloc( ( ulCount - 1 ) * ulItemSize );
    if ( pcArrayNew == NULL )
    {
      debug( "Not enought memory" );
      return FALSE;
    }

    memcpy( pcArrayNew, pcArray, ulIndex * ulItemSize );
    memcpy( &pcArrayNew[ulIndex * ulItemSize],
            &pcArray[( ulIndex + 1 ) * ulItemSize],
            (ulCount - ulIndex - 1) * ulItemSize );

    free( pcArray );
    *ppArray = pcArrayNew;
  }
  else if ( ulIndex < ( ulCount - 1 ) )
  {
    memcpy( &pcArray[ulIndex * ulItemSize],
            &pcArray[ ( ulIndex + 1 ) * ulItemSize],
            (ulCount - ulIndex - 1) * ulItemSize );
  }
#else
  memcpy( &pcArray[ulIndex * ulItemSize],
          &pcArray[ ( ulIndex + 1 ) * ulItemSize],
          (ulCount - ulIndex - 1) * ulItemSize );
#endif

  *pulCount = ulCount - 1;
  return TRUE;
}

static VOID _CalcDirSize(PCALCDIRSIZE psDir)
{
  ULONG			ulRC;
  ULONG			ulPathLength;
  PFILEFINDBUF3L	psFindItem;
  PSCANDIR		psScanDir;

  ulRC = DosWaitMuxWaitSem( psDir->hmuxCancelEvents, SEM_IMMEDIATE_RETURN,
                            &ulPathLength );
  if ( ulRC == NO_ERROR )
  {
    debug( "Cancel. Semaphore ID: %u", ulPathLength );
    psDir->fCancel = TRUE;
    return;
  }
  else if ( ulRC != ERROR_TIMEOUT )
  {
    debug( "DosWaitMuxWaitSem(), rc = %u", ulRC );
    psDir->llBytes = -1;
    psDir->fCancel = TRUE;
    return;
  }

  psScanDir = malloc( sizeof(SCANDIR) );
  if ( psScanDir == NULL )
  {
    debug( "Not enought memory" );
    psDir->llBytes = -1;
    psDir->fCancel = TRUE;
    return;
  }
/*
  ulRC = DosAllocMem( (PVOID *)&psScanDir, sizeof(SCANDIR), fPERM | PAG_COMMIT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosAllocMem(), rc = %u", ulRC );
    psDir->llBytes = -1;
    psDir->fCancel = TRUE;
    return;
  }
*/

  ulPathLength = strlen( &psDir->acPath );
  strcpy( &psDir->acPath[ulPathLength], "\\*" );
  ulPathLength++;

  psScanDir->hDir = HDIR_CREATE;
  psScanDir->ulCount = FIND_BUF_MAX_ITEMS;
  ulRC = DosFindFirst( &psDir->acPath, &psScanDir->hDir,
                       FILE_ARCHIVED | FILE_DIRECTORY | FILE_SYSTEM |
                       FILE_HIDDEN | FILE_READONLY,
                       &psScanDir->sFindBuf,
                       FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L),
                       &psScanDir->ulCount, FIL_STANDARDL);

  if ( ( ulRC != NO_ERROR ) && ( ulRC != ERROR_NO_MORE_FILES  ) )
  {
    debug( "DosFindFirst(), rc = %u", ulRC );
  }

  while( ulRC == NO_ERROR )
  {
    for( psFindItem = &psScanDir->sFindBuf; ;
         psFindItem = (PFILEFINDBUF3L) ( ((PCHAR)psFindItem) + psFindItem->oNextEntryOffset )
       )
    {
      if ( (psFindItem->attrFile & FILE_DIRECTORY) != 0 )
      {
        if ( *(PUSHORT)&psFindItem->achName != 0x002E 
             && ( strcmp( psFindItem->achName, ".." ) != 0 ) )
        {
          strcpy( &psDir->acPath[ulPathLength], psFindItem->achName );
          _CalcDirSize( psDir );
          if ( psDir->fCancel )
            break;
        }
      }
      else
        psDir->llBytes += *(PLLONG)&psFindItem->cbFileAlloc;

      if ( psFindItem->oNextEntryOffset == 0 )
        break;
    }

    if ( psDir->fCancel )
      break;

    psScanDir->ulCount = FIND_BUF_MAX_ITEMS;

    ulRC = DosFindNext( psScanDir->hDir, &psScanDir->sFindBuf,
                        FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L),
                        &psScanDir->ulCount );
    if ( ( ulRC != NO_ERROR ) && ( ulRC != ERROR_NO_MORE_FILES ) )
    {
      debug( "DosFindNext(), rc = %u", ulRC );
    }
  }

  if ( ulRC != ERROR_NO_MORE_FILES )
  {
    debug( "Canceled" );
    psDir->llBytes = -1;
    psDir->fCancel = TRUE;
  }

  ulRC = DosFindClose( psScanDir->hDir );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosFindClose(), rc = %u", ulRC );
  }

  free( psScanDir );
/*
  ulRC = DosFreeMem( psScanDir );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosFreeMem(), rc = %u", ulRC );
  }
*/
}

static BOOL _IsSubDir(PSZ pszPath)
{
  return strchr( pszPath, '\\' ) != NULL;
}

// Mount point's functions
// -----------------------
//
// static ULONG _resCreateMountPoint(PPMOUNTPOINT ppsMountPoint, PSZ pszName,
//                                   LLONG llLimit, PSZ pszPath)
//
//   Description:
//      ppsMountPoint is a pointer to an PMOUNTPOINT where the allocated block
//      of memory for the mount point is returned. 
//   Returns:
//      NO_ERROR
//      ERROR_INVALID_PARAMETER - invalid name of mount point (pszName)
//      ERROR_PATH_NOT_FOUND - invalid local path (pszPath)
//      ERROR_NOT_ENOUGH_MEMORY - out of memory
//
// static VOID _resDestroyMountPoint(PMOUNTPOINT psMountPoint)
//
//   Description:
//	Deallocates mount point.
//
// static VOID _resMountPointFlag(PMOUNTPOINT psMountPoint, ULONG ulFlag,
//                                BOOL fSet)
//
//   Description:
//	fSet is TRUE: set flag ulFlag for psMountPoint
//	fSet is FALSE: clear flag ulFlag for psMountPoint
//
// static BOOL _resTestMountPointFlag(PMOUNTPOINT psMountPoint, ULONG ulFlag)
//
// static LLONG _resGetMountPointBytesFree(PMOUNTPOINT psMountPoint)
//
// static VOID _resSetMountPointLimit(PMOUNTPOINT psMountPoint, LLONG llLimit)
//
// static BOOL _resRequestMountPointWrite(PMOUNTPOINT psMountPoint)
//
//   Description:
//	Requests write operation for mount point.
//   Returns:
//      The return value is TRUE if write operation allowed (m.p. not "dirty").
//
// static VOID _resReleaseMountPointWrite(PMOUNTPOINT psMountPoint)
//
//   Description:
//	Relinquishes write operation for mount point.
//
// static VOID _resIncMountPointBytesAlloc(PMOUNTPOINT psMountPoint,
//                                         LLONG llBytes)


static ULONG _resCreateMountPoint(PPMOUNTPOINT ppsMountPoint, PSZ pszName,
                                  LLONG llLimit, PSZ pszPath)
{
  PMOUNTPOINT	psMountPoint;
  FSALLOCATE	sFSAlloc;
  ULONG		ulRC;
  ULONG		ulDiskNum;
  ULONG		ulPathLen = pszPath == NULL ? 0 : strlen( pszPath );
  LLONG		llByteAlloc = -1;

  if ( ( pszName == NULL ) || ( *pszName == '\0' ) ||
       ( pszName[strcspn( pszName, "\t*?/\\" )] != '\0' ) )
  {
    debug( "Invalid name: %s", pszName );
    return ERROR_INVALID_PARAMETER;
  }

  if ( ulPathLen < 4 )
  {
    debug( "Invalid parameter: local path too short" );
    return ERROR_PATH_NOT_FOUND;
  }

  ulDiskNum = pszPath[0];
  ulDiskNum = ulDiskNum >= 'a' ? ulDiskNum - 'a' : ulDiskNum - 'A';

  if ( ( ulDiskNum > ('z'-'a') ) || ( pszPath[1] != ':' ) )
  {
    debug( "Invalid parameter: invalid path" );
    return ERROR_PATH_NOT_FOUND;
  }

  if ( pszPath[ulPathLen - 1] == '\\' )
    ulPathLen--;

  ulRC = lqlibEAQueryLLong( pszPath, EA_UNIT_ALLOC, &llByteAlloc );
  if ( ulRC != NO_ERROR )
  {
    debug( "Cannot read allocated space from EA. lqlibEAQueryLLong(), rc = %u",
           ulRC );
    return ulRC;
  }

  psMountPoint = (PMOUNTPOINT)calloc( 1, sizeof(MOUNTPOINT) );
  if ( psMountPoint == NULL )
  {
    return ERROR_NOT_ENOUGH_MEMORY;
  }

  psMountPoint->pszName = strdup( pszName );
  if ( psMountPoint->pszName == NULL )
  {
    free( psMountPoint );
    return ERROR_NOT_ENOUGH_MEMORY;
  }

  psMountPoint->pszPath = malloc( ulPathLen + 1 );
  if ( psMountPoint->pszPath == NULL )
  {
    free( psMountPoint->pszName );
    free( psMountPoint );
    return ERROR_NOT_ENOUGH_MEMORY;
  }

  memcpy( psMountPoint->pszPath, pszPath, ulPathLen );
  psMountPoint->pszPath[ulPathLen] = '\0';

  psMountPoint->ulNameLen = strlen( pszName );
  psMountPoint->ulPathLen = ulPathLen;

  ulRC = DosQueryFSInfo( ulDiskNum + 1, FSIL_ALLOC, &sFSAlloc,
                         sizeof(FSALLOCATE) );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosQueryFSInfo(), rc = %u. Fake values are used.", ulRC );
    psMountPoint->ulSectorUnit	= 4;
    psMountPoint->ulByteSector	= 512;
  }
  else
  {
    psMountPoint->ulSectorUnit	= sFSAlloc.cSectorUnit;
    psMountPoint->ulByteSector	= sFSAlloc.cbSector;
  }

  psMountPoint->llLimit = ( llLimit < 0 ) ? (LLONG)(-1) : llLimit;
  psMountPoint->ulWriteRequests = 0;
  psMountPoint->llByteAlloc = llByteAlloc;
  if ( llByteAlloc < 0 )
  {
//dbg    debug( "Mount point \"%s\" (%s) is \"dirty\"", pszName, pszPath );
    psMountPoint->ulFlags = MPFL_DIRTY;
  }
//  else
//    psMountPoint->ulFlags = 0;

  DosCreateMutexSem( NULL, &psMountPoint->hmtxLock, 0, FALSE );

  *ppsMountPoint = psMountPoint;

  return NO_ERROR;
}

static VOID _resDestroyMountPoint(PMOUNTPOINT psMountPoint)
{
  if ( psMountPoint->pszName != NULL )
  {
    free( psMountPoint->pszName );
    psMountPoint->pszName = NULL;
  }

  if ( psMountPoint->pszPath != NULL )
  {
    free( psMountPoint->pszPath );
    psMountPoint->pszPath = NULL;
  }

  DosCloseMutexSem( psMountPoint->hmtxLock );
  free( psMountPoint );
}

static VOID _resMountPointFlag(PMOUNTPOINT psMountPoint, ULONG ulFlag,
                               BOOL fSet)
{
  ULONG		ulRC;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    if ( fSet )
      psMountPoint->ulFlags |= ulFlag;
    else
      psMountPoint->ulFlags &= ~ulFlag;

    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }
}

static BOOL _resTestMountPointFlag(PMOUNTPOINT psMountPoint, ULONG ulFlag)
{
  ULONG		ulRC;
  BOOL		fResult;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
    fResult = FALSE;
  }
  else
  {
    fResult = ( (psMountPoint->ulFlags & ulFlag) != 0 );
    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }

  return fResult;
}

static LLONG _resGetMountPointBytesFree(PMOUNTPOINT psMountPoint)
{
  volatile LLONG	llByteFree;
  ULONG			ulRC;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
    llByteFree = 0;
  }
  else
  {
    llByteFree = psMountPoint->llLimit - psMountPoint->llByteAlloc;
    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }

  return llByteFree;
}

static VOID _resSetMountPointLimit(PMOUNTPOINT psMountPoint, LLONG llLimit)
{
  ULONG			ulRC;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
//    llByteFree = 0;
  }
  else
  {
    psMountPoint->llLimit = llLimit;
    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }
}

static BOOL _resRequestMountPointWrite(PMOUNTPOINT psMountPoint)
{
  ULONG		ulRC;
  BOOL		fResult;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
    fResult = FALSE;
  }
  else
  {
    fResult = ( (psMountPoint->ulFlags & MPFL_DIRTY) == 0 );
    if ( fResult )
      psMountPoint->ulWriteRequests++;

    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }

  return fResult;
}

static VOID _resReleaseMountPointWrite(PMOUNTPOINT psMountPoint)
{
  ULONG		ulRC;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    if ( psMountPoint->ulWriteRequests > 0 )
      psMountPoint->ulWriteRequests--;

    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }
}

static VOID _resIncMountPointBytesAlloc(PMOUNTPOINT psMountPoint,
                                        LLONG llBytes)
{
  ULONG		ulRC;

  if ( llBytes == 0 )
    return;

  ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    psMountPoint->llByteAlloc += llBytes;
    if ( psMountPoint->llByteAlloc < 0 )
      psMountPoint->llByteAlloc = 0;

//dbg    debug( "New allocated space for \"%s\" (add %lld): %llu",
//dbg           psMountPoint->pszName, llBytes, psMountPoint->llByteAlloc );

    if ( (psMountPoint->ulFlags & MPFL_CHANGED) == 0 )
    {
      psMountPoint->ulFlags |= MPFL_CHANGED;
      DosQuerySysInfo( QSV_MS_COUNT, QSV_MS_COUNT,
                       &psMountPoint->ulChangeTimeStamp, sizeof(ULONG) );
//dbg      debug( "Set time stamp: %u", psMountPoint->ulChangeTimeStamp );

      ulRC = lqlibEASetLLong( psMountPoint->pszPath, EA_UNIT_ALLOC, (LLONG)(-1) );
      if ( ulRC != NO_ERROR )
        debug( "lqlibEASetLLong(), rc = %u", ulRC );
    }

    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }
}


// Resource's functions
// --------------------
//
// static BOOL _resSearchMountPoint(PRES psRes, PSZ pszName, PULONG pulIndex)
// static BOOL _resSearchMountPoint2(PRES psRes, PCHAR pcName, ULONG ulNameLen,
//                                   PULONG pulIndex)
//
//   Description:
//	Search the mount point with name pszName.
//   Returns:
//      The return value is FALSE if there no mount point found, pulIndex -
//	index where mount point must be placed. The return value is TRUE if
//      mount point found and pulIndex will be index of that mount point.
//
// static BOOL _resInsertMountPoint(PRES psRes, ULONG ulIndex,
//                                  PMOUNTPOINT psMountPoint)
//
//   Description:
//	Insert a mount point psMountPoint in the psRes at the position ulIndex.
//   Returns:
//	Functions return FALSE if there is insufficient memory available.
//
// static VOID _resClearMountPoints(PRES psRes)
//
//   Description:
//	Remove all mount points from psRes, close opened for writing files.
//
// static VOID _resDeleteMountPoint(PRES psRes, ULONG ulIndex)
//
//   Description:
//	Remove mount points specified by ulIndex, close opened for writing
//	files on this mount points.
//
// static VOID _resReconfig(PRES psRes)
//
//   Description:
//	Read config-file and rebuild list of mountpoints.
//
// static ULONG _resLockWrite(PRES psRes)
//
//   Description:
//	Lock the resource pointed to by psRes for rebuild mount point's list.
//	Any other _resLockWrite() and _resLockRead() calls will blocks the
//	calling thread unless _resUnlockWrite().
//   Returns:
//	System's API error code.
//
// static VOID _resUnlockWrite(PRES psRes)
//
//   Description:
//	Relinquishes lock the psRes.
//
// static ULONG _resLockRead(PRES psRes)
//
//   Description:
//	Requests the resource pointed to by psRes to prevent rebuild mount
//	point's list. Any _resLockWrite() (but not _resLockRead()) calls will
//	blocks the calling thread unless _resUnlockRead().
//   Returns:
//	System's API error code.
//
// static VOID _resUnlockRead(PRES psRes)
//
//   Description:
//	Relinquishes lock the psRes.
//
// static VOID _resDirtyMountPonts(PRES psRes)
//
//   Description:
//	Set "dirty" flag for all mount points.
//
// static VOID _resScanLR(PRES psRes, BOOL fFlush)
//
//   Description:
//	Calc allocated space for each "dirty" mount point. Store allocated
//	units counter to EA. If fFlush is TRUE: do not calc allocated space,
//	store allocated space values to EA immediately.
//
// static VOID _resStoreFileHandle(PRES psRes, HFILE hFile,
//                                 PMOUNTPOINT psMountPoint)
//
//   Description:
//	Stores the handle of file and mount point's pointer pair in resource's
//	internal list.
//
// static VOID _resRemoveFileHandle(PRES psRes, HFILE hFile)
//
//   Description:
//	Rmoves the handle of file from internal list.
//
// static PMOUNTPOINT _resSearchFileHandle(PRES psRes, HFILE hFile)
//
//   Description:
//	Searches the mount point for handle of file specified by a hFile
//	argument.
//   Returns:
//	The _resSearchFileHandle function returns a pointer to the mount point,
//	or NULL if the hFile was not stored for resource.

static LONG __cbMountPointNameComp(PVOID pKey, PVOID pItem)
{
  return stricmp( (PSZ)pKey, (*((PPMOUNTPOINT)pItem))->pszName );
}

static BOOL _resSearchMountPoint(PRES psRes, PSZ pszName, PULONG pulIndex)
{
  return _BinSearch( psRes->ppsMPList, psRes->ulMPListCount, sizeof(PMOUNTPOINT),
                     __cbMountPointNameComp, pszName, pulIndex );
}

static BOOL _resSearchMountPoint2(PRES psRes, PCHAR pcName, ULONG ulNameLen,
                                  PULONG pulIndex)
{
  CHAR		acName[MOUNT_POINT_NAME_MAX_LEN];

  if ( ulNameLen >= ( sizeof(acName) - 1 ) )
  {
    debug( "Too long name of mount point in: %s", pcName );
    return FALSE;
  }

  memcpy( &acName, pcName, ulNameLen );
  acName[ulNameLen] = '\0';

  return _resSearchMountPoint( psRes, &acName, pulIndex );
}

static BOOL _resInsertMountPoint(PRES psRes, ULONG ulIndex,
                                 PMOUNTPOINT psMountPoint)
{
  return _BinInsert( (PVOID *)&psRes->ppsMPList, &psRes->ulMPListCount,
                     sizeof(PMOUNTPOINT), ulIndex, &psMountPoint );
}

static VOID _resClearMountPoints(PRES psRes)
{
  ULONG			ulIdx;
  PRESFILE		psFile;
  ULONG			ulRC;

//dbg  debug( "[%p] destroy %u mount points, close %u file(s)",
//dbg         psRes, psRes->ulMPListCount, psRes->ulFileListCount );

  ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    psFile = psRes->psFileList;
    for( ulIdx = 0; ulIdx < psRes->ulFileListCount; ulIdx++ )
    {
      DosClose( psFile->hFile );
      psFile++;
    }

    if ( psRes->psFileList != NULL )
      free( psRes->psFileList );

    psRes->psFileList = NULL;
    psRes->ulFileListCount = 0;

    DosReleaseMutexSem( psRes->hmtxFileList );
  }

  for( ulIdx = 0; ulIdx < psRes->ulMPListCount; ulIdx++ )
    _resDestroyMountPoint( psRes->ppsMPList[ulIdx] );

  free( psRes->ppsMPList );
  psRes->ppsMPList = NULL;
  psRes->ulMPListCount = 0;
}

static VOID _resDeleteMountPoint(PRES psRes, ULONG ulIndex)
{
  PMOUNTPOINT	psMountPoint = psRes->ppsMPList[ulIndex];
  LONG		lIdx;
  PRESFILE	psFile;
  ULONG		ulRC;

  ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
    return;
  }

  for( lIdx = psRes->ulFileListCount - 1; lIdx >= 0; lIdx-- )
  {
    psFile = &psRes->psFileList[lIdx];
    if ( psFile->psMountPoint == psMountPoint )
    {
      DosClose( psFile->hFile );
      _BinDelete( (PVOID *)&psRes->psFileList, &psRes->ulFileListCount,
                  sizeof(RESFILE), lIdx );
    }
  }

  DosReleaseMutexSem( psRes->hmtxFileList );

  _resDestroyMountPoint( psMountPoint );
  _BinDelete( (PVOID *)&psRes->ppsMPList, &psRes->ulMPListCount,
               sizeof(PMOUNTPOINT), ulIndex );
}

static VOID _resReconfig(PRES psRes)
{
  FILE		*pFile;
  CHAR		acBuf[512];
  PCHAR		pcPtr;
  CHAR		acMountPoint[MOUNT_POINT_NAME_MAX_LEN];
  CHAR		acLimit[32];
  CHAR		acPath[320];
  LONG		lIndex, lPathScanIndex;
  BOOL		fPathInUse;
  PMOUNTPOINT	psMountPoint;
  LLONG		llLimit;
  ULONG		ulRC;

  for( lIndex = 0; lIndex < psRes->ulMPListCount; lIndex++ )
  {
    _resMountPointFlag( psRes->ppsMPList[lIndex], MPFL_NOT_ACTUAL, TRUE );
  }

  pFile = fopen( psRes->pszCfgFile, "r" );
  if ( pFile == NULL )
  {
    debug( "Cannot open config file \"%s\"", psRes->pszCfgFile );
  }
  else
  {
    while( fgets( &acBuf, sizeof(acBuf) - 1, pFile ) != NULL )
    {
      pcPtr = &acBuf;
      while( isspace( *pcPtr ) )
        pcPtr++;

      if ( ( *pcPtr == '\0' ) || ( *pcPtr == '#' ) )
        continue;

      _CopyQuoteString( 
        _CopyQuoteString( 
          _CopyQuoteString( pcPtr, &acMountPoint, sizeof(acMountPoint) ),
          &acLimit, sizeof(acLimit) ),
        &acPath, sizeof(acPath)
      );

      llLimit = atoll( &acLimit ) * 1024;

      if ( _resSearchMountPoint( psRes, &acMountPoint, (PULONG)&lIndex ) )
      {
        psMountPoint = psRes->ppsMPList[lIndex];
        if ( !_resTestMountPointFlag( psMountPoint, MPFL_NOT_ACTUAL ) )
        {
          debug( "Mount point \"%s\" ignored: already exists", &acMountPoint );
          continue;
        }

        if ( stricmp( &acPath, psMountPoint->pszPath ) == 0 )
        {
//dbg          debug( "Mount point \"%s\" exists, new limit: %lld", &acMountPoint,
//dbg                 llLimit );
          _resSetMountPointLimit( psMountPoint, llLimit );
          _resMountPointFlag( psMountPoint, MPFL_NOT_ACTUAL, FALSE );
          continue;
        }

//dbg        debug( "Mount point \"%s\" path changed", &acMountPoint );
        _resDeleteMountPoint( psRes, lIndex );
      }

      fPathInUse = FALSE;
      for( lPathScanIndex = 0; lPathScanIndex < psRes->ulMPListCount;
           lPathScanIndex++ )
      {
        psMountPoint = psRes->ppsMPList[lPathScanIndex];

        if ( !_resTestMountPointFlag( psMountPoint, MPFL_NOT_ACTUAL )
             && ( stricmp( psMountPoint->pszPath, &acPath ) == 0 ) )
        {
          debug( "Mount point \"%s\" ignored: the path \"%s\" was used in \"%s\"",
                 &acMountPoint, &acPath, psMountPoint->pszName );
          fPathInUse = TRUE;
          break;
        }
      }

      if ( !fPathInUse )
      {
        if ( _resCreateMountPoint( &psMountPoint, &acMountPoint, llLimit,
                                   &acPath ) != NO_ERROR )
        {
          debug( "Mount point \"%s\" ignored", &acMountPoint );
        }
        else if ( !_resInsertMountPoint( psRes, lIndex, psMountPoint ) )
        {
          debug( "Not enought memory (insert mount point)" );
          _resDestroyMountPoint( psMountPoint );
        }
      }
    }

    fclose( pFile );

    for( lIndex = psRes->ulMPListCount - 1; lIndex >= 0; lIndex-- )
    {
      psMountPoint = psRes->ppsMPList[lIndex];
      if ( !_resTestMountPointFlag( psMountPoint, MPFL_NOT_ACTUAL ) )
        continue;

      debug( "Delete mount point: \"%s\"", psMountPoint->pszName );
      _resDeleteMountPoint( psRes, lIndex );
    }

    ulRC = DosQueryPathInfo( psRes->pszCfgFile, FIL_STANDARDL,
                             &psRes->sCfgFileStat, sizeof(FILESTATUS3L) );
    if ( ulRC != NO_ERROR )
      debug( "DosQueryPathInfo(), rc = %u", ulRC );

#ifdef DEBUG_CODE
    if ( psRes->ulMPListCount == 0 )
    {
      debug( "No mount points!" );
    }
    else
    {
      debug( "There are %u mount points listed:", psRes->ulMPListCount );

      for( lIndex = 0; lIndex < psRes->ulMPListCount; lIndex++ )
      {
        psMountPoint = psRes->ppsMPList[lIndex];
        debug( "  #%u \"%s\", limit: %lld, allocated: %lld", lIndex,
               psMountPoint->pszName, psMountPoint->llLimit,
               psMountPoint->llByteAlloc );
        psMountPoint++;
      }
    }
#endif
  }
}

static ULONG _resLockWrite(PRES psRes)
{
  ULONG		ulRC;
#ifndef RES_HARD_LOCKS
  ULONG		ulCount;

  for( ; ; )
  {
    ulRC = DosRequestMutexSem( psRes->hmtxMPList, SEM_INDEFINITE_WAIT );
    if ( ulRC != NO_ERROR )
    {
      debug( "DosRequestMutexSem(), rc = %u", ulRC );
      break;
    }

    if ( psRes->ulMPListReadLocks == 0 )
      break;

    DosReleaseMutexSem( psRes->hmtxMPList );

    ulRC = DosWaitEventSem( psRes->hevMPList, 200/*SEM_INDEFINITE_WAIT*/ );

    if ( ulRC == NO_ERROR )
    {
      DosResetEventSem( psRes->hevMPList, &ulCount );
    }
    else if ( ulRC != ERROR_TIMEOUT  )
    {
      debug( "DosWaitEventSem(), rc = %u", ulRC );
      break;
    }
  }
#else
  ulRC = DosRequestMutexSem( psRes->hmtxMPList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
#endif

  return ulRC;
}

static VOID _resUnlockWrite(PRES psRes)
{
#ifndef RES_HARD_LOCKS
  ULONG		ulCount;

  DosReleaseMutexSem( psRes->hmtxMPList );
  DosPostEventSem( psRes->hevMPList );
  DosResetEventSem( psRes->hevMPList, &ulCount );
#else
  DosReleaseMutexSem( psRes->hmtxMPList );
#endif
}

static ULONG _resLockRead(PRES psRes)
{
  ULONG		ulRC;

#ifndef RES_HARD_LOCKS
  ulRC = DosRequestMutexSem( psRes->hmtxMPList, SEM_INDEFINITE_WAIT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    psRes->ulMPListReadLocks++;
    DosReleaseMutexSem( psRes->hmtxMPList );
  }
#else
  ulRC = DosRequestMutexSem( psRes->hmtxMPList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
#endif

  return ulRC;
}

static VOID _resUnlockRead(PRES psRes)
{
#ifndef RES_HARD_LOCKS
  ULONG		ulCount;

  DosRequestMutexSem( psRes->hmtxMPList, SEM_INDEFINITE_WAIT );
  if ( psRes->ulMPListReadLocks > 0 )
  {
    psRes->ulMPListReadLocks--;
  }
  else if ( psRes->ulMPListReadLocks == 0 )
    debug( "unwanted call" );
  ulCount = psRes->ulMPListReadLocks;
  DosReleaseMutexSem( psRes->hmtxMPList );

  if ( ulCount == 0 )
  {
    DosPostEventSem( psRes->hevMPList );
    DosResetEventSem( psRes->hevMPList, &ulCount );
  }
#else
  DosReleaseMutexSem( psRes->hmtxMPList );
#endif
}

static VOID _resDirtyMountPonts(PRES psRes)
{
  ULONG			ulIdx;

  for( ulIdx = 0; ulIdx < psRes->ulMPListCount; ulIdx++ )
  {
    _resMountPointFlag( psRes->ppsMPList[ulIdx], MPFL_DIRTY, TRUE );
  }
}

static ULONG _resGetLocalPath(PRES psRes,
                              PSZ pszMountPointPath, ULONG ulMountPointPathLen,
                              PCHAR pcBuf, ULONG ulBufLen,
                              PMOUNTPOINT *ppsMountPoint)
{
  PCHAR		pcPathLocalPart;
  ULONG		ulMountPointNameLen;
  ULONG		ulPathLocalPartLen;
  BOOL		fFound;
  ULONG         ulIndex;
  PMOUNTPOINT	psMountPoint;

  pcPathLocalPart = memchr( pszMountPointPath, '\\', ulMountPointPathLen );
  if ( pcPathLocalPart != NULL )
  {
    ulMountPointNameLen = pcPathLocalPart - pszMountPointPath;
    ulPathLocalPartLen = ulMountPointPathLen - ulMountPointNameLen;
  }
  else
  {
    ulMountPointNameLen = ulMountPointPathLen;
    ulPathLocalPartLen = 0;
  }

  fFound = _resSearchMountPoint2( psRes, pszMountPointPath,
                                  ulMountPointNameLen, &ulIndex );
  if ( !fFound )
  {
//dbg    debug( "Mount point for \"%s\" not found", pszMountPointPath );
    return ERROR_PATH_NOT_FOUND;
  }

  if ( ( ulPathLocalPartLen != 0 ) &&
       ( pcPathLocalPart[ulPathLocalPartLen - 1] == '\\' ) )
  {
    ulPathLocalPartLen--;
  }

  psMountPoint = psRes->ppsMPList[ulIndex];
  ulIndex = psMountPoint->ulPathLen + ulPathLocalPartLen;
  if ( ulIndex >= ( ulBufLen - 1 ) )
  {
    debug( "Buffer too short: need %u bytes, have %u bytes of space",
            ulIndex + 1, ulBufLen );
    return ERROR_BUFFER_OVERFLOW;
  }

  memcpy( pcBuf, psMountPoint->pszPath, psMountPoint->ulPathLen );
  memcpy( &pcBuf[psMountPoint->ulPathLen], pcPathLocalPart,
          ulPathLocalPartLen );
  pcBuf[ulIndex] = '\0';

  if ( ppsMountPoint != NULL )
    *ppsMountPoint = psMountPoint;

  return NO_ERROR;
}

static ULONG _resGetLocalPathLR(PRES psRes, PSZ pszMountPointPath,
                                ULONG ulMountPointPathLen, PCHAR pcBuf,
                                ULONG ulBufLen)
{
  ULONG		ulRC = _resLockRead( psRes );

  if ( ulRC == NO_ERROR )
  {
    ulRC = _resGetLocalPath( psRes, pszMountPointPath, ulMountPointPathLen,
                             pcBuf, ulBufLen, NULL );
    _resUnlockRead( psRes );
  }

  return ulRC;
}

static BOOL _resCalcMountPointAlloc(PRES psRes, PMOUNTPOINT psMountPoint,
                                    PLLONG pllBytes)
{
  CALCDIRSIZE	sDir;
  SEMRECORD	asSemList[2];
  ULONG		ulRC;

  strcpy( &sDir.acPath, psMountPoint->pszPath );
  sDir.llBytes = 0;
  sDir.fCancel = FALSE;

  asSemList[0].hsemCur = (HSEM)psRes->hevStop;
  asSemList[0].ulUser = ID_STOP_SEM;
  asSemList[1].hsemCur = (HSEM)psRes->hevConf;
  asSemList[1].ulUser = ID_CONF_SEM;
  ulRC = DosCreateMuxWaitSem( NULL, &sDir.hmuxCancelEvents, 2,
                              &asSemList, DCMW_WAIT_ANY );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosCreateMuxWaitSem(), rc = %u", ulRC );
    sDir.fCancel = TRUE;
    *pllBytes = -1;
  }
  else
  {
    _CalcDirSize( &sDir );
    DosCloseMuxWaitSem( sDir.hmuxCancelEvents );
    *pllBytes = sDir.llBytes;
  }

  return !sDir.fCancel;
}

static VOID _resScanLR(PRES psRes, BOOL fFlush)
{
  ULONG		ulNow;
  PMOUNTPOINT	psMountPoint;
  ULONG		ulRC;
  ULONG		ulIdx;
  LLONG		llBytes;

  DosQuerySysInfo( QSV_MS_COUNT, QSV_MS_COUNT, &ulNow, sizeof(ULONG) );

  if ( _resLockRead( psRes ) != NO_ERROR )
    return;

  for( ulIdx = 0; ulIdx < psRes->ulMPListCount; ulIdx++ )
  {
    psMountPoint = psRes->ppsMPList[ulIdx];

    ulRC = DosRequestMutexSem( psMountPoint->hmtxLock,
                               fFlush ? MUTEX_TIMEOUT : SEM_IMMEDIATE_RETURN );
    if ( ulRC != NO_ERROR )
    {
      if ( fFlush || ( ulRC != ERROR_TIMEOUT ) )
        debug( "DosRequestMutexSem(), rc = %u", ulRC );

      continue;
    }

    if ( !fFlush && ( (psMountPoint->ulFlags & MPFL_DIRTY) != 0 ) )
    {
      DosReleaseMutexSem( psMountPoint->hmtxLock );

      if ( !_resCalcMountPointAlloc( psRes, psMountPoint, &llBytes ) )
      {
        if ( llBytes < 0 )
          continue;		// error - continue scan
        else
          break;		// event - stop scan
      }

      ulRC = DosRequestMutexSem( psMountPoint->hmtxLock, MUTEX_TIMEOUT );

      if ( ulRC != NO_ERROR )
      {
        debug( "DosRequestMutexSem(), rc = %u", ulRC );
      }
      else
      {
        psMountPoint->llByteAlloc = llBytes;
        psMountPoint->ulFlags &= ~(MPFL_DIRTY | MPFL_CHANGED);

        ulRC = lqlibEASetLLong( psMountPoint->pszPath, EA_UNIT_ALLOC, llBytes );
        if ( ulRC != NO_ERROR )
          debug( "lqlibEASetLLong(), rc = %u", ulRC );

        DosReleaseMutexSem( psMountPoint->hmtxLock );
      }

      continue;
    }

    if (
         ( (psMountPoint->ulFlags & (MPFL_CHANGED | MPFL_DIRTY)) ==
            MPFL_CHANGED )
       &&
         (
           fFlush ||
           ( (ulNow - psMountPoint->ulChangeTimeStamp) >
             MOUNTPOINT_FLUSH_TIMEOUT )
         )
       )
    {
      ulRC = lqlibEASetLLong( psMountPoint->pszPath, EA_UNIT_ALLOC, 
                              psMountPoint->llByteAlloc );
      if ( ulRC != NO_ERROR )
      {
        debug( "lqlibEASetLLong(), rc = %u", ulRC );
      }
      else
      {
        psMountPoint->ulFlags &= ~MPFL_CHANGED;
//dbg        debug( "ByteAlloc counter for \"%s\" stored to EA",
//dbg               psMountPoint->pszName );
      }
    }

    DosReleaseMutexSem( psMountPoint->hmtxLock );
  }

  _resUnlockRead( psRes );
}

static LONG __cbResFileComp(PVOID pKey, PVOID pItem)
{
  return (LONG)pKey - ((PRESFILE)pItem)->hFile;
}

static VOID _resStoreFileHandle(PRES psRes, HFILE hFile,
                                PMOUNTPOINT psMountPoint)
{
  BOOL		fFind;
  ULONG		ulRC;
  ULONG		ulIndex;
  RESFILE	sResFile;

  ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    fFind = _BinSearch( psRes->psFileList, psRes->ulFileListCount,
                        sizeof(RESFILE), &__cbResFileComp, (PVOID)hFile,
                        &ulIndex );
    if ( fFind )
    {
      debug( "The handle #%u already stored!", hFile );
    }
    else
    {
      sResFile.hFile = hFile;
      sResFile.psMountPoint = psMountPoint;

      if ( !_BinInsert( (PVOID *)&psRes->psFileList,
                        &psRes->ulFileListCount, sizeof(RESFILE), ulIndex,
                        &sResFile ) )
      {
        debug( "Not enought memory (store handle)" );
      }
    }

    DosReleaseMutexSem( psRes->hmtxFileList );
  }
}

static VOID _resRemoveFileHandle(PRES psRes, HFILE hFile)
{
  BOOL		fFind;
  ULONG		ulIndex;
  ULONG		ulRC;

  ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
  }
  else
  {
    fFind = _BinSearch( psRes->psFileList, psRes->ulFileListCount,
                        sizeof(RESFILE), &__cbResFileComp, (PVOID)hFile,
                        &ulIndex );
    if ( !fFind )
    {
//dbg      debug( "The handle was not stored" );
    }
    else
    {
      _BinDelete( (PVOID *)&psRes->psFileList, &psRes->ulFileListCount,
                  sizeof(RESFILE), ulIndex );
    }

    DosReleaseMutexSem( psRes->hmtxFileList );
  }
}

static PMOUNTPOINT _resSearchFileHandle(PRES psRes, HFILE hFile)
{
  ULONG		ulRC;
  ULONG		ulIndex;
  PMOUNTPOINT	psMountPoint;

  ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
  if ( ulRC != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", ulRC );
    psMountPoint = NULL;
  }
  else
  {
    if ( !_BinSearch( psRes->psFileList, psRes->ulFileListCount,
            sizeof(RESFILE), &__cbResFileComp, (PVOID)hFile, &ulIndex ) )
    {
//dbg      debug( "The handle not found." );
      psMountPoint = NULL;
    }
    else
      psMountPoint = psRes->psFileList[ulIndex].psMountPoint;

    DosReleaseMutexSem( psRes->hmtxFileList );
  }

  return psMountPoint;
}


BOOL fnPipeWriteStat(HPIPE hPipe, PCHAR pcBuf, BOOL fCompact)
{
  PRES		psRes;
  ULONG		ulRC;
  ULONG		ulIdx;
  ULONG		ulLength;
  PMOUNTPOINT	psMountPoint;
  PSZ		pszFormat;

  if ( fCompact )
  {
    pszFormat = "\"%s\" \"%s\" %lld %lld %lld %s %s";
  }
  else
  {
    pszFormat =
      "Name: %s\n"\
      "Path: %s\n"\
      "Limit: %lld\n"\	
      "Allocated: %lld\n"\
      "Free: %lld\n"\
      "Flag \"changed\": %s\n"\
      "Flag \"dirty\": %s";
  }

  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT  );

  if ( psResList == NULL )
  {
    DosReleaseMutexSem( hmtxResList );
    return FALSE;
  }

  for( psRes = psResList; psRes != NULL; psRes = psRes->psNext )
  {
    ulRC = DosRequestMutexSem( psRes->hmtxFileList, MUTEX_TIMEOUT );
    if ( ulRC != NO_ERROR )
    {
      ulIdx = (ULONG)(-1);
      debug( "DosRequestMutexSem(), rc = %u", ulRC );
    }
    else
    {
      ulIdx = psRes->ulFileListCount;
      DosReleaseMutexSem( psRes->hmtxFileList );
    }

    sprintf( pcBuf,
             "Mount points: %s\nWrite-acces files: %u\nConnections: %d\n",
             psRes->pszCfgFile, ulIdx, psRes->lConnectionsCount );
    ulRC = lqlibPipeWriteMessage( hPipe, pcBuf );
    if ( ulRC != NO_ERROR )
      break;

    _resLockRead( psRes );

    for( ulIdx = 0; ulIdx < psRes->ulMPListCount; ulIdx++ )
    {
      psMountPoint = psRes->ppsMPList[ulIdx];

      DosRequestMutexSem( psMountPoint->hmtxLock, SEM_INDEFINITE_WAIT );

      ulLength =
        sprintf( pcBuf, pszFormat,
          psMountPoint->pszName,
          psMountPoint->pszPath,
          psMountPoint->llLimit,
          psMountPoint->llByteAlloc,
          psMountPoint->llLimit - psMountPoint->llByteAlloc,
          (psMountPoint->ulFlags & MPFL_CHANGED) != 0 ? "ON" : "OFF",
          (psMountPoint->ulFlags & MPFL_DIRTY) != 0 ? "ON" : "OFF"
        );

      DosReleaseMutexSem( psMountPoint->hmtxLock );

      if ( !fCompact || ( ulIdx == (psRes->ulMPListCount - 1) ) )
      {
        strcpy( &pcBuf[ulLength], "\n" );
      }

      lqlibPipeWriteMessage( hPipe, pcBuf );
    }

    _resUnlockRead( psRes );
  }

  DosReleaseMutexSem( hmtxResList );

  return TRUE;
}


// Resource's thread
// -----------------
//
// FlushThread thread will be created for an each resource.

VOID _System FlushThread(ULONG ulParam)
{
  PRES		psRes = (PRES)ulParam;
  ULONG		ulEvSemId;
  ULONG		ulRC;
  BOOL		fStop = FALSE;
  FILESTATUS3L	sCfgFileStat;

//dbg  debug( "New thread: psRes = %p (%s)", psRes, psRes->pszCfgFile );

  while( !fStop )
  {
    ulRC = DosWaitMuxWaitSem( psRes->hmuxThreadEvents, 1000, &ulEvSemId );
    if ( ulRC == NO_ERROR )
    {
      switch( ulEvSemId )
      {
        case ID_STOP_SEM:
          debug( "[%p] Stop event", psRes );
          fStop = TRUE;
          break;

        case ID_CONF_SEM:
//dbg          debug( "[%p] Config event", psRes );

          _resScanLR( psRes, TRUE );

          if ( _resLockWrite( psRes ) == NO_ERROR )
          {
            _resReconfig( psRes );
            _resUnlockWrite( psRes );
          }

          DosResetEventSem( psRes->hevConf, &ulEvSemId );
          break;

        case ID_TESTCONFFILE_SEM:
          ulRC = DosQueryPathInfo( psRes->pszCfgFile, FIL_STANDARDL,
                                   &sCfgFileStat, sizeof(FILESTATUS3L) );
          if ( ulRC != NO_ERROR )
          {
            debug( "DosQueryPathInfo(), rc = %u", ulRC );
          }
          else if
          ( 
            ( memcmp( &sCfgFileStat.ftimeLastWrite,
                &psRes->sCfgFileStat.ftimeLastWrite, sizeof(FTIME) ) != 0 )
          ||
            ( memcmp( &sCfgFileStat.fdateLastWrite,
                &psRes->sCfgFileStat.fdateLastWrite, sizeof(FDATE) ) != 0 )
          )
          {
            debug( "[%p] Config file %s was changed", psRes, psRes->pszCfgFile );
            DosPostEventSem( psRes->hevConf );
          }

          DosResetEventSem( psRes->hevTestConfFile, &ulEvSemId );
          break;

        default:
//dbg          debug( "[%p] Scan event", psRes );
          _resScanLR( psRes, FALSE );
          DosResetEventSem( psRes->hevScan, &ulEvSemId );
      }
    }
    else if ( ulRC == ERROR_TIMEOUT )
    {
      _resScanLR( psRes, FALSE );
    }
    else
    {
      debug( "[%p] DosWaitMuxWaitSem(), rc=%u", psRes, ulRC );
      break;
    }
  }
}


// Public Routines
// ---------------

ULONG resInit()
{
  ULONG		ulRC;

  ulRC = DosCreateMutexSem( NULL, &hmtxResList, 0, FALSE );
 
  if ( ulRC == NO_ERROR )
  {
    ulRC = lqlibPipeServInit( &sLQPipe, LQTREE_PIPE_NAME, fnPipeWriteStat );

    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "lqlibPipeServInit(), rc = %u", ulRC );
      DosCloseMutexSem( hmtxResList );
    }
  }

  return ulRC;
}

VOID resDone()
{
  debug( "shutdown..." );

  if ( psResList != NULL )
    debug( "Not all of resources was unmounted!" );

  lqlibPipeServDone( &sLQPipe );
  DosCloseMutexSem( hmtxResList );
  debug( "Done" );
}

ULONG resNew(PSZ pszCfgFile, PRES *ppsRes)
{
  PRES		psRes;
  ULONG		ulRC;
  SEMRECORD	asSemList[4];
  FILE		*pFile;

  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT  );
  for( psRes = psResList; psRes != NULL; psRes = psRes->psNext )
  {
    if ( stricmp( psRes->pszCfgFile, pszCfgFile ) == 0 )
      break;
  }
  DosReleaseMutexSem( hmtxResList );

  if ( psRes != NULL )
  {
    psRes->ulMountCount++;
    ulRC = NO_ERROR;
  }
  else
  {
    pFile = fopen( pszCfgFile, "r" );
    if ( pFile == NULL )
      return ERROR_FILE_NOT_FOUND;

    fclose( pFile );

    psRes = (PRES)calloc( 1, sizeof(RES) );
    if ( psRes == NULL )
    {
      debug( "Not enought memory" );
      ulRC = ERROR_NOT_ENOUGH_MEMORY;
    }
    else
    {
      psRes->pszCfgFile = strdup( pszCfgFile );
      if ( psRes->pszCfgFile == NULL )
      {
        debug( "Not enought memory" );
        free( psRes );
        ulRC = ERROR_NOT_ENOUGH_MEMORY;
      }
      else
      {
        psRes->tidFlush = (TID)(-1);
        psRes->hmuxThreadEvents = (HMUX)(-1);
        psRes->htimerTestConfFile = (HTIMER)(-1);

        DosCreateMutexSem( NULL, &psRes->hmtxMPList, 0, FALSE );
        DosCreateEventSem( NULL, &psRes->hevMPList, 0, FALSE );
        DosCreateMutexSem( NULL, &psRes->hmtxFileList, 0, FALSE );
        DosCreateEventSem( NULL, &psRes->hevStop, 0, FALSE );
        DosCreateEventSem( NULL, &psRes->hevScan, 0, FALSE );
        DosCreateEventSem( NULL, &psRes->hevConf, 0, TRUE );
        DosCreateEventSem( NULL, &psRes->hevTestConfFile, DC_SEM_SHARED, FALSE );

        asSemList[0].hsemCur = (HSEM)psRes->hevStop;
        asSemList[0].ulUser = ID_STOP_SEM;
        asSemList[1].hsemCur = (HSEM)psRes->hevScan;
        asSemList[1].ulUser = ID_SCAN_SEM;
        asSemList[2].hsemCur = (HSEM)psRes->hevConf;
        asSemList[2].ulUser = ID_CONF_SEM;
        asSemList[3].hsemCur = (HSEM)psRes->hevTestConfFile;
        asSemList[3].ulUser = ID_TESTCONFFILE_SEM;
        ulRC = DosCreateMuxWaitSem( NULL, &psRes->hmuxThreadEvents, 4,
                                    &asSemList, DCMW_WAIT_ANY );

        if ( ulRC != NO_ERROR )
        {
          debug( "DosCreateMuxWaitSem(), rc=%d", ulRC );
        }
        else
        {
          ulRC = DosStartTimer( TEST_CONFIG_INTERVAL,
                   (HSEM)psRes->hevTestConfFile, &psRes->htimerTestConfFile ); 
          if ( ulRC != NO_ERROR )
            debug( "DosStartTimer(), rc=%d", ulRC );

          ulRC = DosCreateThread( &psRes->tidFlush, &FlushThread, (ULONG)psRes,
                                  CREATE_READY | STACK_SPARSE, 65536 );
          if ( ulRC != NO_ERROR )
          {
            debug( "DosCreateThread(), rc=%d", ulRC );
          }
        }

        if ( ulRC != NO_ERROR )
        {
          resFree( psRes );
          psRes = NULL;
        }
        else
        {
          psRes->ulMountCount = 1;

          DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT  );

          psRes->ppsSelf = ppsResLast;
          *ppsResLast = psRes;
          ppsResLast = &psRes->psNext;

          DosReleaseMutexSem( hmtxResList );
        }
      }
    }
  }

  *ppsRes = psRes;

  return ulRC;
}

VOID resFree(PRES psRes)
{
  psRes->ulMountCount--;

  if ( psRes->ulMountCount == 0 )
  {
    if ( psRes->htimerTestConfFile != (HTIMER)(-1) )
      DosStopTimer( psRes->htimerTestConfFile );

    DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT  );

    if ( psRes->ppsSelf != NULL )
    {
      *psRes->ppsSelf = psRes->psNext;
      if ( psRes->psNext != NULL )
      {
        psRes->psNext->ppsSelf = psRes->ppsSelf;
        psRes->psNext = NULL;
      }
      else
        ppsResLast = psRes->ppsSelf;
    }

    DosReleaseMutexSem( hmtxResList );

    DosPostEventSem( psRes->hevStop );

    if ( psRes->tidFlush != (TID)(-1) )
    {
      DosWaitThread( &psRes->tidFlush, DCWW_WAIT );
    }

    _resScanLR( psRes, TRUE );
    _resClearMountPoints( psRes );

    if ( psRes->pszCfgFile != NULL )
      free( psRes->pszCfgFile );

    if ( psRes->hmuxThreadEvents != (HMUX)(-1) )
      DosCloseMuxWaitSem( psRes->hmuxThreadEvents );

    DosCloseEventSem( psRes->hevStop );
    DosCloseEventSem( psRes->hevScan );
    DosCloseEventSem( psRes->hevConf );
    DosCloseEventSem( psRes->hevTestConfFile );
    DosCloseMutexSem( psRes->hmtxMPList );
    DosCloseEventSem( psRes->hevMPList );
    DosCloseMutexSem( psRes->hmtxFileList );

    free( psRes );

    debug( "Done" );
  }
}

VOID resConnectionNew(PRES psRes)
{
  psRes->lConnectionsCount++;
}

VOID resConnectionFree(PRES psRes)
{
  psRes->lConnectionsCount--;
}

ULONG resCompare(PRES psRes, PRES psResTest)
{
  ULONG		fEqual;

  fEqual = ( stricmp( psRes->pszCfgFile, psResTest->pszCfgFile ) == 0 );

  return fEqual ? NO_ERROR : ERROR_NOT_SAME_DEVICE;
}

ULONG resGetInfo(PRES psRes, PCHAR pcBuf, ULONG ulBufLen, PULONG pulActual,
                 PULONG pulMounPoints)
{
  ULONG		ulLength;
  ULONG		ulRC;

  if ( pulMounPoints != NULL )
  {
    _resLockRead( psRes );
    *pulMounPoints = psRes->ulMPListCount;
    _resUnlockRead( psRes );
  }

  ulLength = strlen( psRes->pszCfgFile ) + 1;

  if ( ulLength > ulBufLen )
  {
    ulRC = ERROR_BUFFER_OVERFLOW;
    debug( "Buffer too small (need %u bytes, but have only %u)", ulLength,
           ulBufLen );
  }
  else
  {
    *pulActual = ulLength;
    memcpy( pcBuf, psRes->pszCfgFile, ulLength );
    ulRC = NO_ERROR;
  }

  return ulRC;
}

ULONG resQueryFSAllocate(PRES psRes, PULONG pulSectorUnit, PULONG pulUnit,
                         PULONG pulUnitAvail, PUSHORT pusSector)
{
/*
#define RESFS_SECTOR_UNIT		psRes->ulSectorUnit
#define RESFS_BYTE_SECTOR		psRes->ulByteSector
#define RESFS_BYTE_ALLOC		psRes->llByteAlloc
#define RESFS_LIMIT			psRes->llLimit
*/
#define RESFS_SECTOR_UNIT		4
#define RESFS_BYTE_SECTOR		512
#define RESFS_BYTE_ALLOC		0
#define RESFS_LIMIT			0x7FFFFFFF

  ULONG		ulByteUnit = RESFS_SECTOR_UNIT * RESFS_BYTE_SECTOR;
  ULONG		ulUnitAlloc = RESFS_BYTE_ALLOC / ulByteUnit;
  ULONG		ulUnit = RESFS_LIMIT / ulByteUnit;

  if ( pulSectorUnit != NULL )
    *pulSectorUnit	= RESFS_SECTOR_UNIT;

  if ( pulUnit != NULL )
    *pulUnit 		= ulUnit;

  if ( pulUnitAvail != NULL )
    *pulUnitAvail	= ulUnit > ulUnitAlloc ? ( ulUnit - ulUnitAlloc ) : 0;

  if ( pusSector != NULL )
    *pusSector		= RESFS_BYTE_SECTOR;

  return NO_ERROR;
}

ULONG resRefresh(PRES psRes, PSZ pszMountPointPath, BOOL fSubdir)
{
  ULONG		ulIndex;
  ULONG		ulRC;
  PCHAR		pcPtr;

  if ( pszMountPointPath != NULL )
  {
    while( *pszMountPointPath == '\\' )
      pszMountPointPath++;
  }

  if ( ( pszMountPointPath == NULL ) || ( *pszMountPointPath == '\0' ) )
  {
//dbg    debug( "Read mount points list from %s", psRes->pszCfgFile );
    ulRC = DosPostEventSem( psRes->hevConf );
    if ( ( ulRC == NO_ERROR ) && fSubdir )
    {
      do
      {
        DosSleep( 1 );

        ulRC = DosQueryEventSem( psRes->hevConf, &ulIndex );
        if ( ulRC != NO_ERROR )
        {
          debug( "DosQueryEventSem(), rc = %u", ulRC );
          break;
        }
      }
      while( ulIndex != 0 );

      if ( ulRC == NO_ERROR )
      {
//dbg        debug( "Mark all mount points \"dirty\"" );
        if ( _resLockRead( psRes ) == NO_ERROR )
        {
          _resDirtyMountPonts( psRes );
          DosPostEventSem( psRes->hevScan );
          _resUnlockRead( psRes );
        }
      }
    }
//dbg    debug( "Done" );
  }
  else
  {
    pcPtr = strchr( pszMountPointPath, '\\' );
    if ( pcPtr == NULL )
      pcPtr = strchr( pszMountPointPath, '\0' );

    ulRC = _resLockRead( psRes );
    if ( ulRC == NO_ERROR )
    {
      if ( !_resSearchMountPoint2( psRes, pszMountPointPath,
                                   pcPtr - pszMountPointPath, &ulIndex ) )
      {
//dbg        debug( "Mount point for \"%s\" not found", pszMountPointPath );
        ulRC = ERROR_PATH_NOT_FOUND;
      }
      else
      {
        PMOUNTPOINT	psMountPoint = psRes->ppsMPList[ulIndex];

//dbg        debug( "start refresh: %s", psMountPoint->pszPath );
        _resMountPointFlag( psMountPoint, MPFL_DIRTY, TRUE );
        ulRC = DosPostEventSem( psRes->hevScan );
      }

      _resUnlockRead( psRes );
    }
  }

  return ulRC;
}

ULONG resFlush(PRES psRes)
{
  _resScanLR( psRes, TRUE );

  return NO_ERROR;
}

ULONG resQueryPathInfo(PRES psRes, PSZ pszMountPointPath, PFILESTATUS3L psStat)
{
  PCHAR		pcPtr;
  ULONG		ulRC;
  CHAR		acBuf[CCHMAXPATH];
  BOOL		fWildCards;

  fWildCards = ( strchr( pszMountPointPath, '*' ) != NULL ) ||
               ( strchr( pszMountPointPath, '?' ) != NULL );

  if ( !fWildCards )
  {
    ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
                               strlen( pszMountPointPath ),
                               &acBuf, sizeof(acBuf) );

    if ( ulRC == NO_ERROR )
    {
      ulRC = DosQueryPathInfo( &acBuf, FIL_STANDARDL, psStat,
                               sizeof(FILESTATUS3L) );
      if ( ulRC != NO_ERROR )
      {
//dbg        debug( "DosQueryPathInfo(%s) #1, rc = %u", &acBuf, ulRC );
      }
    }
  }

  if ( fWildCards || ( ulRC == ERROR_PATH_NOT_FOUND ) )
  {
    pcPtr = strrchr( pszMountPointPath, '\\' );

    if ( pcPtr == NULL )
    {
      ulRC = ERROR_FILE_NOT_FOUND;
    }
    else
    {
      ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
                                 pcPtr - pszMountPointPath,
                                 &acBuf, sizeof(acBuf) );

      if ( ulRC == NO_ERROR )
      {
        ulRC = DosQueryPathInfo( &acBuf, FIL_STANDARDL, psStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryPathInfo(%s) #2, rc = %u", &acBuf, ulRC );
        }
        else
        {
          ulRC = (psStat->attrFile & FILE_DIRECTORY) != 0 ?
                   ERROR_FILE_NOT_FOUND : ERROR_PATH_NOT_FOUND;
        }
      }
    }
  }

  return ulRC;
}

ULONG resSetPathInfo(PRES psRes, PSZ pszMountPointPath, PFILESTATUS3L psStat)
{
  ULONG		ulRC;
  CHAR		acBuf[CCHMAXPATH];

  if ( !_IsSubDir( pszMountPointPath ) )
    return ERROR_ACCESS_DENIED;

  ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
                             strlen( pszMountPointPath ),
                             &acBuf, sizeof(acBuf) );

  if ( ulRC == NO_ERROR )
  {
    ulRC = DosSetPathInfo( &acBuf, FIL_STANDARDL, psStat, sizeof(FILESTATUS3L),
                           DSPI_WRTTHRU );
    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosSetPathInfo(%s), rc = %u", &acBuf, ulRC );
    }
  }

  return ulRC;
}

ULONG resDelete(PRES psRes, PSZ pszFile)
{
  ULONG		ulRC;
  CHAR		acFile[CCHMAXPATH];
  PMOUNTPOINT   psMountPoint;
  FILESTATUS3L	sStat;

  ulRC = _resLockRead( psRes );
  if ( ulRC == NO_ERROR )
  {
    ulRC = _resGetLocalPath( psRes, pszFile, strlen( pszFile ),
                             &acFile, sizeof(acFile), &psMountPoint );
    if ( ulRC == NO_ERROR )
    {
      if ( !_resRequestMountPointWrite( psMountPoint ) )
      {
//dbg        debug( "Mount point \"%s\" is \"dirty\"", psMountPoint->pszName );
        ulRC = ERROR_ACCESS_DENIED;
      }
      else
      {
        ulRC = DosQueryPathInfo( &acFile, FIL_STANDARDL, &sStat,
                               sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryPathInfo(%s), rc = %d", &acFile, ulRC );
        }
        else
        {
          ulRC = DosForceDelete( &acFile );

          if ( ulRC != NO_ERROR )
          {
//dbg            debug( "DosForceDelete(%s), rc = %d", &acFile, ulRC );
          }
          else
          {
//dbg            debug( "Deleted: %s, (free %lld bytes)", &acFile,
//dbg                   *(PLLONG)&sStat.cbFileAlloc );
            _resIncMountPointBytesAlloc( psMountPoint,
                                         -(*(PLLONG)&sStat.cbFileAlloc) );
          }
        }
       
        _resReleaseMountPointWrite( psMountPoint );
      }
    }

    _resUnlockRead( psRes );
  }

  return ulRC;
}

ULONG resCreateDir(PRES psRes, PSZ pszMountPointPath)
{
  ULONG		ulRC;
  CHAR		acBuf[CCHMAXPATH];

  if ( !_IsSubDir( pszMountPointPath ) )
    return ERROR_ACCESS_DENIED;

  ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
                             strlen( pszMountPointPath ),
                             acBuf, sizeof(acBuf) );
  if ( ulRC == NO_ERROR )
  {
    ulRC = DosCreateDir( &acBuf, NULL );

    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosCreateDir(%s), rc = %d", &acBuf, ulRC );
    }
  }

  return ulRC;
}

ULONG resDeleteDir(PRES psRes, PSZ pszMountPointPath)
{
  ULONG		ulRC;
  CHAR		acBuf[CCHMAXPATH];

  if ( !_IsSubDir( pszMountPointPath ) )
    return ERROR_ACCESS_DENIED;

  ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
                             strlen( pszMountPointPath ),
                             acBuf, sizeof(acBuf) );
  if ( ulRC == NO_ERROR )
  {
    ulRC = DosDeleteDir( &acBuf );

    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosDeleteDir(%s), rc = %d", &acBuf, ulRC );
    }
  }

  return ulRC;
}

ULONG resList(PRES psRes, PSZ pszMountPointPath, ULONG ulAttribute,
              PFNLISTITEM pfnListItem, PVOID pFNData)
{
  CHAR		acPath[CCHMAXPATH];
  ULONG		ulFileNames;
  ULONG		ulRC, ulRC2;

  if ( strchr( pszMountPointPath, '\\' ) == NULL )
  {
    BOOL		fWildCards;
    PMOUNTPOINT		psMountPoint;
    FILEFINDBUF3L	sFind;

    if ( ( ulAttribute != 0 ) &&
         (
           ( (ulAttribute & MUST_HAVE_DIRECTORY) == 0 ) ||
           ( (ulAttribute & (MUST_HAVE_ARCHIVED | MUST_HAVE_SYSTEM |
              MUST_HAVE_HIDDEN | MUST_HAVE_READONLY) & 0xFF00) != 0 )
         )
       )
    {
      return NO_ERROR;
    }

    sFind.oNextEntryOffset = 0;
    memcpy( &sFind.fdateCreation, &psRes->sCfgFileStat.fdateCreation,
            sizeof(FDATE) );
    memcpy( &sFind.ftimeCreation, &psRes->sCfgFileStat.ftimeCreation,
            sizeof(FTIME) );
    memcpy( &sFind.fdateLastAccess, &psRes->sCfgFileStat.fdateLastAccess,
            sizeof(FDATE) );
    memcpy( &sFind.ftimeLastAccess, &psRes->sCfgFileStat.ftimeLastAccess,
            sizeof(FTIME) );
    memcpy( &sFind.fdateLastWrite, &psRes->sCfgFileStat.fdateLastWrite,
            sizeof(FDATE) );
    memcpy( &sFind.ftimeLastWrite, &psRes->sCfgFileStat.ftimeLastWrite,
            sizeof(FTIME) );
    *(PLLONG)&sFind.cbFile = (LLONG)0;
    *(PLLONG)&sFind.cbFileAlloc = (LLONG)0;
    sFind.attrFile = FILE_DIRECTORY;

    fWildCards = ( strchr( pszMountPointPath, '*' ) != NULL ) ||
                 ( strchr( pszMountPointPath, '?' ) != NULL );

    ulRC = _resLockRead( psRes );

    if ( ulRC == NO_ERROR )
    {
      for( ulFileNames = 0; ulFileNames < psRes->ulMPListCount; ulFileNames++ )
      {
        psMountPoint = psRes->ppsMPList[ulFileNames];

        if ( fWildCards )
        {
          ulRC2 = DosEditName( 1, psMountPoint->pszName, pszMountPointPath,
                               acPath, sizeof(acPath) );
          if ( ulRC2 != NO_ERROR )
          {
//dbg            debug( "DosEditName(%s,%s), rc = %u", psMountPoint->pszName,
//dbg                   pszMountPointPath, ulRC );
            continue;
          }

          if ( stricmp( psMountPoint->pszName, &acPath ) != 0 )
            continue;
        }
        else if ( stricmp( psMountPoint->pszName, pszMountPointPath ) != 0 )
          continue;

        sFind.cchName = strlen( psMountPoint->pszName );
        strcpy( &sFind.achName, psMountPoint->pszName );

        if ( !pfnListItem( &sFind, pFNData ) )
          break;
      }

      _resUnlockRead( psRes );
    }
  }
  else
  {
    HDIR		hDir = HDIR_CREATE;
    PFILEFINDBUF3L	pasFind, psFindItem;

    ulRC = _resGetLocalPathLR( psRes, pszMountPointPath,
             strlen( pszMountPointPath ), &acPath, sizeof(acPath) );
    if ( ulRC != NO_ERROR )
      return ulRC;

    pasFind = malloc( FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L) );
    if ( pasFind == NULL )
    {
      debug( "Not enought memory" );
    }
/*
    ulRC = DosAllocMem( (PVOID *)&pasFind,
             FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L), fPERM | PAG_COMMIT );
    if ( ulRC != NO_ERROR )

    {
      debug( "DosAllocMem(), rc = %u", ulRC );
    }
*/
    else
    {
      ulFileNames = FIND_BUF_MAX_ITEMS;
      ulRC = DosFindFirst( &acPath, &hDir, ulAttribute & 0x00003737, pasFind,
                           FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L),
                           &ulFileNames, FIL_STANDARDL );

      while( ulRC == NO_ERROR )
      {
        for( psFindItem = pasFind; ;
             psFindItem = (PFILEFINDBUF3L) ( ((PCHAR)psFindItem) + psFindItem->oNextEntryOffset )
           )
        {
          if ( !pfnListItem( psFindItem, pFNData ) ||
               ( psFindItem->oNextEntryOffset == 0 ) )
          {
            break;
          }
        }

        ulFileNames = FIND_BUF_MAX_ITEMS;
        ulRC = DosFindNext( hDir, pasFind,
                 FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L), &ulFileNames );
      }

      if ( ulRC == ERROR_NO_MORE_FILES )
      {
        ulRC = NO_ERROR;
      }
      else
        debug( "DosFind*(%s), rc = %u", &acPath, ulRC );

      DosFindClose( hDir );

      free( pasFind );
/*
      ulRC = DosFreeMem( pasFind );
      if ( ulRC != NO_ERROR )
      {
        debug( "DosFreeMem(), rc = %u", ulRC2 );
      }
*/
    }
  }

  return ulRC;
}

ULONG resCopy(PRES psRes, PRES psResDst, PSZ pszFile, PSZ pszFileDest,
              ULONG ulOption)
{
  ULONG		ulRC;
  CHAR		acSrcName[CCHMAXPATH];
  CHAR		acDstName[CCHMAXPATH];
  PCHAR		pcPtr;
  FILESTATUS3L	sStat;
  PMOUNTPOINT   psDstMountPoint;
  LLONG		llFileSize;

  ulRC = _resGetLocalPathLR( psRes, pszFile, strlen( pszFile ),
                             &acSrcName, sizeof(acSrcName) );
  if ( ulRC != NO_ERROR )
    return ulRC;

  ulRC = DosQueryPathInfo( &acSrcName, FIL_STANDARDL, &sStat,
                           sizeof(FILESTATUS3L) );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosQueryPathInfo(%s) #1, rc=%d", &acSrcName, ulRC );
    return ulRC;
  }

  llFileSize = *(PLLONG)&sStat.cbFileAlloc;

  ulRC = _resLockRead( psResDst );
  if ( ulRC == NO_ERROR )
  {
    ulRC = _resGetLocalPath( psResDst, pszFileDest, strlen( pszFileDest ),
                             &acDstName, sizeof(acDstName), &psDstMountPoint );
    if ( ulRC == NO_ERROR )
    {
      if ( !_resRequestMountPointWrite( psDstMountPoint ) )
      {
//dbg        debug( "Mount point \"%s\" is \"dirty\"", psDstMountPoint->pszName );
        ulRC = ERROR_ACCESS_DENIED;
      }
      else
      {
        LLONG	llBytes = _resGetMountPointBytesFree( psDstMountPoint );

        if ( llFileSize > llBytes )
        {
//dbg          debug( "Not enought space in \"%s\". Need: %lld, Have: %lld",
//dbg            psDstMountPoint->pszName, *(PLLONG)&sStat.cbFileAlloc, llBytes );
          ulRC = ERROR_DISK_FULL;
        }
        else
        {
          ulRC = DosCopy( &acSrcName, &acDstName, ulOption );
          if ( ulRC != NO_ERROR )
          {
//dbg            debug( "DosCopy(%s,%s), rc = %u", &acSrcName, &acDstName, ulRC );
          }
          else
          {
            ulRC = DosQueryPathInfo( &acDstName, FIL_STANDARDL, &sStat,
                                     sizeof(FILESTATUS3L) );
            if ( ulRC != NO_ERROR )
            {
//dbg              debug( "DosQueryPathInfo(%s) #2, rc = %u", &acDstName, ulRC );
            }
            else
            {
              if ( (sStat.attrFile & FILE_DIRECTORY) != 0 )
              {
                pcPtr = strrchr( &acSrcName, '\\' );
                if ( pcPtr != NULL )
                {
                  strcat( &acDstName, pcPtr == NULL ? &acSrcName : pcPtr );
                  ulRC = DosQueryPathInfo( &acDstName, FIL_STANDARDL, &sStat,
                                           sizeof(FILESTATUS3L) );
                  if ( ulRC == NO_ERROR )
                  {
                    llFileSize = *(PLLONG)&sStat.cbFileAlloc;
                  }
                  else
                  {
//dbg                    debug( "DosQueryPathInfo(%s) #3, rc = %u", &acDstName,
//dbg                           ulRC );
                  }
                }
              }
              else
                llFileSize = *(PLLONG)&sStat.cbFileAlloc;

              _resIncMountPointBytesAlloc( psDstMountPoint, llFileSize );
            }
          }
        }

        _resReleaseMountPointWrite( psDstMountPoint );
      }
    }

    _resUnlockRead( psResDst );
  }

  return ulRC;
}

ULONG resMove(PRES psRes, PRES psResDst, PSZ pszFile, PSZ pszFileDest)
{
  ULONG		ulRC;
  CHAR		acSrcName[CCHMAXPATH];
  CHAR		acDstName[CCHMAXPATH];
  PMOUNTPOINT   psSrcMountPoint;
  PMOUNTPOINT   psDstMountPoint;
  FILESTATUS3L	sStat;
  LLONG		llFileSize;
  LLONG		llBytes;

  ulRC = _resLockRead( psRes );
  if ( ulRC != NO_ERROR )
    return ulRC;

  ulRC = _resLockRead( psResDst );
  if ( ulRC != NO_ERROR )
  {
    _resUnlockRead( psRes );
    return ulRC;
  }

  do
  {
    ulRC = _resGetLocalPath( psRes, pszFile, strlen( pszFile ),
                             &acSrcName, sizeof(acSrcName), &psSrcMountPoint );
    if ( ulRC != NO_ERROR )
      break;

    ulRC = _resGetLocalPath( psResDst, pszFileDest, strlen( pszFileDest ),
                             &acDstName, sizeof(acDstName), &psDstMountPoint );
    if ( ulRC != NO_ERROR )
      break;

    ulRC = DosQueryPathInfo( &acSrcName, FIL_STANDARDL, &sStat,
                             sizeof(FILESTATUS3L) );
    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosQueryPathInfo(%s) #1, rc=%d", &acSrcName, ulRC );
      break;
    }

    llFileSize = *(PLLONG)&sStat.cbFileAlloc;
    llBytes = _resGetMountPointBytesFree( psDstMountPoint );

    if ( llFileSize > llBytes )
    {
//dbg      debug( "Not enought space in \"%s\". Need: %lld, Have: %lld",
//dbg        psDstMountPoint->pszName, llFileSize, llBytes );
      ulRC = ERROR_DISK_FULL;
      break;
    }

    if ( !_resRequestMountPointWrite( psDstMountPoint ) )
    {
//dbg      debug( "Mount point \"%s\" is \"dirty\"", psDstMountPoint->pszName );
      ulRC = ERROR_ACCESS_DENIED;
    }
    else
    {
      ulRC = DosMove( &acSrcName, &acDstName );
      if ( ulRC != NO_ERROR )
      {
//dbg        debug( "DosMove(%s,%s), rc=%d", &acSrcName, &acDstName, ulRC );
      }
      else
      {
        ulRC = DosQueryPathInfo( &acDstName, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryPathInfo(%s) #2, rc=%d - Get original size of file",
//dbg                 &acDstName, ulRC );
        }
        else
          llFileSize = *(PLLONG)&sStat.cbFileAlloc;

        _resIncMountPointBytesAlloc( psSrcMountPoint, -llFileSize );
        _resIncMountPointBytesAlloc( psDstMountPoint, llFileSize );
      }

      _resReleaseMountPointWrite( psDstMountPoint );
    }
  }
  while( FALSE );

  _resUnlockRead( psResDst );
  _resUnlockRead( psRes );

  return ulRC;
}

ULONG resOpen(PRES psRes, PSZ pszFile, PHFILE phFile, LONGLONG llSize,
              ULONG ulAttribute, ULONG ulOpenFlags, ULONG ulOpenMode)
{
  ULONG		ulRC;
  CHAR		acBuf[CCHMAXPATH];
  ULONG		ulAction;
  FILESTATUS3L	sStat;
  LLONG		llFileByteAllocOld;
  LLONG		llByteFree;
  PMOUNTPOINT   psMountPoint;
  BOOL		fWriteMode;

  if ( !_IsSubDir( pszFile ) )
    return ERROR_ACCESS_DENIED;

  ulRC = _resLockRead( psRes );
  if ( ulRC != NO_ERROR )
    return ulRC;

  do
  {
    ulRC = _resGetLocalPath( psRes, pszFile, strlen( pszFile ),
                             &acBuf, sizeof(acBuf), &psMountPoint );
    if ( ulRC != NO_ERROR )
      break;

    fWriteMode = ( ulOpenMode & (OPEN_ACCESS_WRITEONLY | OPEN_ACCESS_READWRITE) ) != 0;

    if (
         fWriteMode ||
         ( ulOpenFlags &
           (OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS) ) != 0
       )
    {
      llByteFree = _resGetMountPointBytesFree( psMountPoint );

      if ( llByteFree <= 0 )
      {
        ulRC = ERROR_DISK_FULL;
        break;
      }

      ulRC = DosQueryPathInfo( &acBuf, FIL_STANDARDL, &sStat,
                               sizeof(FILESTATUS3L) );
      if ( ulRC != NO_ERROR )
      {
        if ( ulRC != ERROR_FILE_NOT_FOUND )
        {
//dbg          debug( "DosQueryPathInfo(%s), rc = %u", &acBuf, ulRC );
          break;
        }

        llFileByteAllocOld = 0;
      }
      else
        llFileByteAllocOld = *(PLLONG)(&sStat.cbFileAlloc);

      if ( ( ulOpenFlags & OPEN_ACTION_REPLACE_IF_EXISTS ) != 0 )
        llByteFree += llFileByteAllocOld;

      if ( *(PLLONG)&llSize > llByteFree )
      {
//dbg        debug( "Requested %lld bytes but have only %lld",
//dbg               *(PLLONG)&llSize, llByteFree );
        *(PLLONG)&llSize = llByteFree > 0 ? llByteFree : 0;
//dbg        debug( "...new size: %llu", *(PLLONG)&llSize );
      }
      else
      {
//dbg        debug( "Requested %lld bytes for file size", *(PLLONG)&llSize );
      }
    }

    if ( fWriteMode && !_resRequestMountPointWrite( psMountPoint ) )
    {
//dbg        debug( "Mount point \"%s\" is \"dirty\"", psMountPoint->pszName );
      ulRC = ERROR_ACCESS_DENIED;
      break;
    }

    ulRC = DosOpenL( &acBuf, phFile, &ulAction, llSize, ulAttribute,
                     ulOpenFlags, ulOpenMode, NULL );

    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosOpenL(%s), rc = %u", &acBuf, ulRC );
    }
    else if ( fWriteMode )
    {
      if ( ulRC == NO_ERROR )
      {
        ulRC = DosQueryFileInfo( *phFile, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg            debug( "DosQueryFileInfo(), rc = %u", ulRC );
          DosClose( *phFile );
          DosForceDelete( &acBuf );
        }
        else
        {
          _resIncMountPointBytesAlloc( psMountPoint,
            *(PLLONG)&sStat.cbFileAlloc - llFileByteAllocOld );
          _resStoreFileHandle( psRes, *phFile, psMountPoint );
        }
      }

    }
  }
  while( FALSE );

  if ( fWriteMode )
    _resReleaseMountPointWrite( psMountPoint );

  _resUnlockRead( psRes );

  return ulRC;
}

ULONG resFileClose(PRES psRes, HFILE hFile)
{
  ULONG		ulRC;

  if ( _resLockWrite( psRes ) == NO_ERROR )
  {
    ulRC = DosClose( hFile );

    if ( ulRC == NO_ERROR )
    {
      _resRemoveFileHandle( psRes, hFile );
    }
    else
      debug( "DosClose(%u), rc=%u", hFile, ulRC );

    _resUnlockWrite( psRes );
  }
  else
    ulRC = ERROR_ACCESS_DENIED;

/*
  ulRC = DosClose( hFile );

  if ( ulRC != NO_ERROR )
  {
    debug( "DosClose(%u), rc=%u", hFile, ulRC );
  }
  else
  {
    if ( _resLockRead( psRes ) == NO_ERROR )
    {
      _resRemoveFileHandle( psRes, hFile );
      _resUnlockRead( psRes );
    }
//dbg    debug( "File closed, handle=%u", hFile );
  }
*/

  return ulRC;
}

ULONG resSetFileAttribute(PRES psRes, PSZ pszFile, USHORT usAttr)
{
  ULONG		ulRC;
  CHAR		acFile[CCHMAXPATH];
  HFILE		hFile;
  ULONG		ulAction;
  FILESTATUS3L	sStat;
  LONGLONG	llSize = {0};

  ulRC = _resGetLocalPathLR( psRes, pszFile, strlen( pszFile ),
                             &acFile, sizeof(acFile) );
  if ( ulRC != NO_ERROR )
    return ulRC;

  ulRC = DosOpenL( &acFile, &hFile, &ulAction, llSize, FILE_NORMAL,
                   OPEN_EXISTING, OPEN_FLAGS_FAIL_ON_ERROR |
                   OPEN_SHARE_DENYNONE | OPEN_ACCESS_WRITEONLY, NULL );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosOpenL(%s), rc = %u", &acFile, ulRC );
    return ulRC;
  }

  ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosQueryFileInfo(), rc = %u", ulRC );
  }
  else
  {
    sStat.attrFile = usAttr;
    ulRC = DosSetFileInfo( hFile, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
    if ( ulRC != NO_ERROR )
    {
//dbg      debug( "DosSetFileInfo(), rc = %u", ulRC );
    }
  }
  DosClose( hFile );

  return ulRC;
}

ULONG resFileQueryInfo(PRES psRes, HFILE hFile, PFILESTATUS3L psStat)
{
  ULONG		ulRC;

  ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, psStat, sizeof(FILESTATUS3L) );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosQueryFileInfo(), rc = %u", ulRC );
  }

  return ulRC;
}

ULONG resFileSetInfo(PRES psRes, HFILE hFile, PFILESTATUS3L psStat)
{
  ULONG		ulRC;

  ulRC = DosSetFileInfo( hFile, FIL_STANDARDL, psStat, sizeof(FILESTATUS3L) );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosSetFileInfo(), rc = %u", ulRC );
  }

  return ulRC;
}

ULONG resFileSetFilePtr(PRES psRes, HFILE hFile, LONGLONG llOffset,
                        ULONG ulMethod, LONGLONG *pllActual)
{
  ULONG		ulRC;

  ulRC = DosSetFilePtrL( hFile, llOffset, ulMethod, pllActual );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosSetFilePtrL(%u,,,), rc = %u", hFile, ulRC );
  }
  else
  {
//dbg    debug( "The pointer moved: %lld bytes from the %s, file handle=%u",
//dbg           *(PLLONG)&llOffset,
//dbg           ulMethod == FILE_BEGIN ? "beginning of the file" :
//dbg             ulMethod == FILE_CURRENT ? "current location" : "end of the file",
//dbg           hFile );
  }

  return ulRC;
}

ULONG resFileNewSize(PRES psRes, HFILE hFile, LONGLONG llSize)
{
  ULONG		ulRC;
  PMOUNTPOINT	psMountPoint;
  FILESTATUS3L	sStat;
  LLONG		llByteFree;
  LLONG		llFileByteAllocOld;

  ulRC = _resLockRead( psRes );
  if ( ulRC != NO_ERROR )
    return ulRC;

  psMountPoint = _resSearchFileHandle( psRes, hFile );

  if ( psMountPoint == NULL )
  {
    ulRC = ERROR_INVALID_HANDLE;
  }
  else
  {
    if ( !_resRequestMountPointWrite( psMountPoint ) )
    {
//dbg      debug( "Mount point \"%s\" is \"dirty\"", psMountPoint->pszName );
      ulRC = ERROR_ACCESS_DENIED;
    }
    else
    {
      do
      {
        llByteFree = _resGetMountPointBytesFree( psMountPoint );
        if ( *(PLLONG)&llSize > llByteFree )
        {
//dbg          debug( "Not enought space (free: %lld, need: %lld)", llByteFree,
//dbg                 *(PLLONG)&llSize );

          ulRC = ERROR_DISK_FULL;
          break;
        }

        ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryFileInfo() #1, rc = %u", ulRC );
          break;
        }

        ulRC = DosSetFileSizeL( hFile, llSize );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosSetFileSize(), rc = %u", ulRC );
          break;
        }
//dbg        debug( "New file size (handle=%u): %llu", hFile, llSize );

        llFileByteAllocOld = *(PLLONG)(&sStat.cbFileAlloc);

        ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryFileInfo() #2, rc = %u", ulRC );
          break;
        }

        _resIncMountPointBytesAlloc( psMountPoint,
          *(PLLONG)&sStat.cbFileAlloc - llFileByteAllocOld );
      }
      while( FALSE );

      _resReleaseMountPointWrite( psMountPoint );
    }
  }

  _resUnlockRead( psRes );

  return ulRC;
}

ULONG resFileRead(PRES psRes, HFILE hFile, PVOID pBuffer, ULONG ulLength,
                  ULONG *pulActual)
{
  ULONG		ulRC;

  ulRC = DosRead( hFile, pBuffer, ulLength, pulActual );
  if ( ulRC != NO_ERROR )
  {
//dbg    debug( "DosRead(), rc = %u", ulRC );
  }

  return ulRC;
}

ULONG resFileWrite(PRES psRes, HFILE hFile, PVOID pBuffer, ULONG ulLength,
                   ULONG *pulActual)
{
  ULONG		ulRC;
  PMOUNTPOINT	psMountPoint;
  FILESTATUS3L	sStat;
  LLONG		llByteFree;
  LLONG		llFileByteAlloc;
  LONGLONG	llFileByteSizeOld;

  ulRC = _resLockRead( psRes );
  if ( ulRC != NO_ERROR )
    return ulRC;

  psMountPoint = _resSearchFileHandle( psRes, hFile );

  if ( psMountPoint == NULL )
  {
    ulRC = ERROR_INVALID_HANDLE;
  }
  else
  {
    if ( !_resRequestMountPointWrite( psMountPoint ) )
    {
//dbg      debug( "Mount point \"%s\" is \"dirty\"", psMountPoint->pszName );
      ulRC = ERROR_ACCESS_DENIED;
    }
    else
    {
      do
      {
        llByteFree = _resGetMountPointBytesFree( psMountPoint );
        if ( llByteFree <= 0 )
        {
          ulRC = ERROR_DISK_FULL;
          break;
        }

        if ( ulLength > llByteFree )
        {
/*          debug( "Not enought space (free: %lld, need: %u)", llByteFree,
                 ulLength );

          ulRC = ERROR_DISK_FULL;
          break;*/

//dbg          debug( "%u bytes requested to write, reduce to %lld bytes",
//dbg                 ulLength, llByteFree );
          ulLength = llByteFree;
        }

        ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryFileInfo() #1, rc = %u", ulRC );
          break;
        }

        ulRC = DosWrite( hFile, pBuffer, ulLength, pulActual );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosWrite(), rc = %u", ulRC );
          break;
        }

        llFileByteAlloc = *(PLLONG)(&sStat.cbFileAlloc);
        llFileByteSizeOld = sStat.cbFile;

        ulRC = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat,
                                 sizeof(FILESTATUS3L) );
        if ( ulRC != NO_ERROR )
        {
//dbg          debug( "DosQueryFileInfo() #2, rc = %u", ulRC );
          break;
        }

        llFileByteAlloc = *(PLLONG)&sStat.cbFileAlloc - llFileByteAlloc;

        if ( llFileByteAlloc > llByteFree )
        {
//dbg          debug( "New allocated size of file out of limit - roll back (%lld bytes)",
//dbg                 llFileByteSizeOld );
          DosSetFileSizeL( hFile, llFileByteSizeOld );
          ulRC = ERROR_DISK_FULL;
          break;
        }

        _resIncMountPointBytesAlloc( psMountPoint, llFileByteAlloc );
      }
      while( FALSE );

      _resReleaseMountPointWrite( psMountPoint );
    }
  }

  _resUnlockRead( psRes );

  return ulRC;
}
