/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, I1.HR                           *
*  Date        : Feb 92                                                   *
*  Last Update : Apr 93                                                   *
*                                                                         *
*  Module      : makefiles                                                *
*                                                                         *
*  Function    : generate makefile for the different machines             *
*                                                                         *
*  Export :                                                               *
*                                                                         *
*   - gen_makefile                                                        *
*                                                                         *
**************************************************************************/

# include <stdio.h>
# ifdef alliant
# include <fcntl.h>             /* for testing file access */
# else
# include <unistd.h>             /* for testing file access */
# endif
# include "Idents.h"
# include "StringMe.h"
# include "Tree.h"

/*******************************************************************
*                                                                  *
*  More general functions to generate a correct Makefile           *
*                                                                  *
*******************************************************************/

     /***********************************************
     *                                              *
     *  generate the header                         *
     *                                              *
     *   BEWARE : ...                               *
     *                                              *
     *   PHOME = ...                                *
     *   DALIB = ...                                *
     *                                              *
     ***********************************************/

void gen_header (mf, lib)
FILE *mf;
char lib[];


{
  char *r;

  fprintf (mf,"#\n");
  fprintf (mf,
    "# BEWARE: this Makefile was automatically generated by Adaptor\n");
  fprintf (mf,
    "#         Adaptor will not overwrite any changes you make here,\n");
  fprintf (mf,
    "#         however this Makefile may become out of date\n");

  fprintf (mf,"# Target machine is %s (%s)\n",machine_items[target_machine],
           model_items[target_model]);

  fprintf (mf,"#\n");
/*
  fprintf (mf,"# PHOME is Home Directory of Adaptor \n");
*/
  fprintf (mf,"# DALIB is directory for ADAPTOR libraries and object files\n");
  fprintf (mf,"#\n");
/*
  fprintf (mf,"PHOME =  $(HOME)/adaptor\n", PHOME;
*/
  if (!(r = getenv("DALIB")))
     r = "/adaptor/lib";

  fprintf (mf,"DALIB = %s\n", r);
/*
  fprintf (mf,"DALIB  = $(PHOME)/dalib/%s\n", lib);
*/
  fprintf (mf,"#\n");
  fprintf (mf,"# Standard commands\n");
  fprintf (mf,"ADAPTOR = fadapt\n");
  fprintf (mf,"FSPLIT  = fsplit\n");
  fprintf (mf,"FSTRIP  = fstrip\n");
  fprintf (mf,"#\n");
} /* gen_header */

     /***********************************************
     *                                              *
     *  Generating a list of names for              *
     *  all units (host/node) (obj/src)             *
     *                                              *
     ***********************************************/

void gen_obj_entries (mf, host, obj)
bool host;
bool obj;
FILE *mf;
{ tTree units, unit;
  tIdent name;
  char c, str[50];
  int i, len;
  int nnames = 0;
  int pure;        /* pure subprograms not in host */

  units = TreeRoot->COMP_UNIT.COMP_ELEMENTS;

  /* traversing is done after passing the node adaption */

  while (units->Kind != kDECL_EMPTY )
    { unit = units->DECL_LIST.Elem;
      name = unit->DECL_NODE.Name;
      pure = 0;
      if (unit->Kind == kFUNC_DECL)
         pure = unit->FUNC_DECL.IsPure;
      if (unit->Kind == kPROC_DECL)
         pure = unit->PROC_DECL.IsPure;
      GetString (name, str);
      /*  are the names of node program SUBNAME_N, NODEMODULE */
      /* fsplit generates names with lower letters */
      len = strlen (str);
      for (i=0; i < len; i++)
         str[i] = tolower (str[i]);
      if (host)
         { if (strcmp (str, "nodemodule") == 0)
             strcpy (str, "hostmodule");
            else
             str[len-1] = 'h'; /* h */
         }
      if (obj)
         strcat (str, ".o");
       else
         strcat (str, ".f");
      if (++nnames % 6 == 0)
          fprintf(mf," \\\n\t");
      if (!host || !pure)
         fprintf (mf, " %s", str);
      units = units->DECL_LIST.Next;
    }
} /* gen_obj_entries */

     /***********************************************
     *                                              *
     *   NSRC = ....                                *
     *   HSRC = ....                                *
     *                                              *
     *   NODE_OBJ = ....                            *
     *   HOST_OBJ = ....                            *
     *   CUBE_OBJ = ....                            *
     *   OBJ1 = ....                                *
     *                                              *
     ***********************************************/

void gen_OBJS (mf, split_flag)
FILE *mf;
bool split_flag;

{ /* Collect all OBJ files

     NODE_OBJ = n_unit1.o n_unit2.o ...
     HOST_OBJ = h_unit1.o h_unit2.o ...        */

  fprintf (mf,"# Object Codes\n");
  fprintf (mf,"#\n");

  if (split_flag)
    { fprintf (mf,"NODE_OBJ ="); gen_obj_entries (mf, false, true);
      fprintf (mf,"\n");
      fprintf (mf,"NSRC ="); gen_obj_entries (mf, false, false);
      fprintf (mf,"\n");
      fprintf (mf,"#\n");
      fprintf (mf,"HOST_OBJ ="); gen_obj_entries (mf, true, true );
      fprintf (mf,"\n");
      fprintf (mf,"HSRC ="); gen_obj_entries (mf, true, false);
      fprintf (mf,"\n");
      fprintf (mf,"#\n");
      fprintf (mf,"CUBE_OBJ = $(NODE_OBJ)\n");
      fprintf (mf,"OBJ1 = $(NODE_OBJ)\n");
      fprintf (mf,"#\n");
    }
  else
    { fprintf (mf,"NODE_OBJ = node.o\n");
      fprintf (mf,"NSRC = node.f\n");
      fprintf (mf,"#\n");
      fprintf (mf,"HOST_OBJ = host.o\n");
      fprintf (mf,"HSRC = host.f\n");
      fprintf (mf,"#\n");
      fprintf (mf,"CUBE_OBJ = cube.o\n");
      fprintf (mf,"#\n");
      fprintf (mf,"OBJ1 = node1.o\n");
      fprintf (mf,"#\n");
    }
}

void gen_split (mf)
FILE *mf;

{ /* splitting host.f and node.f to generate sources */

  char node_source [10];

  if (target_model == ONLY_NODE)
     sprintf (node_source, "cube");
  else if (target_model == HOST_NODE)
     sprintf (node_source, "node");
  else
     sprintf (node_source, "node1");

  fprintf (mf,"$(HSRC): host.stp host.f\n");
  fprintf (mf,"\trm -f $(HSRC) zzz*.f\n");
  fprintf (mf,"\t$(FSPLIT) host.f\n");
  fprintf (mf,"#\n");
  fprintf (mf,"$(NSRC): %s.stp %s.f\n", node_source, node_source);
  fprintf (mf,"\trm -f $(NSRC) zzz*.f\n");
  fprintf (mf,"\t$(FSPLIT) %s.f\n", node_source);
  fprintf (mf,"#\n");
}

          /***************************************************
          *                                                  *
          *  generate Library entries                        *
          *                                                  *
          ***************************************************/

void gen_libraries (mf)
FILE *mf;
{
/*
  fprintf (mf,"XLIB  = $(DALIB)/libadpx.a -lX11\n");
*/
  fprintf (mf,"NLIB  = $(DALIB)/dalib.a\n");
  fprintf (mf,"HLIB  = $(DALIB)/hostlib.a\n");
  fprintf (mf,"LIB1  = $(DALIB)/unilib.a\n");

  fprintf (mf,"#\n");
  fprintf (mf,"# main programs of host, node, cube\n");
  fprintf (mf,"#\n");

  fprintf (mf,"MAIN_HOST  = $(DALIB)/mhost.o\n");
  fprintf (mf,"MAIN_NODE  = $(DALIB)/mnode.o\n");
  fprintf (mf,"MAIN_CUBE  = $(DALIB)/mcube.o\n");
  fprintf (mf,"MAIN_NODE1 = $(DALIB)/mnode1.o\n");
  fprintf (mf,"#\n");
}

          /***************************************************
          *                                                  *
          *  generate compiling entries                      *
          *                                                  *
          ***************************************************/

void gen_compile (mf)          /* Compiling host and node */
FILE *mf;
{
  /* Make rules so that we don't get a problem from the implicit ones */

  fprintf (mf,"# So that we don't get the default rules\n#\n");
  fprintf (mf, ".SUFFIXES:\n");
  fprintf (mf, ".SUFFIXES: .f .o .stp\n");
  fprintf (mf, "# Fortran compilation rule\n");
  fprintf (mf,".f.o:\n");
#ifdef OS2
  fprintf (mf,"\tf2c -w $<\n");
  fprintf (mf,"\t$(CC) $(NOPT) $<\n");
#else
  fprintf (mf,"\t$(NFC) $(NOPT) -c $<\n");
#endif
  fprintf (mf,"#\n");
  fprintf (mf, "# Width stripping rule\n");
  fprintf (mf,".f.stp:\n");
  fprintf (mf,"\t$(FSTRIP) $< 72\n");
  fprintf (mf,"\ttouch $@\n");
  fprintf (mf,"#\n");

  fprintf (mf,"#\n");
  fprintf (mf,"# Compilation rules: host sources, node sources\n");
  fprintf (mf,"#\n");
#ifdef OS2
  fprintf (mf,"host.o: host.stp host.f\n");
  fprintf (mf,"\t$(FC) host.f\n");
  fprintf (mf,"\t$(CC) $(OPT) host.c\n");
  fprintf (mf,"#\n");
  fprintf (mf,"node.o: node.stp node.f\n");
  fprintf (mf,"\t$(NFC) node.f\n");
  fprintf (mf,"\t$(CC) $(OPT) node.c\n");
  fprintf (mf,"#\n");
  fprintf (mf,"cube.o: cube.stp cube.f\n");
  fprintf (mf,"\t$(NFC) cube.f\n");
  fprintf (mf,"\t$(CC) $(OPT) cube.c\n");
  fprintf (mf,"#\n");
  fprintf (mf,"node1.o: node1.stp node1.f\n");
  fprintf (mf,"\t$(NFC) node1.f\n");
  fprintf (mf,"\t$(CC) $(OPT) node1.c\n");
#else
  fprintf (mf,"host.o: host.strip host.f\n");
  fprintf (mf,"\t$(FC) $(OPT) -c host.f\n");
  fprintf (mf,"#\n");
  fprintf (mf,"node.o: node.strip node.f\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -c node.f\n");
  fprintf (mf,"#\n");
  fprintf (mf,"cube.o: cube.strip cube.f\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -c cube.f\n");
  fprintf (mf,"#\n");
  fprintf (mf,"node1.o: node1.strip node1.f\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -c node1.f\n");
#endif
  fprintf (mf,"#\n");

  fprintf (mf,"# Target to perform adaption\n#\n");
  fprintf (mf,"adapt: frc\n");
  fprintf (mf,"\t$(ADAPTOR) ");
  if (interactive)
  {
      /* Should really pull out all the relevant options */
      char * p = filename + strlen(filename);

      for (; *p != '/' && p >= filename; p--)
          ;
      if (p != filename)
          p++;

      fprintf(mf,"%s",p);
  } else
  {
      int i;

      for (i=1; i < fad_argc; i++)
          fprintf(mf,"%s ", fad_argv[i]);
  }
  fprintf(mf,"\n#\nfrc:\n#\n");
}
          /***************************************************
          *                                                  *
          *  generate linking entries (shared memory)        *
          *                                                  *
          ***************************************************/

void gen_link_shared (mf)
FILE *mf;
{
  /* Linking the shared memory host program */

  fprintf (mf,"#\n");
  fprintf (mf,"# host is the shared memory program for host+nodes \n");
  fprintf (mf,"#\n");
#ifdef OS2
  fprintf (mf,"host.exe: $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) \n");
  fprintf (mf,"\t${CC} $(LDFLAGS) -o host.exe $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) $(NLIB)\n");
#else
  fprintf (mf,"host: $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t${FC} $(LDFLAGS) -o host $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) $(NLIB)\n");
#endif
  fprintf (mf,"#\n");

  /* Linking the shared memory cube program */

  fprintf (mf,"#\n");
  fprintf (mf,"# cube is the shared memory program for only nodes \n");
  fprintf (mf,"#\n");
#ifdef OS2
  fprintf (mf,"cube.exe: $(MAIN_CUBE) $(CUBE_OBJ) \n");
  fprintf (mf,"\t${CC} $(LDFLAGS) -o cube.exe $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
#else
  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t${FC} $(LDFLAGS) -o cube $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
#endif
  fprintf (mf,"#\n");

  /* Linking the uniproc program */

  fprintf (mf,"#\n");
  fprintf (mf,"#  node is the program for running on one processor\n");
  fprintf (mf,"#\n");
#ifdef OS2
  fprintf (mf,"node1.exe: $(MAIN_NODE1) $(OBJ1) \n");
  fprintf (mf,"\t${CC} $(LDFLAGS) -o node1.exe $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
#else
  fprintf (mf,"node1: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t${FC} $(LDFLAGS) -o node1 $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
#endif
  fprintf (mf,"#\n");
}

          /***************************************************
          *                                                  *
          *  Makefile for the Parsytec GC                    *
          *                                                  *
          ***************************************************/

void gen_makefile_GC (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "GC");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77.px\n");
  fprintf (mf,"NFC    = f77.px\n");
  fprintf (mf,"CC     = cc.px\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"#\n");
  fprintf (mf,"OPT  = -W0,-Xextensions=1\n");
  fprintf (mf,"NOPT = -W0,-Xextensions=1\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

  if (target_model == HOST_NODE)
     fprintf (mf,"all: host.px\n");
    else if (target_model == ONLY_NODE)
     fprintf (mf,"all: cube.px\n");
    else
     fprintf (mf,"all: node1.px\n");
  fprintf (mf,"#\n");

  /* Linking the executable */

  fprintf (mf,"host.px: $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t${FC} -o host.px $(MAIN_HOST) $(HOST_OBJ) $(NODE_OBJ) $(NLIB)\n");
  fprintf (mf,"#\n");
  fprintf (mf,"cube.px: $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t${FC} -o cube.px $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"#\n");
  fprintf (mf,"node1.px: $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t${FC} -o node1.px $(OBJ1) $(LIB1)\n");

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Running entry */

  fprintf (mf,"run:\n");
  if (target_model == HOST_NODE)
    fprintf (mf,"\t run -s0 -g1 1 4 host\n");
   else if (target_model == ONLY_NODE)
    fprintf (mf,"\t run -s0 -g1 1 4 cube\n");
   else
    fprintf (mf,"\t run -s0 -g1 1 1 cube\n");

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t rm -f *.o host.px cube.px node1.px core");
  fprintf (mf," $(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_GC */

          /***************************************************
          *                                                  *
          *  Makefile for the iPSC/860 (distributed mem.)    *
          *                                                  *
          ***************************************************/

void gen_makefile_IPSC (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf,"IPSC");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = if77\n");
  fprintf (mf,"CC     = cc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf, "# Compiler Options\n");
  fprintf (mf, "OPT  = -e\n");
  fprintf (mf, "NOPT = -O -Mvect -Knoieee -Mextend\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

#ifdef OS2
  if (target_model == HOST_NODE)
      fprintf (mf,"all: host.exe node.exe\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube.exe\n");
    else
      fprintf (mf,"all: node1.exe\n");
  fprintf (mf,"#\n");
#else
  if (target_model == HOST_NODE)
      fprintf (mf,"all: host node\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube\n");
    else
      fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");
#endif

  /* Linking the host program */

  fprintf (mf,"host: $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB)\n");
  fprintf (mf,"\t$(FC) -o host $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB) -host\n");
  fprintf (mf,"#\n");

  /* Linking the node program */

  fprintf (mf,"node: $(MAIN_NODE) $(NODE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t$(NFC) -Knoieee -o node $(MAIN_NODE) $(NODE_OBJ) $(NLIB)\n");
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */

  fprintf (mf,"node1: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t$(NFC) -Knoieee -o node1 $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"#\n");

  /* Linking the cube program */

  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"\t$(NFC) -Knoieee -o cube $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"#\n");

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

 fprintf (mf,"clean:\n");
 fprintf (mf,"\t-rm -f *.o host node node1 cube core $(NSRC) $(HSRC) zzz*.f\n");
 fprintf (mf,"#\n");
 fclose (mf);

} /* gen_makefile_IPSC */

          /***************************************************
          *                                                  *
          *  Makefile for the ALLIANT (shared-memory)        *
          *                                                  *
          ***************************************************/

void gen_makefile_ALLIANT (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "ALLIANT");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = fortran\n");
  fprintf (mf,"NFC    = fortran\n");
  fprintf (mf,"CC     = fxc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = -Oguv -uniproc -extend_source -w\n");
  fprintf (mf,"NOPT = -Oguv -uniproc -extend_source -w\n");
  fprintf (mf,"LDFLAGS = -uniproc\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

#ifdef OS2
  if (target_model == HOST_NODE)
    { fprintf (mf,"all: host.exe\n");
    }
   else if (target_model == ONLY_NODE)
    { fprintf (mf,"all: cube.exe\n");
    }
   else
    fprintf (mf,"all: node1.exe\n");
  fprintf (mf,"#\n");
#else
  if (target_model == HOST_NODE)
    { fprintf (mf,"all: host\n");
    }
   else if (target_model == ONLY_NODE)
    { fprintf (mf,"all: cube\n");
    }
   else
    fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");
#endif

  /* Generate link entries */

  gen_link_shared (mf);

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-rm -f *.o host node1 cube core ");
  fprintf (mf,"$(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_ALLIANT */

          /***************************************************
          *                                                  *
          *  Makefile for the SGI     (shared-memory)        *
          *                                                  *
          ***************************************************/

void gen_makefile_SGI (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "SGI");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = f77\n");
  fprintf (mf,"CC     = cc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = -O -extend_source -w\n");
  fprintf (mf,"NOPT = -O -extend_source -w\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

  if (target_model == HOST_NODE)
    { fprintf (mf,"all: host\n");
    }
   else if (target_model == ONLY_NODE)
    { fprintf (mf,"all: cube\n");
    }
   else
    fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");

  /* Generate link entries */

  gen_link_shared (mf);

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-rm -f *.o host node1 cube core ");
  fprintf (mf,"$(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_SGI */

          /***************************************************
          *                                                  *
          *  Makefile for the KSR 1   (shared-memory)        *
          *                                                  *
          ***************************************************/

void gen_makefile_KSR1 (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "KSR1");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = f77\n");
  fprintf (mf,"CC     = cc\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = -O2 -e -i4 -r8\n");
  fprintf (mf,"NOPT = -O2 -e -i4 -r8\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

  if (target_model == HOST_NODE)
    { fprintf (mf,"all: host\n");
    }
   else if (target_model == ONLY_NODE)
    { fprintf (mf,"all: cube\n");
    }
   else
    fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");

  /* Generate link entries */

  gen_link_shared (mf);

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-rm -f *.o host node1 cube core ");
  fprintf (mf,"$(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_KSR1 */

          /***************************************************
          *                                                  *
          *  Makefile for Net of Workstations (PVM version)  *
          *                                                  *
          ***************************************************/

void gen_makefile_PVM (mf, split_flag, arch)

bool split_flag;     /* true: sources will be splitted for compilation */
FILE *mf;
char arch[];

{    int i;
     char *r;

  /* PHOME Directory , FC, ... is defined */

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"ARCH = %s\n", arch);
  if (!(r = getenv("PVM_ROOT")))
     r = "/pvm3";

  fprintf (mf,"PVM = %s\n", r);

  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  fprintf (mf,"PVMLIB = -lpvm3\n");
  fprintf (mf,"#\n");

  /* Collect all OBJ files  */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

#ifdef OS2
  if (target_model == HOST_NODE)
     { fprintf (mf,"all: host.exe node.exe\n");
       fprintf (mf,"\tmv node.exe $(PVM)/bin/$(ARCH)\n");
     }
    else if (target_model == ONLY_NODE)
     { /* pnode for start and load */
       fprintf (mf,"all: cube.exe\n");
       fprintf (mf,"\tcp cube.exe $(PVM)/bin/$(ARCH)\n");
     }
    else /* uniproc is executable node */
     { fprintf (mf,"all: node1.exe\n");
     }
  fprintf (mf,"#\n");
#else
  if (target_model == HOST_NODE)
     { fprintf (mf,"all: host node\n");
       fprintf (mf,"\tmv node $(PVM)/bin/$(ARCH)\n");
     }
    else if (target_model == ONLY_NODE)
     { /* pnode for start and load */
       fprintf (mf,"all: cube\n");
       fprintf (mf,"\tcp cube $(PVM)/bin/$(ARCH)\n");
     }
    else /* uniproc is executable node */
     { fprintf (mf,"all: node1\n");
     }
  fprintf (mf,"#\n");
#endif

  /* Linking the PVM host program */

#ifdef OS2
  fprintf (mf,"host.exe: $(HOST_OBJ) $(MAIN_HOST) $(NLIB) \n");
  fprintf (mf,"\t${CC} -s -o host.exe $(MAIN_HOST) $(HOST_OBJ) $(NLIB) $(PVMLIB) -lso32 -lxdr -ltcp32 -los2 -lf2c \n");
#else
  fprintf (mf,"host: $(HOST_OBJ) $(MAIN_HOST) $(NLIB) $(PVMLIB)\n");
  fprintf (mf,"\t${FC} -o host $(MAIN_HOST) $(HOST_OBJ) $(NLIB) $(PVMLIB)\n");
#endif
  fprintf (mf,"#\n");

  /* Linking the PVM node program */

#ifdef OS2
  fprintf (mf,"node.exe: $(NODE_OBJ) $(MAIN_NODE) $(NLIB) \n");
  fprintf (mf,"\t${CC} -s -o node.exe $(MAIN_NODE) $(NODE_OBJ) $(NLIB) $(PVMLIB) -lso32 -lxdr -ltcp32 -los2 -lf2c \n");
#else
  fprintf (mf,"node: $(NODE_OBJ) $(MAIN_NODE) $(NLIB) $(PVMLIB)\n");
  fprintf (mf,"\t${FC} -o node $(MAIN_NODE) $(NODE_OBJ) $(NLIB) $(PVMLIB)\n");
#endif
  fprintf (mf,"#\n");

  /* Linking the PVM cube program */

#ifdef OS2
  fprintf (mf,"cube.exe: $(CUBE_OBJ) $(MAIN_CUBE) $(NLIB) \n");
  fprintf (mf,"\t${CC} -s -o cube.exe $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB) $(PVMLIB) -lso32 -lxdr -ltcp32 -los2 -lf2c \n");
#else
  fprintf (mf,"cube: $(CUBE_OBJ) $(MAIN_CUBE) $(NLIB) $(PVMLIB)\n");
  fprintf (mf,"\t${FC} -o cube $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB) $(PVMLIB)\n");
#endif
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */

#ifdef OS2
  fprintf (mf,"node1.exe: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t${CC} -s -o node1.exe $(MAIN_NODE1) $(OBJ1) $(LIB1) -lf2c -los2\n");
#else
  fprintf (mf,"node1: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t${FC} -o node1 $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
#endif
  fprintf (mf,"#\n");

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
#ifdef OS2
  fprintf (mf,"\t-rm -f *.o node.exe host.exe node1.exe cube.exe core ");
#else
  fprintf (mf,"\t-rm -f *.o node host node1 cube core ");
#endif
  fprintf (mf,"$(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");

} /* gen_makefile_PVM */

          /***************************************************
          *                                                  *
          *  Makefile for the ALLIANT (PVM)                  *
          *                                                  *
          ***************************************************/

void gen_makefile_ALL_PVM (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf,"PVM3.1");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = fortran\n");
  fprintf (mf,"NFC    = fortran\n");
  fprintf (mf,"CC     = fxc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"#\n");
  fprintf (mf,"OPT  = -Oguv -uniproc -extend_source -w\n");
  fprintf (mf,"NOPT = -Oguv -uniproc -extend_source -w\n");
  fprintf (mf,"#\n");

  gen_makefile_PVM (mf, split_flag, "FX2800");

  fclose (mf);

} /* gen_makefile_ALL_PVM */

          /***************************************************
          *                                                  *
          *  Makefile for the KSR1    (PVM)                  *
          *                                                  *
          ***************************************************/

void gen_makefile_KSR1_PVM (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "PVM3.1");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = f77\n");
  fprintf (mf,"CC     = cc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = -O2 -e -i4 -r8");
  fprintf (mf,"NOPT = -O2 -e -i4 -r8");
  fprintf (mf,"#\n");

  gen_makefile_PVM (mf, split_flag, "KSR1");

  fclose (mf);

} /* gen_makefile_KSR1_PVM */

          /***************************************************
          *                                                  *
          *  Makefile for Net of SUN4 (PVM version)          *
          *                                                  *
          ***************************************************/

void gen_makefile_SUN4_PVM (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "PVM3.1");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC  = /vol/lang/f77\n");
  fprintf (mf,"NFC = /vol/lang/f77\n");
  fprintf (mf,"FSPLIT = /vol/lang/fsplit\n");
  fprintf (mf,"CC  = cc\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = %s\n", SUN4_PVM_OPTIONS);
  fprintf (mf,"NOPT = %s\n", SUN4_PVM_OPTIONS);
  fprintf (mf,"#\n");

  gen_makefile_PVM (mf, split_flag, "SUN4");

  fclose (mf);

} /* gen_makefile_SUN4_PVM */


          /***************************************************
          *                                                  *
          *  Makefile for Net of OS2 (PVM version)           *
          *                                                  *
          ***************************************************/

void gen_makefile_OS2_PVM (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "PVM3.2");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC  = f2c -w\n");
  fprintf (mf,"NFC = f2c -w\n");
  fprintf (mf,"FSPLIT = fsplit\n");
  fprintf (mf,"CC  = gcc\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = -c -O2\n");
  fprintf (mf,"NOPT = -c\n");
  fprintf (mf,"#\n");

  split_flag = 0;
  gen_makefile_PVM (mf, split_flag, "OS2");

  fclose (mf);

} /* gen_makefile_OS2_PVM */


          /***************************************************
          *                                                  *
          *  Makefile for Net of IBM RISC (PVM version)      *
          *                                                  *
          ***************************************************/

void gen_makefile_RIOS_PVM (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "PVM3.1");

  /* variables for compiler, pvm architecturs */
  fprintf (mf,"#\n");
  fprintf (mf,"FC  = xlf\n");
  fprintf (mf,"NFC = xlf\n");
  fprintf (mf,"CC  = cc\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# Compiler Options\n");
  fprintf (mf,"OPT  = %s\n", RIOS_PVM_OPTIONS);
  fprintf (mf,"NOPT = %s\n", RIOS_PVM_OPTIONS);
  fprintf (mf,"#\n");

  gen_makefile_PVM (mf, split_flag, "RIOS");

  fclose (mf);

} /* gen_makefile_RIOS_PVM */

          /*****************************************************
          *                                                    *
          *  Makefile for the Meiko CS 1   (distributed mem.)  *
          *                                                    *
          *****************************************************/

void gen_makefile_MEIKO_CS1 (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "MEIKO-CS1");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = pgf77\n");
  fprintf (mf,"NFC    = pgf77\n");
  fprintf (mf,"FSTRIP = $(PHOME)/bin/fstrip\n");

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf, "# Compiler Options\n");
  fprintf (mf, "OPT  = -Mextend\n");
  fprintf (mf, "NOPT = -Minfo=loop -O4 -Mextend -Mvect -Mnoframe\n");
  fprintf (mf,"#\n");

  /* Libraries */

/*
  fprintf (mf,"XLIB    = $(DALIB)/libadpx.a -lX11\n");
*/
  fprintf (mf,"LIBS    = -lmpsc -lcs -lcsn\n");
  fprintf (mf,"NODELIB = $(DALIB)/libnode.a\n");
  fprintf (mf,"HOSTLIB = $(DALIB)/libhost.a\n");
  fprintf (mf,"#\n");
  fprintf (mf,"LIB1    = $(DALIB)/unilib.a\n");
  fprintf (mf,"MAIN_NODE   = $(DALIB)/mnode.o\n");
  fprintf (mf,"MAIN_CUBE   = $(DALIB)/mcube.o\n");
  fprintf (mf,"MAIN_HOST   = $(DALIB)/mhost.o\n");
  fprintf (mf,"MAIN_NODE1  = $(DALIB)/mnode1.o\n");
  fprintf (mf,"#\n");

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

#ifdef OS2
  if (target_model == HOST_NODE)
      fprintf (mf,"all: host.exe node.exe\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube.exe\n");
    else
      fprintf (mf,"all: node1.exe\n");
  fprintf (mf,"#\n");
#else
  if (target_model == HOST_NODE)
      fprintf (mf,"all: host node\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube\n");
    else
      fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");
#endif

  /* Linking the host program */

  fprintf (mf,"host: $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB)\n");
  fprintf (mf,"\t$(FC) -o host $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the node program */

  fprintf (mf,"node: $(MAIN_NODE) $(NODE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(NFC) -o node $(MAIN_NODE) $(NODE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the cube program */

  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(NFC) -o cube $(MAIN_CUBE) $(CUBE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */

  fprintf (mf,"node1: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t$(NFC) -o node1 $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"#\n");

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-/bin/rm -f *.o vpu/*.o *.strip adaptor.* *~ host node \\\n");
  fprintf (mf,"\t            cube vcube core $(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_MEIKO_CS1 */

          /*****************************************************
          *                                                    *
          *  Makefile for the Meiko CS 2   (distributed mem.)  *
          *                                                    *
          *****************************************************/

void gen_makefile_MEIKO_CS2 (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "MEIKO-CS2");

  /* variables for compiler, pvm architecturs */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = f77\n");
  fprintf (mf,"VFC    = pgf77\n");
  fprintf (mf,"MHOME  = /opt/MEIKOcs2\n");  /* Root of CS2 code */

  /* setting the compiler options for compiling generated functions */
  fprintf (mf,"#\n");
  fprintf (mf,"# OPT : Compiler Options for SunPRO f77\n");
  fprintf (mf,"OPT  = -Nl100 -e -w -fast\n");
  fprintf (mf,"NOPT = $(OPT)\n");
  fprintf (mf,"#\n");
  fprintf (mf,"# VOPT : Compiler Options for PGI vectorising f77\n");
  fprintf (mf,"#\n");
  fprintf (mf,"VOPT = -Mextend -O4 -Mvect -Mx,80,1 -Mx,98,2\n");
  fprintf (mf,"#\n");

  /* Libraries */

/*
  fprintf (mf,"XLIB        = $(DALIB)/libadpx.a -lX11\n");
*/
  fprintf (mf,"LIBS        = -L$(MHOME)/lib -lmpsc -lew -lelan -laio\n");
  fprintf (mf,"NODELIB     = $(DALIB)/libnode.a\n");
  fprintf (mf,"HOSTLIB     = $(DALIB)/libhost.a\n");
  fprintf (mf,"LIB1        = $(DALIB)/libuni.a\n");
  fprintf (mf,"#\n");
  fprintf (mf,"# Special MAIN Needed for Sun's f77 compiler\n");
  fprintf (mf,"MAIN_SUNF77 = $(DALIB)/cubemain.o\n");
  fprintf (mf,"MAIN_NODE   = $(DALIB)/mnode.o\n");
  fprintf (mf,"MAIN_CUBE   = $(DALIB)/mcube.o\n");
  fprintf (mf,"MAIN_HOST   = $(DALIB)/mhost.o\n");
  fprintf (mf,"MAIN_NODE1  = $(DALIB)/mnode1.o\n");
  fprintf (mf,"#\n");

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  fprintf(mf,"# Object files for the vector node\n");
  fprintf(mf,"VNODE_OBJ = $(NODE_OBJ:%%=vpu/%%)\n");
  fprintf(mf,"VCUBE_OBJ = $(CUBE_OBJ:%%=vpu/%%)\n");
  fprintf(mf,"VOBJ1     = $(OBJ1:%%=vpu/%%)\n#\n");

  /* main or starting entry */

  if (target_model == HOST_NODE)
      fprintf (mf,"all: host node\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube\n");
    else
      fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");


  /* Linking the host program */
  fprintf (mf,"host: $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB)\n");
  fprintf (mf,"\t$(FC) $(OPT) -o $@ $(MAIN_SUNF77) $(MAIN_HOST) $(HOST_OBJ) $(HOSTLIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the node program */
  fprintf (mf,"node: $(MAIN_NODE) $(NODE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -o $@ $(MAIN_SUNF77) $(MAIN_NODE) $(NODE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");
  fprintf (mf,"vnode: $(MAIN_NODE) $(VNODE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(VFC) $(VOPT) -o $@ $(MAIN_NODE) $(VNODE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the cube program */
  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -o $@ $(MAIN_SUNF77) $(MAIN_CUBE) $(CUBE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");
  /* Linking the vectorised cube program */
  fprintf (mf,"vcube: $(MAIN_CUBE) $(VCUBE_OBJ) $(NODELIB)\n");
  fprintf (mf,"\t$(VFC) $(VOPT) -o $@ $(MAIN_CUBE) $(VCUBE_OBJ) $(NODELIB) $(LIBS)\n");
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */
  fprintf (mf,"node1: $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -o $@ $(MAIN_SUNF77) $(MAIN_NODE1) $(OBJ1) $(LIB1)\n");
  fprintf (mf,"#\n");
  /* Linking the vectorised uniproc node program */
  fprintf (mf,"vnode1: $(MAIN_NODE1) $(VOBJ1) $(LIB1)\n");
  fprintf (mf,"\t$(VFC) $(VOPT) -o $@ $(MAIN_NODE1) $(VOBJ1) $(LIB1)\n");
  fprintf (mf,"#\n");


  /* Compiling host and node */
  gen_compile (mf);
  /* Additional rules */
  fprintf (mf,"vpu/%%.o: %%.f\n");
  fprintf (mf,"\t$(VFC) $(VOPT) -c -o $@ $<\n#\n");

  /* Splitting node.f and host.f to generate sources */
  if (split_flag) gen_split (mf);

  fprintf (mf,"$(VNODE_OBJ) $(VCUBE_OBJ) $(VOBJ1): vpu \n");
  fprintf (mf,"#\n");

  fprintf (mf,"vpu: \n\tmkdir vpu\n#\n");

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-/bin/rm -f *.o vpu/*.o *.strip adaptor.* *~ host node \\\n");
  fprintf (mf,"\t            cube vcube node1 vnode1 core $(NSRC) $(HSRC) node1.f cube.f zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_MEIKO_CS2 */

          /***************************************************
          *                                                  *
          *  Makefile for the CM 5     (distributed mem.)    *
          *                                                  *
          ***************************************************/

void gen_makefile_CM5 (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "CM5");

  /* variables for compiler */

  fprintf (mf,"FC     = f77\n");
  fprintf (mf,"NFC    = f77\n");
  fprintf (mf,"CC     = cc\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf, "# Compiler Options\n");
  fprintf (mf, "OPT  = -e -w -O\n");
  fprintf (mf, "NOPT = -e -w -O\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

  if (target_model == HOST_NODE)
      fprintf (mf,"all: host\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube\n");
    else
      fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");

  /* Linking the host and node program */

  fprintf (mf,"host: $(MAIN_HOST) $(MAIN_NODE) $(HOST_OBJ) $(NODE_OBJ) $(HLIB) $(NLIB)\n");
  fprintf (mf,"\tcmmd-ld -comp $(FC) -v -g -o host \\\n");
  fprintf (mf,"\t-node $(MAIN_NODE) $(NODE_OBJ) $(NLIB) \\\n");
  fprintf (mf,"\t-host $(MAIN_HOST) $(HOST_OBJ) $(HLIB) \n");
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */

  fprintf (mf,"node1: $(MAIN_NODE1) $(NODE_OBJ) $(LIB1)\n");
  fprintf (mf,"\t$(NFC) -o node $(MAIN_NODE1) $(NODE_OBJ) $(LIB1)\n");
  fprintf (mf,"#\n");

  /* Linking the cube program */

  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"\tcmmd-ld -comp f77 -o cube $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"#\n");

  /* Compiling host and node */

  gen_compile (mf);

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-rm -f *.o host cube host.trace $(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_CM5 */

          /***************************************************
          *                                                  *
          *  Makefile for the CM 5     (distributed mem.)    *
          *                                                  *
          *  this solution is for using VECTOR UNITS         *
          *  and using the CM Fortran compiler for nodes     *
          *                                                  *
          ***************************************************/

void gen_makefile_CM5a (split_flag)

bool split_flag;     /* true: sources will be splitted for compilation */

{ FILE *mf;
  int i;

  mf = fopen ("Makefile","w");

  gen_header (mf, "CM5a");
  fprintf (mf,"CMMD_ROOT = /usr/cmmd/3.0\n");

  /* variables for compiler */

  fprintf (mf,"FC     = cmf\n");
  fprintf (mf,"NFC    = cmf\n");

  /* setting the compiler options for compiling generated functions */

  fprintf (mf,"#\n");
  fprintf (mf, "# Compiler Options\n");
  fprintf (mf, "OPT  = -e -w -O\n");
  fprintf (mf, "NOPT = -e -w -O\n");
  fprintf (mf,"#\n");

  /* Libraries */

  gen_libraries (mf);

  /* Collect all OBJ files */

  gen_OBJS (mf, split_flag);

  /* main or starting entry */

  if (target_model == HOST_NODE)
      fprintf (mf,"all: host\n");
    else if (target_model == ONLY_NODE)
      fprintf (mf,"all: cube\n");
    else
      fprintf (mf,"all: node1\n");
  fprintf (mf,"#\n");

  /* Linking the uniproc node program */

  fprintf (mf,"node1: $(NODE_OBJ) $(LIB1)\n");
  fprintf (mf,"\t$(NFC) -o node $(NODE_OBJ) $(LIB1)\n");
  fprintf (mf,"#\n");

  /* Linking the cube program */

  fprintf (mf,"cube: $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB)\n");
  fprintf (mf,"\tcmf -node -cmmd_root $(CMMD_ROOT)  $(MAIN_CUBE) $(CUBE_OBJ) $(NLIB) -v -o cube\n");
  fprintf (mf,"#\n");

  /* Compiling host and node */

  fprintf (mf,"cube.o: cube.fcm\n");
  fprintf (mf,"\t$(NFC) $(NOPT) -c cube.fcm\n");
  fprintf (mf,"#\n");
  fprintf (mf,"cube.fcm: cube.f\n");
  fprintf (mf,"\tmv cube.f cube.fcm\n");

  /* Splitting node.f and host.f to generate sources */

  if (split_flag) gen_split (mf);

  /* Cleaning entry */

  fprintf (mf,"clean:\n");
  fprintf (mf,"\t-rm -f *.o host cube host.trace $(NSRC) $(HSRC) zzz*.f\n");
  fprintf (mf,"#\n");
  fclose (mf);

} /* gen_makefile_CM5 */

void gen_makefile ()

{ /* split_flag is global from global.h */

    /* Only write a Makefile if it doesn't already exist */

   /*    if (access("Makefile", R_OK) != 0)   */

    switch(target_machine)
    {
      case OS2_PVM:     gen_makefile_OS2_PVM   (split_flag); break;
      case SUN4_PVM:    gen_makefile_SUN4_PVM  (split_flag); break;
      case RIOS_PVM:    gen_makefile_RIOS_PVM  (split_flag); break;
      case ALLIANT :    gen_makefile_ALLIANT   (split_flag); break;
      case IPSC_860:    gen_makefile_IPSC      (split_flag); break;
      case PARSYTEC_GC: gen_makefile_GC        (split_flag); break;
      case MEIKO_CS1:   gen_makefile_MEIKO_CS1 (split_flag); break;
      case MEIKO_CS2:   gen_makefile_MEIKO_CS2 (split_flag); break;
      case KSR1:        gen_makefile_KSR1      (split_flag); break;
      case KSR1_PVM:    gen_makefile_KSR1_PVM  (split_flag); break;
      case ALL_PVM:     gen_makefile_ALL_PVM   (split_flag); break;
      case CM5:         if (target_language == FORTRAN_90)
                          { gen_makefile_CM5a      (split_flag); break; }
                         else
                          { gen_makefile_CM5       (split_flag); break; }
      case SGI:         gen_makefile_SGI       (split_flag); break;

      default : fprintf (stderr,"gen_makefile failed, unknown machine\n");
                break;
    }

} /* gen_makefile */

