/* cmds.c
 *
 * Copyright (c) 1996 Mike Gleason, NCEMRSoft.
 * All rights reserved.
 *
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF NCEMRSOFT.
 * The copyright notice above does not evidence any actual or intended
 * publication of such source code.
 */

#include "syshdrs.h"

int
FTPChdir(const FTPCIPtr cip, const char *cdCwd)
{
	int result;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((cdCwd == NULL) || (cdCwd[0] == '\0')) {
		result = kErrInvalidDirParam;
		cip->errNo = kErrInvalidDirParam;
	} else {
		if (strcmp(cdCwd, "..") == 0)
			result = FTPCmd(cip, "CDUP");
		else
			result = FTPCmd(cip, "CWD %s", cdCwd);
		if (result >= 0) {
			if (result == 2) {
				result = 0;
			} else {
				result = kErrCWDFailed;
				cip->errNo = kErrCWDFailed;
			}
		}
	}
	return (result);
}	/* FTPChdir */




int
FTPChmod(const FTPCIPtr cip, const char *pattern, const char *mode, int doGlob)
{
	LineList fileList;
	LinePtr filePtr;
	char *file;
	int onceResult, batchResult;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	(void) FTPRemoteGlob(cip, &fileList, pattern, doGlob);
	for (batchResult = kNoErr, filePtr = fileList.first;
		filePtr != NULL;
		filePtr = filePtr->next)
	{
		file = filePtr->line;
		if (file == NULL) {
			batchResult = kErrBadLineList;
			cip->errNo = kErrBadLineList;
			break;
		}
		onceResult = FTPCmd(cip, "SITE CHMOD %s %s", mode, file); 	
		if (onceResult < 0) {
			batchResult = onceResult;
			break;
		}
		if (onceResult != 2) {
			batchResult = kErrChmodFailed;
			cip->errNo = kErrChmodFailed;
		}
	}
	DisposeLineListContents(&fileList);
	return (batchResult);
}	/* FTPChmod */




int
FTPDelete(const FTPCIPtr cip, const char *pattern, int recurse, int doGlob)
{
	LineList fileList;
	LinePtr filePtr;
	char *file;
	int onceResult, batchResult;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	/* recurse option will not be implemented yet.
	 *
	 * When it is, you can remove the file or directory, but also
	 * subdirectories in it (i.e. rm -r).
	 */
	if (recurse != kRecursiveNo) {
		cip->errNo = kErrUnimplementedOption;
		return (kErrUnimplementedOption);
	}

	(void) FTPRemoteGlob(cip, &fileList, pattern, doGlob);
	for (batchResult = kNoErr, filePtr = fileList.first;
		filePtr != NULL;
		filePtr = filePtr->next)
	{
		file = filePtr->line;
		if (file == NULL) {
			batchResult = kErrBadLineList;
			cip->errNo = kErrBadLineList;
			break;
		}
		onceResult = FTPCmd(cip, "DELE %s", file);
		if (onceResult < 0) {
			batchResult = onceResult;
			break;
		}
		if (onceResult != 2) {
			batchResult = kErrDELEFailed;
			cip->errNo = kErrDELEFailed;
		}
	}
	DisposeLineListContents(&fileList);
	return (batchResult);
}	/* FTPDelete */




int
FTPGetCWD(const FTPCIPtr cip, char *newCwd, size_t newCwdSize)
{
	ResponsePtr rp;
	char *l, *r;
	int result;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((newCwd == NULL) || (newCwdSize == 0)) {
		result = kErrInvalidDirParam;
		cip->errNo = kErrInvalidDirParam;
	} else {
		rp = InitResponse();
		if (rp == NULL) {
			result = kErrMallocFailed;
			cip->errNo = kErrMallocFailed;
			Error(cip, kDontPerror, "Malloc failed.\n");
		} else {
			result = RCmd(cip, rp, "PWD");
			if (result == 2) {
				if ((r = strrchr(rp->msg.first->line, '"')) != NULL) {
					/* "xxxx" is current directory.
					 * Strip out just the xxxx to copy into the remote cwd.
					 */
					l = strchr(rp->msg.first->line, '"');
					if ((l != NULL) && (l != r)) {
						*r = '\0';
						++l;
						(void) Strncpy(newCwd, l, newCwdSize);
						*r = '"';	/* Restore, so response prints correctly. */
					}
				} else {
					/* xxxx is current directory.
					 * Mostly for VMS.
					 */
					if ((r = strchr(rp->msg.first->line, ' ')) != NULL) {
						*r = '\0';
						(void) Strncpy(newCwd, (rp->msg.first->line), newCwdSize);
						*r = ' ';	/* Restore, so response prints correctly. */
					}
				}
				result = kNoErr;
			} else if (result > 0) {
				result = kErrPWDFailed;
				cip->errNo = kErrPWDFailed;
			}
			DoneWithResponse(cip, rp);
		}
	}
	return (result);
}	/* FTPGetCWD */




int
FTPChdirAndGetCWD(const FTPCIPtr cip, const char *cdCwd, char *newCwd, size_t newCwdSize)
{
	ResponsePtr rp;
	char *l, *r;
	int result;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((newCwd == NULL) || (cdCwd == NULL) || (cdCwd[0] == '\0')) {
		result = kErrInvalidDirParam;
		cip->errNo = kErrInvalidDirParam;
	} else {
		rp = InitResponse();
		if (rp == NULL) {
			result = kErrMallocFailed;
			cip->errNo = kErrMallocFailed;
			Error(cip, kDontPerror, "Malloc failed.\n");
		} else {
			if (strcmp(cdCwd, "..") == 0)
				result = RCmd(cip, rp, "CDUP"); 	
			else
				result = RCmd(cip, rp, "CWD %s", cdCwd);
			if (result == 2) {
				l = strchr(rp->msg.first->line, '"');
				if ((l == rp->msg.first->line) && ((r = strrchr(rp->msg.first->line, '"')) != NULL) && (l != r)) {
					/* "xxxx" is current directory.
					 * Strip out just the xxxx to copy into the remote cwd.
					 *
					 * This is nice because we didn't have to do a PWD.
					 */
					*r = '\0';
					++l;
					(void) Strncpy(newCwd, l, newCwdSize);
					*r = '"';	/* Restore, so response prints correctly. */
					DoneWithResponse(cip, rp);
					result = kNoErr;
				} else {
					DoneWithResponse(cip, rp);
					result = FTPGetCWD(cip, newCwd, newCwdSize);
				}
			} else if (result > 0) {
				result = kErrCWDFailed;
				cip->errNo = kErrCWDFailed;
				DoneWithResponse(cip, rp);
			} else {
				DoneWithResponse(cip, rp);
			}
		}
	}
	return (result);
}	/* FTPChdirAndGetCWD */




int
FTPMkdir(const FTPCIPtr cip, const char *newDir, int recurse)
{
	int result, result2;
	char *cp, *newTreeStart;
	char dir[512];
	char dir2[512];

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((newDir == NULL) || (newDir[0] == '\0')) {
		cip->errNo = kErrInvalidDirParam;
		return (kErrInvalidDirParam);
	}

	if (recurse == kRecursiveNo) {
		result = FTPCmd(cip, "MKD %s", newDir);
		if (result > 0) {
			if (result != 2) {
				Error(cip, kDontPerror, "MKD %s failed; [%s]\n", newDir, cip->lastFTPCmdResultStr);
				result = kErrMKDFailed;
				cip->errNo = kErrMKDFailed;
				return (result);
			} else {
				result = kNoErr;
			}
		}
	} else {
		/* Preserve old working directory. */
		(void) FTPGetCWD(cip, cip->buf, cip->bufSize);

		result = FTPChdir(cip, newDir);
		if (result == 0)
			goto goback;	/* Already there. */

		(void) STRNCPY(dir, newDir);

		/* Strip trailing slashes. */
		cp = dir + strlen(dir) - 1;
		while (1) {
			if (cp <= dir) {
				if ((newDir == NULL) || (newDir[0] == '\0')) {
					cip->errNo = kErrInvalidDirParam;
					result = kErrInvalidDirParam;
					goto goback;
				}
			}
			if (*cp != '/') {
				cp[1] = '\0';
				break;
			}
			--cp;
		}
		(void) STRNCPY(dir2, dir);

		for (;;) {
			cp = strrchr(dir, '/');
			if (cp == NULL) {
				cp = dir + strlen(dir) - 1;
				if (dir[0] == '\0') {
					result = kErrMKDFailed;
					cip->errNo = kErrMKDFailed;
					goto goback;
				}
				cp = dir - 1;
				break;
			}
			if (cp == dir) {
				result = kErrMKDFailed;
				cip->errNo = kErrMKDFailed;
				goto goback;
			}
			*cp = '\0';
			result = FTPChdir(cip, dir);
			if (result == 0)
				break;	/* Found a valid parent dir. */
		}

		newTreeStart = dir2 + ((cp + 1) - dir);
		for (cp = newTreeStart; ; ) {
			cp = strchr(cp, '/');
			if (cp != NULL) {
				*cp = '\0';
				if (cp[1] == '\0') {
					/* Done, if they did "mkdir /tmp/dir/" */
					break;
				}
			}
			result = FTPCmd(cip, "MKD %s", newTreeStart);
			if (result < 0) {
				return (result);
			}
			if (result != 2) {
				Error(cip, kDontPerror, "Cwd=%s; MKD %s failed; [%s]\n", cip->buf, newTreeStart, cip->lastFTPCmdResultStr);
				result = kErrMKDFailed;
				cip->errNo = kErrMKDFailed;
				goto goback;
			}
			if (cp == NULL)
				break;	/* No more to make, done. */
			*cp++ = '/';
		}
		result = kNoErr;

goback:
		result2 = FTPChdir(cip, cip->buf);
		if ((result == 0) && (result2 < 0)) {
			result = kErrCannotGoToPrevDir;
			cip->errNo = kErrCannotGoToPrevDir;
		}
	}
	return (result);
}	/* FTPMkdir */




int
FTPFileModificationTime(const FTPCIPtr cip, const char *file, time_t *mdtm)
{
	int result;
	ResponsePtr rp;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((mdtm == NULL) || (file == NULL))
		return (kErrBadParameter);
	*mdtm = kModTimeUnknown;

	if (cip->hasMDTM == kCommandNotAvailable) {
		cip->errNo = kErrMDTMNotAvailable;
		result = kErrMDTMNotAvailable;
	} else {
		rp = InitResponse();
		if (rp == NULL) {
			result = kErrMallocFailed;
			cip->errNo = kErrMallocFailed;
			Error(cip, kDontPerror, "Malloc failed.\n");
		} else {
			result = RCmd(cip, rp, "MDTM %s", file);
			if (result < 0) {
				DoneWithResponse(cip, rp);
				return (result);
			} else if (result == 2) {
				*mdtm = UnMDTMDate(rp->msg.first->line);
				cip->hasMDTM = kCommandAvailable;
				result = kNoErr;
			} else if (UNIMPLEMENTED_CMD(rp->code)) {
				cip->hasMDTM = kCommandNotAvailable;
				cip->errNo = kErrMDTMNotAvailable;
				result = kErrMDTMNotAvailable;
			} else {
				cip->errNo = kErrMDTMFailed;
				result = kErrMDTMFailed;
			}
			DoneWithResponse(cip, rp);
		}
	}
	return (result);
}	/* FTPFileModificationTime */




int
FTPRename(const FTPCIPtr cip, const char *oldname, const char *newname)
{
	int result;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	if ((oldname == NULL) || (oldname[0] == '\0'))
		return (kErrBadParameter);
	if ((newname == NULL) || (oldname[0] == '\0'))
		return (kErrBadParameter);

	
	result = FTPCmd(cip, "RNFR %s", oldname);
	if (result < 0)
		return (result);
	if (result != 3) {
		cip->errNo = kErrRenameFailed;
		return (cip->errNo);
	}
	
	result = FTPCmd(cip, "RNTO %s", newname);
	if (result < 0)
		return (result);
	if (result != 2) {
		cip->errNo = kErrRenameFailed;
		return (cip->errNo);
	}
	return (kNoErr);
}	/* FTPRename */




int
FTPRemoteHelp(const FTPCIPtr cip, const char *pattern, LineListPtr llp)
{
	int result;
	ResponsePtr rp;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	rp = InitResponse();
	if (rp == NULL) {
		result = kErrMallocFailed;
		cip->errNo = kErrMallocFailed;
		Error(cip, kDontPerror, "Malloc failed.\n");
	} else {
		if ((pattern == NULL) || (*pattern == '\0'))
			result = RCmd(cip, rp, "HELP");
		else
			result = RCmd(cip, rp, "HELP %s", pattern);
		if (result < 0) {
			DoneWithResponse(cip, rp);
			return (result);
		} else if (result == 2) {
			if (CopyLineList(llp, &rp->msg) < 0) {
				result = kErrMallocFailed;
				cip->errNo = kErrMallocFailed;
				Error(cip, kDontPerror, "Malloc failed.\n");
			} else {
				result = kNoErr;
			}
		} else {
			cip->errNo = kErrHELPFailed;
			result = kErrHELPFailed;
		}
		DoneWithResponse(cip, rp);
	}
	return (result);
}	/* FTPRemoteHelp */




int
FTPRmdir(const FTPCIPtr cip, const char *pattern, int recurse, int doGlob)
{
	LineList fileList;
	LinePtr filePtr;
	char *file;
	int onceResult, batchResult;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	/* recurse option will not be implemented yet.
	 *
	 * When it is, you can remove not only the directory, but
	 * all files and subdirectories in it (i.e. rm -r).
	 * When you don't supply the recurse flag, it just acts as
	 * rmdir, so if the directory is non-empty it won't work.
	 */
	if (recurse != kRecursiveNo) {
		cip->errNo = kErrUnimplementedOption;
		return (kErrUnimplementedOption);
	}

	(void) FTPRemoteGlob(cip, &fileList, pattern, doGlob);
	for (batchResult = kNoErr, filePtr = fileList.first;
		filePtr != NULL;
		filePtr = filePtr->next)
	{
		file = filePtr->line;
		if (file == NULL) {
			batchResult = kErrBadLineList;
			cip->errNo = kErrBadLineList;
			break;
		}
		onceResult = FTPCmd(cip, "RMD %s", file); 	
		if (onceResult < 0) {
			batchResult = onceResult;
			break;
		}
		if (onceResult != 2) {
			batchResult = kErrRMDFailed;
			cip->errNo = kErrRMDFailed;
		}
	}
	DisposeLineListContents(&fileList);
	return (batchResult);
}	/* FTPRmdir */




int
FTPSetTransferType(const FTPCIPtr cip, int type)
{
	int result;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if (cip->curTransferType != type) {
		switch (type) {
			case kTypeAscii:
			case kTypeBinary:
			case kTypeEbcdic:
				break;
			case 'i':
			case 'b':
			case 'B':
				type = kTypeBinary;
				break;
			case 'e':
				type = kTypeEbcdic;
				break;
			case 'a':
				type = kTypeAscii;
				break;
			default:
				/* Yeah, we don't support Tenex.  Who cares? */
				Error(cip, kDontPerror, "Bad transfer type [%c].\n", type);
				cip->errNo = kErrBadTransferType;
				return (kErrBadTransferType);
		}
		result = FTPCmd(cip, "TYPE %c", type);
		if (result != 2) {
			result = kErrTYPEFailed;
			cip->errNo = kErrTYPEFailed;
			return (result);
		}
		cip->curTransferType = type;
	}
	return (kNoErr);
}	/* FTPSetTransferType */




/* If the remote host supports the SIZE command, we can find out the exact
 * size of a remote file, depending on the transfer type in use.  SIZE
 * returns different values for ascii and binary modes!
 */
int
FTPFileSize(const FTPCIPtr cip, const char *file, longest_int *size, int type)
{
	int result;
	ResponsePtr rp;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((size == NULL) || (file == NULL))
		return (kErrBadParameter);
	*size = kSizeUnknown;

	result = FTPSetTransferType(cip, type);
	if (result < 0)
		return (result);

	if (cip->hasSIZE == kCommandNotAvailable) {
		cip->errNo = kErrSIZENotAvailable;
		result = kErrSIZENotAvailable;
	} else {
		rp = InitResponse();
		if (rp == NULL) {
			result = kErrMallocFailed;
			cip->errNo = kErrMallocFailed;
			Error(cip, kDontPerror, "Malloc failed.\n");
		} else {
			result = RCmd(cip, rp, "SIZE %s", file);
			if (result < 0) {
				DoneWithResponse(cip, rp);
				return (result);
			} else if (result == 2) {
#if defined(HAVE_LONG_LONG) && defined(SCANF_LONG_LONG)
				(void) sscanf(rp->msg.first->line, SCANF_LONG_LONG, size);
#elif defined(HAVE_LONG_LONG) && defined(HAVE_STRTOQ)
				*size = (longest_int) strtoq(rp->msg.first->line, NULL, 0);
#else
				(void) sscanf(rp->msg.first->line, "%ld", size);
#endif
				cip->hasSIZE = kCommandAvailable;
				result = kNoErr;
			} else if (UNIMPLEMENTED_CMD(rp->code)) {
				cip->hasSIZE = kCommandNotAvailable;
				cip->errNo = kErrSIZENotAvailable;
				result = kErrSIZENotAvailable;
			} else {
				cip->errNo = kErrSIZEFailed;
				result = kErrSIZEFailed;
			}
			DoneWithResponse(cip, rp);
		}
	}
	return (result);
}	/* FTPFileSize */




int
FTPSymlink(const FTPCIPtr cip, const char *const lfrom, const char *const lto)
{
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	if ((cip == NULL) || (lfrom == NULL) || (lto == NULL))
		return (kErrBadParameter);
	if ((lfrom[0] == '\0') || (lto[0] == '\0'))
		return (kErrBadParameter);
	if (FTPCmd(cip, "SITE SYMLINK %s %s", lfrom, lto) == 2)
		return (kNoErr);
	return (kErrSYMLINKFailed);
}	/* FTPSymlink */




int
FTPUmask(const FTPCIPtr cip, const char *umask)
{
	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	if ((umask == NULL) || (umask[0] == '\0'))
		return (kErrBadParameter);
	if (FTPCmd(cip, "SITE UMASK %s", umask) == 2)
		return (kNoErr);
	return (kErrUmaskFailed);
}	/* FTPUmask */




static void
GmTimeStr(char *dst, size_t dstsize, time_t t)
{
	char buf[64];
	struct tm *gtp;

	gtp = gmtime(&t);
	if (gtp == NULL) {
		dst[0] = '\0';
	} else {
#ifdef HAVE_SNPRINTF
		buf[sizeof(buf) - 1] = '\0';
		(void) snprintf(buf, sizeof(buf) - 1, "%04d%02d%02d%02d%02d%02d",
#else
		(void) sprintf(buf, "%04d%02d%02d%02d%02d%02d",
#endif
			gtp->tm_year + 1900,
			gtp->tm_mon + 1,
			gtp->tm_mday,
			gtp->tm_hour,
			gtp->tm_min,
			gtp->tm_sec
		);
		(void) Strncpy(dst, buf, dstsize);
	}
}	/* GmTimeStr */




int
FTPUtime(const FTPCIPtr cip, const char *file, time_t actime, time_t modtime, time_t crtime)
{
	char mstr[64], astr[64], cstr[64];
	int result;
	ResponsePtr rp;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if (cip->hasUTIME == kCommandNotAvailable) {
		cip->errNo = kErrUTIMENotAvailable;
		result = kErrUTIMENotAvailable;
	} else {
		if ((actime == (time_t) 0) || (actime == (time_t) -1))
			(void) time(&actime);
		if ((modtime == (time_t) 0) || (modtime == (time_t) -1))
			(void) time(&modtime);
		if ((crtime == (time_t) 0) || (crtime == (time_t) -1))
			crtime = modtime;

		(void) GmTimeStr(astr, sizeof(astr), actime);
		(void) GmTimeStr(mstr, sizeof(mstr), modtime);
		(void) GmTimeStr(cstr, sizeof(cstr), crtime);

		rp = InitResponse();
		if (rp == NULL) {
			result = kErrMallocFailed;
			cip->errNo = kErrMallocFailed;
			Error(cip, kDontPerror, "Malloc failed.\n");
		} else {
			result = RCmd(cip, rp, "SITE UTIME %s %s %s %s UTC", file, astr, mstr, cstr); 	
			if (result < 0) {
				DoneWithResponse(cip, rp);
				return (result);
			} else if (result == 2) {
				cip->hasUTIME = kCommandAvailable;
				result = kNoErr;
			} else if (UNIMPLEMENTED_CMD(rp->code)) {
				cip->hasUTIME = kCommandNotAvailable;
				cip->errNo = kErrUTIMENotAvailable;
				result = kErrUTIMENotAvailable;
			} else {
				cip->errNo = kErrUTIMEFailed;
				result = kErrUTIMEFailed;
			}
			DoneWithResponse(cip, rp);
		}
	}
	return (result);
}	/* FTPUtime */
