// Copyright (c) 1995-1996, Information Technology Solutions. 
// All rights reserved.

#import "HRTopController.h"

#import "webrex_access.h"
#import "HRPage.h"

#import <foundation/NSException.h>
#import <foundation/NSAutoreleasePool.h>
#import <foundation/NSUtilities.h>
#import <foundation/NSPathUtilities.h>

#import <stdio.h>

#define ERROR_PAGE				@"error_msg"
#define EMP_ENTRY_PAGE			@"HRE_employee_entry"
#define EMP_ACCESS_PAGE			@"HRE_employee_access"
#define EMP_ELECTION_PAGE		@"HRE_employee_elections"
#define EMP_UPDATED_PAGE		@"HRE_employee_updated"

#define HTML_TYPE				@"html"
#define HR_APPLET_CACHE_NAME	@"HR_applet"
#define HR_DBNAME				@"humanresources"
#define ENTITY_EMPLOYEE			@"EMP_RECORD"

#define DBC_EMP_ID				@"EMP_ID"
#define DBC_PASSWORD			@"PASSWORD"
#define DBC_FIRSTNAME			@"FIRSTNAME"
#define DBC_LASTNAME			@"LASTNAME"
#define DBC_IND_PLAN			@"IND_PLAN"
#define DBC_STD_PLAN			@"STD_PLAN"
#define DBC_LTD_PLAN			@"LTD_PLAN"
#define DBC_PPO_PLAN			@"PPO_PLAN"
#define DBC_DNTL_PLAN			@"DNTL_PLAN"


#define CREATE_NEW_RECORD_MSG	@"No employee with the specified id exists in the database.  To create an employee benefits record, enter the last and first name for the employee specified and then press the \"Update Employee Record\" button."

#define INVALID_ID_MSG			@"You must specify a valid employee id to access an existing record, or create one."

#define INVALID_PASSWORD_MSG	@"The password entered does not match the password for the specified employee id, the employee record cannot be modified."

#define INSERT_FAILED_MSG		@"Could not insert employee record into database."

#define UPDATE_FAILED_MSG		@"Database error:  Unable to update employee record with new information."

@implementation HRTopController

- (WWResponse *)responseForRequest:(WWRequest *)request
	/*" my superclass implements this by default, but I can 
	override it if I want to do something special, like was done
	in the timeoff demo.  Perhaps your usename and password
	could be verified against the db or something.  Or the connection
	can be checked.
	PERFORMANCE: in checking this way to see if we are connected
		to the db, on an objectstation communicating with oracle
		on an old solaris box, it adds ~8000 uS to the average retrieve
		time.  This might improve if it were running PDO on the same
		machine as the db.
	"*/
{
	// check to see if we are still connected to the db in 
	// the fastest way possible.  i have found by experimentation
	// that a begin will succeed even if we lost our connected,
	// but the commit will surely fail.
	if ((![_context beginTransaction]) || (![_context commitTransaction])) {
		// remove yourself from the cache and go away and return an error.
		[_cache removeObjectForKey:HR_APPLET_CACHE_NAME];
		[_cache autorelease];
		return [self responseWithErrorMsg:@"Cannot connect to database."];
	}

	return [super responseForRequest:request];
}


- realInitWithServer:(WWServer *)server baseLocator:(NSString *)locator
	/*" Get the resources we need when we are init'd.  WWServer is the object
	that allocs us.  If we implement this method, we are not just init'd, but
	we are init'dWithServer.  This gives us a chance to get some information 
	about what's going on in the world... information we can use while initing. 
	Set up all the state that the other methods of this object assume. 
	(like the db connetion)
	"*/
{
    NSBundle			*bundle		= nil;
    NSString			*path		= nil;
    EOModel				*model		= nil;
    EODatabase			*database	= nil;
    EODatabaseChannel 	*channel	= nil;
	BOOL				aChanOpen	= NO;
	
    // This is a potential problem when having a model fragment in each part
    // of the server tree.  There needs to be a web framework convenience
    // method or two to do this common kind of stuff.
    bundle 		= [server bundleForBaseLocator:locator];
    path 		= [bundle pathForResource:HR_DBNAME ofType:@"eomodel"];
    model 		= [[[EOModel alloc] initWithContentsOfFile:path] autorelease];
    database 	= [[[EODatabase alloc] initWithModel:model] autorelease];
    _context 	= [[[EODatabaseContext alloc] initWithDatabase:database] 
					autorelease];
    channel 	= [[[EODatabaseChannel alloc] initWithDatabaseContext:_context]
					autorelease];
    _dataSource = [[EODatabaseDataSource alloc] initWithDatabaseChannel:channel
					entityNamed:ENTITY_EMPLOYEE];
	aChanOpen 	= [[channel adaptorChannel] openChannel];
	
    if (!_dataSource) {
		NSLog (@"*** Error creating data source! ***");
		[self dealloc];
		return nil;
    }
	if (aChanOpen == NO) {
		NSLog (@"*** Error opening adaptor channel, cannot connect to database.");
		[self dealloc];
		return nil;
	}
	_employeeEntity = [model entityNamed:ENTITY_EMPLOYEE];

	return self;
}

- initWithServer:(WWServer *)server baseLocator:(NSString *)baseLocator
	/*" This is the method with which we are normally init'd by the
	server.  A new instance of this class will be sent this message every
	time a new http request comes in.  Therefore I will 'cache' myself
	in this method the first time i am inited and every other time i will
	retrieve the instance in the cache and dealloc myself.  

	All of the real	work on initializing is done in a separate init method, 
	#realInitWithServer:baseLocator:.  NOTE:  since 
	the http connection is stateless, make sure you don't cache state that
	doesn't make sense to cache... At this point, basically all i am 
	caching is my connection to the database.
	"*/
{
	id					cachedInstance	= nil;
	id					ourSelves		= nil;
    NSMutableDictionary *cache			= nil;

	[super initWithServer:server baseLocator:baseLocator];

	// get the cache.  debugtrieger how many cache's are there?  how
	// many different baseLocator's, one for every applet?
    cache = [server cachedResourcesForKey:baseLocator];

	// see if there's been an object cached with our unique name.
	cachedInstance = [cache objectForKey:HR_APPLET_CACHE_NAME];
    if (cachedInstance) {
		// blow away ourselves and return the one from the cache.
		[self autorelease];
		return [cachedInstance retain];
    }
	
	// guess not, we must be the first.  init yourself.
    ourSelves = [self realInitWithServer:server baseLocator:baseLocator];
	if (ourSelves == nil) {
		// initialization must have failed.
		return nil;
	} else {
		// store yourself in the cache. (ourSelves == self here)
		[cache setObject:ourSelves forKey:HR_APPLET_CACHE_NAME];
		_cache = [cache retain];
		return self;
	}
}

- (void)dealloc
	/*" Foundation thing.  called implicitly when our refcount has gone to 0.
	No need to return.  Clean up all my ivars, and then super.
	"*/
{
	// clean up my ivars.

	// all done, call super dealloc.
	[super dealloc];
}

- (WWResponse *)sendMainPage:(WWRequest *)request
	/*" value, needs to implement htmldescription.  if this is just 
	the flat index.html page with no bindings done to it, use
	wwresponse responseforresource oftype: or something like that.
	"*/
{
	return [WWResponse responseWithPage:
		[WWPage pageWithResource:@"index" ofType:@"html"]];
}


- (EOGenericRecord *)retrieveEmployeeRecordWithId:(NSString *)employeeId
	/*" Lookup the employee with the employee id specified by employeeId 
	  and return that generic record.  If no matches were found, or if the
	  database is inaccessible, nil is returned.  If there is no transaction
	  in progress, one is started automagically.
	  "*/
{
	EOQualifier			*q 			= nil;
	NSArray				*matches	= nil;
	EOGenericRecord		*empRecord	= nil;
	
	if (!employeeId || ([employeeId length] == 0))
		return nil;

	q = [[EOQualifier alloc]
			 initWithEntity:_employeeEntity
			 qualifierFormat:@"%A = '%@'", @"EMP_ID", employeeId];
	[_dataSource setQualifier:q];
	matches = [_dataSource fetchObjects];

	if (matches) {
		if ([matches count] == 1) {
			empRecord = [matches lastObject];
		} 
	} else {
		NSLog(@"Could not access \"%@\" database", HR_DBNAME);
	}
	return empRecord;
}

- (WWResponse *)responseWithElectionPageForEmployee:(EOGenericRecord *)employeeRecord
	/*" Return the employee election page with the values filled in from the 
	  employeeRecord.
	  "*/
{
	WWPage					*returnPage		= nil;
	NSDictionary			*idDict			= nil;

	returnPage = [HRPage pageWithResource:EMP_ELECTION_PAGE 
						 ofType:HTML_TYPE];
	if (!returnPage)
		return [self responseWithErrorMsg:@"Unable to load election page."];

	[returnPage bindAnchorValues:
				[employeeRecord valuesForKeys:
								[_employeeEntity classPropertyNames]]];

	idDict = [NSDictionary dictionaryWithObjects:
						   [NSArray arrayWithObject:
									[employeeRecord objectForKey:DBC_EMP_ID]]
						   forKeys:[NSArray arrayWithObject:DBC_EMP_ID]];
	[returnPage bindValues:idDict forFormWithAction:@"setEmployeeElections"];
	return [WWResponse responseWithPage:returnPage];
	
}

- (WWResponse *)getEmployeeEntry:(WWRequest *)request
	/*" Lookup the employee with the employee id specified in the URL.
	Try to match the passwd.  If the passwd doesn't match, alert the user to
	that fact, otherwise load the employee data entry page and set the 
	appropriate default values for each of the employee elected benefits.
	"*/
{
	EOGenericRecord			*employee		= nil;
	NSDictionary			*urlQuery		= nil;
	WWPage					*returnPage		= nil;
	NSString				*empId			= nil;
	
	urlQuery	= [request urlQuery];
	empId 		= [urlQuery objectForKey:DBC_EMP_ID];
	employee 	= [self retrieveEmployeeRecordWithId:empId];
	
	if (employee) {
		NSObject	*empPasswd 		= [employee objectForKey:DBC_PASSWORD];
		NSString	*enteredPasswd 	= [urlQuery objectForKey:DBC_PASSWORD];
		BOOL		noPasswd 	= ![empPasswd isKindOfClass:[NSString class]];
		
		if (noPasswd || [(NSString *)empPasswd isEqualToString:enteredPasswd]){
			return [self responseWithElectionPageForEmployee:employee];
		} else {
			return [self responseWithErrorMsg:INVALID_PASSWORD_MSG];
		}
	} else {
		if (empId && [empId length]) {
			returnPage = [HRPage pageWithResource:EMP_ENTRY_PAGE 
								 ofType:HTML_TYPE];
			[returnPage bindValue:CREATE_NEW_RECORD_MSG
						forAnchorWithName:@"why-here"];
			[returnPage bindAnchorValues:[request urlQuery]];
			[returnPage bindValues:[request urlQuery] 
						forFormWithAction:@"setEmployeeEntry"];
			return [WWResponse responseWithPage:returnPage];
		} else {
			return [self responseWithErrorMsg:INVALID_ID_MSG];
		}
	}
}

- (WWResponse *)setEmployeeEntry:(WWRequest *)request
	/*" Lookup the employee with the employee id specified in the URL.
	Try to match the passwd.  If the passwd doesn't match, alert the user to
	that fact, otherwise update the employee data entry in the 
	database and then load the employee elections page.
	"*/
{
	EOGenericRecord			*employee		= nil;
	NSDictionary			*row			= nil;
	NSDictionary			*urlQuery		= nil;
	NSString				*empId			= nil;
	
	urlQuery	= [request urlQuery];
	empId 		= [urlQuery objectForKey:DBC_EMP_ID];
	
	[_context beginTransaction];
	employee 	= [self retrieveEmployeeRecordWithId:empId];
	row			= [_employeeEntity rowWithSubsetOfDictionary:urlQuery];

	if (employee) {
		NSObject	*empPasswd 		= [employee objectForKey:DBC_PASSWORD];
		NSString	*enteredPasswd 	= [urlQuery objectForKey:DBC_PASSWORD];
		BOOL		noPasswd 	= ![empPasswd isKindOfClass:[NSString class]];
		
		if (noPasswd || [(NSString *)empPasswd isEqualToString:enteredPasswd]){
			[employee takeValuesFromDictionary:row];
			[_dataSource updateObject:employee];
			if ([_context commitTransaction]) {
				return [self responseWithElectionPageForEmployee:employee];
			} else {
				[_context rollbackTransaction];
				return [self responseWithErrorMsg:UPDATE_FAILED_MSG];
			}
		} else {	
			[_context rollbackTransaction];
			return [self responseWithErrorMsg:INVALID_PASSWORD_MSG];
		}	
	} else {
		if (empId && [empId length]) {
			employee = [[EOGenericRecord alloc]
							initWithPrimaryKey:nil entity:_employeeEntity];
			[employee takeValuesFromDictionary:row];
			[employee autorelease];

			if ([_dataSource insertObject:employee]) {
				[_context commitTransaction];
				return [self responseWithElectionPageForEmployee:employee];
			} else {
				[_context rollbackTransaction];
				return [self responseWithErrorMsg:INSERT_FAILED_MSG];
			}
		} else {
			[_context rollbackTransaction];
			return [self responseWithErrorMsg:
						 @"You specified an invalid employee id"];
		}
	}
}


- setEmployeeElections:(WWRequest *)request
{
	EOGenericRecord			*employee		= nil;
	NSDictionary			*row			= nil;
	NSDictionary			*urlQuery		= nil;
	WWPage					*returnPage		= nil;
	NSString				*empId			= nil;
	
	urlQuery	= [request urlQuery];
	empId 		= [urlQuery objectForKey:DBC_EMP_ID];
	
	[_context beginTransaction];
	employee 	= [self retrieveEmployeeRecordWithId:empId];
	row			= [_employeeEntity rowWithSubsetOfDictionary:urlQuery];

	if (employee) {
		[employee takeValuesFromDictionary:row];
		[_dataSource updateObject:employee];
		if ([_context commitTransaction]) {
			returnPage = [HRPage pageWithResource:EMP_UPDATED_PAGE 
							   ofType:HTML_TYPE];
			[returnPage bindAnchorValues:
						[employee valuesForKeys:
								  [_employeeEntity classPropertyNames]]];
			return [WWResponse responseWithPage:returnPage];
		} else {
			[_context rollbackTransaction];
			return [self responseWithErrorMsg:UPDATE_FAILED_MSG];
		}
	} else {
		[_context rollbackTransaction];
		return [self responseWithErrorMsg:
					 @"An unknown database error has occured."];
	}
}

- (WWResponse *)getAdminSummary:(WWRequest *)request
{
	WWPage				*returnPage		= nil;
	NSArray				*matches		= nil;
	WWTable				*table			= nil;

	[_dataSource setQualifier:[_employeeEntity qualifier]];
	matches = [_dataSource fetchObjects];
	table 	= [[[WWTable alloc] init] autorelease];

	// got the widths from the eomodel.
	[table addColumn:@"Emp ID"		key:DBC_EMP_ID width:12];
	[table addColumn:@"First"		key:DBC_FIRSTNAME width:12];
	[table addColumn:@"Last"		key:DBC_LASTNAME width:12];
	[table addColumn:@"Indiv."		key:DBC_IND_PLAN width:7];
	[table addColumn:@"Long"		key:DBC_LTD_PLAN width:6];
	[table addColumn:@"Short"		key:DBC_STD_PLAN width:6];
	[table addColumn:@"PPO"			key:DBC_PPO_PLAN width:6];
	[table addColumn:@"Dental"		key:DBC_DNTL_PLAN width:6];
	[table setObjects:matches];

	// binding
	returnPage = [HRPage pageWithResource:@"HRA_admin_summary"
						 ofType:HTML_TYPE];
	[returnPage bindValue:[NSNumber numberWithInt:[matches count]]
				forAnchorWithName:@"summary_total"];
	[returnPage bindValue:table forAnchorWithName:@"summary_table"];

	return [WWResponse responseWithPage:returnPage];;
}


- (WWResponse *)responseWithErrorMsg:(NSString *)errorMsg
	/*" Binds the errorMsg to the errorMsg anchor in the html file: 
	ERROR_PAGE, and returns the newly formed page. "*/
{
	WWPage	*page 		= nil;
	
	if (!errorMsg) 
		errorMsg = @"An error occured on the WebRex server.";
	page = [HRPage pageWithResource:ERROR_PAGE ofType:HTML_TYPE];
	[page bindValue:errorMsg forAnchorWithName:@"errorMsg"];
	return [WWResponse responseWithPage:page];
}

@end


@implementation EOEntity (humanresources_stuff)

- (NSDictionary *)rowWithSubsetOfDictionary:(NSDictionary *)dict
	/*" This takes a dictionary of stuff and extracts the parts 
		appropriate to this entity and converts the values to the 
		proper model types.
	"*/
{
	int 					i, count;
	id 						value;
	NSString				*name;
	NSMutableDictionary 	*row;
	NSArray 				*attributes;
	EOAttribute 			*attribute;

	row 		= [[[NSMutableDictionary alloc] init] autorelease];
	attributes 	= [self attributes];
	count 		= [attributes count];

	for (i = 0; i < count; i++) {
		attribute 	= [attributes objectAtIndex:i];
		name 		= [attribute name];
		value 		= [dict objectForKey:name];
		if (!value)
			continue;

		value = [self convertValue:value attribute:attribute];
		if (!value)
			continue;
		
		[row setObject:value forKey:name];
	}
	return row;
}


@end


