/** ColEm: portable Coleco emulator **************************/
/**                                                         **/
/**                         Common.h                        **/
/**                                                         **/
/** This file contains screen refresh drivers which are     **/
/** common for both X11 and VGA implementations. It is      **/
/** included either from Unix.c or MSDOS.c.                 **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-1998                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

static void RefreshSprites(byte Y);
static void RefreshBorder(byte Y);

/** RefreshScreen() ******************************************/
/** Refresh screen. This function is called in the end of   **/
/** refresh cycle to show the entire screen.                **/
/*************************************************************/
void RefreshScreen(void) { PutImage(); }

/** RefreshBorder() ******************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** the screen border.                                      **/
/*************************************************************/
void RefreshBorder(register byte Y)
{
  if(!Y)
    memset(XBuf,BGColor,WIDTH*(HEIGHT-192)/2);
//    memset(XBuf,XPal[BGColor],WIDTH*(HEIGHT-192)/2);
  if(Y==191)
    memset(XBuf+WIDTH*(HEIGHT+192)/2,BGColor,WIDTH*(HEIGHT-192)/2);
//    memset(XBuf+WIDTH*(HEIGHT+192)/2,XPal[BGColor],WIDTH*(HEIGHT-192)/2);
}

/** RefreshSprites() *****************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** sprites.                                                **/
/*************************************************************/
void RefreshSprites(register byte Y)
{
  register byte C,H;
  register byte *P,*PT,*AT;
  register int L,K;
  register unsigned int M;

  H=Sprites16x16? 16:8;
  C=0;M=0;L=0;AT=SprTab-4;
  do
  {
    M<<=1;AT+=4;L++;    /* Iterating through SprTab */
    K=AT[0];            /* K = sprite Y coordinate */
    if(K==208) break;   /* Iteration terminates if Y=208 */
    if(K>256-H) K-=256; /* Y coordinate may be negative */

    /* Mark all valid sprites with 1s, break at 4 sprites */
    if((Y>K)&&(Y<=K+H)) { M|=1;if(++C==4) break; }
  }
  while(L<32);

  for(;M;M>>=1,AT-=4)
    if(M&1)
    {
      C=AT[3];                  /* C = sprite attributes */
      L=C&0x80? AT[1]-32:AT[1]; /* Sprite may be shifted left by 32 */
      C&=0x0F;                  /* C = sprite color */

      if((L<256)&&(L>-H)&&C)
      {
        K=AT[0];                /* K = sprite Y coordinate */
        if(K>256-H) K-=256;     /* Y coordinate may be negative */

        P=XBuf+WIDTH*(HEIGHT-192)/2+(WIDTH-256)/2+WIDTH*Y+L;
        PT=SprGen+((int)(H>8? AT[2]&0xFC:AT[2])<<3)+Y-K-1;
//        C=XPal[C];

        /* Mask 1: clip left sprite boundary */
        K=L>=0? 0x0FFFF:(0x10000>>-L)-1;
        /* Mask 2: clip right sprite boundary */
        if(L>256-H) K^=((0x00200>>(H-8))<<(L-257+H))-1;
        /* Get and clip the sprite data */
        K&=((int)PT[0]<<8)|(H>8? PT[16]:0x00);

        /* Draw left 8 pixels of the sprite */
        if(K&0xFF00)
        {
          if(K&0x8000) P[0]=C;if(K&0x4000) P[1]=C;
          if(K&0x2000) P[2]=C;if(K&0x1000) P[3]=C;
          if(K&0x0800) P[4]=C;if(K&0x0400) P[5]=C;
          if(K&0x0200) P[6]=C;if(K&0x0100) P[7]=C;
        }

        /* Draw right 8 pixels of the sprite */
        if(K&0x00FF)
        {
          if(K&0x0080) P[8]=C; if(K&0x0040) P[9]=C;
          if(K&0x0020) P[10]=C;if(K&0x0010) P[11]=C;
          if(K&0x0008) P[12]=C;if(K&0x0004) P[13]=C;
          if(K&0x0002) P[14]=C;if(K&0x0001) P[15]=C;
        }
      }
    }
}

/** RefreshLine0() *******************************************/
/** Refresh line Y (0..191) of SCREEN0, including sprites   **/
/** in this line.                                           **/
/*************************************************************/
void RefreshLine0(register byte Y)
{
  register byte X,K,Offset,FC,BC;
  register byte *P,*T;

  P=XBuf+WIDTH*(HEIGHT-192)/2+WIDTH*Y;
//  XPal[0]=BGColor? XPal[BGColor]:XPal0;

//  if(!ScreenON) memset(P,XPal[BGColor],WIDTH);
  if(!ScreenON) memset(P,BGColor,WIDTH);
  else
  {
//    BC=XPal[BGColor];
    BC=BGColor;
//    FC=XPal[FGColor];
    FC=FGColor;
    T=ChrTab+(Y>>3)*40;
    Offset=Y&0x07;

    memset(P,BC,(WIDTH-240)/2);
    P+=(WIDTH-240)/2;

    for(X=0;X<40;X++)
    {
      K=ChrGen[((int)*T<<3)+Offset];
      P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
      P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
      P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
      P+=6;T++;
    }

    memset(P,BC,(WIDTH-240)/2);
  }

  RefreshBorder(Y);
}

/** RefreshLine1() *******************************************/
/** Refresh line Y (0..191) of SCREEN1, including sprites   **/
/** in this line.                                           **/
/*************************************************************/
void RefreshLine1(register byte Y)
{
  register byte X,K,Offset,FC,BC;
  register byte *P,*T;

  P=XBuf+WIDTH*(HEIGHT-192)/2+WIDTH*Y;
//  XPal[0]=BGColor? XPal[BGColor]:XPal0;

//  if(!ScreenON) memset(P,XPal[BGColor],WIDTH);
  if(!ScreenON) memset(P,BGColor,WIDTH);
  else
  {
    T=ChrTab+(Y>>3)*32;
    Offset=Y&0x07;

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    P+=(WIDTH-256)/2;

    for(X=0;X<32;X++)
    {
      K=*T;
      BC=ColTab[K>>3];
      K=ChrGen[((int)K<<3)+Offset];
//      FC=XPal[BC>>4];
//      BC=XPal[BC&0x0F];
      FC=BC>>4;
      BC=BC&0x0F;
      P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
      P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
      P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
      P[6]=K&0x02? FC:BC;P[7]=K&0x01? FC:BC;
      P+=8;T++;
    }

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    RefreshSprites(Y);
  }

  RefreshBorder(Y);
}

/** RefreshLine2() *******************************************/
/** Refresh line Y (0..191) of SCREEN2, including sprites   **/
/** in this line.                                           **/
/*************************************************************/
void RefreshLine2(register byte Y)
{
  register byte X,K,FC,BC,Offset;
  register byte *P,*T,*PGT,*CLT;
  register int I;

  P=XBuf+WIDTH*(HEIGHT-192)/2+WIDTH*Y;
//  XPal[0]=BGColor? XPal[BGColor]:XPal0;

//  if(!ScreenON) memset(P,XPal[BGColor],WIDTH);
  if(!ScreenON) memset(P,BGColor,WIDTH);
  else
  {
    I=(int)(Y&0xC0)<<5;
    PGT=ChrGen+I;
    CLT=ColTab+I;
    T=ChrTab+(Y>>3)*32;
    Offset=Y&0x07;

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    P+=(WIDTH-256)/2;

    for(X=0;X<32;X++)
    {
      I=((int)*T<<3)+Offset;
      K=PGT[I];
      BC=CLT[I];
//      FC=XPal[BC>>4];
//      BC=XPal[BC&0x0F];
      FC=BC>>4;
      BC=BC&0x0F;
      P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
      P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
      P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
      P[6]=K&0x02? FC:BC;P[7]=K&0x01? FC:BC;
      P+=8;T++;
    }

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    RefreshSprites(Y);
  }

  RefreshBorder(Y);
}

/** RefreshLine3() *******************************************/
/** Refresh line Y (0..191) of SCREEN3, including sprites   **/
/** in this line.                                           **/
/*************************************************************/
void RefreshLine3(register byte Y)
{
  register byte X,K,Offset;
  register byte *P,*T;

  P=XBuf+WIDTH*(HEIGHT-192)/2+WIDTH*Y;
//  XPal[0]=BGColor? XPal[BGColor]:XPal0;

//  if(!ScreenON) memset(P,XPal[BGColor],WIDTH);
  if(!ScreenON) memset(P,BGColor,WIDTH);
  else
  {
    T=ChrTab+(Y>>3)*32;
    Offset=(Y&0x1C)>>2;

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    P+=(WIDTH-256)/2;

    for(X=0;X<32;X++)
    {
      K=ChrGen[((int)*T<<3)+Offset];
//      P[0]=P[1]=P[2]=P[3]=XPal[K>>4];
//      P[4]=P[5]=P[6]=P[7]=XPal[K&0x0F];
      P[0]=P[1]=P[2]=P[3]=K>>4;
      P[4]=P[5]=P[6]=P[7]=K&0x0F;
      P+=8;T++;
    }

//    memset(P,XPal[BGColor],(WIDTH-256)/2);
    memset(P,BGColor,(WIDTH-256)/2);
    RefreshSprites(Y);
  }

  RefreshBorder(Y);
}
