//====//====//====//====//====//====//====//====//====//====//
/*
	Project: YATE (Yet Another Text Editor)
	File: “Yate.cpp”

	Copyright © 1996-1998 Acid Software, All Rights Reserved.

	This file houses the high-level BApplication class.
	Also contains several glue functions and manages
	the YateWindows to prevent DeadLocks.
*/
//====//====//====//====//====//====//====//====//====//====//

#include "Actions.h"
#include "cb_list.h"
#include "UBuffer.h"    //the high-level text representation
#include "YateWindow.h"
#include "Yate.h"
#include "FFont.h"
#include "BufList.h"
#include "Utils.h"


#include <FilePanel.h>
#include <Application.h>
#include <Autolock.h>
#include <Alert.h>  //about grrrrrrr
#include <Entry.h> //suffix
#include <Beep.h>

#include <stdio.h>
#include <string.h>


extern void BroadCast(Buff*,BWindow*,int32 type);

BFont *be_plain_font2=NULL;
BFont *be_bold_font2=NULL;
BFont *be_fixed_font2=NULL;

	class YateApplication : public BApplication //standard fare
	{
		public:
							YateApplication();
							virtual ~YateApplication();
			virtual void    ReadyToRun(void);
			virtual void	RefsReceived(BMessage *message);
			virtual void    ArgvReceived(int32 argc, char **argv);
			virtual void    MessageReceived(BMessage *msg);
			virtual void    AboutRequested();
			virtual bool    QuitRequested();
			virtual void    Quit();
		private:
			void            RefsReceived2(BMessage *msg,YateWindow *ww);
			void            CheckBuffStat();

			void            OpenSel(YateWindow *,Buff *cb);
			void            DiscardBuff(Buff*);
			YateWindow*     OpenWin(BRect Place,Buff *StartBuff);
			void            CloseWin(YateWindow*);
			void            Transmit(BMessage*);

			void            NewFontReq(BMessage*);
			void            DoNextError(YateWindow*);

			BRect           aRect;      //default window position..
			BFilePanel      *openpanel;
			CList           *winlist,*buflist;
			int32           quitnow;

//------------------------------------
			Buff *GetBuff(entry_ref *);//NULL if unable
			Buff *GetBuff(char *path);
			Buff *FirstShell();

			Buff *NewBuff();
			Buff *NewShell();

			void SetName(Buff *);
			void Discard(Buff*);

			Buff *BuffAt(int32 index);
			Buff *Shift(Buff*,int32 ofs);
			int32   CountItems();

			void    MakePerfect();
			void    MySave(YAction type);
			void    YExec(YateWindow*,char *);
			void    CheckAlone(Buff *);
			void    CheckTitle(YateWindow*,Buff*);
			void    CheckAllTitle();
			void    Sort();
			int32   bufcnt,shellcnt;
			void    SendMsg(Buff *,BMessage*);
			void    DoDocText();

	};


int main(int a,char **b)      //pretty standard stuff
{
	YateApplication *ya_app = new YateApplication();
	ya_app->Run();
	delete ya_app;
	return 0;
}


YateApplication::YateApplication() : BApplication("application/x-vnd.acid-YATE")
{
	new BufList();

	be_plain_font2=new BFont();
	be_bold_font2=new BFont();
	be_fixed_font2=new BFont();
	*be_plain_font2=*be_plain_font;
	openpanel=NULL;
	aRect.Set(320, 96, 760, 500);   //we should grab this out of the preferences
	quitnow=FALSE;
	winlist=new CList(100);
	buflist=new CList(100);
	shellcnt=bufcnt=1;
}

YateApplication::~YateApplication(){
	delete winlist;
//    delete bl;
}

bool YateApplication::QuitRequested(){
//  return TRUE;
	if (quitnow)return TRUE;

	int mx=CountItems();
	bool happy=TRUE;
	Buff *badb=NULL;
	for (int idx=0;idx<mx;idx++){
		Buff *bb=BuffAt(idx);
		if (bb->alone && !bb->fried){
			if (bb->lastchanged!=0){
				happy=FALSE;
				if (!badb)badb=bb;
			}
		}
	}

	mx=winlist->Num();

	if (!happy){

		BAlert *bal=new BAlert("Quitting Alert",
			"Yate: Some Files are Not Saved!\n",
			"Don't Save",
			"Cancel",
			"Save");

		if (mx==0){
			OpenWin(aRect,badb);
		}else{
			YateWindow *yw=(YateWindow*)winlist->Item(0);
			CheckTitle(yw,badb);
			{
				BAutolock l1(yw);
				yw->ChangeBuff(badb);
			}
			MakePerfect();
		}
		int32 res=bal->Go();
		switch (res){
			case 0:
				return TRUE;
			case 1:
				return FALSE;
			case 2:
				PostMessage(A_SAVE_CHANGED);
				return FALSE;
		}
	}

	if (mx){
		for (int idx=0;idx<mx;idx++){
			YateWindow *yw=(YateWindow*)winlist->Item(idx);
			yw->PostMessage(B_QUIT_REQUESTED);
		}
		return FALSE;
	}


	return TRUE;
}


void YateApplication::Quit(){
	BApplication::Quit();
}


YateWindow *YateApplication::OpenWin(BRect frame, Buff *bb){
	YateWindow  *yw=new YateWindow(frame,bb);
	winlist->Add(yw);
	CheckTitle(yw,bb);
	yw->Show();
	MakePerfect();
	return yw;
}

void YateApplication::CloseWin(YateWindow *win){
	if (!win) return;
	win->Lock();
	win->Quit();
	winlist->RemIndex(winlist->IndexOf(win));
	if ( !winlist->Num() )  PostMessage(B_QUIT_REQUESTED);
}


/*
#define Y_BROADCAST        0x15f9a
void BroadCast(class Buff *cb,BWindow *win,int32 type){
	BMessage    bcst(Y_BROADCAST);
	bcst.AddPointer("Buff",cb);
	bcst.AddPointer("YWin",win);
	if (type==NULL)type=Y_INVALIDATE;
	bcst.AddInt32("Type",type);
	be_app->PostMessage(&bcst);
}
*/

void YateApplication::Transmit(BMessage *src){
	Buff *bb=NULL;
	YateWindow *notme=NULL;
	int32 type;

	status_t rr;
	rr=src->FindInt32("Type",&type);
	rr=src->FindPointer("Buff",(void**)&bb);
	rr=src->FindPointer("YWin",(void**)&notme);

	BMessage *msg=NULL;

	long mx=winlist->Num();
	for (long idx=0;idx<mx;idx++){
		YateWindow *yw=(YateWindow*)winlist->Item(idx);
		if (yw!=notme){
			int go=FALSE;

			if (bb==NULL)go=TRUE;
			if (yw->MyBuff()==bb) go=TRUE;
			if (go==FALSE) continue;
			BMessageQueue *mq=yw->MessageQueue();
			if (!mq) continue;
			if (mq->FindMessage(type,1)) continue;
			if (!msg) msg=new BMessage(type);
			yw->PostMessage(msg);
		}
	}
	if (msg) delete(msg);
}


void YateApplication::RefsReceived(BMessage *message){
	RefsReceived2(message,NULL);

}

void YateApplication::RefsReceived2(BMessage *msg,YateWindow *ww){
	FILE *f=fopen("/boot/home/mine.txt","wb");
	fprintf(f,"%d,%x\n",msg,1);
	fclose(f);
	Buff   *fb=NULL;
	type_code   tc;
	int32 count;

	if (ww) ww->Lock();

	msg->GetInfo("refs",&tc,&count);
	for ( long i = --count; i >= 0; i-- ) {
		entry_ref   er;
		status_t    rr=msg->FindRef("refs",i,&er);

		if (rr==B_NO_ERROR){
			BNode anode(&er);
			if (!anode.IsDirectory()){
				Buff *hit=GetBuff(&er);
				if (hit && !fb)fb=hit;
			}
		}
	}
	if (fb){
		if (ww){
			CheckTitle(ww,fb);
			ww->ChangeBuff(fb);
		}else{
			OpenWin(aRect,fb);
		}
	}

	if (ww){
		ww->Unlock();
	}
	MakePerfect();
}

//#include <unistd.h>  //getcwd()

/*
	char    cc[512];
	getcwd  (cc,128);
//    int ll=strlen(cc);
//    cc[ll++]='/';
	printf("working in '%s'\n",cc);
*/

void YateApplication::ArgvReceived(int32 argc,char **argv){
	int i,erc;
	BFile       ff;
	Buff        *bb=NULL,*fb=NULL;

	erc=0;
	for (i=1;i<argc;i++){
		bb=GetBuff(argv[i]);

		if (bb && !fb) fb=bb;
		if (!bb)erc++;
	}

	if (erc){
		printf("Try using a more explicit pathname\n");
	}
	if (fb) OpenWin(aRect,fb);   //if we opened a file
}

void YateApplication::ReadyToRun(void)
{
	if (CountItems()==0)    //no buffers?
	{
		Buff *cb=FirstShell();
		YateWindow  *ywin=OpenWin(aRect,cb);
//        BMessage msg(Y_DO_EXEC);
//        msg.AddString("exec","pwd");
//        ywin->PostMessage(&msg);
	}
}

void YateApplication::OpenSel(YateWindow *yw,Buff *cb){
	char sha[256],*spa=sha;
	char more[512];
	sha[0]=0;
	{
		BAutolock l1(cb);
		int32 p1,p2;
		p1=cb->Pos(141);
		p2=cb->Pos(142);
		SORT32(p1,p2);
		if (p1==-1){beep();return;}
//      if (p1==-1 || p1==p2){p1=p2=Pos(cnum);}

		p2=MIN(p1+250,p2);

		int32 ll=cb->Fetch(sha,p2-p1,p1);
		while ( (--ll>0) ){
			int32 f=1;
			if (sha[ll]==10)f=0;
			if (sha[ll]==13)f=0;
			if (sha[ll]=='\t')f=0;
			if (sha[ll]==' ')f=0;
			if (sha[ll]=='\"')f=0;
			if (sha[ll]=='>')f=0;
			if (f) break;
		}
		sha[++ll]=0;

		for (int i=0;i<2;i++){
			if (i && !strncmp(spa,"#include",8)){spa+=8;ll-=8;}
			while (ll>0){
				int32 f=1;
				if (*spa==' ')f=0;
				if (*spa=='\t')f=0;
				if (*spa=='\"')f=0;
				if (*spa=='<')f=0;
				if (f)break;
				spa++;ll--;
			}
		}
	}

	Buff *bb=GetBuff(spa);
	if (!bb){
		beep();
		sprintf(more,"Couldn't open '%'",spa);
		DoMsg(yw,more);
		return;
	}
	if (yw){
		BAutolock l7(yw);
		yw->ChangeBuff(bb);
	}
}


extern void ShowAbout();
void YateApplication::AboutRequested(){
	ShowAbout();
}

void YateApplication::NewFontReq(BMessage *msg){
	FindMessageFont(msg,"be_plain_font",0,be_plain_font2);
	FindMessageFont(msg,"be_bold_font" ,0,be_bold_font2 );
	FindMessageFont(msg,"be_fixed_font",0,be_fixed_font2);
	BMessage txm(Y_SYSFONT_CHANGE);
	Transmit(&txm);
}


void    YateApplication::MessageReceived(BMessage *msg)
{
	Buff    *nb=NULL;
	int32    idx,mx,tt;

	YateWindow *yw=NULL;
	Buff *cb=NULL;
	msg->FindPointer("Buff",(void**)&cb);
	msg->FindPointer("YWin",(void**)&yw);

	tt=msg->what;
	switch (tt){
		case A_NEXT_ERROR:
			DoNextError(yw);
			CheckTitle(yw,yw->cb);
			break;
		case A_RESET_ERROR:
			cb=FirstShell();
			if (!cb){
				beep();
				DoMsg(yw,"No Error List To Reset");
				break;
			}
			cb->PostMessage(A_RESET_ERROR);
			DoMsg(yw,"Error List Reset");
			break;
		case A_OPEN_SELECTION:OpenSel(yw,cb);break;

		case B_SIMPLE_DATA:
			if (msg->CountNames(B_REF_TYPE)){
				RefsReceived2(msg,yw);
			}
			break;

		case B_KEY_DOWN+1:
			tt=0;
			msg->FindInt32("yak:window",&tt);
			if (tt<0 || tt>winlist->Num())tt=0;
			yw=(YateWindow*)winlist->Item(tt);
			if (yw) yw->PostMessage(msg);

			break;
		case A_NEW_WINDOW:
			{
			if (winlist->Num()>15) break;

			BRect   trex=aRect;

			if (yw){
				trex=yw->Frame();
				trex.OffsetBy(30,30);
				nb=yw->MyBuff();
			}else{
				nb=BuffAt(0);
			}

			OpenWin(trex,nb);
			}
			break;
		case A_PREV_BUFF:
		case A_NEXT_BUFF:
			if (yw){
				Buff *bb=Shift(cb,msg->what==A_NEXT_BUFF?1:-1);
				BAutolock l2(yw);
				CheckTitle(yw,bb);
				yw->ChangeBuff(bb);
			}
			break;
		case A_PREV_WINDOW:
		case A_NEXT_WINDOW:
				mx=winlist->Num();
				tt=-1;
				for (idx=0;idx<mx;idx++){
					YateWindow *yw=(YateWindow*)winlist->Item(idx);
					if (yw->IsActive()) tt=idx;
				}
				tt=tt+1;
				if (msg->what==A_PREV_WINDOW) tt+=mx-2;
				((YateWindow*)winlist->Item(tt%mx))->Activate(TRUE);
				break;
		case Y_CLOSE_WIN_REQ:
			msg->FindPointer("YWin",(void**)&yw);
			CloseWin(yw);
			break;
		case Y_CHANGE_BUFF:
			if (yw){
				BAutolock l7(yw);
				yw->ChangeBuff(cb);
				MakePerfect();
			}
			break;

//        case Y_BROADCAST:Transmit(msg);break;
//        case Y_SORT_BUFF:MySortBuf();break;
//        case Y_CHECK_LONELY:CheckBuffStat();break;

		case A_OPEN_PANEL:
			if (!openpanel){
				openpanel=new BFilePanel();
			}
			openpanel->Show();
			break;
		case B_CANCEL:
			//openpanel=NULL:
			break;

		case '!FNT':
			NewFontReq(msg);
			break;
		case Y_CHECK_TITLE:
			MakePerfect();
			break;
		case A_MAKE:YExec(yw,"make");break;
		case A_MAKE_RUN:YExec(yw,"make run");break;
		case A_DISCARD_BUFFER:Discard(cb);break;
		case A_SAVE_CHANGED:
		case A_SAVE_ALL:
			MySave((YAction)msg->what);
			break;
		case  A_NEW_BUFFER:
			if (yw){
				Buff *bb=NewBuff();
				BAutolock l7(yw);
				CheckTitle(yw,bb);
				yw->ChangeBuff(bb);
				DoMsg(yw,"New Buffer");
				MakePerfect();
			}
			break;
		case A_OPEN_SUFFIX:
			{
				int nx=FALSE,fs=TRUE;
				entry_ref bst;
				if (!cb) goto FailSuff;
				{
				int ll=strlen(cb->myname);
				while (ll>0){
					if (cb->myname[--ll]=='.'){break;}
				}
				if (!ll) goto FailSuff;
				ll++;
				{
					entry_ref cdir(cb->myent.device,cb->myent.directory,".");
					BDirectory adir(&cdir);
					entry_ref tst(cdir);
					for (;;){
						if (tst==cb->myent)nx=TRUE;
						status_t ss=adir.GetNextRef(&tst);
						if (ss!=B_NO_ERROR) break;
						if (strncmp(cb->myname,tst.name,ll))continue;
						if (!strcmp("o",tst.name+ll))continue;
						if (!strcmp("xSYM",tst.name+ll))continue;
						BNode anode(&tst);
						if (anode.IsDirectory())continue;
						if (fs){bst=tst;fs=FALSE;}
						if (nx){bst=tst;break;}
					}
				}
				}
				if (!nx)goto FailSuff;
				{
				Buff *nb=GetBuff(&bst);
				if (yw){
					BAutolock l7(yw);
					yw->ChangeBuff(nb);
				}
				MakePerfect();
				break;
				}
				FailSuff:
				DoMsg(yw,"No Suffix File To Open");
				beep();
				break;
			}
			case B_SOME_APP_ACTIVATED:
				break;

		default:
			printf("unknown yate-app message:\n");
			msg->PrintToStream();
			BApplication::MessageReceived(msg);
			break;
	}
}




Buff *YateApplication::BuffAt(int32 idx){
	return (Buff*)buflist->Item(idx);
}

Buff *YateApplication::Shift(Buff *stt,int32 ofs){
	int32 tt=buflist->IndexOf(stt)+ofs;
	int32 mx=buflist->Num();
	tt=(tt+mx)%mx;
	return BuffAt(tt);
}

Buff *YateApplication::NewBuff(){	
	Buff *cb=new Buff();
	cb->Run();
	buflist->Add(cb);
	sprintf(cb->myname,"Untitled %d",bufcnt);
	sprintf(cb->mytitle,"!Untitled %d",bufcnt);
	bufcnt++;
	return cb;
}

Buff *YateApplication::NewShell(){
	Buff *cb=new Buff();
	cb->Run();
	buflist->Add(cb);
	cb->term=TRUE;
	sprintf(cb->myname,"(Shell %d)",shellcnt);
	sprintf(cb->mytitle,">(Shell %d)",shellcnt);
	shellcnt++;
	return cb;
}

void YateApplication::CheckAlone(Buff *b1){
	BAutolock l2(b1);

	int al=TRUE,max=buflist->Num();
	for (int jj=0;jj<max;jj++){
		Buff *b2=BuffAt(jj);
		if (b1!=b2){
			BAutolock l3(b2);
			if (b1->myent==b2->myent){
				al=FALSE;
				if (b2->alone){
					b2->entchanged=TRUE;
					b2->alone=FALSE;
				}
			}
		}
	}
	b1->alone=al;
}


void YateApplication::SetName(Buff *cb){
	BAutolock l2(cb);
	if (buflist->IndexOf(cb)<0){
		return;
	}

	if (cb->entchanged){
		cb->entchanged=FALSE;
		CheckAlone(cb);
		BEntry  ent(&cb->myent);
		cb->fried=ent.InitCheck();
		if (!cb->fried) ent.GetName(cb->myname);

		char *title=cb->mytitle;
		if (strlen(cb->myname)<Y_NAME_LENGTH-3){
			sprintf(title+1,"%s",cb->myname);
		}else{
			memcpy(title+1,cb->myname,Y_NAME_LENGTH-4);
			title[Y_NAME_LENGTH-4]=0xe2;
			title[Y_NAME_LENGTH-3]=0x80;
			title[Y_NAME_LENGTH-2]=0xa6;
			title[Y_NAME_LENGTH-1]=0;
		}

		char st='*';

		if (cb->term){
			st='>';
			if (cb->lastchanged)st='$';
		}else{
			st=cb->fried?'!':'-';
			if (cb->lastchanged)st='+';
		}
		if (!cb->fried && !cb->alone)st='?';
		cb->mytitle[0]=st;
	}
}

void YateApplication::CheckTitle(YateWindow *yw,Buff *bb){
	if (!bb) return;

	char    nutitle[1024];
	SetName(bb);
	sprintf(nutitle,"YATE %d  %c%s",yw->wincnt,bb->mytitle[0],bb->myname);
	if (strcmp(nutitle,yw->oldtitle)){
		BAutolock l7(yw);
		yw->SetTitle(nutitle);
		strcpy(yw->oldtitle,nutitle);
	}
}

void YateApplication::DoDocText(){
	BAutolock l1(gb);
	int32 idx=-1;
	for (;;){
		Buff *tp=BuffAt(++idx);
		if (!tp) break;
		gb->SetBuff(idx,tp);
		gb->SetName(idx,tp->mytitle);
	}
	gb->SetCount(idx);
}


void YateApplication::MakePerfect(){
	int32 mx=buflist->Num();
	for (int32 i=0;i<mx;i++){
		Buff *cb=BuffAt(i);
		BAutolock l1(cb);
		CheckAlone(cb);
		SetName(cb);
	}
	Sort();
	int32 idx=0;
	YateWindow *yw=NULL;
	while ( (yw=(YateWindow*)winlist->Item(idx++))!=0){
		CheckTitle(yw,yw->cb);
	}
	DoDocText();
}

void YateApplication::Discard(Buff *bb){
	if (!bb){
		printf("discard got bad args");
	}

	Buff *aa=Shift(bb,1);
	buflist->RemIndex(buflist->IndexOf(bb));
	if (aa==bb) aa=BuffAt(0);
	if (!aa){aa=NewBuff();}

	int idx=0;
	YateWindow *yw=NULL;
	while ( (yw=bb->WinAt(idx))!=0){
		BAutolock  l7(yw);
		CheckTitle(yw,aa);
		yw->ChangeBuff(aa);
		MakePerfect();
	}


	bb->PostMessage(B_QUIT_REQUESTED);
}

void YateApplication::Sort(){
	int go=TRUE;

	while (go){
		go=FALSE;
		int i,mx=buflist->Num();
		for (i=0;i<mx-1;i++)  {
			Buff *a=BuffAt(i);
			Buff *b=BuffAt(i+1);
			if (strcmp(a->myname,b->myname)>0){
				buflist->SwapWith(b,i);
				buflist->SwapWith(a,i+1);
/*
				void **src=(void**)lst->Items();
				src[i]=b;
				src[i+1]=a;
*/
				go=TRUE;
			}
		}
	}
}

int32 YateApplication::CountItems(){
	return buflist->Num();
}

void YateApplication::MySave(YAction type){

	int mx=buflist->Num();

	for (int idx=0;idx<mx;idx++){
		Buff *bb=BuffAt(idx);
		if (bb->alone && !bb->fried){
			BAutolock l2(bb);
			if (type==A_SAVE_CHANGED){
				if (bb->lastchanged!=0){
					bb->PutToEnt();
				}
			}else{
				bb->PutToEnt();
			}
		}
	}
}


Buff *YateApplication::FirstShell(){
	int32 max=buflist->Num();
	for (int i=0;i<max;i++){
		Buff *nb=BuffAt(i);
		if (!nb)break;
		if (nb->term){return nb;}
	}

	return NewShell();
}

Buff *YateApplication::GetBuff(entry_ref *er){
	for (int ll=0;TRUE;ll++){
		Buff *bb=BuffAt(ll);
		if (!bb)break;
		if (*er==bb->myent){
			return bb;
		}
	}

	BFile   fil(er,B_READ_ONLY);
	if (fil.IsReadable() ){
		Buff *bb=NewBuff();
		BAutolock l7(bb);
		bb->GetFromEnt(er);
		return bb;
	}

	return NULL;
};

Buff *YateApplication::GetBuff(char *path){
	entry_ref   er;
	status_t rr=get_ref_for_path(path,&er);
	if (BEntry(&er).Exists() ) return GetBuff(&er);

	printf("Yate:<sniff> Couldn't open '%s'\n",path);
	return  NULL;
}


void YateApplication::YExec(YateWindow *yw,char *cmd){
	if (!cmd)return;
	MySave(A_SAVE_CHANGED);
	Buff *cb=FirstShell();
	BAutolock l7(cb);
	int ll=strlen(cmd);
	cb->MoveTo(128,1);
	cb->AddChar(128,"$ ",2);
	cb->AddChar(128,cmd,ll);
	cb->AddChar(128,"\n",1);
	for (int i=140;i<160;i+=4){
		if (cb->Pos(i)>=1)cb->MoveTo(i,1);
	}
	cb->MoveTo(129,-1);
	cb->TX(1026);
	cb->StripToPound(128);
	cb->MyExec(yw,128,cmd);
//    YateWindow *yw=(YateWindow*)winlist->Item(0);
//    if (yw)yw->ChangeBuff(cb);
}


void YateApplication::DoNextError(YateWindow *yw){
	int32 linenum=0;
	char myname[1024];
	Buff *cb=FirstShell();
	if (!cb){beep();return;}
	int32 iserr=FALSE;
	{
		BAutolock l1(cb);
		iserr=cb->NextError(yw,myname,&linenum);
	}
//  printf(" got %s, %d,%d\n",myname,linenum,lasterrpos);
	if (!iserr){
		DoMsg(yw,"No More Errors");
		return;
	}
	Buff *bb=GetBuff(myname);
//  printf("open %x\n",myname);
	if (!bb){
		DoMsg(yw,"Couldn't Open Error File");
		return;
	}
	{
		BAutolock l2(yw);
		yw->ChangeBuff(bb);
	}
	{
		BMessage ms(A_GOTO_LINEY);
		ms.AddPointer("YWin",yw);
		ms.AddInt32("Pos",linenum);
		bb->PostMessage(&ms);
	}
}
