/* Clip handling functions.

   Copyright (C) 1993-1996 Sebastiano Vigna

    This file is part of ne, the nice editor.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

In other words, you are welcome to use, share and improve this program.
You are forbidden to forbid anyone else to use, share and improve
what you give them.   Help stamp out software-hoarding!  */


#include "ne.h"


#ifndef min
#define   min(a,b)    ((a) <= (b) ? (a) : (b))
#endif


/* A clip is a numbered node in the global clip list. The contents of the
clip are handled through the stream functions contained in streams.c. */



/* This function allocates a clip descriptor. */

clip_desc *alloc_clip_desc(int n, int size) {

	clip_desc *cd;

	assert(n>=0);
	assert(size>=0);

	if (cd = calloc(1, sizeof(clip_desc))) {
		cd->n = n;
		if (cd->cs = alloc_char_stream(size)) return(cd);

		free(cd);
	}
	return(NULL);
}



/* This function reallocates a clip descriptor of the given size. If cd is
NULL, this is equivalent to calling alloc_clip_desc. */

clip_desc *realloc_clip_desc(clip_desc *cd, int n, int size) {

	char_stream *cs;

	assert(n>=0);
	assert(size>=0);

	if (!cd) return(alloc_clip_desc(n, size));

	assert_clip_desc(cd);

	if (cd->n != n) return(NULL);

	if (cs = realloc_char_stream(cd->cs, size)) {
		cd->cs = cs;
		return(cd);
	}

	return(NULL);
}


/* This function frees a clip descriptor. */

void free_clip_desc(clip_desc *cd) {

	if (!cd) return;

	assert_clip_desc(cd);

	free_char_stream(cd->cs);
	free(cd);
}


/* This function scans the global clip list, searching for a specific
numbered clip. Returns NULL on failure. */

clip_desc *get_nth_clip(int n) {
	clip_desc *cd = (clip_desc *)clips.head;

	while (cd->cd_node.next) {

		assert_clip_desc(cd);

		if (cd->n == n) return(cd);
		cd = (clip_desc *)cd->cd_node.next;
	}

	return(NULL);
}




/* This function copies the characters between the cursor and the block
marker of the given buffer to the nth clip. If the cut flag is true, the
characters are also removed from the text. The code scans the text two
times: the first time in order to determine the exact length of the text,
the second time in order to actually copy it. */

int copy_to_clip(buffer *b, int n, int cut) {

	int i, pass, start_pos, len, clip_len, y = b->cur_line;
	char *p;
	clip_desc *cd, *new_cd;
	line_desc *ld;

	if (!b->marking) return(MARK_BLOCK_FIRST);

	if (b->block_start_line >= b->line_num) return(MARK_OUT_OF_BUFFER);

	ld = b->cur_line_desc;

	/* If the mark and the cursor are on the same line and on the same position
	(or both beyond the line length), we can't copy anything. */

	cd = get_nth_clip(n);

	if (y == b->block_start_line &&
		(b->cur_pos == b->block_start_pos ||
		b->cur_pos >= ld->line_len && b->block_start_pos >= ld->line_len)) {

		if (!(new_cd = realloc_clip_desc(cd, n, 0))) return(OUT_OF_MEMORY);
		if (!cd) add_head(&clips, &new_cd->cd_node);
		return(OK);
	}


	/* We have two different loops for direct or inverse copying. Making this
	conditional code would be cumbersome, awkward, and definitely inefficient. */

	if (y > b->block_start_line || y == b->block_start_line && b->cur_pos > b->block_start_pos) {

		for(pass=clip_len=0; pass<2; pass++) {

			ld = b->cur_line_desc;

			for(i=y; i>=b->block_start_line; i--) {

				start_pos = 0;
				len = ld->line_len;

				if (i == b->block_start_line) {
					start_pos = b->block_start_pos >= ld->line_len ? ld->line_len : b->block_start_pos;
					len -= start_pos;
				}

				if (i == y) {
					len -= (b->cur_pos >= ld->line_len) ? 0 : ld->line_len-b->cur_pos;
				}

				if (pass) {
					assert(!(len != 0 && ld->line == NULL));

					*--p = 0;
					p -= len;
					if (ld->line) memcpy(p, ld->line+start_pos, len);
				}
				else clip_len += len+1;

				ld = (line_desc *)ld->ld_node.prev;
			}

			if (pass) {

				cd->cs->len = clip_len;
				assert_clip_desc(cd);

				if (cut) {
					goto_line(b, b->block_start_line);
					goto_column(b, b->block_start_col);
					delete_stream(b, b->cur_line_desc, b->cur_line, b->cur_pos, clip_len-1);
				}

				return(OK);
			}

			if (!(new_cd = realloc_clip_desc(cd, n, clip_len))) return(OUT_OF_MEMORY);
			if (!cd) add_head(&clips, &new_cd->cd_node);
			cd = new_cd;
			p = cd->cs->stream+clip_len;
		}
	}
	else {

		for(pass=clip_len=0; pass<2; pass++) {

			ld = b->cur_line_desc;

			for(i=y; i<=b->block_start_line; i++) {

				start_pos = 0;
				len = ld->line_len;

				if (i == y) {
					start_pos = b->cur_pos >= ld->line_len ? ld->line_len : b->cur_pos;
					len -= start_pos;
				}

				if (i == b->block_start_line) {
					len -= (b->block_start_pos >= ld->line_len) ? 0 : ld->line_len-b->block_start_pos;
				}

				if (pass) {
					assert(!(len != 0 && ld->line == NULL));

					if (ld->line) memcpy(p, ld->line+start_pos, len);
					p += len;
					*(p++) = 0;
				}
				else clip_len += len+1;

				ld = (line_desc *)ld->ld_node.next;
			}

			if (pass) {
				cd->cs->len = clip_len;
				assert_clip_desc(cd);
				if (cut) delete_stream(b, b->cur_line_desc, b->cur_line, b->cur_pos, clip_len-1);
				return(OK);
			}

			if (!(new_cd = realloc_clip_desc(cd, n, clip_len))) return(OUT_OF_MEMORY);
			if (!cd) add_head(&clips, &new_cd->cd_node);
			cd = new_cd;
			p = cd->cs->stream;
		}
	}
}



/* This function simply erases a block, without putting it in a clip. */

int erase_block(buffer *b) {

	int i, start_pos, len, erase_len = 0, y = b->cur_line;
	line_desc *ld;

	if (!b->marking) return(MARK_BLOCK_FIRST);
	if (b->block_start_line >= b->line_num) return(MARK_OUT_OF_BUFFER);

	ld = b->cur_line_desc;

	if (y == b->block_start_line &&
		(b->cur_pos == b->block_start_pos ||
		b->cur_pos >= ld->line_len && b->block_start_pos >= ld->line_len))
		return(OK);

	if (y > b->block_start_line || y == b->block_start_line && b->cur_pos > b->block_start_pos) {
		for(i=y; i>=b->block_start_line; i--) {
			start_pos = 0;
			len = ld->line_len;
			if (i == b->block_start_line) {
				start_pos = b->block_start_pos >= ld->line_len ? ld->line_len : b->block_start_pos;
				len -= start_pos;
			}
			if (i == y) {
				len -= (b->cur_pos >= ld->line_len) ? 0 : ld->line_len-b->cur_pos;
			}
			erase_len += len+1;
			ld = (line_desc *)ld->ld_node.prev;
		}
		goto_line(b, b->block_start_line);
		goto_column(b, b->block_start_col);
	}
	else {
		for(i=y; i<=b->block_start_line; i++) {
			start_pos = 0;
			len = ld->line_len;
			if (i == y) {
				start_pos = b->cur_pos >= ld->line_len ? ld->line_len : b->cur_pos;
				len -= start_pos;
			}
			if (i == b->block_start_line) {
				len -= (b->block_start_pos >= ld->line_len) ? 0 : ld->line_len-b->block_start_pos;
			}
			erase_len += len+1;
			ld = (line_desc *)ld->ld_node.next;
		}
	}

	delete_stream(b, b->cur_line_desc, b->cur_line, b->cur_pos, erase_len-1);
	return(OK);
}




/* This function pastes a clip into a buffer. Since clips are streams, the
operation is definitely straightforward. */

int paste_to_buffer(buffer *b, int n) {

	clip_desc *cd;

	if (!(cd = get_nth_clip(n))) return(CLIP_DOESNT_EXIST);

	if (!cd->cs->len) return(OK);

	insert_stream(b, b->cur_line_desc, b->cur_line, b->cur_pos, cd->cs->stream, cd->cs->len);

	return(OK);
}



/* This function works like copy_to_clip(), but the region to copy is the
rectangle defined by the cursor and the marker. Same comments apply. Note
that in case of a cut we use start_undo_chain() in order to make the various
deletions a single undo operation. */

int copy_vert_to_clip(buffer *b, int n, int cut) {

	int i, pass, start_pos, len, clip_len, y = b->cur_line, start_x, end_x;
	char *p;
	clip_desc *cd, *new_cd;
	line_desc *ld;

	if (!b->marking) return(MARK_BLOCK_FIRST);
	if (b->block_start_line >= b->line_num) return(MARK_OUT_OF_BUFFER);

	ld = b->cur_line_desc;

	cd = get_nth_clip(n);

	if (b->win_x+b->cur_x  == b->block_start_col ||
		y == b->block_start_line && b->cur_pos >= ld->line_len && b->block_start_pos >= ld->line_len) {

		if (!(new_cd = realloc_clip_desc(cd, n, 0))) return(OUT_OF_MEMORY);
		if (!cd) add_head(&clips, &new_cd->cd_node);
		return(OK);
	}

	start_x = b->block_start_col;
	end_x = b->win_x+b->cur_x;

	if (end_x<start_x) {
		start_x = b->win_x+b->cur_x;
		end_x = b->block_start_col+1;
	}

	if (cut) start_undo_chain(b);

	if (y > b->block_start_line) {

		for(pass=clip_len=0; pass<2; pass++) {

			ld = b->cur_line_desc;

			for(i=y; i>=b->block_start_line; i--) {

				start_pos = calc_pos(ld, start_x, b->tab_size);
				len = calc_pos(ld, end_x, b->tab_size)-start_pos;

				if (pass) {
					*--p = 0;
					p -= len;
					if (len) memcpy(p, ld->line+start_pos, len);
					if (cut) delete_stream(b, ld, i, start_pos, len);
				}

				else clip_len += len+1;
				ld = (line_desc *)ld->ld_node.prev;
			}

			if (pass) {
				cd->cs->len = clip_len;
				assert_clip_desc(cd);
				if (cut) {
					goto_line(b, min(b->block_start_line, b->cur_line));
					goto_pos(b, min(b->block_start_pos, b->cur_pos));
					end_undo_chain(b);
				}
				return(OK);
			}

			if (!(new_cd = realloc_clip_desc(cd, n, clip_len))) return(OUT_OF_MEMORY);
			if (!cd) add_head(&clips, &new_cd->cd_node);
			cd = new_cd;
			p = cd->cs->stream+clip_len;
		}
	}
	else {

		for(pass=clip_len=0; pass<2; pass++) {

			ld = b->cur_line_desc;

			for(i=y; i<=b->block_start_line; i++) {

				start_pos = calc_pos(ld, start_x, b->tab_size);
				len = calc_pos(ld, end_x, b->tab_size)-start_pos;

				if (pass) {
					if (len) memcpy(p, ld->line+start_pos, len);
					p += len;
					*(p++) = 0;
					if (cut) delete_stream(b, ld, i, start_pos, len);
				}

				else clip_len += len+1;
				ld = (line_desc *)ld->ld_node.next;
			}

			if (pass) {
				cd->cs->len = clip_len;
				assert_clip_desc(cd);
				if (cut) {
					goto_line(b, min(b->block_start_line, b->cur_line));
					goto_pos(b, min(b->block_start_pos, b->cur_pos));
					end_undo_chain(b);
				}
				return(OK);
			}

			if (!(new_cd = realloc_clip_desc(cd, n, clip_len))) return(OUT_OF_MEMORY);
			if (!cd) add_head(&clips, &new_cd->cd_node);
			cd = new_cd;
			p = cd->cs->stream;
		}
	}

	if (cut) end_undo_chain(b);
	return(OK);
}


int erase_vert_block(buffer *b) {

	int i, start_pos, len, y = b->cur_line, start_x, end_x;
	line_desc *ld;

	if (!b->marking) return(MARK_BLOCK_FIRST);
	if (b->block_start_line >= b->line_num) return(MARK_OUT_OF_BUFFER);

	ld = b->cur_line_desc;

	if (b->win_x+b->cur_x  == b->block_start_col ||
		y == b->block_start_line && b->cur_pos >= ld->line_len && b->block_start_pos >= ld->line_len)
		return(OK);

	start_x = b->block_start_col;
	end_x = b->win_x+b->cur_x;

	if (end_x<start_x) {
		start_x = b->win_x+b->cur_x;
		end_x = b->block_start_col+1;
	}

	start_undo_chain(b);

	if (y > b->block_start_line) {
		for(i=y; i>=b->block_start_line; i--) {
			start_pos = calc_pos(ld, start_x, b->tab_size);
			len = calc_pos(ld, end_x, b->tab_size)-start_pos;
			delete_stream(b, ld, i, start_pos, len);
			ld = (line_desc *)ld->ld_node.prev;
		}
	}
	else {
		for(i=y; i<=b->block_start_line; i++) {
			start_pos = calc_pos(ld, start_x, b->tab_size);
			len = calc_pos(ld, end_x, b->tab_size)-start_pos;
			delete_stream(b, ld, i, start_pos, len);
			ld = (line_desc *)ld->ld_node.next;
		}
	}

	goto_line(b, min(b->block_start_line, b->cur_line));
	goto_pos(b, min(b->block_start_pos, b->cur_pos));

	end_undo_chain(b);


	return(OK);
}



/* This function performes a vertical paste. It has to be done via an
insert_stream() for each string of the clip. Again, the undo linking feature
makes all these operations a single undo step. */

int paste_vert_to_buffer(buffer *b, int n) {

	int len, i, x, line;
	char *p;
	clip_desc *cd;
	line_desc *ld;

	if (!(cd = get_nth_clip(n))) return(CLIP_DOESNT_EXIST);

	if (!cd->cs->len) return(OK);

	p = cd->cs->stream;
	ld = b->cur_line_desc;
	line = b->cur_line;
	x = b->cur_x+b->win_x;

	start_undo_chain(b);

	while(p-cd->cs->stream < cd->cs->len) {
		len = strlen(p)+1;

		if (!ld->ld_node.next) {
			insert_lin(b, (line_desc *)ld->ld_node.prev, line, ((line_desc *)ld->ld_node.prev)->line_len);
			ld = (line_desc *)ld->ld_node.prev;
		}

		for(n=i=0; i<ld->line_len && n<x; i++) {
			if (ld->line[i] == '\t') n += b->tab_size - n%b->tab_size;
			else n++;
		}

		if (i == ld->line_len && x-n>0) {
			insert_spaces(b, ld, line, ld->line_len, x-n);
			insert_stream(b, ld, line, ld->line_len, p, len);
		}
		else insert_stream(b, ld, line, i, p, len);

		p += len;
		ld = (line_desc *)ld->ld_node.next;
		line++;
	}

	end_undo_chain(b);

	return(OK);
}


/* This function loads a clip. It is just a load_stream, plus an
insertion in the clip list. */

int load_clip(int n, const char *name) {

	clip_desc *cd = get_nth_clip(n);

	if (!cd) {
		if (!(cd = alloc_clip_desc(n, 0))) return(OUT_OF_MEMORY);
		add_head(&clips, &cd->cd_node);
	}

	return(load_stream(cd->cs, name) ? 0 : CANT_OPEN_FILE);
}


/* This function saves a clip to a file. */

int save_clip(int n, const char *name) {

	clip_desc *cd = get_nth_clip(n);

	if (!cd) return(CLIP_DOESNT_EXIST);
	return(save_stream(cd->cs, name));
}
