// TrackView.cpp (nanodot)
// e.moon 27apr98

#include <stdio.h>
#include <string.h>
#include "TrackView.h"
#include "InvocableMenuItem.h"

TrackView::TrackView(Track& track, BRect frame) :
	m_track(track),
	BView(frame, "TrackView", B_FOLLOW_NONE, B_WILL_DRAW) {

	// put myself together:
	BPopUpMenu* pNoteMenu = new BPopUpMenu("note-menu");
	buildNoteMenu(pNoteMenu);
	
	// build & add menu field:
	m_pNoteField = new BMenuField(BRect(3, 7, 95, 40),
		"note-menu-field", "Note:", pNoteMenu);
	AddChild(m_pNoteField);
	
	// edge of menu area:
	m_menuRect = BRect(0, 0, 100, 40);
	
	// build & add track-editing checkboxes
	BRect checkFrame(113, 9, 24, 26);
	
	for(int16 nCount = 0; nCount < m_track.getSeqLength(); nCount++) {
		
		// construct a message:
		BMessage* pMsg = new BMessage(TRACK_EDIT_SEQ);
		
		// create checkbox
		BCheckBox* pCheckBox = new BCheckBox(
			checkFrame,
			NULL,
			"", // no label
			pMsg, // message to fire on click
			B_FOLLOW_NONE,
			B_WILL_DRAW); // turn off keyboard navigation			
		
		// get height
		pCheckBox->ResizeToPreferred();
		pCheckBox->SetValue(m_track.getData(nCount));
			
		// initialize the message:
		pMsg->AddInt16("seq-pos", nCount);
		pMsg->AddPointer("check-box", pCheckBox);
		pMsg->AddPointer("track-view", this);

		// bump frame along for next checkbox:
		checkFrame.OffsetBy(24.0, 0.0);
		
		// bump every 4 a little more:
		if((nCount % 4) == 3)
			checkFrame.OffsetBy(4.0, 0.0);
		
		// add checkbox to view:
		AddChild(pCheckBox);
		
		// add to retarget-list:
		m_controls.AddItem(pCheckBox);
	}
	
	// prepare for drawing
	
	rgb_color bgMenu = {219,219,219,255};
	SetViewColor(bgMenu);
}

TrackView::~TrackView() {;}

// buildNoteMenu: fills a BMenu with note numbers & names
void TrackView::buildNoteMenu(BMenu* pMenu) {
	// start octave
	int nOctave = 1;
	// start pitch (0=C)
	int nPitch = 0;
	// start note-number (36=C1)
	int nNote = 36;
	// last note-number
	int nEndNote = 60;
	
	// pitch names
	char ppszPitchNames[][3] = {
		"C ",
		"C#",
		"D ",
		"D#",
		"E ",
		"F ",
		"F#",
		"G ",
		"G#",
		"A ",
		"A#",
		"B "
	};
	
	char pNameBuffer[16];
	while(nNote <= nEndNote) {
		// name note;
		sprintf(pNameBuffer, "%s %d", ppszPitchNames[nPitch],
			nOctave);

		// Construct message:
		BMessage* pMsg = new BMessage(TRACK_SET_PITCH);
		pMsg->AddInt16("pitch", nNote);
		pMsg->AddPointer("track-view", this);
		
		// Build menu item:
		InvocableMenuItem* pItem = new InvocableMenuItem(pNameBuffer, pMsg);
			
		// Add to menu:
		pMenu->AddItem(pItem, 0);
		
		// add to retarget list:
		//m_controls.AddItem(pItem);
		
		// Reference menu:
		pMsg->AddPointer("menu", pMenu);	
		
		// Bump counts:
		++nNote;
		if(++nPitch > 11) {
			nPitch = 0;
			++nOctave;
		}
	}
}

// target messages:

void TrackView::AllAttached() {
	// walk m_controls; retarget each to myself:
	for(int nCount = 0; nCount < m_controls.CountItems(); nCount++) {
		BControl* pC = (BControl*)m_controls.ItemAt(nCount);
		pC->SetTarget(this);
//		printf("targeted: %d (%p)\n", nCount, pC);
	}
	
	// retarget note menu:
	BMenu* pMenu = m_pNoteField->Menu();
	pMenu->SetTargetForItems(this);
	
	// trigger the right item in note menu:
	// (walk it & compare pitch in message)
	// +++++ ICK
	
	int nLast = pMenu->CountItems();
	int nTarget = m_track.getPitch();
	
	for(int n = 0; n < nLast; n++) {
		InvocableMenuItem* pCur = (InvocableMenuItem*)pMenu->ItemAt(n);
		BMessage* pMsg = pCur->Message();
		
		int16 nPitch;
		if(pMsg->FindInt16("pitch", &nPitch) == B_OK &&
			 nPitch == nTarget) {
			 
			// found it; fire:
			pCur->Fire();
			break;
		}
	}

	// hand off to mommy
	BView::AllAttached();	
}

// handle edit messages:

void TrackView::handleEditSeq(BMessage* pMsg) {
	// get position in sequence
	int16 nSeqPos;
	if(pMsg->FindInt16("seq-pos", &nSeqPos) != B_OK)
		return;
		
	// get checkbox ptr:
	BCheckBox* pCheck;
	if(pMsg->FindPointer("check-box", &pCheck) != B_OK)
		return;
	
	// set value:
	m_track.setData(nSeqPos, (pCheck->Value() != 0));
}

void TrackView::handleSetPitch(BMessage* pMsg)  {
	int16 nPitch;
	if(pMsg->FindInt16("pitch", &nPitch) != B_OK)
		return;
		
	m_track.setPitch(nPitch);
}
	
void TrackView::MessageReceived(BMessage* pMsg) {
	
	switch(pMsg->what) {
		case TrackView::TRACK_EDIT_SEQ:
			handleEditSeq(pMsg);
			break;
		
		case TrackView::TRACK_SET_PITCH:
			handleSetPitch(pMsg);
			break;

		default:
			BView::MessageReceived(pMsg);
			break;
	}
}

// paint method: draws a 3d border

void TrackView::Draw(BRect rect) {
	BRect bounds = Bounds();

	// colors:
	rgb_color borderHi = {153,153,153,255};
	rgb_color borderSh = {87,87,87,255};
	rgb_color edgeHi = {244,244,244,255};
	rgb_color edgeSh = {197,197,197,255};
	rgb_color bgMenu = {219,219,219,255};
	rgb_color bgSequence = {0,0,0,255};
	
	// draw the outer border:
	
	// top-left
	SetHighColor(edgeHi);
	StrokeLine(bounds.LeftTop(), bounds.RightTop());
	StrokeLine(bounds.LeftTop(), bounds.LeftBottom()-BPoint(0,1));
	
	// bottom-right
	SetHighColor(borderSh);
	StrokeLine(bounds.LeftBottom(), bounds.RightBottom());
	StrokeLine(bounds.RightTop(), bounds.RightBottom()-BPoint(0,1));
	SetHighColor(borderHi);
	bounds.bottom--;
	StrokeLine(bounds.LeftBottom(), bounds.RightBottom());
}

// END -- TrackView.cpp --