/* cmdline.cpp

   Command line version of maplay written by John Fehr

   Fixes for HP-UX by Earle F Philhower <earle@geocities.com>

   This can be used for unix systems as well as the Win32 console. */

#ifndef __DATE__
#define __DATE__  Mar 22, 1997
#endif // __DATE__

#ifndef __TIME__
#define __TIME__   12:00:00
#endif // __TIME__

#ifdef __WIN32__
#include <windows.h>
#endif // __WIN32__

#include <stdio.h>
#include <string.h>
#include <iostream.h>

#include "all.h"
#include "crc.h"
#include "args.h"

// What character marks the end of a directory entry? For DOS and
// Windows, it is "\"; in UNIX it is "/".
#if defined(__WIN32__) || defined(OS2) || defined(__DOS__)
#define DIR_MARKER '\\'
#else
#define DIR_MARKER '/'
#endif // __WIN32__

#ifdef IRIX
#define MAPLAY_VERSION "SGI IRIX"
#define MAPLAY_PORTER  "Andres Meyer (ameyer@desun1.epfl.ch)"
#define MAPLAY_PORT_VERSION_NUMBER "1.1"
#else
#ifdef SunOS4_1_1
#define MAPLAY_VERSION "SPARC SunOS 4.1.1"
#define MAPLAY_PORTER  "Anonymous SPARC SunOS 4.1.1 Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef SunOS4_1_3
#define MAPLAY_VERSION "SPARC SunOS 4.1.3"
#define MAPLAY_PORTER  "Anonymous SPARC SunOS 4.1.3 Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef Solaris
#define MAPLAY_VERSION "SPARC Solaris 2.x"
#define MAPLAY_PORTER  "Anonymous SPARC Solaris 2.x Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef LINUX
#ifdef PENTIUM
#define MAPLAY_VERSION "Linux, Pentium CPU"
#else
#define MAPLAY_VERSION "Linux, 486 CPU"
#endif // PENTIUM
#define MAPLAY_PORTER  "Anonymous Linux Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef __FreeBSD__
#ifdef PENTIUM
#define MAPLAY_VERSION "FreeBSD, Pentium CPU"
#else
#define MAPLAY_VERSION "FreeBSD, 486 CPU"
#endif // PENTIUM
#define MAPLAY_PORTER  "Jeff Tsay"
#define MAPLAY_PORT_VERSION_NUMBER "1.1"
#else
#ifdef ULTRIX
#define MAPLAY_VERSION "RISC ULTRIX 4.x"
#define MAPLAY_PORTER  "Anonymous ULTRIX Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef HPUX
#define MAPLAY_VERSION "HP-UX 9.05 or greater"
#define MAPLAY_PORTER  "Earle Philhower (earle@geocities.com)"
#define MAPLAY_PORT_VERSION_NUMBER "1.1"
#else
#ifdef AIX
#define MAPLAY_VERSION "AIX"
#define MAPLAY_PORTER  "Anonymous AIX Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef NeXT
#define MAPLAY_VERSION "NeXTStep 3.2"
#define MAPLAY_PORTER  "Anonymous NeXTStep 3.2 Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#else
#ifdef __WIN32__
#ifdef PENTIUM
#define MAPLAY_VERSION "Win32 Console, Pentium CPU"
#else
#define MAPLAY_VERSION "Win32 Console, 486 CPU"
#endif // PENTIUM
#define MAPLAY_PORTER  "Jeff Tsay"
#define MAPLAY_PORT_VERSION_NUMBER "1.1"
#else
#ifdef BEOS
#define MAPLAY_VERSION "BeOS Preview Release"
#define MAPLAY_PORTER  "Thomas Philippe (tphilippe@sdv.fr)"
#define MAPLAY_PORT_VERSION_NUMBER "1.2"
#else
#ifdef __DOS__
#ifdef PENTIUM
#define MAPLAY_VERSION "DOS, Pentium CPU"
#else
#define MAPLAY_VERSION "DOS, 486 CPU"
#endif // PENTIUM
#define MAPLAY_PORTER  "Jochen Wilhelmy (digisnap@cs.tu-berlin.de)"
#define MAPLAY_PORT_VERSION_NUMBER "1.1"
#else
#define MAPLAY_VERSION "unknown"
#define MAPLAY_PORTER  "Anonymous Porter"
#define MAPLAY_PORT_VERSION_NUMBER "1.0"
#endif  // __DOS__
#endif  // BEOS
#endif  // _WIN32__
#endif  // NeXT
#endif  // AIX
#endif  // HPUX
#endif  // ULTRIX
#endif  // __FreeBSD__
#endif  // LINUX
#endif  // Solaris
#endif  // SunOS4_1_3
#endif  // SunOS4_1_1
#endif  // IRIX

#ifdef __WIN32__
HWAVEOUT hwo;
#endif // __WIN32__

bool verbose_mode = FALSE;
char filename[512] = "";

uint32 maplay(MPEG_Args *);

int ProcessArgs(int argc,char *argv[],MPEG_Args *);

int Usage(char *pname)
{
    cerr << "\nusage: " << pname
	 << " [-v] [-s] [-l] [-r] [-d] "
#if defined(SPARC) || defined(HPUX)
	    "[-us] [-uh] "
#ifndef SunOS4_1_1
	    "[-ul] "
#endif
#endif // SPARC || HPUX

#ifdef HPUX
		 "[-w] "
#endif // HPUX

	    "[-f ushort] filename\n"

	    "  filename   filename of a MPEG audio stream or - for stdin\n"
#ifdef VERBOSE
	    "  -v         verbose mode\n"
#endif
#ifdef ULAW
	    "  -s         write u-law samples at 8 kHz rate to stdout\n"
#else
	    "  -s         write pcm samples to stdout\n"
#endif
	    "  -l         decode only the left channel\n"
	    "  -r         decode only the right channel\n"
	    "  -d         downmix mode (layer III only)\n"
#if defined(SPARC) || defined(HPUX)
	    "  -us        send audio signal to speaker (default)\n"
	    "  -uh        send audio signal to headphone\n"
#ifndef SunOS4_1_1
	    "  -ul        send audio signal to line out\n"
#endif
#endif // SPARC || HPUX
#ifdef HPUX
		 "  -w         wait for audio device to become available\n"
#endif // HPUX
#ifdef ULAW
	    "  -amd       force maplay to treat /dev/audio as an amd device\n"
#endif // ULAW
	    "  -f ushort  use this scalefactor instead of the default value 32768\n\n"
	    "@(#) MPEG Audio Player maplay 1.2+ "
#ifdef ULAW
	    "(8 kHz u-law "
#else
	    "("
#endif // ULAW
		 MAPLAY_VERSION " version)\n"
	    "@(#) Copyright (C) 1996, 1997 Jeff Tsay (ctsay@pasteur.eecs.berkeley.edu)\n"
       "@(#) Decoder engine version 1.90. Command-line code by John Fehr.\n"
       "@(#) This port (version " MAPLAY_PORT_VERSION_NUMBER ") by "
       MAPLAY_PORTER ".\n"
       "@(#)\n"
	    "@(#) Original maplay 1.2:\n"
	    "@(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)\n"
	    "@(#) Layer III code adapted from the ISO MPEG Audio Subgroup Software\n"
       "@(#) Simulation Group\n"
       "@(#)\n"
	    "@(#) Created: " __DATE__ " " __TIME__ "\n"
	    "@(#) This program is free software. See the GNU General Public License\n"
	    "@(#) in the file COPYING for more details.\n\n";
    return(0);
  }

int ProcessArgs(int argc, char *argv[], MPEG_Args *ma)
{

  Crc16 *crc = NULL;
  int i;

  ma->stdout_mode  = FALSE;
  ma->verbose_mode = FALSE;

#if defined(SPARC) || defined(HPUX)
  ma->use_speaker   = FALSE;
  ma->use_headphone = FALSE;
  ma->use_line_out  = FALSE;
#endif // SPARC || HPUX

#ifdef __WIN32__
  ma->phwo = &hwo;
#endif

#ifdef HPUX
  ma->wait_if_busy = FALSE;
#endif

  if (argc < 2 || !strncmp (argv[1], "-h", 2))
  {
	return Usage(argv[0]);
  }
  // parse arguments:
  for (i = 1; i < argc; ++i)
    if (argv[i][0] == '-' && argv[i][1])
      switch ((int32)argv[i][1])
      {
	case 'v':
#ifdef VERBOSE
	  verbose_mode = TRUE;
     ma->verbose_mode = TRUE;
#else
	  cerr << "maplay: not compiled with VERBOSE." << endl;
#endif
	  break;
	case 's':
	  ma->stdout_mode = TRUE;
	  break;
	case 'l':
	  ma->which_c = Left;
	  break;
	case 'r':
	  ma->which_c = Right;
	  break;
   case 'd':
	  ma->which_c = Downmix;
     break;

#if defined(SPARC) || defined(HPUX)
	case 'u':
	  switch (argv[i][2])
	  {
	    case 's':
	      ma->use_speaker = TRUE;
	      break;
	    case 'h':
	      ma->use_headphone = TRUE;
	      break;
#ifndef SunOS4_1_1
	    case 'l':
	      ma->use_line_out = TRUE;
	      break;
#endif
	    default:
	      return Usage(argv[0]);
	  }
	  break;
#endif // SPARC || HPUX
#ifdef HPUX
	case 'w':
     ma->wait_if_busy = TRUE;
     break;
#endif // HPUX
#ifdef ULAW
	case 'a':
	  ma->force_amd = TRUE;
	  break;
#endif
	case 'c':
     cerr << "Sorry, this version does not support filter checking.\n" << endl;
	  break;
	case 'f':
	  if (++i == argc)
	  {
	    cerr << "please specify a new scalefactor after the -f option!\n";
	    exit (1);
	  }
	  ma->use_own_scalefactor = TRUE;
	  sscanf (argv[i], "%f", &ma->scalefactor);
	  break;
	default:
	  return Usage(argv[0]);
      }
    else if (!filename[0])
      strcpy(filename,argv[i]);
    else
      return Usage(argv[0]);

  if (!filename[0])
    return Usage(argv[0]);

#if defined(SPARC) || defined(HPUX)
  if (!(ma->use_speaker || ma->use_headphone || ma->use_line_out))
    ma->use_speaker = TRUE;
#endif // SPARC || HPUX

  if (!strcmp (filename, "-"))
    ma->stream = new Ibitstream (0);		// read from stdin
  else
    ma->stream = new Ibitstream (filename);		// read from file

  ma->MPEGheader = new Header;
  if (!ma->MPEGheader->read_header (ma->stream, &crc))
  {
    cerr << endl <<  "Error : no header found!" << endl;
    return(0);
  }
  return(1);
}

int maplay_setup (MPEG_Args *ma)
{
#ifdef BEOS
	set_thread_priority(find_thread(NULL), B_URGENT_PRIORITY);
#endif // BEOS

#ifdef VERBOSE
  if (verbose_mode)
  {
    // print information about the stream
    char *name = strrchr (filename, DIR_MARKER);
    if (name)
      ++name;
    else
      name = filename;
    cerr << name << " is a layer " << ma->MPEGheader->layer_string () << " "
	 << ma->MPEGheader->mode_string() << " "
    << ma->MPEGheader->version_string() << " audio stream with";
    if (!ma->MPEGheader->checksums())
      cerr << "out";
    cerr << " checksums.\nThe sample frequency is "
	 << ma->MPEGheader->sample_frequency_string () << " at a bitrate of "
	 << ma->MPEGheader->bitrate_string() << ".\n"
	    "This stream is ";
    if (ma->MPEGheader->original())
      cerr << "an original";
    else
      cerr << "a copy";
    cerr << " and is ";
    if (!ma->MPEGheader->copyright())
      cerr << "not ";
    cerr << "copyright protected.\n";
  }
#endif
  return(1);
}

int main(int argc,char *argv[])
{
	MPEG_Args ma;
	if (!ProcessArgs(argc,argv,&ma)) return(0);
	maplay_setup(&ma);
	maplay(&ma);
   return(0);
}
