//
// killteam v1.3
// Original 1.0 by Eric Shepherd
//
// Public domain.
//
// The killteam utility lets you kill all the threads
// in one or more teams by specifying either the team ID
// or a full or partial team name.
//
// Version 1.1 -- Patched by Steven Black <steven@be.com>
//   Added support for usage infomation on error
//   Added support for single/all team killing
//   Added support for verbose/quiet
//   Cleaned up code a little
//   Added support for DR9.1 (while keeping backward support)
//
// Version 1.2 -- Updated by Eric Shepherd <sheppy@be.com>
//   Added interactive mode
//   Made verbose mode the default
//   Messages include both team number and name
//   Cleaned up code a bit
//
// Version 1.3 -- Updated by Eric Shepherd <sheppy@be.com>
//	Now displays the right thread ID when killing by number.
//	Displays the args for teams even when killed by number.
//	Lets you kill teams by signature (full match only).
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <OS.h>
#include <Roster.h>

#define VERSION "1.3"

#define USAGE "Usage : killteam [-e|-s] [-a|-o] team1 [(options) team2] ...\n" \
	"             -e     Use exact matches for team names (default)\n" \
	"             -s     Use substring matches for team names\n" \
	"             -r     Match on signatures instead of names\n" \
	"             -n     Non-interactive (default)\n" \
	"             -i     Interactive\n" \
	"             -o     Kill one team only (default)\n" \
	"             -a     Kill all teams with this name\n" \
	"             -v     Message on success and error (default)\n" \
	"             -q     Message only on error\n" \
	"        teamN is either a team ID or team name\n" \
	"        options affect all team ID/names until counteracted by another option\n" \
	"        use -e or -s to switch back to name-matching mode after using -r.\n"
	
// Precision constants:
//
//	PRECISION_EXACT				Team names must match exactly
//	PRECISION_SUBSTRING			Any substring must match
#define PRECISION_EXACT 0
#define PRECISION_SUBSTRING 1

team_info info;				// Info on the team to be killed is put here

// Prototypes

int is_number(char *s);
team_id find_named_team(char *s, int prec, int reset);
int handle_error_killteam(int thid, int prec, int verbose, int interactive);

//
// is_number
//
// Determine whether or not the string passed in
// is a valid integral number.
//
int is_number(char *s) {
	while(*s && isdigit(*s++));
	return (*s == '\0');   // If we're at the terminator, it's a number
}

//
// find_named_team
//
// Search through all teams until a match is found with the
// specified name.  We use the precision variable to determine
// whether to do a partial match or full match comparison.
//
// On return, the info structure is filled out with the info on
// the team to be killed, and we return the ID of the team, or
// 0 if no match was found.
//
// If reset is non-zero, we start at the top of the team list;
// otherwise we continue the previous search.
//
team_id find_named_team(char *s, int prec, int reset) 
{
	static int32 cookie;
	int len;
	
	if (reset) {
		cookie = 0;				// Start with first team
	}
	
	len = strlen(s);

	while (get_next_team_info(&cookie, &info) == B_OK) {		
		switch(prec) {
			case PRECISION_SUBSTRING:
				if (strstr(info.args, s) && !strstr(info.args, "killteam")) {
					return info.team;
				}
				break;
			
			case PRECISION_EXACT:
				/* Original version was broke with DR9.1. This one should work with both. */
				if(!memcmp(info.args,s,len)){
					if(info.args[len]=='\0' || info.args[len]==' '){
						return(info.team);
					}
				}
				break;
		}
	}
	return 0;				// No matching team found
}


//
// main
//
// Main program.  Handles the interface to the user.
//
int main(int argc, char **argv)
{
	int precision;				// Specifies how precise string matches must be
	int once;
	int verbose;
	int interactive;
	int skip;
	int matchsigs;
	BRoster theRoster;

	team_id   thid;
	
	precision = PRECISION_EXACT;	// Assume exact precision
	once = 1;
	
	/* User can set to true/false to turn on or off matching */
	/* of signatures instead of names. */
	matchsigs = 0;
	
	/* User can set to true/false for more/less verbosity. */
	/* Default is message on error. */
	/* -q == only messages on error. (default) */
	/* -v == default/excessing information. */
	verbose = 1;
	
	// User can turn on interactive mode, where
	// they're asked before each team is killed.
	interactive = 0;

	if( argc<2 || argv[1][1]=='v' || argv[1][1]=='h') printf("%s -- Version %s -- Public Domain\n", argv[0], VERSION);
	if (argc < 2) {
		fprintf(stderr,USAGE);
		exit(1);
	}

	for (argc--, argv++; argc > 0; argc--, argv++) {
		if (*argv[0] == '-') {
			if (*(argv[0]+1) == 's') {
				precision = PRECISION_SUBSTRING;
				matchsigs = 0;
			}
			else if (*(argv[0]+1) == 'e') {
				precision = PRECISION_EXACT;
				matchsigs = 0;
			}
			else if (*(argv[0]+1) == 'r') {
				matchsigs = 1;
			}
			else if(argv[0][1] == 'h'){
				fprintf(stderr,USAGE);
				exit(1);
			}
			else if(argv[0][1] == 'o'){
				once = 1;
			}
			else if(argv[0][1] == 'a'){
				once = 0;
			}
			else if(argv[0][1] == 'q'){
				verbose = -1;
			}
			else if(argv[0][1] == 'v'){
				verbose = 1;
			}
			else if (argv[0][1] == 'i') {
				interactive = 1;
			}
			else if (argv[0][1] == 'n') {
				interactive = 0;
			}
			else {
				fprintf(stderr, "Invalid option: %s\n", argv[0]);
				fprintf(stderr,USAGE);
				exit(1);
			}
			continue;
		}
		
		skip = 1;		// Default: don't need to skip
		if (is_number(argv[0])) {
			thid = strtoul(argv[0], 0, 0);
			handle_error_killteam(thid, precision, verbose, interactive);
		}
		else {
			if (!matchsigs) {
				thid = find_named_team(argv[0], precision, 1);	// start new search
			}
			else {
				thid = theRoster.TeamFor(argv[0]);
				if (thid < 0) {
					thid = 0;
				}
			}
			if (!thid) {
				fprintf(stderr, "No matching team: %s\n", argv[0]);
				continue;
			}
			else {
				do	{
					handle_error_killteam(thid, precision, verbose, interactive);
					if (!once) {
						if (!matchsigs) {
							thid = find_named_team(argv[0], precision, 0);	// next!
						}
						else {
							thid = theRoster.TeamFor(argv[0]);
							if (thid < 0) {
								thid = 0;
							}
						}
					}
					else {
						thid = 0;
					}
				} while (thid);
			}
		}
	}

	return(0);
}

int handle_error_killteam(int thid, int prec, int verbose, int interactive)
{
	if (interactive) {
		char c[5];
		printf("  kill team %d: %s (y/N/q)? ", info.team, info.args);
		fgets(c, 4, stdin);
		if ((c[0]=='q') || (c[0]=='Q')) {
			exit(1);						// user quits
		}
		if ((c[0]!='y') && (c[0]!='Y')) {
			return 0;
		}
	}
	
	switch(kill_team(thid)) {		// Kill the team
		case B_BAD_TEAM_ID:
			fprintf(stderr, "No such team: %d\n", thid);
			return(-1);

		case B_OK:
			get_team_info(thid, &info);
			if (verbose) {
				fprintf(stderr, "Team killed: %d: %s\n", thid, info.args);
			}
			return(0);
	}
	return(-1);
}

