// BitmapStream.cpp
// © 1998 Matt Lewinski
// Streamable Bitmap (from datatypes library)

#include "BitmapStream.h"
#include <Datatypes.h>

BitmapStream::BitmapStream(BBitmap *bitmap)
{
	mMap = bitmap;
	mDetached = false;
	mPosition = 0;
	mSize = 0;
	if (mMap) {
		mHeader.magic = DATA_BITMAP;
		mHeader.bounds = mMap->Bounds();
		mHeader.rowBytes = mMap->BytesPerRow();
		mHeader.colors = mMap->ColorSpace();
		mHeader.dataSize = (mHeader.bounds.Height()+1)*mHeader.rowBytes;
		mSize = sizeof(DATABitmap)+mHeader.dataSize;
	}
}


BitmapStream::~BitmapStream()
{
	if (mMap && !mDetached)
		delete mMap;
}


status_t BitmapStream::DetachBitmap(BBitmap *&outBitmap)
{
	outBitmap = NULL;
	if (!mMap)
		return B_ERROR;
	if (mDetached)
		return B_ERROR;
	mDetached = true;
	outBitmap = mMap;
	return B_NO_ERROR;
}


off_t BitmapStream::Position() const
{
	return mPosition;
}


status_t BitmapStream::ReadAt(off_t pos, void *buffer, size_t size)
{
	if (!mMap)
		return B_ERROR;
	if (!size)
		return B_NO_ERROR;
	if (pos >= mSize)
		return B_ERROR;

	long toRead;
	void *source;

	if (mPosition < sizeof(DATABitmap)) {
		toRead = sizeof(DATABitmap)-pos;
		source = ((char *)&mHeader)+pos;
	} else {
		toRead = mSize-pos;
		source = ((char *)mMap->Bits())+mPosition-sizeof(DATABitmap);
	}
	if (toRead > size)
		toRead = size;
	memcpy(buffer, source, toRead);
	return toRead;
}


off_t BitmapStream::Seek(off_t inPosition, uint32 inWhence)
{
	if (inWhence == SEEK_CUR)
		inPosition += mPosition;
	if (inWhence == SEEK_END)
		inPosition += mSize;
	if (inPosition < 0)
		return B_BAD_VALUE;
	if (inPosition > mSize)
		return B_BAD_VALUE;
	mPosition = inPosition;
	return mPosition;
}


status_t BitmapStream::SetSize(off_t inSize)
{
	if (inSize < 0)
		return B_BAD_VALUE;
	if (mMap && (inSize > mHeader.dataSize+sizeof(DATABitmap)))
		return B_BAD_VALUE;
	if (mMap)
		mSize = inSize;
	return B_NO_ERROR;
}


off_t BitmapStream::Size() const
{
	return mSize;
}


status_t BitmapStream::WriteAt(off_t	 pos, const void *data, size_t size)
{
	if (!size)
		return B_NO_ERROR;
	ssize_t written = 0;
	while (size > 0) {
		long toWrite;
		void *dest;
			if (pos < sizeof(DATABitmap)) {
			toWrite = sizeof(DATABitmap)-pos;
			dest = ((char *)&mHeader)+pos;
		} else {
			toWrite = mHeader.dataSize-pos+sizeof(DATABitmap);
			dest = ((char *)mMap->Bits())+pos-sizeof(DATABitmap);
		}
		if (toWrite > size)
			toWrite = size;
		if (!toWrite && size)
			return B_BAD_VALUE;
		memcpy(dest, data, toWrite);
		pos += toWrite;
		written += toWrite;
		data = ((char *)data)+toWrite;
		size -= toWrite;
		if (pos > mSize)
			mSize = pos;
		if (pos == sizeof(DATABitmap)) {
			if (mMap && ((mMap->Bounds() != mHeader.bounds) ||
					(mMap->ColorSpace() != mHeader.colors) ||
					(mMap->BytesPerRow() != mHeader.rowBytes))) {
				if (!mDetached)
					delete mMap;
				mMap = NULL;
			}
			if (!mMap) {
				mMap = new BBitmap(mHeader.bounds, mHeader.colors);
				if (mMap->BytesPerRow() != mHeader.rowBytes) {
					return B_MISMATCHED_VALUES;
				}
			}
			if (mMap) {
				mSize = sizeof(DATABitmap)+mMap->BitsLength();
			}
		}
	}
	return written;
}