
#include <math.h>
#include <stdio.h>
#include "idx3d.h"

float sinus[360];		//=new float[360];
float cosinus[360];		//=new float[360];	 


// G3DKIT_MATRIX class

		idx3d_matrix::idx3d_matrix()
		{
			matrix[0][0]=(float)1;
			matrix[1][1]=(float)1;
			matrix[2][2]=(float)1;
			matrix[3][3]=(float)1;

			matrix[0][1]=0.;
			matrix[0][2]=0.;
			matrix[0][3]=0.;
			matrix[1][0]=0.;
			matrix[1][2]=0.;
			matrix[1][3]=0.;
			matrix[2][0]=0.;
			matrix[2][1]=0.;
			matrix[2][3]=0.;
			matrix[3][0]=0.;
			matrix[3][1]=0.;
			matrix[3][2]=0.;
		}

		void idx3d_matrix::crossproduct(idx3d_matrix *a, idx3d_matrix *b)
		{
			matrix[0][0]=a->matrix[0][0]*b->matrix[0][0]+a->matrix[0][1]*b->matrix[1][0]+a->matrix[0][2]*b->matrix[2][0]+a->matrix[0][3]*b->matrix[3][0];
			matrix[0][1]=a->matrix[0][0]*b->matrix[0][1]+a->matrix[0][1]*b->matrix[1][1]+a->matrix[0][2]*b->matrix[2][1]+a->matrix[0][3]*b->matrix[3][1];
			matrix[0][2]=a->matrix[0][0]*b->matrix[0][2]+a->matrix[0][1]*b->matrix[1][2]+a->matrix[0][2]*b->matrix[2][2]+a->matrix[0][3]*b->matrix[3][2];
			matrix[0][3]=a->matrix[0][0]*b->matrix[0][3]+a->matrix[0][1]*b->matrix[1][3]+a->matrix[0][2]*b->matrix[2][3]+a->matrix[0][3]*b->matrix[3][3];
			matrix[1][0]=a->matrix[1][0]*b->matrix[0][0]+a->matrix[1][1]*b->matrix[1][0]+a->matrix[1][2]*b->matrix[2][0]+a->matrix[1][3]*b->matrix[3][0];
			matrix[1][1]=a->matrix[1][0]*b->matrix[0][1]+a->matrix[1][1]*b->matrix[1][1]+a->matrix[1][2]*b->matrix[2][1]+a->matrix[1][3]*b->matrix[3][1];
			matrix[1][2]=a->matrix[1][0]*b->matrix[0][2]+a->matrix[1][1]*b->matrix[1][2]+a->matrix[1][2]*b->matrix[2][2]+a->matrix[1][3]*b->matrix[3][2];
			matrix[1][3]=a->matrix[1][0]*b->matrix[0][3]+a->matrix[1][1]*b->matrix[1][3]+a->matrix[1][2]*b->matrix[2][3]+a->matrix[1][3]*b->matrix[3][3];
			matrix[2][0]=a->matrix[2][0]*b->matrix[0][0]+a->matrix[2][1]*b->matrix[1][0]+a->matrix[2][2]*b->matrix[2][0]+a->matrix[2][3]*b->matrix[3][0];
			matrix[2][1]=a->matrix[2][0]*b->matrix[0][1]+a->matrix[2][1]*b->matrix[1][1]+a->matrix[2][2]*b->matrix[2][1]+a->matrix[2][3]*b->matrix[3][1];
			matrix[2][2]=a->matrix[2][0]*b->matrix[0][2]+a->matrix[2][1]*b->matrix[1][2]+a->matrix[2][2]*b->matrix[2][2]+a->matrix[2][3]*b->matrix[3][2];
			matrix[2][3]=a->matrix[2][0]*b->matrix[0][3]+a->matrix[2][1]*b->matrix[1][3]+a->matrix[2][2]*b->matrix[2][3]+a->matrix[2][3]*b->matrix[3][3];
		}

		void idx3d_matrix::init(idx3d_matrix *m)
		{
			memcpy(matrix, (m->matrix), 4*4*sizeof(float));
		}

		void idx3d_matrix::shiftMatrix(float dx, float dy, float dz)
		{
			matrix[0][3] = dx;
			matrix[1][3] = dy;
			matrix[2][3] = dz;

			matrix[1][0]=0.;
			matrix[2][0]=0.;
			matrix[3][0]=0.;
			matrix[0][0]=0.;
			matrix[1][1]=0.;
			matrix[2][1]=0.;
			matrix[3][1]=0.;
			matrix[0][1]=0.;
			matrix[1][2]=0.;
			matrix[2][2]=0.;
			matrix[3][2]=0.;
			matrix[0][2]=0.;
			matrix[3][3]=0.;
		}

		void idx3d_matrix::scaleMatrix(float dx, float dy, float dz)
		{
			matrix[0][0] = dx;
			matrix[1][1] = dy;
			matrix[2][2] = dz;

			matrix[0][1]=0.;
			matrix[0][2]=0.;
			matrix[0][3]=0.;
			matrix[1][0]=0.;
			matrix[1][2]=0.;
			matrix[1][3]=0.;
			matrix[2][0]=0.;
			matrix[2][1]=0.;
			matrix[2][3]=0.;
			matrix[3][0]=0.;
			matrix[3][1]=0.;
			matrix[3][2]=0.;
			matrix[3][3]=0.;
		}

		void idx3d_matrix::rotateMatrix(float dx, float dy, float dz)
		{
			float SIN;
			float COS;
			long angle;
			idx3d_matrix *out = new idx3d_matrix();
			idx3d_matrix *out2 = new idx3d_matrix();

			if (dx!=0)
			{
				idx3d_matrix *m = new idx3d_matrix();
				angle = ((long)dx+1440)%360;
				SIN = sinus[angle];
				COS = cosinus[angle];
				m->matrix[1][1] =  COS;
				m->matrix[1][2] =  SIN;
				m->matrix[2][1] = -SIN;
				m->matrix[2][2] =  COS;
				out2->crossproduct(m,out);
				out->init(out2);
				delete m;
			}
			if (dy!=0)
			{
				idx3d_matrix *m = new idx3d_matrix();
				angle=((long)dy+720)%360;
				SIN = sinus[angle];
				COS = cosinus[angle];
				m->matrix[0][0] =  COS;
				m->matrix[0][2] =  SIN;
				m->matrix[2][0] = -SIN;
				m->matrix[2][2] =  COS;
				out2->crossproduct(m,out);
				out->init(out2);
				delete m;
			}
			if (dz!=0)
			{
				idx3d_matrix *m = new idx3d_matrix();
				angle=((long)dz+720)%360;
				SIN = sinus[angle];
				COS = cosinus[angle];
				m->matrix[0][0] =  COS;
				m->matrix[0][1] =  SIN;
				m->matrix[1][0] = -SIN;
				m->matrix[1][1] =  COS;
				out2->crossproduct(m,out);
				out->init(out2);
				delete m;
			}
			init(out);

			delete out;
			delete out2;
		}

		void idx3d_matrix::afficher()
		{
			printf("%g %g %g\n", matrix[0][0], matrix[1][0], matrix[2][0]);
			printf("%g %g %g\n", matrix[0][1], matrix[1][1], matrix[2][1]);
			printf("%g %g %g\n", matrix[0][2], matrix[1][2], matrix[2][2]);
		}


// G3DKIT_VECTOR class

		idx3d_vector::idx3d_vector()
		{
			x = 0.;
			y = 0.;
			z = 1.;
		}

		idx3d_vector::idx3d_vector(float xval, float yval, float zval)
		{
			x = xval;
			y = yval;
			z = zval;
		}

		idx3d_vector::~idx3d_vector()
		{}


		void idx3d_vector::setValues(float xval, float yval, float zval)
		{
			x = xval;
			y = yval;
			z = zval;
		}

		void idx3d_vector::vectordist(idx3d_vector *v1, idx3d_vector *v2)
		{
			x = v1->x - v2->x;
			y = v1->y - v2->y;
			z = v1->z - v2->z;
		}


		float idx3d_vector::vectorlength()
		{
			return (float)sqrt(x*x + y*y + z*z);
		}

		void idx3d_vector::normalize(idx3d_vector *v)
		{
			float length=v->vectorlength();
			x = v->x/length;
			y = v->y/length;
			z = v->z/length;
		}

		void idx3d_vector::normalize()
		{
			float length=vectorlength();
			x = x/length;
			y = y/length;
			z = z/length;
		}

		void idx3d_vector::getNormal(idx3d_vector *a, idx3d_vector *b, idx3d_vector *c)
		{
			float ax = b->x - a->x;
			float ay = b->y - a->y;
			float az = b->z - a->z;
			float bx = c->x - a->x;
			float by = c->y - a->y;
			float bz = c->z - a->z;
			x = ay*bz - by*az;
			y = az*bx - bz*ax;
			z = ax*by - bx*ay;
			normalize();
		}


		void idx3d_vector::matrixvectorproduct(idx3d_matrix *b, idx3d_vector *a)
		{
			x = a->x*b->matrix[0][0] + a->y*b->matrix[0][1] + a->z*b->matrix[0][2] + b->matrix[0][3];
			y = a->x*b->matrix[1][0] + a->y*b->matrix[1][1] + a->z*b->matrix[1][2] + b->matrix[1][3];
			z = a->x*b->matrix[2][0] + a->y*b->matrix[2][1] + a->z*b->matrix[2][2] + b->matrix[2][3];
		}


// G3DKIT_NODE class

		idx3d_node::idx3d_node(float x, float y, float z)
		{
			v  = new idx3d_vector();
			n  = new idx3d_vector();
			n2 = new idx3d_vector();

			v->x=x;
			v->y=y;
			v->z=z;
			n->x=0.;
			n->y=0.;
			n->z=1.;
		}

		idx3d_node::idx3d_node()
		{
			v  = new idx3d_vector();
			n  = new idx3d_vector();
			n2 = new idx3d_vector();
			v->x=0.;
			v->y=0.;
			v->z=0.;
			n->x=0.;
			n->y=0.;
			n->z=1.;
		}

		idx3d_node::~idx3d_node()
		{
			delete v;
			delete n;
			delete n2;
		}


// G3DKIT_TRIANGLE class

		idx3d_triangle::idx3d_triangle(long q1, long q2, long q3)
		{
			n = new idx3d_vector();
			p1=q1;
			p2=q2;
			p3=q3;
		}

		idx3d_triangle::~idx3d_triangle()
		{
			delete n;
		}


// G3DKIT_OBJECT class

		idx3d_object::idx3d_object(long rm, long col)
		{
			matrix  = new idx3d_matrix();
			matrix2 = new idx3d_matrix();

			mode=rm;
			color=col;

			maxtriangles = 0;
			triangles = 0;
			maxnodes = 0;
			nodes = 0;
			triangle = NULL;
			node = NULL;
		}

		idx3d_object::~idx3d_object()
		{
			delete matrix;
			delete matrix2;
		}


// G3DKIT_LIGHT class

		idx3d_light::idx3d_light(idx3d_vector *lvect, long lmode, long lintensity)
		{
			x = (long)(lvect->x*255);
			y = (long)(lvect->y*255);
			z = (long)(lvect->z*255);
			mode = lmode;
			intensity = lintensity;
		}


// G3DKIT class


	// GENERAL METHODS

		idx3d::idx3d(long width, long height)
		{
			objects=0;
			lights=0;	
			textures=0;
			maxobjects=0;
			maxlights=0;
			maxtextures=0;
			object = NULL;
			light = NULL;
			texture = NULL;

//			worldmatrix.idx3d_matrix();
			
			perspective = 4.;
			zoomcorrection = 0.8;
			ambient=0;				//0..255
			phong=80;				//0..255
			reflectivity=255;		//0..255
			staticLight=false;

			resize(width, height);
		}

		void idx3d::resize(long width, long height)
		{
			w=width;
			h=height;
			centerx=(float)w/2;
			centery=(float)h/2;
			sysinfo=getSysInfo();
			TargetPixel=new long[w*h];
			Background=new long[w*h];
			LEdgeBuffer=new long[h];
			REdgeBuffer=new long[h];
			zBuffer=new long[w*h];
			ZEdgeBufferL=new long[h];
			ZEdgeBufferR=new long[h];
			IEdgeBufferL=new long[h];
			IEdgeBufferR=new long[h];
			NXEdgeBufferL=new long[h];
			NYEdgeBufferL=new long[h];
			NXEdgeBufferR=new long[h];
			NYEdgeBufferR=new long[h];
			TXEdgeBufferL=new long[h];
			TYEdgeBufferL=new long[h];
			TXEdgeBufferR=new long[h];
			TYEdgeBufferR=new long[h];
	
			bgcolor=getIntColor(0,0,0);
			clearBackground();
			init_LightMap();
			init_ColorMap();
			init_TrigTables();
		}


		char *idx3d::getSysInfo()	
		{
			//return (System.getProperty("os.arch")+" on "+System.getProperty("os.name"));
			return ("no info\n");
		}

		void idx3d::init_LightMap()
		{
			float NX;
			float NY;
			float NZ;
			for (long j=0;j<256;j++)
			{
				for (long i=0;i<256;i++)
				{
					NX=((float)i-127)/127;
					NY=((float)j-127)/127;
					NZ=(float)(1-sqrt(NX*NX+NY*NY));
					LightMap[(i<<8)+j]=crop((long)(NZ*255),0,255);
				}
			}
		}

		void idx3d::setStatic()
		{
			for (long j=0;j<256;j++)
			{
				for (long i=0;i<256;i++)
				{
					StaticLightMap[(i<<8)+j]=getIntensity(2*i-255,2*j-255);
					StaticEnvMap[(i<<8)+j]=getEnvironment(2*i-255,2*j-255);
				}
			}
			staticLight=true;
		}

		void idx3d::init_ColorMap()
		{
			long H2=255-phong;
			for(long x=0;x<256;x++)
			{
				for (long y=0;y<H2;y++)
				{
					ColorMap[(y<<8)+x]=(x*y/H2);
				}
				for (long y=H2;y<256;y++)
				{
					ColorMap[(y<<8)+x]=(y+(255-y)*(x-H2)/phong);
				}
			}	
		}

		void idx3d::init_TrigTables()
		{
			for(long i=0;i<360;i++)
			{
				sinus[i]=(float)sin(deg2rad*(float)i);
				cosinus[i]=(float)cos(deg2rad*(float)i);
			}
		}

		void idx3d::setPhong(long p)
		{
			phong = p;
			init_ColorMap();
		}

		float idx3d::crop(float a, float b, float c)
		{
			if (a<b) return (b);
			if (a>c-1) return (c-1);
			return a;
		}

		long idx3d::crop(long a, long b, long c)
		{	
			if (a<b) return (b);
			if (a>c-1) return (c-1);
			return a;
		}

		bool idx3d::inrange(long a, long b, long c)
		{
			return((a>=b)&&(a<c));
		}

		bool idx3d::inrange(float a, float b, float c)
		{
			return((a>=b)&&(a<c));
		}
		

	// DATA MANAGEMENT

		void idx3d::addObject(long rendermode, long color)
		{
			objects+=1;

			if (objects>=maxobjects)
			{
				if (object==NULL)
				{
					maxobjects=10;
					object=new idx3d_object*[11];
				}
				else
				{
					maxobjects +=10;
					idx3d_object **temp=new idx3d_object*[maxobjects+1];
					memcpy(temp, object, objects*sizeof(idx3d_object*));
					delete object;
					object=temp;
				}
			}
			object[objects]=new idx3d_object(rendermode,color);
		}

		void idx3d::addLight(idx3d_vector *v, long mode, long intensity)
		{
			lights+=1;

			if (lights>=maxlights)
			{
				if (light==NULL)
				{
					maxlights=10;
					light=new idx3d_light*[11];
				}
				else
				{
					maxlights+=10;
					idx3d_light **temp=new idx3d_light*[maxlights+1];
					memcpy(temp, light, lights*sizeof(idx3d_light*));
					delete light;
					light=temp;
				}
			}
			if (mode==1) v->normalize();
			light[lights]=new idx3d_light(v,mode,intensity);
		}

		void idx3d::addTexture(BBitmap *img)
		{
			textures += 1;

			if (textures >= maxtextures)
			{
				if (texture == NULL)
				{
					maxtextures = 20;
					texture = new idx3d_texture*[21];
				}
				else
				{
					maxtextures += 10;
					idx3d_texture **temp=new idx3d_texture*[maxtextures+1];
					memcpy(temp, texture, textures*sizeof(idx3d_texture*));
					delete texture;
					texture=temp;
				}
			}
			texture[textures]=new idx3d_texture();
			BRect rect=img->Bounds();
			long width=(long)rect.right + 1;

			long height=(long)rect.bottom + 1;
			texture[textures]->w=width;
			texture[textures]->h=height;
			texture[textures]->pixel=new long[width*height];
			memcpy(texture[textures]->pixel, img->Bits(), img->BitsLength());
		}

		void idx3d::addTriangle(long objectNr, long p1, long p2, long p3)
		{
			object[objectNr]->triangles+=1;

			if (object[objectNr]->triangles>=object[objectNr]->maxtriangles)
			{
				if (object[objectNr]->triangle==NULL)
				{
					object[objectNr]->maxtriangles=10;
					object[objectNr]->triangle=new idx3d_triangle*[11];
				}
				else
				{
					object[objectNr]->maxtriangles +=10;
					idx3d_triangle **temp=new idx3d_triangle*[object[objectNr]->maxtriangles+1];
					memcpy(temp, object[objectNr]->triangle, (object[objectNr]->triangles)*sizeof(idx3d_triangle*));
					delete object[objectNr]->triangle;
					object[objectNr]->triangle=temp;
				}
			}
			object[objectNr]->triangle[object[objectNr]->triangles]=new idx3d_triangle(p1,p2,p3);
			object[objectNr]->triangle[object[objectNr]->triangles]->n->getNormal(object[objectNr]->node[p1]->v, object[objectNr]->node[p3]->v, object[objectNr]->node[p2]->v);
		}

		void idx3d::addNode(long objectNr, idx3d_node *t)
		{
			object[objectNr]->nodes+=1;
			
			if (object[objectNr]->nodes >= object[objectNr]->maxnodes)
			{
				if (object[objectNr]->node==NULL)
				{
					object[objectNr]->maxnodes=10;
					object[objectNr]->node=new idx3d_node*[11];
				}
				else
				{
					object[objectNr]->maxnodes += 10;
					idx3d_node **temp=new idx3d_node*[object[objectNr]->maxnodes+1];
					memcpy(temp, object[objectNr]->node, object[objectNr]->nodes*sizeof(idx3d_node*));
					delete object[objectNr]->node;
					object[objectNr]->node=temp;
				}
			}
			t->n->normalize();
			object[objectNr]->node[object[objectNr]->nodes]=t;
		}

		void idx3d::addNode(long objectNr,float x, float y, float z)
		{
			idx3d_node *t = new idx3d_node(x,y,z);
			addNode(objectNr,t);
		}



	// OBJECT MANIPULATION

		void idx3d::shiftObject(long o, float dx, float dy, float dz)
		{
			idx3d_matrix *mshift = new idx3d_matrix();
			mshift->shiftMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(object[o]->matrix);

			object[o]->matrix->crossproduct(mshift, mold);
			
			delete mshift;
			delete mold;
		}

		void idx3d::scaleObject(long o, float dx, float dy, float dz)
		{
			idx3d_matrix *mscale = new idx3d_matrix();
			mscale->scaleMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(object[o]->matrix);

			object[o]->matrix->crossproduct(mscale, mold);
			
			delete mscale;
			delete mold;
		}

		void idx3d::rotateObject(long o, float dx, float dy, float dz)
		{
			idx3d_matrix *mrotat = new idx3d_matrix();
			mrotat->rotateMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(object[o]->matrix);

			object[o]->matrix->crossproduct(mold, mrotat);
			
			delete mrotat;
			delete mold;
		}

		void idx3d::rotateObjectWorld(long o, float dx, float dy, float dz)
		{
			idx3d_matrix *mrotat = new idx3d_matrix();
			mrotat->rotateMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(object[o]->matrix);

			object[o]->matrix->crossproduct(mrotat, mold);
			
			delete mrotat;
			delete mold;
		}

		void idx3d::rotateObject(long obj, float px, float py, float pz, float dx, float dy, float dz)
		{
			shiftObject(obj,-px,-py,-pz);
			rotateObject(obj,dx,dy,dz);
			shiftObject(obj,px,py,pz);
		}

		void idx3d::shiftWorld(float dx, float dy, float dz)
		{
			idx3d_matrix *mshift = new idx3d_matrix();
			mshift->shiftMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(&worldmatrix);

			worldmatrix.crossproduct(mshift, mold);
			
			delete mshift;
			delete mold;
		}

		void idx3d::scaleWorld(float dx, float dy, float dz)
		{
			idx3d_matrix *mscale = new idx3d_matrix();
			mscale->scaleMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(&worldmatrix);

			worldmatrix.crossproduct(mscale, mold);
			
			delete mscale;
			delete mold;
		}

		void idx3d::rotateWorld(float dx, float dy, float dz)
		{
			idx3d_matrix *mrotat = new idx3d_matrix();
			mrotat->rotateMatrix(dx,dy,dz);
			idx3d_matrix *mold = new idx3d_matrix();
			mold->init(&worldmatrix);

			worldmatrix.crossproduct(mrotat, mold);
			
			delete mrotat;
			delete mold;
		}

		void idx3d::scaleObject(long obj, float d)
		{
			scaleObject(obj,d,d,d);
		}

		void idx3d::scaleWorld(float d)
		{
			scaleWorld(d,d,d);
		}


	// RENDERING

		void idx3d::renderScene()
		{
			clearDoubleBuffer();
			paintObjects();
		}

		void idx3d::clearDoubleBuffer()
		{
			register long zBuf = zInfinite;
			for (long i=w*h-1;i>=0;i--)
			{
				TargetPixel[i] = Background[i];
				zBuffer[i] = zBuf;
			}
		}

		void idx3d::clearBackground()
		{
			for (long i=w*h-1;i>=0;i--)
			{
					Background[i] = bgcolor;
			}
		}

		void idx3d::paintObjects()
		{
			for(long i=objects; i>0; i--)
			{
				object[i]->matrix2->crossproduct(&worldmatrix, object[i]->matrix);
//long min=50000;
//long max=-50000;
				for(long j=1; j<=object[i]->nodes; j++)
				{
					projectNode(i,object[i]->node[j]);
//					if (object[i]->node[j]->zz > max)
//						max = object[i]->node[j]->zz;
//					else if (object[i]->node[j]->zz < min)
//						min = object[i]->node[j]->zz;
				}
//printf("zmin %ld zmax %ld\n", min, max);
				for(long j=object[i]->triangles; j>0; j--)
				{

					idx3d_node *pa = object[i]->node[object[i]->triangle[j]->p1];
					idx3d_node *pb = object[i]->node[object[i]->triangle[j]->p2];
					idx3d_node *pc = object[i]->node[object[i]->triangle[j]->p3];

					float z=(pb->xx-pa->xx)*(pc->yy-pa->yy)-(pb->yy-pa->yy)*(pc->xx-pa->xx);
					//   z>0 --> triangle visible

					if (z>0)
					{
						switch (object[i]->mode)
						{	case 1:        drawWireframe(i,object[i]->triangle[j]); break;
							case 2:       drawFlatshaded(i,object[i]->triangle[j]); break;
							case 3:          drawGouraud(i,object[i]->triangle[j]); break;
							case 4:            drawPhong(i,object[i]->triangle[j]); break;
							case 5:        drawEnvmapped(i,object[i]->triangle[j]); break;
							case 6:   drawGouraudTexture(i,object[i]->triangle[j]); break;
							case 7:     drawPhongTexture(i,object[i]->triangle[j]); break;
							case 8: drawEnvmappedTexture(i,object[i]->triangle[j]); break;
						}
					}
				}
			}
		}

		void idx3d::projectNode(long obj, idx3d_node *q)
		{
			idx3d_vector mvect;
//			mvect.idx3d_vector();
			mvect.matrixvectorproduct(object[obj]->matrix2,q->v);
			
			q->n2->matrixvectorproduct(object[obj]->matrix2,q->n);
			q->n2->normalize();

			float zoom=perspective/(perspective+mvect.z)*zoomcorrection;
	
			q->xx=(long)(mvect.x*zoom*centerx+centerx);
			q->yy=(long)(mvect.y*zoom*centery+centery);
			q->zz=(long)(mvect.z*65536);

		}

		long idx3d::getIntensity(long nx, long ny)
		{
			if (staticLight) return StaticLightMap[((nx/2+127)<<8) + ny/2+127];
			long nz=LightMap[((nx/2+127)<<8) + ny/2+127];
			long intensity=ambient;
			for (long i=1;i<=lights;i++)
			{
				if (light[i]->mode==1)
				{
					intensity+=crop((light[i]->intensity*(long)abs(nx*light[i]->x+ny*light[i]->y+nz*light[i]->z)>>8)>>8,0,255);
				}
				else if (light[i]->mode==2)
				{
					// POINT LIGHT ALGORITHM TO BE IMPLEMENTED HERE
				}
			}
			return crop(intensity,0,255);
		}

		long idx3d::getIntensity(idx3d_vector *v)
		{
			return getIntensity((long)(v->x*255),(long)(v->y*255));
		}

		long idx3d::getEnvironment(long nx, long ny)
		{
			if (staticLight) return StaticEnvMap[((nx/2+127)<<8) + ny/2+127];
			long env=EnvMap[((nx/2+127)<<8) + ny/2+127];
			return crop(ambient+getIntensity(nx,ny)*env*reflectivity/65536,0,255);
		}


		long idx3d::getIntColor(long r, long g, long b)
		{
			return alpha|(r<<16)|(g<<8)|b;
		}

		long idx3d::getColor(long c, long intensity)
		{
			long r=ColorMap[(intensity<<8)+((rbit&c)>>16)];
			long g=ColorMap[(intensity<<8)+((gbit&c)>>8)];
			long b=ColorMap[(intensity<<8)+(bbit&c)];
			return getIntColor(r,g,b);
		}

		long idx3d::getGray(long c)
		{
			return (((rbit&c)>>16)+((gbit&c)>>8)+(bbit&c))/3;
		}

		void idx3d::setBackground(BBitmap *TempImage)
		{
			long *temp = (long*)TempImage->Bits();
			memcpy(Background, temp, w*h*sizeof(long));
		}

		void idx3d::setEnvironment(BBitmap *TempImage)
		{
			long *temp = (long*)TempImage->Bits();

			for (long i=0;i<256;i++)
			{
				for (long j=0;j<256;j++)
				{
					EnvMap[(i<<8)+j]=getGray(temp[i+j*256]);
				}
			}
		}


	// RENDERMODES

		void idx3d::drawLine(idx3d_node *q1,idx3d_node *q2,long color)
		{
			idx3d_node *temp;
			long dx = (long)abs(q1->xx-q2->xx);
			long dy = (long)abs(q1->yy-q2->yy);
			long dz = 0;
			long x,y,z;
			long x2,y2;

			if (dx>dy)
			{
				if (q1->xx > q2->xx)
				{
					temp=q1;
					q1=q2;
					q2=temp;
				}
				if (dx>0)
				{
					dz=(q2->zz - q1->zz)/dx;
					dy=((q2->yy - q1->yy)<<16)/dx;
				}
				z=q1->zz;
				y=q1->yy<<16;
				for(x=q1->xx;x<=q2->xx;x++)
				{
					y2=y>>16;
					if (inrange(x,0,w)&&inrange(y2,0,h))
					{
						if (z<zBuffer[x+y2*w])
						{
							TargetPixel[x+y2*w]=color;
							zBuffer[x+y2*w]=z;
						}
					}
					z+=dz;
					y+=dy;
				}
			}
			else
			{
				if (q1->yy > q2->yy)
				{
					temp=q1;
					q1=q2;
					q2=temp;
				}
				if (dy>0)
				{
					dz=(q2->zz - q1->zz)/dy;
					dx=((q2->xx - q1->xx)<<16)/dy;
				}
				z=q1->zz;
				x=q1->xx<<16;
				for(y=q1->yy;y<=q2->yy;y++)
				{
					x2=x>>16;
					if (inrange(x2,0,w)&&inrange(y,0,h))
					{
						if (z<zBuffer[x2+y*w])
						{
							TargetPixel[x2+y*w]=color;
							zBuffer[x2+y*w]=z;
						}
					}
					z+=dz;
					x+=dx;
				}
			}
		}

		void idx3d::drawWireframe(long obj, idx3d_triangle *t)
		{	
			long c=object[obj]->color;
			idx3d_node *p1 = object[obj]->node[t->p1];
			idx3d_node *p2 = object[obj]->node[t->p2];
			idx3d_node *p3 = object[obj]->node[t->p3];

			drawLine(p1,p2,c);
			drawLine(p2,p3,c);
			drawLine(p1,p3,c);
		}
		
		void idx3d::drawFlatshaded(long obj, idx3d_triangle *t)
		{	
			idx3d_vector n;
			n.matrixvectorproduct(object[obj]->matrix2,t->n);
			n.normalize();
			
			idx3d_node *temp1;

			idx3d_node *p1=object[obj]->node[t->p1];
			idx3d_node *p2=object[obj]->node[t->p2];
			idx3d_node *p3=object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;				
			}
			xL=x2;
			zL=p2->zz;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;			
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawFlatshadedLine(j,getColor(object[obj]->color,getIntensity(&n)));
		}

		void idx3d::drawGouraud(long obj, idx3d_triangle *t)
		{	
			idx3d_node *temp1;

			idx3d_node *p1=object[obj]->node[t->p1];
			idx3d_node *p2=object[obj]->node[t->p2];
			idx3d_node *p3=object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long i1=getIntensity(p1->n2)<<8;
			long i2=getIntensity(p2->n2)<<8;
			long i3=getIntensity(p3->n2)<<8;

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long di1=0;
			long di2=0;
			long di3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				di1=(i2-i1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				di2=(i3-i2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				di3=(i3-i1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long iL=i1;
			long iR=i1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					IEdgeBufferL[k]=iL;
					IEdgeBufferR[k]=iR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				iL+=di1;
				iR+=di3;
			}
			xL=x2;
			zL=p2->zz;
			iL=i2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					IEdgeBufferL[k]=iL;
					IEdgeBufferR[k]=iR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				iL+=di2;
				iR+=di3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawGouraudLine(j,object[obj]->color);
		}
		
		void idx3d::drawPhong(long obj, idx3d_triangle *t)
		{	
			idx3d_node *temp1;

			idx3d_node *p1=object[obj]->node[t->p1];
			idx3d_node *p2=object[obj]->node[t->p2];
			idx3d_node *p3=object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;
			
			long nx1=(long)(65536*p1->n2->x);
			long nx2=(long)(65536*p2->n2->x);
			long nx3=(long)(65536*p3->n2->x);
			long ny1=(long)(65536*p1->n2->y);
			long ny2=(long)(65536*p2->n2->y);
			long ny3=(long)(65536*p3->n2->y);

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long dnx1=0;
			long dnx2=0;
			long dnx3=0;
			long dny1=0;
			long dny2=0;
			long dny3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				dnx1=(nx2-nx1)/dy;
				dny1=(ny2-ny1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				dnx2=(nx3-nx2)/dy;
				dny2=(ny3-ny2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				dnx3=(nx3-nx1)/dy;
				dny3=(ny3-ny1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long nxL=nx1;
			long nxR=nx1;
			long nyL=ny1;
			long nyR=ny1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				nxL+=dnx1;
				nxR+=dnx3;
				nyL+=dny1;
				nyR+=dny3;
			}
			xL=x2;
			zL=p2->zz;
			nxL=nx2;
			nyL=ny2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				nxL+=dnx2;
				nxR+=dnx3;
				nyL+=dny2;
				nyR+=dny3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawPhongLine(j,object[obj]->color);
		}
		
		void idx3d::drawEnvmapped(long obj, idx3d_triangle *t)
		{	
			idx3d_node *temp1;

			idx3d_node *p1=object[obj]->node[t->p1];
			idx3d_node *p2=object[obj]->node[t->p2];
			idx3d_node *p3=object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;
			
			long nx1=(long)(65536*p1->n2->x);
			long nx2=(long)(65536*p2->n2->x);
			long nx3=(long)(65536*p3->n2->x);
			long ny1=(long)(65536*p1->n2->y);
			long ny2=(long)(65536*p2->n2->y);
			long ny3=(long)(65536*p3->n2->y);

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long dnx1=0;
			long dnx2=0;
			long dnx3=0;
			long dny1=0;
			long dny2=0;
			long dny3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				dnx1=(nx2-nx1)/dy;
				dny1=(ny2-ny1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				dnx2=(nx3-nx2)/dy;
				dny2=(ny3-ny2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				dnx3=(nx3-nx1)/dy;
				dny3=(ny3-ny1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long nxL=nx1;
			long nxR=nx1;
			long nyL=ny1;
			long nyR=ny1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				nxL+=dnx1;
				nxR+=dnx3;
				nyL+=dny1;
				nyR+=dny3;
			}
			xL=x2;
			zL=p2->zz;
			nxL=nx2;
			nyL=ny2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				nxL+=dnx2;
				nxR+=dnx3;
				nyL+=dny2;
				nyR+=dny3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawEnvmappedLine(j,object[obj]->color);
		}

		void idx3d::drawGouraudTexture(long obj, idx3d_triangle *t)
		{	
			idx3d_node *temp1;

			idx3d_node *p1=object[obj]->node[t->p1];
			idx3d_node *p2=object[obj]->node[t->p2];
			idx3d_node *p3=object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long i1=getIntensity(p1->n2)<<8;
			long i2=getIntensity(p2->n2)<<8;
			long i3=getIntensity(p3->n2)<<8;

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;
			float tw=(float)texture[object[obj]->texture]->w*65536;
			float th=(float)texture[object[obj]->texture]->h*65536;
			long tx1=(long)(tw*p1->tx);
			long tx2=(long)(tw*p2->tx);
			long tx3=(long)(tw*p3->tx);
			long ty1=(long)(th*p1->ty);
			long ty2=(long)(th*p2->ty);
			long ty3=(long)(th*p3->ty);

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long di1=0;
			long di2=0;
			long di3=0;
			long dtx1=0;
			long dtx2=0;
			long dtx3=0;
			long dty1=0;
			long dty2=0;
			long dty3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				di1=(i2-i1)/dy;
				dtx1=(tx2-tx1)/dy;
				dty1=(ty2-ty1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				di2=(i3-i2)/dy;
				dtx2=(tx3-tx2)/dy;
				dty2=(ty3-ty2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				di3=(i3-i1)/dy;
				dtx3=(tx3-tx1)/dy;
				dty3=(ty3-ty1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long iL=i1;
			long iR=i1;
			long txL=tx1;
			long txR=tx1;
			long tyL=ty1;
			long tyR=ty1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					IEdgeBufferL[k]=iL;
					IEdgeBufferR[k]=iR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				iL+=di1;
				iR+=di3;
				txL+=dtx1;
				txR+=dtx3;
				tyL+=dty1;
				tyR+=dty3;
			}
			xL=x2;
			zL=p2->zz;
			iL=i2;
			txL=tx2;
			tyL=ty2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					IEdgeBufferL[k]=iL;
					IEdgeBufferR[k]=iR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				iL+=di2;
				iR+=di3;
				txL+=dtx2;
				txR+=dtx3;
				tyL+=dty2;
				tyR+=dty3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawGouraudTextureLine(j,object[obj]->texture);
		}

		void idx3d::drawPhongTexture(long obj, idx3d_triangle *t)
		{
//			long temp;
			idx3d_node *temp1;

			idx3d_node *p1 = object[obj]->node[t->p1];
			idx3d_node *p2 = object[obj]->node[t->p2];
			idx3d_node *p3 = object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;

			long nx1=(long)(65536*p1->n2->x);
			long nx2=(long)(65536*p2->n2->x);
			long nx3=(long)(65536*p3->n2->x);
			long ny1=(long)(65536*p1->n2->y);
			long ny2=(long)(65536*p2->n2->y);
			long ny3=(long)(65536*p3->n2->y);

			float tw=(float)texture[object[obj]->texture]->w*65536;
			float th=(float)texture[object[obj]->texture]->h*65536;
			long tx1=(long)(tw*p1->tx);
			long tx2=(long)(tw*p2->tx);
			long tx3=(long)(tw*p3->tx);
			long ty1=(long)(th*p1->ty);
			long ty2=(long)(th*p2->ty);
			long ty3=(long)(th*p3->ty);

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long dnx1=0;
			long dnx2=0;
			long dnx3=0;
			long dny1=0;
			long dny2=0;
			long dny3=0;
			long dtx1=0;
			long dtx2=0;
			long dtx3=0;
			long dty1=0;
			long dty2=0;
			long dty3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				dnx1=(nx2-nx1)/dy;
				dny1=(ny2-ny1)/dy;
				dtx1=(tx2-tx1)/dy;
				dty1=(ty2-ty1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				dnx2=(nx3-nx2)/dy;
				dny2=(ny3-ny2)/dy;
				dtx2=(tx3-tx2)/dy;
				dty2=(ty3-ty2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				dnx3=(nx3-nx1)/dy;
				dny3=(ny3-ny1)/dy;
				dtx3=(tx3-tx1)/dy;
				dty3=(ty3-ty1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long nxL=nx1;
			long nxR=nx1;
			long nyL=ny1;
			long nyR=ny1;
			long txL=tx1;
			long txR=tx1;
			long tyL=ty1;
			long tyR=ty1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				nxL+=dnx1;
				nxR+=dnx3;
				nyL+=dny1;
				nyR+=dny3;
				txL+=dtx1;
				txR+=dtx3;
				tyL+=dty1;
				tyR+=dty3;
			}
			xL=x2;
			zL=p2->zz;
			nxL=nx2;
			nyL=ny2;
			txL=tx2;
			tyL=ty2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				nxL+=dnx2;
				nxR+=dnx3;
				nyL+=dny2;
				nyR+=dny3;
				txL+=dtx2;
				txR+=dtx3;
				tyL+=dty2;
				tyR+=dty3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawPhongTextureLine(j,object[obj]->texture);
		}
		
		void idx3d::drawEnvmappedTexture(long obj, idx3d_triangle *t)
		{	
			idx3d_node *temp1;

			idx3d_node *p1 = object[obj]->node[t->p1];
			idx3d_node *p2 = object[obj]->node[t->p2];
			idx3d_node *p3 = object[obj]->node[t->p3];

			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}
			if (p2->yy>p3->yy)
			{
				temp1=p2;
				p2=p3;
				p3=temp1;
			}
			if (p1->yy>p2->yy)
			{
				temp1=p1;
				p1=p2;
				p2=temp1;
			}

			long x1=p1->xx<<16;
			long x2=p2->xx<<16;
			long x3=p3->xx<<16;
			long y1=p1->yy<<16;
			long y2=p2->yy<<16;
			long y3=p3->yy<<16;

			long nx1=(long)(65536*p1->n2->x);
			long nx2=(long)(65536*p2->n2->x);
			long nx3=(long)(65536*p3->n2->x);
			long ny1=(long)(65536*p1->n2->y);
			long ny2=(long)(65536*p2->n2->y);
			long ny3=(long)(65536*p3->n2->y);

			float tw=(float)texture[object[obj]->texture]->w*65536;
			float th=(float)texture[object[obj]->texture]->h*65536;
			long tx1=(long)(tw*p1->tx);
			long tx2=(long)(tw*p2->tx);
			long tx3=(long)(tw*p3->tx);
			long ty1=(long)(th*p1->ty);
			long ty2=(long)(th*p2->ty);
			long ty3=(long)(th*p3->ty);

			long dx1=0;
			long dx2=0;
			long dx3=0;			
			long dz1=0;
			long dz2=0;
			long dz3=0;
			long dnx1=0;
			long dnx2=0;
			long dnx3=0;
			long dny1=0;
			long dny2=0;
			long dny3=0;
			long dtx1=0;
			long dtx2=0;
			long dtx3=0;
			long dty1=0;
			long dty2=0;
			long dty3=0;
			long dy;
			if (y2!=y1) 
			{
				dy=(y2-y1)>>16;
				dx1=(x2-x1)/dy;
				dz1=(p2->zz-p1->zz)/dy;
				dnx1=(nx2-nx1)/dy;
				dny1=(ny2-ny1)/dy;
				dtx1=(tx2-tx1)/dy;
				dty1=(ty2-ty1)/dy;
			}
			if (y3!=y2) 
			{
				dy=(y3-y2)>>16;
				dx2=(x3-x2)/dy;
				dz2=(p3->zz-p2->zz)/dy;
				dnx2=(nx3-nx2)/dy;
				dny2=(ny3-ny2)/dy;
				dtx2=(tx3-tx2)/dy;
				dty2=(ty3-ty2)/dy;
			}
			if (y3!=y1) 
			{
				dy=(y3-y1)>>16;
				dx3=(x3-x1)/dy;
				dz3=(p3->zz-p1->zz)/dy;
				dnx3=(nx3-nx1)/dy;
				dny3=(ny3-ny1)/dy;
				dtx3=(tx3-tx1)/dy;
				dty3=(ty3-ty1)/dy;
			}
			
			long xL=x1;
			long xR=x1;
			long zL=p1->zz;
			long zR=p1->zz;
			long nxL=nx1;
			long nxR=nx1;
			long nyL=ny1;
			long nyR=ny1;
			long txL=tx1;
			long txR=tx1;
			long tyL=ty1;
			long tyR=ty1;

			y1=y1>>16;
			y2=y2>>16;
			y3=y3>>16;

			for (long k=y1;k<y2;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx1;
				xR+=dx3;
				zL+=dz1;
				zR+=dz3;
				nxL+=dnx1;
				nxR+=dnx3;
				nyL+=dny1;
				nyR+=dny3;
				txL+=dtx1;
				txR+=dtx3;
				tyL+=dty1;
				tyR+=dty3;
			}
			xL=x2;
			zL=p2->zz;
			nxL=nx2;
			nyL=ny2;
			txL=tx2;
			tyL=ty2;
			for (long k=y2;k<=y3;k++)
			{
				if ((k>=0)&&(k<h))
				{
					LEdgeBuffer[k]=xL>>16;
					REdgeBuffer[k]=xR>>16;
					ZEdgeBufferL[k]=zL;
					ZEdgeBufferR[k]=zR;
					NXEdgeBufferL[k]=nxL;
					NXEdgeBufferR[k]=nxR;
					NYEdgeBufferL[k]=nyL;
					NYEdgeBufferR[k]=nyR;
					TXEdgeBufferL[k]=txL;
					TXEdgeBufferR[k]=txR;
					TYEdgeBufferL[k]=tyL;
					TYEdgeBufferR[k]=tyR;
				}
				xL+=dx2;
				xR+=dx3;
				zL+=dz2;
				zR+=dz3;		
				nxL+=dnx2;
				nxR+=dnx3;
				nyL+=dny2;
				nyR+=dny3;
				txL+=dtx2;
				txR+=dtx3;
				tyL+=dty2;
				tyR+=dty3;
			}		

			y1=crop(y1,0,h);
			y3=crop(y3,0,h);

			for (long j=y1;j<=y3;j++)
				drawEnvmappedTextureLine(j,object[obj]->texture);
		}


		

		void idx3d::drawFlatshadedLine(long y, long color)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp;
						temp=xL; xL=xR; xR=temp;
						temp=zL; zL=zR;	zR=temp;
					}					
					long dz=(zR-zL)/(xR-xL);
					long z=zL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							TargetPixel[x+offset]=color;
							zBuffer[x+offset]=z;
						}
						z+=dz;
					}
				}
			}
		}

		void idx3d::drawGouraudLine(long y, long color)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long iL=IEdgeBufferL[y];
				long iR=IEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp;
						temp=xL; xL=xR; xR=temp;
						temp=zL; zL=zR;	zR=temp;
						temp=iL; iL=iR; iR=temp;
					}	
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long di=(iR-iL)/dx;
					long z=zL;
					long i=iL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							TargetPixel[x+offset]=getColor(color,i>>8);
							zBuffer[x+offset]=z;
						}
						z+=dz;
						i+=di;
					}
				}
			}
		}
		
		void idx3d::drawPhongLine(long y, long color)
		{

			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long nxL=NXEdgeBufferL[y];
				long nxR=NXEdgeBufferR[y];
				long nyL=NYEdgeBufferL[y];
				long nyR=NYEdgeBufferR[y];

				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp; 
						temp=xL; xL=xR; xR=temp; 
						temp=zL; zL=zR; zR=temp; 
						temp=nxL; nxL=nxR; nxR=temp;
						temp=nyL; nyL=nyR; nyR=temp;
					}
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long dnx=(nxR-nxL)/dx;
					long dny=(nyR-nyL)/dx;
					long z=zL;
					long nx=nxL;
					long ny=nyL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							TargetPixel[x+offset]=getColor(color,getIntensity(nx/256,ny/256));
							zBuffer[x+offset]=z;
						}
						z+=dz;
						nx+=dnx;
						ny+=dny;
					}
				}
			}
		}

		void idx3d::drawEnvmappedLine(long y, long color)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long nxL=NXEdgeBufferL[y];
				long nxR=NXEdgeBufferR[y];
				long nyL=NYEdgeBufferL[y];
				long nyR=NYEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp; 
						temp=xL; xL=xR; xR=temp; 
						temp=zL; zL=zR; zR=temp; 
						temp=nxL; nxL=nxR; nxR=temp;
						temp=nyL; nyL=nyR; nyR=temp;
					}	
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long dnx=(nxR-nxL)/dx;
					long dny=(nyR-nyL)/dx;
					long z=zL;
					long nx=nxL;
					long ny=nyL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							TargetPixel[x+offset]=getColor(color,getEnvironment(nx/256,ny/256));
							zBuffer[x+offset]=z;
						}
						z+=dz;
						nx+=dnx;
						ny+=dny;
					}
				}
			}
		}

		void idx3d::drawGouraudTextureLine(long y, long t)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long iL=IEdgeBufferL[y];
				long iR=IEdgeBufferR[y];
				long txL=TXEdgeBufferL[y];
				long txR=TXEdgeBufferR[y];
				long tyL=TYEdgeBufferL[y];
				long tyR=TYEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp;
						temp=xL; xL=xR; xR=temp;
						temp=zL; zL=zR;	zR=temp;
						temp=iL; iL=iR; iR=temp;
						temp=txL; txL=txR; txR=temp;
						temp=tyL; tyL=tyR; tyR=temp;
					}	
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long di=(iR-iL)/dx;
					long dtx=(txR-txL)/dx;
					long dty=(tyR-tyL)/dx;
					long z=zL;
					long i=iL;
					long tx=txL;
					long ty=tyL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							tx=(long)abs(tx);
							ty=(long)abs(ty);
							TargetPixel[x+offset]=getColor(texture[t]->pixel[((tx>>16)%texture[t]->w) + ((ty>>16)%texture[t]->h)*texture[t]->w],i>>8);
							zBuffer[x+offset]=z;
						}
						z+=dz;
						i+=di;
						tx+=dtx;
						ty+=dty;
					}
				}
			}
		}

		void idx3d::drawPhongTextureLine(long y, long t)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long nxL=NXEdgeBufferL[y];
				long nxR=NXEdgeBufferR[y];
				long nyL=NYEdgeBufferL[y];
				long nyR=NYEdgeBufferR[y];
				long txL=TXEdgeBufferL[y];
				long txR=TXEdgeBufferR[y];
				long tyL=TYEdgeBufferL[y];
				long tyR=TYEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp;
						temp=xL; xL=xR; xR=temp;
						temp=zL; zL=zR;	zR=temp;
						temp=nxL; nxL=nxR; nxR=temp;
						temp=nyL; nyL=nyR; nyR=temp;
						temp=txL; txL=txR; txR=temp;
						temp=tyL; tyL=tyR; tyR=temp;
					}	
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long dtx=(txR-txL)/dx;
					long dty=(tyR-tyL)/dx;
					long dnx=(nxR-nxL)/dx;
					long dny=(nyR-nyL)/dx;
					long z=zL;
					long nx=nxL;
					long ny=nyL;
					long tx=txL;
					long ty=tyL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							tx=(long)abs(tx);
							ty=(long)abs(ty);
							TargetPixel[x+offset]=getColor(texture[t]->pixel[((tx>>16)%texture[t]->w) + (((ty>>16)%texture[t]->h) * texture[t]->w)],getIntensity(nx/256,ny/256));
							zBuffer[x+offset]=z;
						}
						z+=dz;
						nx+=dnx;
						ny+=dny;
						tx+=dtx;
						ty+=dty;
					}
				}
			}
		}

		void idx3d::drawEnvmappedTextureLine(long y, long t)
		{
			if ((y>=0)&&(y<h))
			{	
				long xL=LEdgeBuffer[y];
				long xR=REdgeBuffer[y];
				long zL=ZEdgeBufferL[y];
				long zR=ZEdgeBufferR[y];
				long nxL=NXEdgeBufferL[y];
				long nxR=NXEdgeBufferR[y];
				long nyL=NYEdgeBufferL[y];
				long nyR=NYEdgeBufferR[y];
				long txL=TXEdgeBufferL[y];
				long txR=TXEdgeBufferR[y];
				long tyL=TYEdgeBufferL[y];
				long tyR=TYEdgeBufferR[y];
				if ((xL!=xR))
				{
					if (xL>xR)
					{
						long temp;
						temp=xL; xL=xR; xR=temp;
						temp=zL; zL=zR;	zR=temp;
						temp=nxL; nxL=nxR; nxR=temp;
						temp=nyL; nyL=nyR; nyR=temp;
						temp=txL; txL=txR; txR=temp;
						temp=tyL; tyL=tyR; tyR=temp;
					}	
					long dx=(xR-xL);
					long dz=(zR-zL)/dx;
					long dtx=(txR-txL)/dx;
					long dty=(tyR-tyL)/dx;
					long dnx=(nxR-nxL)/dx;
					long dny=(nyR-nyL)/dx;
					long z=zL;
					long nx=nxL;
					long ny=nyL;
					long tx=txL;
					long ty=tyL;

					xL=crop(xL,0,w);
					xR=crop(xR,0,w);

					long offset=y*w;
							
					for (long x=xL;x<xR;x++)
					{
						if (inrange(z,zNear,zBuffer[x+offset]))
						{
							tx=(long)abs(tx);
							ty=(long)abs(ty);
							TargetPixel[x+offset]=getColor(texture[t]->pixel[((tx>>16)%texture[t]->w) + (((ty>>16)%texture[t]->h) * texture[t]->w)],getEnvironment(nx/256,ny/256));
							zBuffer[x+offset]=z;
						}
						z+=dz;
						nx+=dnx;
						ny+=dny;
						tx+=dtx;
						ty+=dty;
					}
				}
			}
		}


	// SPECIAL FUNCTIONS TO CREATE OBJECTS
		
		void idx3d::generateRotationObject(long obj, long steps)
		{
			double alpha=2*pi/((double)steps);

			long start=1;
			long end=object[obj]->nodes+1;
			long t_nodes=object[obj]->nodes;

			//Create new nodes
			
			for (long j=1;j<steps;j++)
			{
				for (long i=start;i<end;i++)
				{
					float qx=(float)( object[obj]->node[i]->v->x * cos(j*alpha) + object[obj]->node[i]->v->z * sin(j*alpha));
					float qz=(float)( object[obj]->node[i]->v->z * cos(j*alpha) - object[obj]->node[i]->v->x * sin(j*alpha));

					idx3d_node *w=new idx3d_node(qx, object[obj]->node[i]->v->y, qz);

					w->ty=(float)(i-1)/(float)(t_nodes-1);
					w->tx=(float)(j-1)/(float)(steps-1);
					addNode(obj,w);
				}
			}

			// Create triangles
			for (long col=0;col<steps;col++)
			{
				for (long row=1;row<t_nodes;row++)
				{
					addTriangle(obj,(t_nodes*col+row),(t_nodes*((col+1) % steps)+row),(t_nodes*col+row+1));
					addTriangle(obj,(t_nodes*col+row+1),(t_nodes*((col+1) % steps)+row),(t_nodes*((col+1) % steps)+row+1));
				}
			}
		}


void idx3d::generateNodeNormals(long obj)
{
	float xr,yr,zr;
	unsigned short nbf;
	long IndTVectN;

	idx3d_vector **TVectN = new idx3d_vector*[object[obj]->triangles];
	for (long s=0; s<object[obj]->triangles; s++)
	{
		TVectN[s] = new idx3d_vector();
	}

	for(long s=1; s<=object[obj]->nodes; s++)
	{
		nbf=0;
		xr = yr = zr = 0.;
		IndTVectN=0;

		/* looking for triangles linked with node <s> */

		for(long f=1; f<=object[obj]->triangles; f++)
		{
			bool a = (s==object[obj]->triangle[f]->p1);
			bool b = (s==object[obj]->triangle[f]->p2);
			bool c = (s==object[obj]->triangle[f]->p3);

			if(a || b || c)
			{
			/* triangle linked with node <s> */

			/* normal vector not already used?? */

				float xn,yn,zn;
				xn = object[obj]->triangle[f]->n->x;
				yn = object[obj]->triangle[f]->n->y;
				zn = object[obj]->triangle[f]->n->z;

				bool existe = false;
				for(long i=0; i<IndTVectN & !existe; i++)
				{
					if((TVectN[i]->x == xn) &&
					   (TVectN[i]->y == yn) &&
					   (TVectN[i]->z == zn))
						existe = true;
		 		}

				if(!existe)
				{
					if((xn!=0.) || (yn!=0.) || (zn!=0.))
					{
						/* add x,y and z */
	
						xr += xn;
						yr += yn;
						zr += zn;
						nbf++;
	
						/* mark this vector as used */
						TVectN[IndTVectN]->x = xn;
						TVectN[IndTVectN]->y = yn;
						TVectN[IndTVectN]->z = zn;
						IndTVectN++;
					}
				}
			}
		}

		if(nbf)
		{
			object[obj]->node[s]->n->setValues(xr/nbf, yr/nbf, zr/nbf);
		}
		else
		{
			/* no triangle associated with the node */
			object[obj]->node[s]->n->setValues(0., 0., 1.);
		}
		object[obj]->node[s]->n->normalize();
	}

	for (long s=0; s<object[obj]->triangles; s++)
		delete TVectN[s];
	delete TVectN;
}


