#include <StorageKit.h>
#include <be_apps/NetPositive/NetPositive.h>
#include <stdio.h>


#include "StudioModel.h"
#include "ViewerSettings.h"
#include "MenuView.h"
#include "ControlView.h"
#include "GlView.h"
#include "PAKView.h"
#include "YColorControlWindow.h"
#include "HLMViewer.h"


const float xdim = 500.0;
const float ydim = 400.0;


HLMViewerWindow::HLMViewerWindow(BRect frame) : 
   BDirectWindow( frame, "Half-Life Model Viewer v1.23", B_TITLED_WINDOW, 0 )
{
   menubar = new MenuView();
   AddChild(menubar);

   BRect control_frame = Bounds();
   control_frame.top = control_frame.bottom - 120;

   control = new ControlView(control_frame);
   AddChild(control);

   SetSizeLimits(312, 10000, 
		 menubar->Bounds().Height() + control_frame.Height() + 2, 10000);
   
   BRect pak_frame = frame;
   pak_frame.InsetBy(100, 50);   

   pak_win = new BWindow(pak_frame, "PAK Viewer", B_TITLED_WINDOW, B_NOT_CLOSABLE);

   pak_win->Lock();
   pak = new PAKView(pak_win->Bounds());
   pak_win->AddChild(pak->ScrollView());
   pak->SetTarget(this);
   pak_win->Show();
   pak_win->Hide();
   pak_win->Unlock();

   BRect glview_frame = Bounds();
   glview_frame.top += menubar->Bounds().Height() + 1;
   glview_frame.bottom -= 122;

   glview = new GlView(glview_frame);
   AddChild(glview);

   file_panel = new BFilePanel(B_OPEN_PANEL, NULL, NULL, 0, 
			       false, NULL, NULL, true, true);

   InitViewerSettings();

   glview->loadTexture(g_viewerSettings.backgroundTexFile, 0);
   glview->loadTexture(g_viewerSettings.groundTexFile, 1);
   
   g_studioModel.FreeModel();

   SetPulseRate(1);
}


bool HLMViewerWindow::QuitRequested(void)
{
   be_app->PostMessage(B_QUIT_REQUESTED);

   return true;
}


void HLMViewerWindow::MessageReceived(BMessage *message)
{
   entry_ref ref;
   BSlider *slider;
   BMenu *menu;
   BMenuItem *item;
   BMessage newMsg;
   BMessenger bellhop(this);
   bool selection = true;

   switch (message->what) {

   case IDC_FILE_LOADMODEL:
      newMsg.what = B_REFS_RECEIVED;
      file_panel->SetMessage(&newMsg);
      file_panel->SetTarget(be_app_messenger);
      file_panel->Show();
      break;

   case IDC_FILE_LOADBACKGROUNDTEX:
      newMsg.what = IDC_HAVE_BACKGROUND_ENTRY;
      file_panel->SetMessage(&newMsg);
      file_panel->SetTarget(bellhop);
      file_panel->Show();
      break;

   case IDC_HAVE_BACKGROUND_ENTRY: 
      if (message->FindRef("refs", &ref) == B_OK) {
	 BEntry entry(&ref);
	 
	 BPath path;
	 entry.GetPath(&path);
	 
	 glview->loadTexture(path.Path(), 0);
	 control->setShowBackground(true);
      }
      break;

   case IDC_FILE_LOADGROUNDTEX:
      newMsg.what = IDC_HAVE_GROUNDTEX_ENTRY;
      file_panel->SetMessage(&newMsg);
      file_panel->SetTarget(bellhop);
      file_panel->Show();
      break;

   case IDC_HAVE_GROUNDTEX_ENTRY:
      if (message->FindRef("refs", &ref) == B_OK) {	 
	 BEntry entry(&ref);
	 
	 BPath path;
	 entry.GetPath(&path);
	 
	 glview->loadTexture(path.Path(), 1);
	 control->setShowGround(true);
      }
      break;

   case IDC_FILE_UNLOADGROUNDTEX:
      glview->loadTexture(0, 1);
      control->setShowGround(false);
      break;

   case IDC_OPTIONS_COLOR_BG_G:
      selection = true;
      if (cWin1 == NULL) {
	 float *col = g_viewerSettings.bgColor;
	 int r = (int) (col[0] * 255.0f);
	 int g = (int) (col[1] * 255.0f);
	 int b = (int) (col[2] * 255.0f);
			
	 rgb_color colorBg = {r, g, b, 255};
		     
	 col = g_viewerSettings.gColor;
	 r = (int) (col[0] * 255.0f);
	 g = (int) (col[1] * 255.0f);
	 b = (int) (col[2] * 255.0f);
		
	 rgb_color colorG = {r, g, b, 255};

	 float left = Frame().left + ((Frame().Width() - 260) / 2);
	 float top = Frame().top + ((Frame().Height() - 70) / 2);

	 BRect screen_frame = BScreen(B_MAIN_SCREEN_ID).Frame();

	 if (!(screen_frame.Contains(BRect(left, top, left+260, top+70)))) {
	    left = (screen_frame.right - 260) / 2;
	    top = (screen_frame.bottom - 70) / 2;
	 } 

	 BPoint corner(left, top);
	 
	 cWin1 = new YColorControlWindow(corner, "Background and Ground Colors", true);
	 cWin1->SetTarget(this);
	 cWin1->SetFgColor(colorBg);
	 cWin1->SetBgColor(colorG);
	 cWin1->SetModificationMessage(new BMessage(IDC_BACKGROUND_COLOR_CHG), 
				       new BMessage(IDC_GROUND_COLOR_CHG));
	 cWin1->SetCloseMessage(new BMessage(IDC_BG_G_COLOR_QUIT));
      }		
      cWin1->SetSelection(selection);
      break;

   case IDC_OPTIONS_COLORLIGHT:
      if (cWin2 == NULL) {
	 float *col = g_viewerSettings.lColor;
	 int r = (int) (col[0] * 255.0f);
	 int g = (int) (col[1] * 255.0f);
	 int b = (int) (col[2] * 255.0f);
		
	 rgb_color colorL = {r, g, b, 255};
	    
	 float left = Frame().left + ((Frame().Width() - 260) / 2);
	 float top = Frame().top + ((Frame().Height() - 70) / 2);

	 BRect screen_frame = BScreen(B_MAIN_SCREEN_ID).Frame();

	 if (!(screen_frame.Contains(BRect(left, top, left+260, top+70)))) {
	    left = (screen_frame.right - 260) / 2;
	    top = (screen_frame.bottom - 70) / 2;
	 } 

	 BPoint corner(left, top);
	 
	 cWin2 = new YColorControlWindow(corner, "Light Color", false);
	 cWin2->SetTarget(this);
	 cWin2->SetFgColor(colorL);
	 cWin2->SetModificationMessage(new BMessage(IDC_LIGHT_COLOR_CHG));
	 cWin2->SetCloseMessage(new BMessage(IDC_L_COLOR_QUIT));
      }
      break;

   case IDC_BG_G_COLOR_QUIT:
      cWin1 = NULL;
      break;

   case IDC_L_COLOR_QUIT:
      cWin2 = NULL;
      break;
      
   case IDC_BACKGROUND_COLOR_CHG:
   case IDC_GROUND_COLOR_CHG:
   case IDC_LIGHT_COLOR_CHG:
      int16 r, g, b;
      float *col;
		
      switch(message->what) {
      case IDC_BACKGROUND_COLOR_CHG:
	 col = g_viewerSettings.bgColor;
	 break;
      case IDC_GROUND_COLOR_CHG:
	 col = g_viewerSettings.gColor;
	 break;
      case IDC_LIGHT_COLOR_CHG:		
	 col = g_viewerSettings.lColor;
	 break;
      }

      message->FindInt16("red", &r);
      message->FindInt16("green", &g);
      message->FindInt16("blue", &b);

      col[0] = (float) r / 255.0f;
      col[1] = (float) g / 255.0f;
      col[2] = (float) b / 255.0f;
      break;

   case IDC_OPTIONS_CENTERVIEW:
      centerView();
      break;

   case IDC_HELP_GOTOHOMEPAGE: 
   {
      BMessage msg(B_NETPOSITIVE_OPEN_URL);
      msg.AddString("be:url", "http://www.swissquake.ch/chumbalum-soft/index.html");

      BRoster roster;

      team_id netpos = roster.TeamFor(B_NETPOSITIVE_APP_SIGNATURE);

      if (netpos) {
	 roster.Launch(B_NETPOSITIVE_APP_SIGNATURE, &msg, &netpos);
      } else {
	 roster.Launch(B_NETPOSITIVE_APP_SIGNATURE, &msg);
      }
      break;
   }

   case IDC_HELP_ABOUT:
      BAlert *alert;
      alert = new BAlert("", "Half-Life Model Viewer v1.23 (c) 1999 by Mete Ciragan\n\n"
			 "BeOS port (with some missing features) by Wesley Dungan",
			 "OK", NULL, NULL, B_WIDTH_AS_USUAL, B_INFO_ALERT);
      alert->Go();
      break;

   //
   // pak viewer messages
   //

   case IDC_FILE_OPENPAKFILE:
      newMsg.what = IDC_HAVE_PAKFILE_ENTRY;
      file_panel->SetMessage(&newMsg);
      file_panel->SetTarget(bellhop);
      file_panel->Show();
      break;

   case IDC_HAVE_PAKFILE_ENTRY:
      if (message->FindRef("refs", &ref) == B_OK) {
	 BEntry entry(&ref);
	 
	 BPath path;
	 entry.GetPath(&path);

	 pak_win->Lock();

	 if (pak->openPAKFile(path.Path())) {
	    if (pak_win->IsHidden()) pak_win->Show();
	 } else {
	    if (!pak_win->IsHidden()) pak_win->Hide();
	 }

	 pak_win->Unlock();
      }
      break;

   case IDC_FILE_CLOSEPAKFILE:
      pak_win->Lock();
      if (!pak_win->IsHidden()) pak_win->Hide();
      while(pak->FullListCountItems() > 0)
	 delete pak->RemoveItem((int32)0);
      pak_win->Unlock();
      break;

   case IDC_PAKVIEWER:
      if (pak->OnPAKViewer()) {
	 loadModel(pak->OnLoadModel());
	 centerView ();
      }
      break;
   }
}


void HLMViewerWindow::loadModel(const char *filename)
{
   glview->LockGL();
   g_studioModel.FreeModel ();
   if (g_studioModel.LoadModel ((char *) filename)) {
      if (g_studioModel.PostLoadModel ((char *) filename)) {
	 control->initSequences ();
	 control->initBodyparts ();
	 control->initBoneControllers ();
	 centerView ();
	 strcpy(g_viewerSettings.modelFile, filename);
	 g_viewerSettings.speedScale = 1.0f;
	 int i;
	 for (i = 0; i < 32; i++)
	    g_viewerSettings.submodels[i] = 0;
	 for (i = 0; i < 8; i++)
	    g_viewerSettings.controllers[i] = 0;
      }
   } else {
      BAlert *alert;
      alert = new BAlert("", "Error opening model file." ,
			 "OK", NULL, NULL, B_WIDTH_AS_USUAL, B_WARNING_ALERT);
      alert->Go();

      control->initSequences ();
      control->initBodyparts ();
      control->initBoneControllers ();
   }	    
   glview->UnlockGL();
}


void HLMViewerWindow::centerView(void)
{
   studiohdr_t *hdr = g_studioModel.getStudioHeader ();
   if (hdr) {
      float min[3], max[3];
      g_studioModel.ExtractBbox (min, max);
      
      float dx = max[0] - min[0];
      float dy = max[1] - min[1];
      float dz = max[2] - min[2];
      float d = dx;
      
      if (dy > d)
	 d = dy;
      if (dz > d)
	 d = dz;
      
      g_viewerSettings.trans[0] = 0;
      g_viewerSettings.trans[1] = min[2] + dz / 2;
      g_viewerSettings.trans[2] = d * 1.0f;
      g_viewerSettings.rot[0] = -90.0f;
      g_viewerSettings.rot[1] = -90.0f;
      g_viewerSettings.rot[2] = 0.0f;
   }
}


void HLMViewerApp::ReadyToRun(void)
{
   BRect screen_frame = BScreen(B_MAIN_SCREEN_ID).Frame();

   float left = (screen_frame.right - xdim) / 2;
   float top = (screen_frame.bottom - ydim) / 2;

   BRect window_frame(left, top, left + xdim, top + ydim);
   
   window = new HLMViewerWindow(window_frame);

   window->Show();
}


void HLMViewerApp::RefsReceived(BMessage *message)
{
   entry_ref ref;

   if (message->FindRef("refs", &ref) == B_OK) {
      BEntry entry(&ref);

      BPath path;
      entry.GetPath(&path);
      
      window->loadModel(path.Path());
   }
}


int main( void )
{
   HLMViewerApp *app = new HLMViewerApp();
   app->Run();
   delete app;

   return B_OK;
}
