//*********************************************************************
//	
//	                        WhatColor application
#define APP_SHORT_INFO "To see what color is where is the cursor."
//
// TODO
// - what you need
//
// DONE
// 06/04/97 - V0.0 
//          - creation, with idea from Locate and Magnify apps
// 07/04/97 - use snooze instead pulse, that's not a real time 
//          application
// 20/06/97 - V1.0 
//          - clean and parametrize
// 13/07/97 - V1.1 
//          - added submenus for size and scale
// 20/07/97 - V2.0 
//          - for DR9 (aka PR), just the minimum
// 21/07/97 - V2.1 
//          - old main menu is now in window
//          - BUG marking of items in submenus dont work
//          - BUG view dont receive msg from submenus
// 22/07/97 - NB: dont delete message of a menu item
// 23/07/97 - C files shall be rooted
//          - app MenusWillShow is replaced by BWindow::MenusBeginning/Ending
// 06/09/97 - V2.2
//          - use pulse.
//          - set app icons and infos.
// 14/09/97 - added B_COLOR_8_BIT index.
//          - avoid blinking when draw.
//          - added menu item for grid.
//          - uses arrow keys for small displacement of cursor when
//          mouse isnt moved.
//          - save current settings on demand
//          - load saved settings if any at launch
//          - in screen minus half bitmap size
//
//*********************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#if 0
#include <algobase.h> // min et max in C++
#else
#define min min_c     // in SupportDefs.h, because I want
#define max max_c     // C, not C++ for mixed mode arithmetic
#endif

#include <Application.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <Alert.h>
#include <Bitmap.h>
#include <Screen.h>
#include <Region.h>
#include <AppFileInfo.h>
#include <FindDirectory.h>
#include <Path.h>

//---------------------------- MACROS ---------------------------------

#define CAST( type_, value_) ((type_)(value_))

#define CHAR_AT( i_, n_) ((char *)(i_))[n_]
#define M_4C( p_)                     \
    CHAR_AT( p_, 0), CHAR_AT( p_, 1), \
    CHAR_AT( p_, 2), CHAR_AT( p_, 3)
    
#define VIEW_COLOR 240, 240, 240

#define SETTINGS_WRITE( file_, var_) {         \
	fprintf (file_, "%s = %d\n", #var_, var_); \
}
#define IS_SETTINGS_ITEM( var_, item_, value_) ( \
	(strcmp (#var_, item_) != 0)? false:         \
		(var_ = value_, true)                    \
)

// NB: Bitmap bounds shall be even to have a pixel at center, not a
// boundary between pixels. Scaling shall be a power of 2, for correct
// scaling with DrawBitmap().

#define SCALE( x_)       ((x_) << pixel_scale)
#define BITMAP_SRC_SIZE (bitmap_halfsize * 2)
#define BITMAP_DST_SIZE (SCALE (1 + BITMAP_SRC_SIZE) - 1)

//---------------------------- CONSTANTS ------------------------------

#define APP_NAME       "WhatColor"
#define APP_VERSION    "2.2" // shall be number.number
#define APP_SIGNATURE  "application/x-vnd.Be-E-2289-" APP_NAME
#define APP_SETTINGS   APP_NAME "_settings"
#define APP_LONG_INFO  APP_SHORT_INFO
#define TIME_RATE      (250 * 1000)
#define MSG_SIZE       'size'
#define MSG_SCALE      'scal'
#define MSG_GRID       'grid'
#define MSG_SAVE       'save'
#define LINES_NUMBER   7

#define SETTINGS_SEP    " =\n\r\t" // for strtok()

// constant defining size of items in view and window
#define FONT_NAME          be_fixed_font // was "Kate" in DR8
#define FONT_SIZE       12.0
#define TEXT_FORMAT        " % 5s % 4d " // for text line in view
#define BORDER_MARGIN      2 // between an item and others in pixels

//---------------------------- TYPES ----------------------------------

typedef struct { // near colors in a B_RGB_32_BIT bitmap
	uint8 blue;
	uint8 green;
	uint8 red;
	uint8 alpha;
	uint8 index;
} what_color;

typedef struct { // item in a submenu
	char  *name;
	int32  value;
} submenu_item;

//---------------------------- CLASS WHATCOLOR BITMAP -----------------

class WhatColorBitmap: public BBitmap {

public:
	WhatColorBitmap (BRect bounds);
	
	what_color get_color (BPoint where);

private:
	BPoint center;
};

//---------------------------- CLASS WHATCOLOR VIEW -------------------

class WhatColorView: public BView {

public:
	WhatColorView (BRect frame);
	~WhatColorView (void);
	
	virtual void AttachedToWindow (void);
    virtual void Draw             (BRect update_rect);
	virtual	void Pulse            (void);
    void         set_offset       (BPoint delta);

private:
	void   draw_text (BPoint *location, char *name, int value);
	BPoint in_screen (BPoint where);
	
	BRegion         *region; // except bitmap
	WhatColorBitmap *bitmap; // under the cursor
	what_color       color;  // under the cursor
	BPoint           where;  // is the cursor
	BPoint           old_offset, offset; // to the cursor if not moving
	BPoint           text_location;
	BRect            bitmap_dst;
	char             text[256];
	int              line_height;
};

//---------------------------- CLASS WHATCOLOR WINDOW -----------------

class WhatColorWindow : public BWindow {

public:
	WhatColorWindow (BRect frame); 
	
	virtual	bool QuitRequested   (void);
	virtual void MenusBeginning  (void);
	virtual	void MessageReceived (BMessage *message);

private:
	WhatColorView *view;
	BMenuBar      *menu_bar;
	BMenu         *menu_size;
	BMenu         *menu_scale;
    BMenuItem     *item_grid;
	
	void   create_view    (void);
	void   create_menu    (void);
	BMenu *create_submenu (const submenu_item *items, int32 what, char *name);
	void   update_submenu (const submenu_item *items, int32 value, BMenu *menu);
	void   save_settings  (void);
};

//---------------------------- CLASS WHATCOLOR APPLICATION ------------

class WhatColor : public BApplication {

public:
	WhatColor (void);
	
	virtual void AboutRequested  (void);

private:
	WhatColorWindow *window;

	void set_app_info  (void);
	void load_settings (void);
};

//---------------------------- PROTOTYPES -----------------------------

//---------------------------- VARIABLES ------------------------------

// default settings
static int bitmap_halfsize = 8; // of area under cursor in pixels
static int pixel_scale     = 3; // log2 of, cf SCALE macro
static int draw_grid       = false;

static FILE *spyfile  = NULL;

static const submenu_item menu_size_items[] = { // name is "value * 2 + 1"
	{ " 3",  1}, // very small
	{ " 5",  2},
	{ " 7",  3},
	{ " 9",  4},
	{ "11",  5},
	{ "13",  6},
	{ "17",  8}, // around small icon
	{ "23", 11},
	{ "33", 16}, // around large icon
	{ "41", 20},
	{ NULL }
};

static const submenu_item menu_scale_items[] = { // name is "1 << value"
	{ " 2",  1}, // very small
	{ " 4",  2},
	{ " 8",  3},
	{ "16",  4},
//	{ "32",  5}, // too big
	{ NULL }
};

// 0xff is transparent, 0x3f is white, 0x00 is black
static const unsigned char icon16[256] = { 
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x2a,0x9c,0x9a,0x9a,0xb9,0xb9,0xb9,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0x7d,0x9c,0x9c,0x7a,0x7a,0x79,0x79,0xc0,0xe0,0xff,0xff,0xff,0xff,
	0xff,0xff,0x7d,0x7c,0x5b,0x7b,0x7a,0x79,0x79,0xa0,0xc0,0xc6,0xe6,0xff,0xff,0xff,
	0xff,0xff,0xbd,0x5c,0x5b,0x5a,0x5a,0xda,0x59,0xa0,0xa6,0xa6,0xec,0xff,0xff,0xff,
	0xff,0xff,0xe5,0xe5,0xd8,0xd9,0x3f,0x3f,0xdb,0x80,0xa6,0xa6,0xd2,0xff,0xff,0xff,
	0xff,0xff,0xfa,0xfc,0xfc,0xd9,0x3f,0x00,0x00,0x86,0x86,0xd2,0xd2,0xff,0xff,0xff,
	0xff,0xff,0x45,0x44,0x43,0x42,0x41,0x00,0x3f,0x00,0x8c,0x8c,0xbe,0xff,0xff,0xff,
	0xff,0xff,0x44,0x4a,0x49,0x49,0x48,0x47,0x00,0x3f,0x00,0x00,0x00,0xff,0xff,0xff,
	0xff,0xff,0x51,0x4a,0x50,0x4f,0x4e,0x00,0x00,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,
	0xff,0xff,0xff,0x57,0x57,0x55,0x55,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,
	0xff,0xff,0xff,0xff,0x34,0x56,0x55,0x54,0x00,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x3f,0x3f,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
};
static const unsigned char icon32[1024] = {
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xbc,0xbc,0xbb,0xba,0xba,0xba,0xb9,0xb9,0xb9,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x2a,0xbc,0xbc,0x9b,0xbb,0x9a,0xba,0x9a,0xb9,0xb9,0xb9,0xb8,0xb9,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x7d,0x9d,0x9c,0x9c,0x9b,0x9a,0x9a,0x7a,0x99,0x99,0x99,0x99,0x99,0xb8,0xe0,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x7d,0x9c,0x9c,0x9c,0x9b,0x9b,0x7a,0x7a,0x99,0x7a,0x99,0x99,0x79,0x99,0xc0,0xc0,0xe0,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x7d,0x9c,0x7c,0x7c,0x7b,0x7b,0x7b,0x7a,0x7a,0x7a,0x79,0x79,0x78,0x78,0x99,0xc0,0xc0,0xc6,0xe6,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0x7d,0x7d,0x5c,0x7c,0x5b,0x7c,0x5b,0x7b,0x7a,0x7a,0x79,0x79,0x79,0x78,0x78,0xa0,0xc0,0xc6,0xc6,0xe6,0xe6,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0x5d,0x5d,0x5c,0x5c,0x5b,0x5b,0x5a,0x5a,0x5a,0x59,0x59,0x59,0x58,0x78,0x78,0xa0,0xa0,0xc6,0xc6,0xc6,0xec,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xbd,0x5d,0xbd,0xbd,0x5b,0x5b,0x5a,0x5b,0x5a,0x5a,0x59,0x59,0x58,0x59,0x59,0x80,0xa0,0xa6,0xa6,0xc6,0xc6,0xec,0xec,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xbd,0xf8,0xbd,0x5b,0xf8,0x5b,0xd8,0x5a,0x5a,0x5a,0x59,0x59,0x59,0xdb,0x58,0x80,0x86,0xa6,0xa6,0xa6,0xcc,0xcc,0xf2,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xf9,0xbd,0xe5,0xe5,0xe5,0xd8,0xd9,0xd9,0xda,0xda,0xda,0x1d,0x1d,0x60,0x80,0x86,0x86,0xa6,0xa6,0xac,0xcc,0xd2,0xf2,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xf9,0xfb,0xe5,0xe5,0xe5,0xfc,0xd9,0xd9,0xfe,0x1d,0x3f,0x3f,0x1d,0xdb,0x60,0x86,0x86,0x86,0xac,0xac,0xd2,0xd2,0xf2,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xfa,0xfa,0xfb,0xfb,0xe5,0xfc,0xfd,0xfc,0xfe,0xfe,0x3f,0x00,0x00,0x00,0x66,0x86,0x86,0x8c,0x8c,0xd2,0xd2,0xf2,0x20,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xfa,0xfb,0xfb,0x43,0x43,0xfc,0x42,0xfe,0x42,0x1d,0x3f,0x00,0x3f,0x3f,0x00,0x66,0x86,0x86,0x8c,0xac,0xb2,0xd2,0xbe,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x45,0x45,0x44,0x44,0x43,0x43,0x42,0x42,0x42,0x41,0x41,0x00,0x3f,0x3f,0x00,0x66,0x66,0x8c,0x8c,0xb2,0xb2,0xd2,0xbe,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x45,0x44,0x44,0x44,0x49,0x43,0x42,0x42,0x48,0x48,0x47,0x47,0x00,0x3f,0x3f,0x00,0x00,0x00,0x00,0x00,0xb2,0xb2,0xbe,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x4b,0x4a,0x4a,0x4a,0x4a,0x49,0x49,0x48,0x48,0x48,0x47,0x41,0x00,0x3f,0x3f,0x00,0x3f,0x3f,0x00,0x3f,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0x4a,0x4a,0x4a,0x4a,0x4f,0x49,0x49,0x4e,0x48,0x48,0x47,0x47,0x00,0x3f,0x3f,0x00,0x3f,0x3f,0x00,0x3f,0x00,0x3f,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0x4b,0x51,0x4a,0x50,0x4f,0x4f,0x49,0x4e,0x4e,0x4e,0x4d,0x4d,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x51,0x50,0x51,0x50,0x50,0x55,0x4f,0x4e,0x4e,0x00,0x00,0x00,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x51,0x57,0x56,0x56,0x56,0x55,0x55,0x00,0x3f,0x3f,0x00,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x57,0x57,0x56,0x55,0x55,0x55,0x00,0x3f,0x3f,0x3f,0x3f,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x34,0x34,0x56,0x56,0x55,0x55,0x00,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x34,0x56,0x55,0x55,0x54,0x54,0x00,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x00,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x3f,0x3f,0x3f,0x3f,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
};

/*-------------------------------------------------------------------*/

#define nspy if (0) spy

static void spy (const char *format, ...) 
{
	static bool dont_work = false;
	va_list     args;
	
	if (dont_work) return;
	if (spyfile == NULL) spyfile = fopen ("/tmp/Spy." APP_NAME, "w");
	if (spyfile == NULL) {
		dont_work = true;
		return;
	}
	
	va_start (args, format);
	vfprintf (spyfile, format, args);
	va_end (args);
	fflush (spyfile);
}

/*-------------------------------------------------------------------*/

static void alert (const char *format, ...)
{
	static  char  message[1024];
	BAlert       *alert;
	va_list       args;
	
	va_start (args, format);
	vsprintf (message, format, args);
	va_end (args);

	alert = new BAlert ("",	message, "Click me");
	alert->Go();
}

//---------------------------------------------------------------------

int main()
{	
	WhatColor *application = new WhatColor();

	application->Run();	
	
	delete application;
    if (spyfile != NULL) fclose (spyfile);
	return 0;
}

//---------------------------------------------------------------------

WhatColor::WhatColor (void):
    BApplication (APP_SIGNATURE)
{
	nspy ("WhatColor::WhatColor\n");

	// -- initialize object datas
	window = NULL;
	set_app_info();
	load_settings();
	
	// -- create window with some size, somewhere in screen
	window = new WhatColorWindow (BRect (100, 100, 120, 120));

	// --- show everythings
	window->Show();
}

//---------------------------------------------------------------------

void WhatColor::AboutRequested (void)
{
	BAlert *alert;

	nspy ("WhatColor::AboutRequested\n");

	alert = new BAlert ("About " APP_NAME,	
		APP_NAME " version " APP_VERSION "\n\n"
		APP_SHORT_INFO "\n",
		"Click me"
	);
	alert->Go();
}

//---------------------------------------------------------------------

void WhatColor::set_app_info (void)
{
	BAppFileInfo  app;
	version_info  app_version;
	status_t      status;
	BRect         rect;
	BBitmap      *icon = NULL;
	app_info      info; 
	BFile         file; 
	ulong         major, middle;
	char          *end;

	nspy ("WhatColor::set_app_info\n");

	// get application file info about me: be_app.
	status = be_app->GetAppInfo (&info); 
	if (status != B_OK) {
		alert ("cant be_app->GetAppInfo(), status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}
	status = file.SetTo (&info.ref, B_READ_WRITE); 
	if (status != B_OK) {
		alert ("cant file.SetTo(), status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}
	status = app.SetTo (&file);
	if (status != B_OK) {
		alert ("cant app.SetTo(), status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}

	// application signature is not always installed by launching
	BMimeType mime_type;
	status = mime_type.SetTo (APP_SIGNATURE);
	if (status != B_OK) {
		alert ("cant set mime to %s, status 0x%08x (%s)", 
			APP_SIGNATURE, status, strerror (status));
		return;
	}
	nspy ("set_app_info, %s installed, signature %s\n", 
		mime_type.IsInstalled()? "is": "isnt", mime_type.Type());
	if (!mime_type.IsInstalled()) mime_type.Install();

#if 0
	// dump app infos
	char   mime_str[B_MIME_TYPE_LENGTH]; 
	uint32 flags;
	
	status = app.GetSignature (mime_str);
	if (status == B_OK) spy ("set_app_info: signature %s\n", mime_str);
	status = app.GetAppFlags (&flags);
	if (status == B_OK) spy ("set_app_info: flags 0x%08x\n", flags); // is B_SINGLE_LAUNCH;
	status = app.GetType (mime_str);
	if (status == B_OK) spy ("set_app_info: type %s\n", mime_str); // is "application/x-be-executable"
	status = app.GetVersionInfo (&app_version, B_APP_VERSION_KIND);
	if (status == B_OK) spy ("set_app_info: app_version.major      %d\n", app_version.major);
	if (status == B_OK) spy ("set_app_info: app_version.middle     %d\n", app_version.middle);
	if (status == B_OK) spy ("set_app_info: app_version.minor      %d\n", app_version.minor);
	if (status == B_OK) spy ("set_app_info: app_version.variety    %d\n", app_version.variety);
	if (status == B_OK) spy ("set_app_info: app_version.internal   %d\n", app_version.internal);
	if (status == B_OK) spy ("set_app_info: app_version.short_info \n    %s\n", app_version.short_info);
	if (status == B_OK) spy ("set_app_info: app_version.long_info  \n    %s\n", app_version.long_info);
	status = app.GetVersionInfo (&app_version, B_SYSTEM_VERSION_KIND);
	if (status == B_OK) spy ("set_app_info: sys_version.major      %d\n", app_version.major);
	if (status == B_OK) spy ("set_app_info: sys_version.middle     %d\n", app_version.middle);
	if (status == B_OK) spy ("set_app_info: sys_version.minor      %d\n", app_version.minor);
	if (status == B_OK) spy ("set_app_info: sys_version.variety    %d\n", app_version.variety);
	if (status == B_OK) spy ("set_app_info: sys_version.internal   %d\n", app_version.internal);
	if (status == B_OK) spy ("set_app_info: sys_version.short_info \n    %s\n", app_version.short_info);
	if (status == B_OK) spy ("set_app_info: sys_version.long_info  \n    %s\n", app_version.long_info);
#endif

	// ? is update needed
	status = app.GetVersionInfo (&app_version, B_APP_VERSION_KIND);
	if ((status != B_OK) && (status != B_ENTRY_NOT_FOUND)) {
		alert ("cant GetVersionInfo, status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}
	major = strtoul (APP_VERSION, &end, 10);
	if (*end == '.') middle = strtoul (end + 1, &end, 10);
	else middle = 0;
	if (*end != '\0') {
		alert ("bad APP_VERSION: \"%s\"", APP_VERSION);
		return;
	}
	if ((major == app_version.major) && (middle == app_version.middle))
		return;
		
	// set signature
	status = app.SetSignature (APP_SIGNATURE);
	if (status != B_OK) {
		alert ("cant SetSignature, status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}

	// set version
	app_version.major    = major;
	app_version.middle   = middle;
	app_version.minor    = 0;
	app_version.variety  = 0; // 0 = dev, 5 = final
	app_version.internal = 0;
	strncpy (app_version.short_info, APP_SHORT_INFO, sizeof(app_version.short_info));
	strncpy (app_version.long_info,  APP_LONG_INFO,  sizeof(app_version.long_info));

	status = app.SetVersionInfo (&app_version, B_APP_VERSION_KIND);
	if (status != B_OK) {
		alert ("cant SetVersionInfo, status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}
	status = app.SetVersionInfo (&app_version, B_SYSTEM_VERSION_KIND);
	if (status != B_OK) {
		alert ("cant SetVersionInfo, status 0x%08x (%s)", 
			status, strerror (status));
		return;
	}

	// set large icon
	rect.Set (0, 0, B_LARGE_ICON - 1, B_LARGE_ICON - 1);
	icon = new BBitmap (rect, B_COLOR_8_BIT);
	if (sizeof (icon32) != icon->BitsLength()) {
		alert ("bad large icon size (%d instead %d)", 
			sizeof (icon32), icon->BitsLength());
		goto do_return;
	}
	memcpy (icon->Bits(), icon32, sizeof (icon32));
	status = app.SetIcon (icon, B_LARGE_ICON);
	if (status != B_OK) {
		alert ("cant set large icon, status 0x%08x (%s)", 
			status, strerror (status));
		goto do_return;
	}
	
	// set mini icon
	rect.Set (0, 0, B_MINI_ICON - 1, B_MINI_ICON - 1);
	icon = new BBitmap (rect, B_COLOR_8_BIT);
	if (sizeof (icon16) != icon->BitsLength()) {
		alert ("bad large icon size (%d instead %d)", 
			sizeof (icon16), icon->BitsLength());
		goto do_return;
	}
	memcpy (icon->Bits(), icon16, sizeof (icon16));
	status = app.SetIcon (icon, B_MINI_ICON);
	if (status != B_OK) {
		alert ("cant set mini icon, status 0x%08x (%s)", 
			status, strerror (status));
		goto do_return;
	}
do_return:
	if (icon != NULL) delete icon;
}

//---------------------------------------------------------------------

void WhatColor::load_settings (void)
{
	static char  line[1024];
	FILE        *file;
	char        *item, *token, *end;
	long         value; // for strtol
	int          lineno;
	status_t     status;
	BPath        path;

	// if settings file exists, read it
    status = find_directory (B_USER_SETTINGS_DIRECTORY, &path);
    if (status != B_OK) {
		alert ("cant find B_USER_SETTINGS_DIRECTORY, status 0x%08x (%s)", 
			status, strerror (status));
    	return;
    }
	path.Append (APP_SETTINGS);
	file = fopen (path.Path(), "r");
	if (file == NULL) return;

	lineno = 0;
	while (NULL != fgets (line, sizeof (line), file)) {
		lineno++;
		item = strtok (line, SETTINGS_SEP);
		if (item == NULL) goto error;
		token = strtok (NULL, SETTINGS_SEP);
		if (token == NULL) goto error;
		value = strtol (token, &end, 0);
		if (*end != '\0') goto error;
		if      (IS_SETTINGS_ITEM (bitmap_halfsize, item, value)) continue;
		else if (IS_SETTINGS_ITEM (pixel_scale,     item, value)) continue;
		else if (IS_SETTINGS_ITEM (draw_grid,       item, value)) continue;
error:
		alert ("settings file %s, error on line %d", APP_SETTINGS, lineno);
	}
	fclose (file);
}

//---------------------------------------------------------------------

WhatColorWindow::WhatColorWindow (BRect frame):
	BWindow (frame, APP_NAME, B_TITLED_WINDOW, 
		B_WILL_ACCEPT_FIRST_CLICK | B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	// -- initialize object datas
	view       = NULL;
	menu_bar   = NULL;
	menu_size  = NULL;
	menu_scale = NULL;
    item_grid  = NULL;

	// -- create window's menu_bar, at window origin, with any size 
	menu_bar = new BMenuBar (Bounds(), APP_NAME " menu bar");
	create_menu();
	AddChild (menu_bar);
	
	// -- now create view, keeping care of menu bar
	create_view();
}

//---------------------------------------------------------------------

void WhatColorWindow::create_view (void)
{
	WhatColorView *old = view;

	// -- disable refresh and lock
	SetPulseRate (0);
    if (!Lock()) {
		alert ("%s %d cant lock\n", __FILE__, __LINE__);
		return;
	}

	// -- create view, at window origin, with any size, but unattached
	view = new WhatColorView (BRect (0, 0, 20, 20));

	// -- put view at a nice offset from top, keeping care of menu
	// height, and now attach it, so view can compute geometry
	view->MoveTo (0, menu_bar->Bounds().bottom + 1);
	AddChild (view);

	// -- remove old view
	if (old != NULL) {
		old->Flush(); // required ?? 
		old->Sync(); // required ?? 
		RemoveChild (old);
		delete old;
	}

	// -- unlock and refresh rate
	Unlock();
	SetPulseRate (TIME_RATE);
}

//---------------------------------------------------------------------

BMenu *WhatColorWindow::create_submenu (
	const submenu_item *items, int32 what, char *name)
{
	BMenuItem *item;
	BMenu     *menu;
	BMessage  *msg;
	int        i;
	
	menu = new BMenu (name, B_ITEMS_IN_COLUMN);
	for (i = 0; items[i].name != NULL; i++) {
		msg  = new BMessage (what);
		msg->AddInt32 ("value", items[i].value);
		item = new BMenuItem (items[i].name, msg);
		item->SetTarget (this);
		item->SetEnabled (true);
		item->SetMarked (false);
		menu->AddItem (item);
	}
	return menu;
}

//---------------------------------------------------------------------

void WhatColorWindow::create_menu (void)
{
    BMenuItem *item;
    BMessage  *message;
	BMenu     *menu;

	nspy ("Window::create_menu\n");

	menu = new BMenu ("Try me", B_ITEMS_IN_COLUMN);
	
	message = new BMessage (B_ABOUT_REQUESTED);
	item    = new BMenuItem ("About " APP_NAME " " B_UTF8_ELLIPSIS, message);
	item->SetTarget (be_app);
	item->SetEnabled (true);
	item->SetMarked (false);
	menu->AddItem (item);
	
	message = new BMessage (MSG_SAVE);
	item = new BMenuItem ("Save settings", message, 'S');
	item->SetTarget (this);
	item->SetEnabled (true);
	item->SetMarked (false);
	menu->AddItem (item);
	item_grid = item;

	menu->AddSeparatorItem();

	message = new BMessage (MSG_GRID);
	item = new BMenuItem ("Grid", message, 'G');
	item->SetTarget (this);
	item->SetEnabled (true);
	item->SetMarked (false);
	menu->AddItem (item);
	item_grid = item;

#if 1
	menu_size = create_submenu (menu_size_items, MSG_SIZE, "Size in pixels");
	item = new BMenuItem (menu_size, NULL);
	item->SetMarked (false);
	menu->AddItem (item);

	menu_scale = create_submenu (menu_scale_items, MSG_SCALE, "Pixel scale");
	item = new BMenuItem (menu_scale, NULL);
	item->SetMarked (false);
	menu->AddItem (item);

	menu->AddSeparatorItem();
#endif
   
	message = new BMessage (B_QUIT_REQUESTED);
	item = new BMenuItem ("Quit", message, 'Q');
	item->SetTarget (be_app);
	item->SetEnabled (true);
	item->SetMarked (false);
	menu->AddItem (item);

	menu_bar->AddItem (menu);

}

//---------------------------------------------------------------------

void WhatColorWindow::update_submenu (
	const submenu_item *items, int32 value, BMenu *menu)
{
	BMenuItem *item;		
	int        i;
	
	// -- unmark/mark items in submenu
	for (i = 0; items[i].name != NULL; i++) {
		item = menu->ItemAt (i);
		if (item == NULL) continue;
		item->SetMarked (items[i].value == value);
	}
}

//---------------------------------------------------------------------

void WhatColorWindow::MenusBeginning (void)
{
	if (item_grid != NULL) item_grid->SetMarked (draw_grid);
	if (menu_size != NULL) update_submenu (
		menu_size_items, bitmap_halfsize, menu_size);
	if (menu_scale != NULL) update_submenu (
		menu_scale_items, pixel_scale, menu_scale);
}

//---------------------------------------------------------------------

void WhatColorWindow::MessageReceived (BMessage *message)
{
	status_t status;
	int32    value;
	BPoint   delta;
	int32    key;
	
	switch (message->what) {
		default:
			inherited::MessageReceived (message);
			break;
		case B_KEY_UP: 
			break;
		case B_KEY_DOWN: 
			delta.Set (0, 0);
			status = message->FindInt32 ("raw_char", &key);
			if (status == B_OK) switch (key) {
				default:                          break;
				case B_LEFT_ARROW:  delta.x -= 1; break;
				case B_RIGHT_ARROW: delta.x += 1; break;
				case B_UP_ARROW:    delta.y -= 1; break;
				case B_DOWN_ARROW:  delta.y += 1; break;
			}
			view->set_offset (delta);
			break;
		case MSG_SIZE:
			status = message->FindInt32 ("value", &value);
			if (status != B_OK) {
				alert ("Window::MessageReceived '%c%c%c%c' without value (0x%08x)", 
	    			M_4C (&message->what), status);
				break;
			}
			if (value == bitmap_halfsize) break;
			bitmap_halfsize = value;
			create_view();
			break;
		case MSG_SCALE:
			status = message->FindInt32 ("value", &value);
			if (status != B_OK) {
				alert ("Window::MessageReceived '%c%c%c%c' without value (0x%08x)", 
	    			M_4C (&message->what), status);
				break;
			}
			if (value == pixel_scale) break;
			pixel_scale = value;
			create_view();
			break;
		case MSG_GRID:
			draw_grid = !draw_grid; // next refresh will do it 
			create_view();          // but ...
			break;
		case MSG_SAVE:
			save_settings();
			break;
	}
}

//---------------------------------------------------------------------

void WhatColorWindow::save_settings (void)
{
	FILE     *file;
	BPath     path;
	status_t  status;
	
    status = find_directory (B_USER_SETTINGS_DIRECTORY, &path);
    if (status != B_OK) {
		alert ("cant find B_USER_SETTINGS_DIRECTORY, status 0x%08x (%s)", 
			status, strerror (status));
    	return;
    }
	path.Append (APP_SETTINGS);
	file = fopen (path.Path(), "w");
	if (file == NULL) {
		alert ("cant open %s status 0x%08x (%s)", 
			path.Path(), status, strerror (status));
		return;
	}
	SETTINGS_WRITE (file, bitmap_halfsize);
	SETTINGS_WRITE (file, pixel_scale);
	SETTINGS_WRITE (file, draw_grid);
	fclose (file);
// TODO set settings file icon/ref
}

//---------------------------------------------------------------------

bool WhatColorWindow::QuitRequested (void)
{
	be_app->PostMessage (B_QUIT_REQUESTED);
	return true;
}

//---------------------------------------------------------------------

WhatColorView::WhatColorView (BRect frame):
	BView (frame, APP_NAME, 
		B_FOLLOW_ALL, B_WILL_DRAW | B_PULSE_NEEDED)
{
	BRect rect;
	
	bitmap = NULL;
	region = NULL;
	where.Set (-1, -1); // out of screen
	offset.Set (0, 0);  // none
	old_offset.Set (0, 0);
		
	// create bitmap
	rect.Set (0, 0, BITMAP_SRC_SIZE, BITMAP_SRC_SIZE);
	bitmap = new WhatColorBitmap (rect);
}

//---------------------------------------------------------------------

WhatColorView::~WhatColorView (void)
{
	if (bitmap != NULL) delete bitmap;
	if (region != NULL) delete region;
}

//---------------------------------------------------------------------

void WhatColorView::AttachedToWindow (void)
{
	BFont       font;
	font_height font_height;
	int         bitmap_size;
	int         space;
	int         text_width, text_height;	
	int         width, height;
	BPoint      point;
	BRect       dst;
	
	nspy ("View::AttachedToWindow\n");

	SetViewColor (VIEW_COLOR); // sure, isnt what I hope
	SetHighColor (0, 0, 0);    // black
	SetLowColor  (VIEW_COLOR);

	font = FONT_NAME;
	font.SetSize (FONT_SIZE);
	SetFont (&font);
	
	// -- compute items sizes, and so, window/view sizes
	bitmap_size = 2 * BORDER_MARGIN + BITMAP_DST_SIZE;
	sprintf (text, TEXT_FORMAT, "x", 0);
	text_width = 2 * BORDER_MARGIN + StringWidth (text);
	GetFontHeight (&font_height);
	nspy ("font leading %f, ascent %f, descent %f\n", 
		font_height.leading, font_height.ascent, font_height.descent);
	line_height = max (1.0, font_height.leading) + font_height.ascent + font_height.descent;
	space       = (BITMAP_DST_SIZE - LINES_NUMBER * line_height) / (LINES_NUMBER - 1);
	if (space < 0) space = 0;
	line_height += min (space, font_height.ascent);
	text_height  = 2 * BORDER_MARGIN + (LINES_NUMBER - 1) * line_height;
	text_height += font_height.ascent + font_height.descent;

	// -- set view & window real size (window first else strange results)
	width  = 1 + bitmap_size + text_width;
	height = 1 + max (bitmap_size, text_height);
	nspy ("bitmap size %d, text wh %d %d, width %d, height %d\n", 
		bitmap_size, text_width, text_height, width, height);
	point.Set (0, 0); // cant found easier way
	ConvertToScreen (&point);
	Window()->ConvertFromScreen (&point);
	Window()->ResizeTo (width, height + point.y);
	ResizeTo (width, height); // view shall be resized last 

	// -- where to draw text
	text_location.Set (
		bitmap_size + BORDER_MARGIN, BORDER_MARGIN + font_height.ascent);

	// -- bitmap destination rect
	bitmap_dst.Set (0, 0, BITMAP_DST_SIZE , BITMAP_DST_SIZE);
	bitmap_dst.OffsetTo (BORDER_MARGIN, BORDER_MARGIN);

	// region is view excepted bitmap
	region = new BRegion();
	region->Include (BRect (0, 0, width, height));
	region->Exclude (bitmap_dst);
}

//---------------------------------------------------------------------

BPoint WhatColorView::in_screen (BPoint where)
{
	BRect rect   = BScreen().Frame();
	int   margin = bitmap_halfsize;
	
	rect.InsetBy (margin, margin);
	if (where.x < rect.left)   where.x = rect.left;
	if (where.x > rect.right)  where.x = rect.right;
	if (where.y < rect.top)    where.y = rect.top;
	if (where.y > rect.bottom) where.y = rect.bottom;

	return where;
}

//---------------------------------------------------------------------

void WhatColorView::draw_text (
	BPoint *location, char *name, int value)
{
	sprintf (text, TEXT_FORMAT, name, value);
	DrawString (text, *location);
	location->y += line_height;
}

//---------------------------------------------------------------------

void WhatColorView::Draw (BRect update_rect)
{
#pragma unused (update_rect)
	BPoint       location, point;
	BRect        dst;
	int          nb_lines; // to verify LINES_NUMBER value
	int          i, step, end;
	BPoint       pa, pb;
	drawing_mode mode;
	BRect        rect;
			
	old_offset = offset; // avoid next draw if same
	if (region == NULL) FillRect (Bounds(), B_SOLID_LOW);
	else FillRegion (region, B_SOLID_LOW);
	
	// display color and cursor position
	location = text_location;
	nb_lines = __LINE__;
	draw_text (&location, "red",   color.red);
	draw_text (&location, "green", color.green);
	draw_text (&location, "blue",  color.blue);
	draw_text (&location, "alpha", color.alpha);
	draw_text (&location, "index", color.index);
	draw_text (&location, "x",     where.x + offset.x);
	draw_text (&location, "y",     where.y + offset.y);
	if (LINES_NUMBER != (__LINE__ - nb_lines - 1)) {
		alert ("line %d, bad LINES_NUMBER value %d\n", 
			__LINE__, LINES_NUMBER);
		be_app->PostMessage (B_QUIT_REQUESTED);
		return;
	}

	// draw scaled bitmap
	if (bitmap == NULL) {
		alert ("Draw NULL bitmap\n");
		be_app->PostMessage (B_QUIT_REQUESTED);
		return;
	}
	DrawBitmap (bitmap, bitmap_dst);

	// draw bitmap grid
	step = draw_grid? SCALE (1): BITMAP_DST_SIZE;
	end  = BORDER_MARGIN + BITMAP_DST_SIZE + (draw_grid? 1: 0);
	for (i = 0; i <= (BITMAP_DST_SIZE + 1); i += step) {
		// vertical line
		pa.Set (BORDER_MARGIN + i, BORDER_MARGIN);
		pb.Set (BORDER_MARGIN + i, end);
		StrokeLine (pa, pb);
		// horizontal line
		pa.Set (BORDER_MARGIN, BORDER_MARGIN + i);
		pb.Set (end, BORDER_MARGIN + i);
		StrokeLine (pa, pb);
	}

	// draw square around target pixel
	mode = DrawingMode();
	step = draw_grid? SCALE (1): SCALE (1) - 1;
	rect.Set (0, 0, step, step);
	if (draw_grid) rect.InsetBy (-2, -2);
	else           rect.InsetBy (-1, -1);
	rect.OffsetBy (
		BORDER_MARGIN + SCALE (bitmap_halfsize), 
		BORDER_MARGIN + SCALE (bitmap_halfsize));
	SetDrawingMode (B_OP_INVERT);
	StrokeRect (rect);
	SetDrawingMode (mode);

	Sync(); // required ?? 
}

//---------------------------------------------------------------------

void WhatColorView::set_offset (BPoint delta)
{
	// stricto sensu, update of offset/old_offset shall be 
	// exclusive and requires a semaphore, but this is without 
	// damage except noise on screen
	old_offset  = offset;
	offset     += delta;
	offset = in_screen (offset + where) - where;
}

//---------------------------------------------------------------------

void WhatColorView::Pulse (void)
{
	what_color old_color  = color;
	BPoint     old_where  = where;
	BPoint     here;
	uint32     buttons;
	bool       same_color, same_where;
	BRect      rect;
	
	// get cursor position
	GetMouse (&here, &buttons);
	where = ConvertToScreen (here);
	if (where != old_where) {
		offset.Set (0, 0);
		old_offset.Set (0, 0);
	}
	
	// avoid drawing if cursor is inside bitmap
	rect.Set (0, 0, 
		BITMAP_DST_SIZE + 1 + 2 * BORDER_MARGIN, 
		BITMAP_DST_SIZE + 1 + 2 * BORDER_MARGIN);
	if (rect.Contains (here + offset)) return;

	// get color under cursor in screen
	where = in_screen (offset + where) - offset;
	color = bitmap->get_color (where + offset);

	// avoid drawing if no change
	same_color = 
		(color.red   == old_color.red  ) &&
		(color.green == old_color.green) &&
		(color.blue  == old_color.blue ) &&
		(color.alpha == old_color.alpha) &&
		(color.index == old_color.index);
	same_where = (where == old_where) && (offset == old_offset);
	if (same_color && same_where) return;
	
    if (Window()->Lock()) {
    	Draw (Bounds());
    	Window()->Unlock();
	} else alert ("%s %d cant lock\n", __FILE__, __LINE__);
}

//---------------------------------------------------------------------

WhatColorBitmap::WhatColorBitmap (BRect bounds):
	BBitmap (bounds, B_RGB_32_BIT) 
	// NB: bitmap shall be B_RGB_32_BIT because magic formula 
	// after for any screen depth, so this work ever.
{
	nspy ("Bitmap::Bitmap\n");

	center.Set (
		(bounds.left + bounds.right ) / 2,
		(bounds.top  + bounds.bottom) / 2);
}

//---------------------------------------------------------------------

what_color WhatColorBitmap::get_color (BPoint where)
{
	BScreen     screen;
	uint8      *row;
	what_color  color;
	BRect       src;
	rgb_color   pixel;
	
	// get part of screen bitmap
	src = Bounds();
	src.OffsetBy (where - center);

	// undocumented Be feature from interface/bitmap.h
	_get_screen_bitmap_ (this, src, false); 

	// get center pixel color with magic formula 
	// NB: work only if bitmap is B_RGB_32_BIT
	row  = CAST (uint8 *, Bits())+ BytesPerRow() * bitmap_halfsize 
		+ bitmap_halfsize * sizeof (rgb_color);

	// colors in bitmap are in order BGRA, not as in rgb_color.
	color.blue  = pixel.blue  = *row++;
	color.green = pixel.green = *row++;
	color.red   = pixel.red   = *row++;
	color.alpha = pixel.alpha = *row++;

	// get quickly index in B_COLOR_8_BIT
	screen = BScreen (B_MAIN_SCREEN_ID);
	color.index = screen.IndexForColor (pixel);
	
	return color;
}

//**************************** END ************************************
