// CramDisk version 6.00 by Earl Colby Pottinger.

#include <OS.h>
#include <Drivers.h>
#include <KernelExport.h>
#include <Mime.h>
#include <stdlib.h>
#include <Errors.h>
#include <string.h>

status_t vd_open(const char *name, uint32 flags, void **cookie);
status_t vd_free(void *cookie);
status_t vd_close(void *cookie);
status_t vd_control(void *cookie, uint32 msg, void *buf, size_t size);
status_t vd_read(void *cookie, off_t pos, void *buf, size_t *count);
status_t vd_write(void *cookie, off_t pos, const void *buf, size_t *count);

enum { RAM_SECTORS =  512, // Number of blocks per track.
       RAM_TRACKS =    32, // Number of cylinders per platter.
       RAM_HEADS =      8, // Number of surfaces.
       RAM_BLOCKS =   512, // Bytes per sector. Set to 512 to support DOS formatting.
       RAM_SIZE =  ((RAM_SECTORS)*(RAM_TRACKS)*(RAM_HEADS)*(RAM_BLOCKS)), // RAMDrive size in bytes.

       CHUNK =          65536,        // Size of memory chunks to allocate.
       CHUNKY =   (CHUNK + 1),        // Chunk+1.  Used for testing read/writes to allocated chunks.
       LASTBYTE = (CHUNK - 1),        // Used in testing.
       NEWCHUNK = (CHUNK + (CHUNK/8)),// Allocation to take account of compression overhead. 
       INDEXS = ((RAM_SIZE)/(CHUNK))  // Number of allocation chunks needed to support size of drive.
       } ;

// null-terminated array of device names supported by this driver.
const char *vd_name[] = { "disk/virtual/cam", NULL } ;
const char *vd_path[] = { "boot/home/config/add-ons/kernel/drivers/bin/", NULL };

// Buffers and pointers needed for reading and writing to drive.
static uchar* B_Index[INDEXS+1]; // Index Array of allocated RAMDISK storage Buffers.
static long Lenghts[256];             /* Convertion Array for Tokens to Lenght. */
#include "icongraphics.h"        // Icon graphics in array.

static device_hooks vd_devices = {
 vd_open,    // -> open entry point
 vd_close,   // -> close entry point
 vd_free,    // -> free cookie
 vd_control, // -> control entry point
 vd_read,    // -> read entry point
 vd_write,   // -> write entry point
 NULL,       // my_device_select,   -> select entry point
 NULL,       // my_device_deselect, -> deselect entry point
 NULL,       // my_device_readv,    -> posix read entry point
 NULL } ;    // my_device_writev    -> posix write entry point

int32 api_version = B_CUR_DRIVER_API_VERSION;
  
status_t init_hardware(void) {
 long i;
 for (i=0; i<INDEXS; i++) { B_Index[i]=NULL; } // Clear the allocated Buffer indexs.
 for (i=0; i<127; i++) { Lenghts[i] = i + 2; } // Set up RLL Tokens. Range 2-128.
 Lenghts[127]=  256; Lenghts[128]=  512; Lenghts[129]= 1024; // Special compressed lenghts.
 Lenghts[130]= 2048; Lenghts[131]= 4096; Lenghts[132]= 8192;
 Lenghts[133]=16384; Lenghts[134]=32768; Lenghts[135]=65536; 
 for (i=136; i<256; i++) { Lenghts[i] = i - 135; } // Set up Linear Tokens. Range 1-120.
 return B_OK; }
status_t init_driver(void) { return B_OK; }

void uninit_driver(void) { return; }

const char** publish_devices() { return vd_name; }

device_hooks* find_device(const char* name) { return &vd_devices; }

status_t vd_open(const char *dname, uint32 flags, void **cookie) { return B_NO_ERROR; }

status_t vd_free (void *cookie) { return B_NO_ERROR; }

status_t vd_close(void *cookie) { return B_NO_ERROR; }
   
status_t vd_read(void *cookie, off_t pos, void *buf, size_t *count) {
 long Index; off_t Packed; off_t UnPacked; uchar c; long z;
 off_t i; off_t BufferOffset; size_t len; size_t Hunk;
 uchar* Buffer; uchar* Expanded; // Buffers for reading data.

 Index = pos / CHUNK; pos = pos - (Index * CHUNK); len = (*count); BufferOffset = 0; // Find proper buffer indexs.

 while ( len > 0 ) {
    Buffer = B_Index[Index]; // If there is data to read from RAMDISK Buffers. Get pointer to a pre-allocated buffer.
    // Create buffer if buffer does not exists already. Return error code if no buffer created.
    if ( Buffer == NULL ) { // dprintf("<Read a blank Buffer (%ld).\n",Index);
       Buffer = malloc(CHUNK); if ( Buffer == NULL ) { return B_ERROR; }
       for (i=0; i<CHUNK; i++) { Buffer[i] = (uchar)0; } // Zero the contents of the buffer.
       // Data all inside this Buffer. Read from buffer. Flag length, End_of_Data.
       if ( (pos+len) < CHUNKY ) { memcpy(buf+BufferOffset, Buffer+pos, len); len = 0; }
       // Data is larger than buffer. How much from buffer? Reduce lenght by amount.
       // Write data to the end of buffer. Increase write pointer. Reset position. Index to next buffer.
       if ( (pos+len) > CHUNK ) { Hunk = CHUNK-pos;  memcpy(buf+BufferOffset, Buffer+pos, Hunk);
          len = len - Hunk; BufferOffset = BufferOffset + Hunk; pos = 0; Index++; }
       free(Buffer); } // Delete buffer that is no longer needed.
    else {
       // dprintf("<Read a filled Buffer (%ld).\n",Index);
       // Expand compressed data before reading data from it.
       Expanded = malloc(CHUNKY); if ( Expanded == NULL ) { return B_ERROR; } // Create buffer to decompress data.
       UnPacked = 0; Packed =0; // Clear pointers to expand buffer.
       while ( UnPacked < CHUNK ) {
          c = Buffer[Packed]; Packed++; z = (long)c; // dprintf("<Read c=(%c) z=(%ld) >\n", c,z);
          if ( z == 0 ) {
             for (i=0; i<256; i++) { Expanded[UnPacked] = (uchar)0; UnPacked++; } } // Zero the contents of the buffer.
          else {
             for (i=0; i<256; i++) { Expanded[UnPacked] = Buffer[Packed]; UnPacked++; Packed++; } } }
       // Data all inside this Buffer. Read from buffer. Flag length, End_of_Data.
       if ( (pos+len) < CHUNKY ) { memcpy(buf+BufferOffset, Expanded+pos, len); len = 0; }
       // Data is larger than buffer. How much from buffer? Reduce lenght by amount.
       // Write data to the end of buffer. Increase write pointer. Reset position.
       if ( (pos+len) > CHUNK ) { Hunk = CHUNK-pos; memcpy(buf+BufferOffset, Expanded+pos, Hunk);
          len = len - Hunk;BufferOffset = BufferOffset + Hunk; pos = 0; Index++; }
       free (Expanded); }
 } 
 return B_NO_ERROR; } // Return error status.

status_t vd_write(void *cookie, off_t pos, const void *buf, size_t *count) { 
 long Index; long Temp; long flag; off_t Packed; off_t UnPacked; off_t Start; uchar c; long z;
 off_t i; off_t BufferOffset; size_t len; size_t Hunk;
 uchar* Buffer; uchar* Expanded; uchar* Compressed; // Buffers for writing data.

 Index = pos / CHUNK; pos = pos - (Index * CHUNK); len = (*count); BufferOffset = 0; // Find proper buffer index.

 while ( len > 0 ) {
    Buffer = B_Index[Index]; // If there is data to write to RAMDISK Buffers. Get pointer to a pre-allocated buffer.
    // Create buffer if buffer does not exists already. Return error code if no buffer created.
    if ( Buffer == NULL ) { // dprintf("<Write on a blank Buffer.\n");
       Expanded = malloc(CHUNKY); if ( Expanded == NULL ) { return B_ERROR; }
       // Zero out the contents of the read buffer before it is read.
       for (i=0; i<CHUNKY; i++) { Expanded[i] = (uchar)0; i++; } }
    else {
       // dprintf("<Write a filled Buffer (%ld).\n",Index);
       // Expand compressed data before writting data to it.
       Expanded = malloc(CHUNKY); if ( Expanded == NULL ) { return B_ERROR; } // Create buffer to decompress data.
       UnPacked = 0; Packed =0; // Clear pointers to expand buffer.
       while ( UnPacked < CHUNK ) {
          c = Buffer[Packed]; Packed++; z = (long)c; // dprintf("<Read c=(%c) z=(%ld) >\n", c,z);
          if ( z == 0 ) {
             for (i=0; i<256; i++) { Expanded[UnPacked] = (uchar)0; UnPacked++; } } // Zero the contents of the buffer.
          else {
             for (i=0; i<256; i++) { Expanded[UnPacked] = Buffer[Packed]; UnPacked++; Packed++; } } }
       free(Buffer); }

    Temp = Index; // Store present buffer index.
    // Data will fit inside this Buffer. Write to buffer. Flag length, End_of_Data.
    if ( (pos+len) < CHUNKY ) { memcpy(Expanded+pos, buf+BufferOffset, len); len = 0; }
    // Data is larger than buffer. How much fits into buffer? Reduce lenght by amount.
    // Write data to the end of buffer. Increase write pointer. Reset position. Index to next buffer.
    if ( (pos+len) > CHUNK ) { Hunk = CHUNK-pos; memcpy(Expanded+pos, buf+BufferOffset, Hunk);
       len = len - Hunk; BufferOffset = BufferOffset + Hunk; pos = 0; Index++; }

    // Compress data, replace zeros that repeat 256 times with one zero.
    Compressed = malloc(NEWCHUNK); if ( Compressed == NULL ) { return B_ERROR; } // Create buffer to compress modified data.
    UnPacked = 0; Packed =0;
    while ( UnPacked < CHUNK ) {
       Start = Packed; Packed++;  z= 0; flag = 0;
       for (i=0; i<256; i++) {
          if ( Expanded[UnPacked+i] != (uchar)0 )
             { flag = 1; i = 255; } }
       if ( flag == 0 )
          { Compressed[Start] = (uchar)0; UnPacked = UnPacked + 256; }
       else {
          Compressed[Start] = (uchar)128;
          for (i=0; i<256; i++)
             { Compressed[Packed] = Expanded[UnPacked]; Packed++; UnPacked++; } } }
    free(Expanded);

    // Create buffer if buffer does not exists already. Return error code if no buffer created.
    Buffer = malloc(Packed); if ( Buffer == NULL ) { return B_ERROR; }
    // Copy into smaller buffer.
    for (i=0; i<Packed; i++) { Buffer[i] = Compressed[i]; } ; B_Index[Temp] = Buffer; free(Compressed);
 }
 return B_NO_ERROR; } // Return error status.

status_t vd_control(void *cookie, uint32 ioctl, void *arg1, size_t len)
{ device_geometry *dinfo; device_icon *dicon; long i;
  switch (ioctl) {                     // generic mass storage device IO control codes.

  case B_GET_BIOS_GEOMETRY:            // Gets the BIOS settings. Not useful for a RAMDISK I think? Testing.
  case B_GET_GEOMETRY:                 // Get real drive geometry.
     dinfo = (device_geometry *) arg1; // Fills out the specified device_geometry structure to describe the device.
     dinfo->bytes_per_sector = RAM_BLOCKS; 
     dinfo->sectors_per_track = RAM_SECTORS; 
     dinfo->cylinder_count = RAM_TRACKS; 
     dinfo->head_count = RAM_HEADS; 
     dinfo->device_type = B_DISK; 
     dinfo->removable = FALSE; 
     dinfo->read_only = FALSE; 
     dinfo->write_once = FALSE;
     return B_OK; // Function supported. Returns device geometry.

  case B_FORMAT_DEVICE:                                // Low-Level formats the drive if boolean data is true.
     if (!*((char *) arg1)) return B_NO_ERROR;         // Flag = false, do not format. <<unclear what should be done >>
        for (i=0; i<INDEXS; i++) {                     // Flag = true, do low-level format.
           if (B_Index[i] != NULL) {                   // Test if a Buffer is allocated.
              free(B_Index[i]); B_Index[i] = NULL; } } // Free allocated Buffer. Clear index pointer.
     return B_OK;                                      // Function supported.

  case B_GET_ICON:                 // Returns a ICON_info structure for the device.
     dicon = (device_icon *) arg1; // Pointer to ICON_info structure.
     switch (dicon->icon_size) {

     case B_LARGE_ICON: // Function supported. Returns large icon image.
        memcpy(dicon->icon_data, icon_disk, B_LARGE_ICON * B_LARGE_ICON);    return B_OK;
     case B_MINI_ICON:  // Function supported. Returns small icon image.
        memcpy(dicon->icon_data, icon_disk_mini, B_MINI_ICON * B_MINI_ICON); return B_OK;
     default:           // Unknown icon size, returns small icon image.
        memcpy(dicon->icon_data, icon_disk_mini, B_MINI_ICON * B_MINI_ICON); return B_BAD_TYPE; }

  case B_GET_DEVICE_SIZE:                        // A size_t indicating the device size in bytes is returned.
     *(size_t*)arg1 = RAM_SIZE; return B_OK ;    // Function supported. Signals valid drive size.

  case B_GET_MEDIA_STATUS:                       // Gets the status of the media in the device by returning
     *(status_t*)arg1 = B_OK; return B_NO_ERROR; // a status_t at the location pointed to. Function is under test.

  case B_FLUSH_DRIVE_CACHE:                      // Flushes the drive's cache. Not useful for a RAMDISK. */
     *(status_t*)arg1 = B_OK; return B_NO_ERROR; // Returns B_OK status_t to function under test.

  case B_GET_DRIVER_FOR_DEVICE:                  // Returns the path of the driver executable handling the device.
     return B_ERROR;                             // Function is not presently supported.

  case B_GET_NEXT_OPEN_DEVICE: return B_ERROR;   // ??? Function not supported.

  case 10199:
     *(size_t*)arg1 = B_OK; return B_NO_ERROR;   // Totally unknown function, found while debugging.                         

  default: 
     return B_ERROR; }} // Returns B_OK status_t to unknown/unsupportted function.
     