#ifndef __dirmon_h
#define __dirmon_h

#include <StorageKit.h>
#include <sys/stat.h>

/**
The Node Monitor is a service that lets you ask to be notified of certain
file system changes. You can ask to be told when a change is made to... 

<UL>
<LI>The contents of a specific directory. 
<LI>The name of a specific entry. 
<LI>Any stat field of a specific entry. 
<LI>Any attribute of a specific entry.
</UL>

The hook functions in this class are by default implemented to do nothing.  The lone
exception is NodeMonitor::EntryDeleted, which is implemented to simply turn off monitoring
for the node since it is now deleted from the files system.

<p><strong><em>Hook functions must not block.</em></strong>  Learn to use inter-thread
communication and semaphores.  It's easy.

@author Stephen van Egmond wrote the C++ classes, after
@author Be, Inc. (probably Dominic Giampaolo)
*/
class NodeMonitor : private BMessenger {
public:
	/**
	Begins watching the given file system entity. The notification functions in this class will be called
	when the given changes occur to the node.  You should override only those notifications
	you're interested in getting.
	
	<p>Remember that BStatable is the ancestor to both BNode and BEntry (which is, in turn,
	the ancestor of BFile, BDirectory, and BSymLink.
    If you have an entry_ref, reassure yourself that it is not abstract (BEntry::Exists), and then you
	can turn it into a BStatable derivative.  If you are working from a BNode only, ensure
	that it's properly initialized by calling InitCheck() before building a NodeMonitor on it.
	
	@param towatch The entity to begin observing.
	*/
	NodeMonitor(const BStatable &toWatch);

	/**
	Stops watching the node that this monitor represents.
	*/
	virtual ~NodeMonitor();

	/**
	These are utility functions to make common storage kit conversions easier than
	before, particularly as they relate to node monitors.  If the conversion you
	want to make is not supported here, you should make <em>certain</em> that you
	have read and understead the storage kit's introduction, and understand
	the difference between a <b>node</b> and an <b>entry</b>.
	
	@name File system utility functions
	*/
	//@{
	
		/**
		This is a utility function that obtains an entry_ref for a name in a directory node.
		
		<p>Keep in mind that the reference may or may not be abstract.  That's up to you
		to find out.
		
		@see GetEntry
		*/
		static entry_ref GetEntryRef(node_ref &from_directory, const char *name);
		
		/**
		This is a utility function that obtains a BEntry for a name in the given directory node.  If the
		corresponding entry is for a symbolic link, it is not traversed.  There are two ways that can go wrong:
		
		<p>First, the reference may come out invalid.  This would be the case if the directory
		node given has been unlinked from the directory it is in.
		
		<p>Second the reference may come out abstract. That is, refer to a file that does not
		exist in a directory that does.  That's up to you
		to find out, perhaps by locking the directory node and testing the entry
		you just created with BEntry::Exists.
		
		<pre>
		BDirectory directory;
		BEntry entry;
		directory = GetDirectory(dir_node_ref);
		if (directory.InitCheck() == B_NO_ERROR) {
			directory.Lock();
		
			BEntry entry = GetEntry(dir_node_ref, name)));
			if (entry.Exists()) {
				...
		  }
		
			directory.Unlock();
		}
		</pre>
		
		@param directory_node MThe node_ref that is known to point to a valid directory.
		@param name The name of the directory to refer to in the directory node.
		@return The BEntry that represents the given name in the given directory.
		*/
		static BEntry GetEntry(node_ref &from_directory, const char *name);
	
	
		/**
		This is a utility function that converts a directory node_ref to a BDirectory object.
		
		<p>In various node monitor hook functions, a parameter is passed in that is a
		node_ref to a directory.  For subtle reasons in the storage kit, you are allowed
		to turn a directory node_ref into its corresponding entry.
		
		<p>Note that the directory object may not be valid if the directory that
		contains the node_ref has not been locked.  You should test
		it (with BDirectory::InitCheck) to make sure.
		
		@param directory_ref The node_ref that is known to point to a valid directory.
		@return The BDirectory object, configured for the node reference.
		*/
		static BDirectory GetDirectory(node_ref &directory_ref);	
	//@}

	/**
		Returns the result of constructing the node monitor.
	*/
	status_t InitCheck();
	
protected:
	/**
	This hook function is called when the node is deleted from the file system.
	Note that, by the time this function is called, the node has already been removed, and the
	node_ref in this class is not guaranteed to be valid any more.
	
	<p>For files, this can be overcome by maintaining BFile objects that point to nodes you
	want to keep around.  Refer to the Storage Kit's documentation on the node
	monitor system for an example.

	<p><b>Note:</b> It is a valid operation to <code>delete this</code> from this hook
	function, once you've dealt with the disappearance of this node.  This is,
	in fact, the default behaviour.
	
	@param fromDirectory The directory that the node that is being monitored is removed from.
	*/
	virtual void EntryDeleted(node_ref &fromDirectory);
	
	/**
	This hook function is called when the node is moved to a new directory and/or gets a new name.
	If the file has simply been renamed in place, from_directory will equal to_directory.
	*/
	virtual void EntryMoved(node_ref &from_directory, node_ref &to_directory, const char *newName);

	/**
	This hook function is called when certain parts of the stat structure have changed.
	The stat structure is described in "The stat Structure" in the BStatable class.
	If you are really interested in this, you will probably want to keep track of the
	parts of the stat structure you're interested in watching change.

    <P>
	Recall that few parts of stat can change:<ul>
	<li>Owner (st_uid), group (st_gid), and permissions (low four bytes of st_mode). 
	<li>Creation (st_ctime), modification (st_mtime), and access times (st_atime; currently unused). 
	<li>The size of the node's data (st_size). The measurement doesn't include attributes.
	</ul>
	@see BStatable
	*/	
	virtual void StatChanged();

	/**
	This hook function is called when an attribute has been added, rewritten, or deleted.
	@see Attribute
	@param which The name of the attribute which was added, rewritten, or deleted.
	*/	
	virtual void AttributeChanged(const char *which);
	
	/**
	The node_ref that is being watched by this object.  It should be equal to the result of
	mWatched->GetNodeRef().
	*/
	node_ref mTargetEntry;

	/**
	The BStatable that is being watched by this object. It's up to you to remember if it was
	a directory, file, or symlink; you can also use the dynamic_cast<> operator.
	*/
	const BStatable *mWatched;
	
	// Ignore that man behind the curtain!
	friend class NodeMonitorManager;
	friend class NodeMonitorMessenger;

private:
	status_t mStatus;
};

/**
Extends the node monitor class by adding directory-specific notifications.
*/
class DirectoryMonitor : public NodeMonitor {
public:
	DirectoryMonitor(const BDirectory &toWatch);

protected:	
	/**
	This hook function is called when a completely new entry is created in the monitored
	directory. 	This doesn't include entries that are moved into this directory from some
	other directory--see EntryMovedIn.)
	@see EntryMovedIn
	@param created_name The name of the entry that was just created in the monitored directory.
	@param created_node The node number for the entry that was just created in the monitored directory.
	*/
	virtual void EntryCreated(const char *created_name, ino_t created_node);

	/**
	This hook function is called when a node is removed (deleted) from the monitored directory.
	
	<p>Note that the <em>node number</em> is most likely
	no longer valid, unless a reference is being kept to keep it around.
	For instance, this would be done just by keeping a valid BFile object pointing to the entry.
	
	<p>The <em>entry</em> that the node
	is associated with, however, is long gone, and that is why there is no name
	parameter to this function.
	
	@param node The number of the entry removed from this directory.
	*/
	virtual void EntryRemoved(ino_t node);

	/**
	This hook function is called when a node was moved to the monitored directory.
	Recall that nodes can move only inside the same volume: you can reuse the
	volume identifiers from mTargetEntry.
	
	@param from_directory A node_ref to the directory that the node moved out of to
	                      come to the monitored directory.
	@param name_of_node The name of entry that moved in.
	@param node The node number of the entry that moved in.
	*/
	virtual void EntryMovedIn(node_ref &from_directory, const char *name_of_node, ino_t node);
	

	/**
	This hook function is called when a node was moved out of the monitored
	directory. Recall that nodes can move only inside the same volume: you
	can reuse the volume identifiers from mTargetEntry.
	@param from_directory A node_ref to the directory that the node moved to
	                      from the monitored directory.
	@param name_of_node The name of entry that moved out.
	@param node The node number of the entry that moved out.
	*/
	virtual void EntryMovedOut(node_ref &to_directory, const char *name_of_node, ino_t node);
	
protected:
	// Ignore that man behind the curtain!
	friend class NodeMonitorManager;
	friend class NodeMonitorMessenger;
};

/**
If true, this flag will cause the node monitor to generate debugging output.
Feel free to change this variable as often as you like.  All regular output for a node
monitor will start with the notation <code>[number]:[number]</code>, where the first number
is the volume ID (an int32) and the second is the unique node number (an int64) on that volume of that node.
*/
extern bool sNodeMonitorDebugging;

#endif
