#include <DataIO.h>


#ifndef _included_Attribute_h
#define _included_Attribute_h

/**
This class provides a friendly representation of an attribute.
 It models this by using the BPositionIO class
from the Support Kit, which lets you seek, read, and write at arbitrary points
within the attribute's data stream.

<p>You can use Attribute in two ways: first, as a stand-alone utility class that you
instantiate, use to manipulate an attribute, then throw away.  Second, as parent class
for an object in your application that corresponds directly to an attribute on a file.
You can use the Attribute methods to update your attribute when your object's state
changes, and call Flush as needed.

<p>This class is not well-suited for especially large attributes, since the entire
data stream is pulled into memory.

<p>To remove an attribute, see BNode::RemoveAttr.
*/
class Attribute
	/* extends */ :public BPositionIO
{
public:
	/** Opens (or creates) an attribute of the given name and type on a given node.
		Refer to the BNode class documentation to see what attributes can do for you.
		* @param node The node to open the attribute on.
		* @param name The name of the attribute.
		* @param type One of the B_*_TYPE constants.
	*/
	Attribute(const BNode *node, const char *name, bool createNew=false);

	/**
	Rewrites the attribute to disk, then destroys the object.  Note that writing may fail
	for a variety of reasons, including read-only files and volumes, and a lack
	of disk space. For this reason it's better to use Flush.
	@see Flush
	*/
	~Attribute();
	
	/**
	Returns true if the named attribute exists for the given node.
	*/
	static bool Exists(const BNode *node, const char *name);
	
	/// Inherited from BPositionIO.
	virtual	ssize_t		Read(void *buffer, size_t size);

	/// Inherited from BPositionIO.
	virtual	ssize_t		Write(const void *buffer, size_t size);

	/// Inherited from BPositionIO.
	virtual	ssize_t		ReadAt(off_t pos, void *buffer, size_t size);

	/// Inherited from BPositionIO.
	virtual	ssize_t		WriteAt(off_t pos, const void *buffer, size_t size);

	/// Inherited from BPositionIO.
	virtual off_t		Seek(off_t position, uint32 seek_mode);

	/// Inherited from BPositionIO.
	virtual	off_t		Position() const;

	/// The size of data contained in the attribute.
	off_t Size();

	/**
	Writes the attribute data in memory out to disk.
	@return The result of the operation, as indicated by BNode::WriteAttr.
	*/
	status_t Flush();
	

private:

	/// Loads the attribute from disk into the memory storage if it exists.
	void Load();
	
private:
	const BNode *m_node;
	const char *m_attributeName;
	const type_code m_attributeType;
	BMallocIO *m_storage;
};

#endif

