/* ArpQuantize.cpp
 */
#include "ArpQuantize.h"

#include <stdio.h>
#include <stdlib.h>
#include <be/interface/CheckBox.h>
#include <be/interface/MenuField.h>
#include <be/interface/MenuItem.h>
#include <be/interface/RadioButton.h>
#include <be/interface/StringView.h>
#include <experimental/ResourceSet.h>
#include "ArpKernel/ArpDebug.h"
//#include "ArpViewsPublic/ArpPrefsI.h"
#include "ArpViews/ArpIntControl.h"
#include "ArpViews/ArpRangeControl.h"
#include "ArpLayout/ArpViewWrapper.h"
#include "AmPublic/AmControls.h"
#include "AmPublic/AmFilterConfig.h"
#include "AmPublic/AmFilterConfigLayout.h"
#include "AmPublic/AmSongObserver.h"

static const uint32		ABSOLUTE_MSG		= 'iabs';
static const uint32		SCALE_MSG			= 'iscl';

static const AmTime		QUANT_PRIMES		= 3*5*7;

static const char*		QUANTIZE_STR		= "quantize";
static const char*		MODIFIER_STR		= "modifier";
static const char*		DO_DURATION_STR		= "do_duration";

ArpMOD();

static BResourceSet gResources;
static int32 gInitResources = 0;
BResourceSet& Resources()
{
	if (atomic_or(&gInitResources, 1) == 0) {
		gResources.AddResources((void*)Resources);
		atomic_or(&gInitResources, 2);
	} else {
		while ((gInitResources&2) == 0) snooze(20000);
	}
	return gResources;
}

/*************************************************************************
 * _QUANTIZE-CONTROL
 * A control that displays a quick-access menu and text control for editing
 * some duration filter properties.
 *************************************************************************/
/* These are the hardcoded duration constants, all based on 4/4 time.
 */
static const float	WHOLE						= PPQN * 4;
static const float	WHOLE_HALF					= PPQN * 3;
static const float	HALF						= PPQN * 2;
static const float	HALF_QUARTER				= PPQN * 1.5;
static const float	QUARTER						= PPQN;
static const float	QUARTER_EIGHTH				= PPQN * 0.75;
static const float	EIGHTH						= PPQN * 0.5;
static const float	EIGTH_SIXTEENTH				= PPQN * 0.375;
static const float	SIXTEENTH					= PPQN * 0.25;
static const float	SIXTEENTH_THIRTYSECOND		= PPQN * 0.1875;
static const float	THIRTYSECOND				= PPQN * 0.125;
static const float	THIRTYSECOND_SIXTYFOURTH	= PPQN * 0.09375;
static const float	SIXTYFOURTH					= PPQN * 0.0625;

class _QuantizeControl : public AmFilterConfig
{
public:
	_QuantizeControl(	AmFilterHolderI* target,
						const BMessage& initSettings,
						const char* initialName );

	void			SetQuantizeTime(AmTime quantizeTime);
	void			SetModifier(int32 modifier);
	
	void			RefreshDurationControl(const BMessage& settings);
	
	virtual void	AttachedToWindow();
	virtual void	MessageReceived(BMessage *msg);

private:
	typedef AmFilterConfig	inherited;
		
	void			AddViews(const char* initialName);
};

/*****************************************************************************
 * ARP-QUANTIZE-FILTER
 *****************************************************************************/
ArpQuantizeFilter::ArpQuantizeFilter(	ArpQuantizeFilterAddOn* addon,
										AmFilterHolderI* holder,
										const BMessage* config)
		: AmFilterI(addon),
		  mAddOn(addon), mHolder(holder),
		  mMultiplier(1), mQuantizeTime(PPQN), mDivider(2), mFullTime(mMultiplier*((mQuantizeTime*2*QUANT_PRIMES)/mDivider)),
		  mIncludeDuration(true)
{
	if( config ) PutConfiguration( config );
}

ArpQuantizeFilter::~ArpQuantizeFilter()
{
}

static inline AmTime quantize(AmTime inTime, AmTime fullTime)
{
	const int64 t = ((int64)inTime)*QUANT_PRIMES;
	return (t-(t%fullTime)) / QUANT_PRIMES;
}

AmEvent* ArpQuantizeFilter::HandleEvent(AmEvent* event, const am_filter_params* /*params*/)
{
	if( !event ) return event;
	ArpVALIDATE(mAddOn != NULL && mHolder != NULL, return event);

	AmTime startTime = quantize(event->Time(), mFullTime);
	event->SetTime(startTime);
//	printf("Quantized %lld to %lld\n", event->Time(), startTime);
	
	if (mIncludeDuration) {
		#if 0
		// This is how I think it should work, but it needs the
		// create tool to generate new events with a duration of 1.
		AmTime duration = quantize(event->Duration(), mFullTime);
		if (duration <= 0) duration = mMultiplier*((mQuantizeTime*2)/mDivider);
		if (--duration <= 0) duration = 1;
		event->SetDuration(duration);
		#else
		// This is the old way.
		/* I subtract one because otherwise absolute times run into each other.
		 */
		const int64		quantized = mMultiplier*((mQuantizeTime*2)/mDivider);
		AmTime		newTime = quantized - 1;
		if( newTime < 1 ) newTime = 1;
		event->SetDuration( newTime );
		#endif
	}

	return event;
}

BView* ArpQuantizeFilter::NewEditView(BPoint requestedSize) const
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return NULL;
	return new _QuantizeControl( mHolder, config, Label() );
}

class ArpQuantizeFilterSettings : public AmFilterConfigLayout
{
public:
	ArpQuantizeFilterSettings(	AmFilterHolderI* target,
								const BMessage& initSettings)
		: AmFilterConfigLayout(target, initSettings),
		  mDurCtrl(NULL)
	{
		try {
			AddLayoutChild((new ArpRunningBar("TopVBar"))
				->SetParams(ArpMessage()
					.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
					.SetFloat(ArpRunningBar::IntraSpaceP, .5)
				)
				->AddLayoutChild((new ArpTextControl(
										SZ_FILTER_LABEL, "Label:","",
										mImpl.AttachTextControl(SZ_FILTER_LABEL)))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "8")
						.SetString(ArpTextControl::PrefTextStringP, "8888888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
				->AddLayoutChild((mDurCtrl = new AmDurationControl("quantize_to", "Quantize to:", this, initSettings,
														AM_SHOW_DURATION_MULTIPLIER | AM_SHOW_DURATION_TEXT,
														QUANTIZE_STR, MODIFIER_STR))
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
				->AddLayoutChild((new ArpViewWrapper(
						new BCheckBox(BRect(0,0,10,10),
										DO_DURATION_STR,
										"Quantize Duration",
										mImpl.AttachCheckBox(DO_DURATION_STR),
										B_FOLLOW_NONE,
										B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE|B_NAVIGABLE) ))
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,1)
						.SetInt32(ArpRunningBar::FillC,ArpWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
			);
		} catch(...) {
			throw;
		}
		
		Implementation().RefreshControls(mSettings);
	}
	virtual	void MessageReceived(BMessage *msg)
	{
		switch (msg->what) {
			case ARP_PUT_CONFIGURATION_MSG:
				{
					BMessage	settings;
					if (msg->FindMessage("settings", &settings) == B_OK)
						if (mDurCtrl) mDurCtrl->Refresh(settings);
				}
				// Note: no break on purpose
			default:
				inherited::MessageReceived( msg );
		}
	}

protected:
	typedef AmFilterConfigLayout inherited;
	AmDurationControl*		mDurCtrl;
};

status_t ArpQuantizeFilter::GetConfiguration(BMessage* values) const
{
	status_t err = AmFilterI::GetConfiguration(values);
	if (err != B_OK) return err;
	
	if( (err = values->AddInt32(AM_MULTIPLIER_CONTROL_KEY_STR, mMultiplier)) != B_OK ) return err;
	if( (err = add_time(*values, QUANTIZE_STR, mQuantizeTime)) != B_OK ) return err;
	if( (err = values->AddInt32(MODIFIER_STR, mDivider)) != B_OK ) return err;
	if( (err = values->AddBool(DO_DURATION_STR, mIncludeDuration)) != B_OK ) return err;
	return B_OK;
}

status_t ArpQuantizeFilter::PutConfiguration(const BMessage* values)
{
	status_t err = AmFilterI::PutConfiguration(values);
	if (err != B_OK) return err;
	
	AmTime		t;
	int32		i;
	bool		b;
	if( values->FindInt32(AM_MULTIPLIER_CONTROL_KEY_STR, &i) == B_OK) mMultiplier = i;
	if( find_time(*values, QUANTIZE_STR, &t) == B_OK) mQuantizeTime = t;
	if( values->FindInt32(MODIFIER_STR, &i) == B_OK) {
		if (i <= 2) i = 2;
		else if (i <= 4) i = 3;
		else if (i <= 6) i = 5;
		else i = 7;
		mDivider = i;
	}
	if( values->FindBool(DO_DURATION_STR, &b) == B_OK ) mIncludeDuration = b;
	mFullTime = mMultiplier*((mQuantizeTime*2*QUANT_PRIMES)/mDivider);
	return B_OK;
}

status_t ArpQuantizeFilter::Configure(ArpVectorI<BView*>& panels)
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return err;
	panels.push_back(new ArpQuantizeFilterSettings(mHolder, config));
	return B_OK;
}

/*****************************************************************************
 * ARP-QUANTIZE-FILTER-ADDON
 *****************************************************************************/
void ArpQuantizeFilterAddOn::GetVersion(int32* major, int32* minor) const
{
	*major = 1;
	*minor = 1;
}

BBitmap* ArpQuantizeFilterAddOn::Image(BPoint requestedSize) const
{
	const BBitmap* bm = Resources().FindBitmap("Class Icon");
	if (bm) return new BBitmap(bm);
	return NULL;
}

AmFilterI* ArpQuantizeFilterAddOn::NewInstance(	AmFilterHolderI* holder,
												const BMessage* config)
{
	return new ArpQuantizeFilter( this, holder, config );
}

extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id /*you*/,
												  const void* cookie, uint32 /*flags*/, ...)
{
	if (n == 0) return new ArpQuantizeFilterAddOn(cookie);
	return NULL;
}

/*************************************************************************
 * _QUANTIZE-CONTROL
 *************************************************************************/
static const uint32		QUANTIZE_FINISHED_MSG = 'iqnf';
static const uint32		MODIFIER_FINISHED_MSG = 'imdf';

_QuantizeControl::_QuantizeControl(	AmFilterHolderI* target,
									const BMessage& initSettings,
									const char* initialName )
		: inherited(target, initSettings, be_plain_font,
					B_FOLLOW_LEFT | B_FOLLOW_TOP, B_WILL_DRAW )
{
	AddViews( initialName );
	RefreshDurationControl(initSettings);
	float	right = 0, bottom = 0;
	BView*	view;
	for( view = ChildAt(0); view != 0; view = view->NextSibling() ) {
		BRect	f = view->Frame();
		if( f.right > right ) right = f.right;
		if( f.bottom > bottom ) bottom = f.bottom;
	}
	ResizeTo(right, bottom);
}

void _QuantizeControl::SetQuantizeTime(AmTime quantizeTime)
{
	BMessage upd;
	if( add_time(upd, QUANTIZE_STR, quantizeTime) == B_OK ) {
		Implementation().SendConfiguration(&upd);
	}
}

void _QuantizeControl::SetModifier(int32 modifier)
{
	BMessage upd;
	if( upd.AddInt32(MODIFIER_STR, modifier) == B_OK ) {
		Implementation().SendConfiguration(&upd);
	}
}

void _QuantizeControl::RefreshDurationControl(const BMessage& settings)
{
	AmDurationControl* dc = dynamic_cast<AmDurationControl*>(FindView("duration_ctrl"));
	if (dc) dc->Refresh(settings);
}
	
void _QuantizeControl::AttachedToWindow()
{
	inherited::AttachedToWindow();
	if ( Parent() ) SetViewColor( Parent()->ViewColor() );
}

void _QuantizeControl::MessageReceived(BMessage *msg)
{
	switch( msg->what ) {
		case DUR_QUANTIZE_FINISHED_MSG: {
			float		yValue;
			if (msg->FindFloat( "y_value", &yValue ) == B_OK )
				SetQuantizeTime( (AmTime)yValue );
		} break;
		case DUR_EIGHTHS_FINISHED_MSG: {
			float		yValue;
			if (msg->FindFloat( "y_value", &yValue ) == B_OK )
				SetModifier( (int32)yValue );
		} break;

		case ARP_PUT_CONFIGURATION_MSG: {
			BMessage settings;
			msg->FindMessage("settings", &settings);
			RefreshDurationControl(settings);
		} break;
	
		default:
			inherited::MessageReceived( msg );
	}
}

void _QuantizeControl::AddViews(const char* initialName)
{
	float			labelH = 10;
	BPoint			origin(0, labelH + 5);

	AmDurationControl*	ctrl = new AmDurationControl(	origin, "duration_ctrl", NULL, 0,
														QUANTIZE_STR, MODIFIER_STR);
	if (ctrl) {
		AddChild(ctrl);
	}

	if( initialName ) {
		float			labelW = StringWidth( initialName );
		BRect			frame(0, 1, labelW, labelH + 1);
		BStringView*	sv = new BStringView( frame, initialName, initialName );
		if( sv ) {
			sv->SetFontSize( labelH );
			AddChild( sv );
		}
	}
}
