/*
 *  main.cpp - Main program
 *
 *  SIDPlayer (C) 1996-1998 Christian Bauer
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <support/UTF8.h>
#include <Path.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "CPU.h"
#include "SID.h"


// Constants
const char APP_SIGNATURE[] = "application/x-vnd.cebix-SIDPlayer";

const rgb_color light_color = {255, 255, 255, 0};
const rgb_color fill_color = {216, 216, 216, 0};
const rgb_color dark_color = {184, 184, 184, 0};

const uint32 MSG_PLAY_PAUSE = 'plpa';
const uint32 MSG_STOP = 'stop';
const uint32 MSG_NEXT = 'next';
const uint32 MSG_PREV = 'prev';
const uint32 MSG_LOAD_MODULE = 'load';
const uint32 MSG_FILTERS = 'filt';
const uint32 MSG_FASTER = 'fast';
const uint32 MSG_SLOWER = 'slow';


// Application object
class SIDPlayer : public BApplication {
public:
	SIDPlayer();
	virtual void ArgvReceived(int32 argc, char **argv);
	virtual void RefsReceived(BMessage *msg);
	virtual void ReadyToRun(void);
	virtual void AboutRequested(void);
};


// Main window object
class MainWindow : public BWindow {
public:
	MainWindow();
	virtual bool QuitRequested(void);
	virtual void MessageReceived(BMessage *msg);

private:
	BStringView *make_name_display(BRect frame, char *label_text, BView *parent);
	bool load_psid(char *filename);
	void select_song(int num);

	uint8 *the_ram;		// Pointer to 64K RAM
	MOS6510 *the_cpu;	// Pointer to 6510
	MOS6581 *the_sid;	// Pointer to 6581

	int current_song;			// Song currently playing
	int number_of_songs;		// Number of songs in module
	uint16 init_adr, play_adr;	// C64 init/play routine addresses
	uint32 song_speeds;			// Speed flags
	int current_freq;			// Current replay frequency in Hz
	char module_name[64];		// Module name
	char author[64];			// Author
	char copyright[64];			// Copyright info

	BStringView *name_view;
	BStringView *author_view;
	BStringView *copyright_view;
	BStringView *position_view;
	BCheckBox *filter_view;
};


// Top view object (handles drag&drop)
class TopView : public BView {
public:
	TopView(BRect frame, const char *name, uint32 resizingMode, uint32 flags);
	virtual void MessageReceived(BMessage *msg);
	virtual void KeyDown(const char *bytes, int32 numBytes);
};


// Buttons
class PlayPauseButton : public BButton {
public:
	PlayPauseButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class StopButton : public BButton {
public:
	StopButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class NextButton : public BButton {
public:
	NextButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class PrevButton : public BButton {
public:
	PrevButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};


// Panning slider
class PanSlider : public BSlider {
public:
	PanSlider(BRect frame, const char *name, float &panning) : BSlider(frame, name, NULL, NULL, -100, 100), p(panning)
	{
		SetHashMarks(B_HASH_MARKS_BOTH);
		SetHashMarkCount(5);
		SetBarColor(fill_color);
	}
	virtual ~PanSlider() {}
	virtual void SetValue(int32 value)
	{
		BSlider::SetValue(value);
		p = float(value + 100) / 200.0;
	}

private:
	float &p;
};


// Volume slider
class VolumeSlider : public BSlider {
public:
	VolumeSlider(BRect frame, const char *name, float &volume) : BSlider(frame, name, NULL, NULL, 0, 100), v(volume)
	{
		SetHashMarks(B_HASH_MARKS_BOTH);
		SetHashMarkCount(5);
		SetBarColor(fill_color);
	}
	virtual ~VolumeSlider() {}
	virtual void SetValue(int32 value)
	{
		BSlider::SetValue(value);
		v = float(value) / 100.0;
	}

private:
	float &v;
};


// PSID file header (big-endian)
struct PSIDHeader {
	uint32	id;				// 'PSID'
	uint16	version;		// Version 1 or 2
	uint16	length;			// Length of header
	uint16	start;			// C64 load address
	uint16	init;			// C64 address of init routine
	uint16	main;			// C64 address of replay routine
	uint16	number;			// Number of subsongs
	uint16	defsong;		// Main subsong number (1..255)
	uint16	speedhi;		// Speed flags (1 bit/song)
	uint16	speedlo;
	uint8	name[32];		// Module name
	uint8	author[32];		// Author
	uint8	copyright[32];	// Copyright info
	uint16	flags;			// Flags (only in version 2 header)
	uint32	reserved;
};


// Global: path name of currently loaded PSID module
char ModuleName[256] = "";


/*
 *  Create application object and start it
 */

int main(int argc, char **argv)
{	
	SIDPlayer *the_app = new SIDPlayer();
	the_app->Run();
	delete the_app;
	return 0;
}


/*
 *  Constructor
 */

SIDPlayer::SIDPlayer() : BApplication(APP_SIGNATURE) {}


/*
 *  Shell arguments received
 */

void SIDPlayer::ArgvReceived(int32 argc, char **argv)
{
	if (argc >= 2)
		strncpy(ModuleName, argv[1], 255);
	else
		printf("Usage: %s [name of PSID file]\n", argv[0]);
}


/*
 *  Browser arguments received
 */

void SIDPlayer::RefsReceived(BMessage *msg)
{
	entry_ref the_ref;
	BEntry the_entry;

	if (msg->FindRef("refs", &the_ref) == B_NO_ERROR)
		if (the_entry.SetTo(&the_ref) == B_NO_ERROR)
			if (the_entry.IsFile()) {
				BPath the_path;
				the_entry.GetPath(&the_path);
				strncpy(ModuleName, the_path.Path(), 255);
			}
}


/*
 *  Arguments processed, open player window
 */

void SIDPlayer::ReadyToRun(void)
{
	new MainWindow();
}


/*
 *  About requested
 */

void SIDPlayer::AboutRequested(void)
{
	BAlert *the_alert = new BAlert("",
		"SIDPlayer V3.0 by Christian Bauer\n<cbauer@iphcip1.physik.uni-mainz.de>\n"
		"\nStereo support by Marco Nelissen <marcone@xs4all.nl>\n"
		"\nFreely distributable.", "OK");
	the_alert->Go();
}


/*
 *  Window constructor
 */

MainWindow::MainWindow() : BWindow(BRect(0, 0, 480, 94), "SIDPlayer", B_TITLED_WINDOW, B_NOT_ZOOMABLE)
{
	Lock();
	BRect b = Bounds();

	// Create objects for emulation
	the_ram = new uint8[0x10000];
	the_cpu = new MOS6510();
	the_cpu->RAM = the_ram;
	the_sid = new MOS6581();
	the_cpu->TheSID = the_sid;
	the_sid->TheCPU = the_cpu;
	the_sid->TheRAM = the_ram;
	current_freq = 50;

	// Move window to right position
	MoveTo(80, 80);
	SetSizeLimits(284, 480, 94, 94);

	// Light gray background
	TopView *top = new TopView(BRect(0, 0, b.right, b.bottom), "main", B_FOLLOW_NONE, B_WILL_DRAW);
	AddChild(top);
	top->SetViewColor(fill_color);

	// Name/author/copyright display
	name_view = make_name_display(BRect(0, 5, 279, 21), "Name", top);
	author_view = make_name_display(BRect(0, 25, 279, 41), "Author", top);
	copyright_view = make_name_display(BRect(0, 45, 279, 61), "Copyright", top);

	// Buttons
	top->AddChild(new PlayPauseButton(BRect(6, 66, 46, 90), new BMessage(MSG_PLAY_PAUSE)));
	top->AddChild(new StopButton(BRect(47, 66, 87, 90), new BMessage(MSG_STOP)));
	top->AddChild(new PrevButton(BRect(88, 66, 128, 90), new BMessage(MSG_PREV)));
	top->AddChild(new NextButton(BRect(129, 66, 169, 90), new BMessage(MSG_NEXT)));
	top->AddChild(new BButton(BRect(170, 66, 184, 90), NULL, "+", new BMessage(MSG_FASTER)));
	top->AddChild(new BButton(BRect(185, 66, 199, 90), NULL, "-", new BMessage(MSG_SLOWER)));

	// Position indicator
	top->AddChild(position_view = new BStringView(BRect(210, 61, 279, 75), "position", ""));
	position_view->SetViewColor(fill_color);
	position_view->SetLowColor(fill_color);
	position_view->SetFont(be_plain_font);

	// Filter enable/disable checkbox
	top->AddChild(filter_view = new BCheckBox(BRect(210, 75, 279, 90), "filters", "Filters", new BMessage(MSG_FILTERS)));
	filter_view->SetViewColor(fill_color);
	filter_view->SetLowColor(fill_color);
	filter_view->SetFont(be_plain_font);
	filter_view->SetValue(B_CONTROL_ON);

	// Volume/panning sliders
	BStringView *label;
	top->AddChild(label = new BStringView(BRect(284, 2, 384, 15), "panning_title", "Panning"));
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_CENTER);
	label->SetFont(be_plain_font);

	BSlider *slider;
	top->AddChild(slider = new PanSlider(BRect(284, 15, 384, 16), "pan_0", the_sid->Panning[0]));
	slider->SetValue(-25);
	top->AddChild(slider = new PanSlider(BRect(284, 35, 384, 36), "pan_1", the_sid->Panning[1]));
	slider->SetValue(0);
	top->AddChild(slider = new PanSlider(BRect(284, 55, 384, 56), "pan_2", the_sid->Panning[2]));
	slider->SetValue(25);
	top->AddChild(slider = new PanSlider(BRect(284, 75, 384, 76), "pan_3", the_sid->Panning[3]));
	slider->SetValue(0);

	top->AddChild(label = new BStringView(BRect(394, 2, 472, 15), "volume_title", "Volume"));
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_CENTER);
	label->SetFont(be_plain_font);

	top->AddChild(slider = new VolumeSlider(BRect(389, 15, 477, 16), "volume_0", the_sid->Volume[0]));
	slider->SetValue(75);
	top->AddChild(slider = new VolumeSlider(BRect(389, 35, 477, 36), "volume_1", the_sid->Volume[1]));
	slider->SetValue(75);
	top->AddChild(slider = new VolumeSlider(BRect(389, 55, 477, 56), "volume_2", the_sid->Volume[2]));
	slider->SetValue(75);
	top->AddChild(slider = new VolumeSlider(BRect(389, 75, 477, 76), "volume_3", the_sid->Volume[3]));
	slider->SetValue(75);

	// Show window
	top->MakeFocus();
	Show();
	Unlock();

	// Load PSID module
	PostMessage(MSG_LOAD_MODULE);
}


/*
 *  Create name display field
 */

BStringView *MainWindow::make_name_display(BRect frame, char *label_text, BView *parent)
{
	// Label to the left of the display field
	BRect label_rect = frame;
	label_rect.right = label_rect.left + 65;

	BStringView *label = new BStringView(label_rect, "", label_text);
	parent->AddChild(label);
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_RIGHT);
	label->SetFont(be_bold_font);

	// Box around display field
	BRect frame_rect = frame;
	frame_rect.left += 70;

	BBox *box = new BBox(frame_rect);
	parent->AddChild(box);
	box->SetViewColor(fill_color);
	box->SetLowColor(fill_color);

	// The display field
	BRect textview_rect = frame_rect;
	textview_rect.OffsetTo(0, 0);
	textview_rect.InsetBy(4, 2);

	BStringView *text = new BStringView(textview_rect, "", "");
	box->AddChild(text);
	text->SetViewColor(fill_color);
	text->SetLowColor(fill_color);
	text->SetFont(be_plain_font);

	return text;
}


/*
 *  Quit requested, close all and quit program
 */

bool MainWindow::QuitRequested(void)
{
	// Delete objects
	if (the_sid != NULL)
		delete the_sid;
	if (the_cpu != NULL)
		delete the_cpu;
	if (the_ram != NULL)
		delete[] the_ram;

	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}


/*
 *  Message received
 */

void MainWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {

		case MSG_PLAY_PAUSE:
			if (the_sid->IsPaused())
				the_sid->ResumeSound();
			else
				the_sid->PauseSound();
			break;

		case MSG_STOP:
			the_sid->PauseSound();
			the_cpu->Emulate(init_adr, current_song, 0, 0);
			break;

		case MSG_NEXT:
			if (current_song < number_of_songs-1)
				select_song(current_song + 1);
			break;

		case MSG_PREV:
			if (current_song > 0)
				select_song(current_song - 1);
			break;

		case MSG_LOAD_MODULE:
			the_sid->PauseSound();

			if (load_psid(ModuleName)) {
				Lock();
				name_view->SetText(module_name);
				author_view->SetText(author);
				copyright_view->SetText(copyright);
				Unlock();

				select_song(current_song);

				if (play_adr)
					the_sid->PlayAdr = play_adr;
				else
					if (the_ram[1] & 2)
						the_sid->PlayAdr = (the_ram[0x0315] << 8) | the_ram[0x0314];
					else
						the_sid->PlayAdr = (the_ram[0xffff] << 8) | the_ram[0xfffe];

				the_sid->ResumeSound();
			}
			break;

		case MSG_FILTERS:
			the_sid->EnableFilters(filter_view->Value());
			break;

		case MSG_FASTER:
			current_freq += 5;
			the_sid->SetReplayFreq(current_freq);
			break;

		case MSG_SLOWER:
			if (current_freq > 5) {
				current_freq -= 5;
				the_sid->SetReplayFreq(current_freq);
			}
			break;

		default:
			BWindow::MessageReceived(msg);
			break;
	}
}


/*
 *  Load PSID module
 */

bool MainWindow::load_psid(char *filename)
{
	// Open PSID file
	FILE *file = fopen(filename, "rb");
	if (file) {

		// Read header
		PSIDHeader header;
		if (fread(&header, 1, sizeof(PSIDHeader), file) == sizeof(PSIDHeader)) {

			// Check header ID and version
			if (B_BENDIAN_TO_HOST_INT32(header.id) == 'PSID' && (B_BENDIAN_TO_HOST_INT16(header.version) == 1 || B_BENDIAN_TO_HOST_INT16(header.version) == 2)) {

				// OK, seek to start of module data
				fseek(file, B_BENDIAN_TO_HOST_INT16(header.length), SEEK_SET);

				// Extract data
				current_song = B_BENDIAN_TO_HOST_INT16(header.defsong);
				if (current_song)
					current_song--;

				number_of_songs = B_BENDIAN_TO_HOST_INT16(header.number);
				if (!number_of_songs)
					number_of_songs = 1;
				if (current_song >= number_of_songs)
					current_song = 0;

				init_adr = B_BENDIAN_TO_HOST_INT16(header.init);
				play_adr = B_BENDIAN_TO_HOST_INT16(header.main);

				song_speeds = (B_BENDIAN_TO_HOST_INT16(header.speedhi) << 16) | B_BENDIAN_TO_HOST_INT16(header.speedlo);

				long sl = 32, dl = 64;
				convert_to_utf8(B_ISO1_CONVERSION, (char *)header.name, &sl, module_name, &dl);
				sl = 32; dl = 64;
				convert_to_utf8(B_ISO1_CONVERSION, (char *)header.author, &sl, author, &dl);
				sl = 32; dl = 64;
				convert_to_utf8(B_ISO1_CONVERSION, (char *)header.copyright, &sl, copyright, &dl);
				module_name[63] = 0;
				author[63] = 0;
				copyright[63] = 0;

				// Initialize memory
				memset(the_ram, 0, 0xe000);
				memset(the_ram + 0xe000, 0x40, 0x2000);
				the_ram[1] = 7;

				// Load module into memory
				uint16 load_adr = B_BENDIAN_TO_HOST_INT16(header.start);
				if (!load_adr) {
					uint8 hi, lo;
					fread(&lo, 1, 1, file);
					fread(&hi, 1, 1, file);
					load_adr = (hi << 8) | lo;
				}
				fread(the_ram + load_adr, 1, 0x10000 - load_adr, file);
				fclose(file);

				if (!init_adr)
					init_adr = load_adr;

				return TRUE;
			}
		}
		fclose(file);
	}
	return FALSE;
}


/*
 *  Select a subsong for playing
 */

void MainWindow::select_song(int num)
{
	bool need_resume = FALSE;
	
	// Pause sound if running
	if (!the_sid->IsPaused()) {
		need_resume = TRUE;
		the_sid->PauseSound();
	}

	current_song = num;
	the_sid->Reset();

	// Set replay frequency
	if (num < 32)
		current_freq = song_speeds & (1 << num) ? 60 : 50;
	else
		current_freq = 50;
	the_sid->SetReplayFreq(current_freq);

	the_cpu->Emulate(init_adr, current_song, 0, 0);

	// Resume sound if it was running
	if (need_resume)
		the_sid->ResumeSound();

	// Update position indicator
	char str[16];
	sprintf(str, "Song %d/%d", current_song + 1, number_of_songs);
	position_view->SetText(str);
}


/*
 *  TopView handles dropped messages (load new PSID module) and keypresses
 */

TopView::TopView(BRect frame, const char *name, uint32 resizingMode, uint32 flags)
 : BView(frame, name, resizingMode, flags) {};

void TopView::MessageReceived(BMessage *msg)
{
	if (msg->what == B_SIMPLE_DATA) {
		entry_ref the_ref;
		BEntry the_entry;
		if (msg->FindRef("refs", &the_ref) == B_NO_ERROR)
			if (the_entry.SetTo(&the_ref) == B_NO_ERROR)
				if (the_entry.IsFile()) {
					BPath the_path;
					the_entry.GetPath(&the_path);
					strncpy(ModuleName, the_path.Path(), 255);
					Window()->PostMessage(MSG_LOAD_MODULE);
				}
	} else
		BView::MessageReceived(msg);
}

void TopView::KeyDown(const char *bytes, int32 numBytes)
{
	switch (bytes[0]) {
		case 'p':
		case 'P':
			Window()->PostMessage(MSG_PLAY_PAUSE);
			break;

		case B_ESCAPE:
		case B_SPACE:
		case 's':
		case 'S':
			Window()->PostMessage(MSG_STOP);
			break;

		case B_LEFT_ARROW:
			Window()->PostMessage(MSG_PREV);
			break;

		case B_RIGHT_ARROW:
		case 'n':
		case 'N':
			Window()->PostMessage(MSG_NEXT);
			break;

		case B_UP_ARROW:
			Window()->PostMessage(MSG_FASTER);
			break;

		case B_DOWN_ARROW:
			Window()->PostMessage(MSG_SLOWER);
			break;
	}
}


/*
 *  Play/Pause button
 */

PlayPauseButton::PlayPauseButton(BRect frame, BMessage *msg) : BButton(frame, "play", "", msg) {};

void PlayPauseButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw play/pause image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 18, 16));
	FillTriangle(BPoint(21, 8), BPoint(21, 16), BPoint(25, 12));
}


/*
 *  Stop button
 */

StopButton::StopButton(BRect frame, BMessage *msg) : BButton(frame, "stop", "", msg) {};

void StopButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw stop image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 25, 16));
}


/*
 *  "Next" button
 */

NextButton::NextButton(BRect frame, BMessage *msg) : BButton(frame, "next", "", msg) {};

void NextButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw "next" image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillTriangle(BPoint(17, 8), BPoint(17, 16), BPoint(21, 12));
	FillRect(BRect(22, 8, 24, 16));
}


/*
 *  "Prev" button
 */

PrevButton::PrevButton(BRect frame, BMessage *msg) : BButton(frame, "prev", "", msg) {};

void PrevButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw "prev" image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(17, 8, 19, 16));
	FillTriangle(BPoint(24, 8), BPoint(24, 16), BPoint(20, 12));
}
