/*
 *  SID.h - 6581 emulation
 *
 *  SIDPlayer (C) 1996-1998 Christian Bauer
 */

#ifndef _SID_H
#define _SID_H

#include <MediaKit.h>


// Define this if you want an emulation of a 8580
// (affects combined waveforms)
#undef EMUL_MOS8580


// Structure for one voice
struct DRVoice {
	int wave;		// Selected waveform
	int eg_state;	// Current state of EG
	DRVoice *mod_by;	// Voice that modulates this one
	DRVoice *mod_to;	// Voice that is modulated by this one

	uint32 count;	// Counter for waveform generator, 8.16 fixed
	uint32 add;		// Added to counter in every frame

	uint16 freq;		// SID frequency value
	uint16 pw;		// SID pulse-width value

	uint32 a_add;	// EG parameters
	uint32 d_sub;
	uint32 s_level;
	uint32 r_sub;
	uint32 eg_level;	// Current EG level, 8.16 fixed

	uint32 noise;	// Last noise generator output value

	bool gate;		// EG gate bit
	bool ring;		// Ring modulation bit
	bool test;		// Test bit
	bool filter;	// Flag: Voice filtered

					// The following bit is set for the modulating
					// voice, not for the modulated one (as the SID bits)
	bool sync;		// Sync modulation bit
};


class MOS6510;

// Class for 6581 emulation
class MOS6581 {
public:
	MOS6581();
	~MOS6581();

	void Reset(void);
	void SetReplayFreq(uint32 freq);
	uint8 ReadRegister(uint16 adr);
	void WriteRegister(uint16 adr, uint8 byte);
	void EnableFilters(bool enable);
	void PauseSound(void);
	void ResumeSound(void);
	bool IsPaused(void);

	MOS6510 *TheCPU;	// Pointer to CPU
	uint8 *TheRAM;		// Pointer to 64K RAM
	uint16 PlayAdr;		// Start address of play routine

	float Panning[4];	// Panning settings (0..1)
	float Volume[4];	// Volume settings (0..x)

private:
	static bool stream_func(void *arg, char *buf, size_t count, void *header);
	void calc_buffer(int16 *buf, size_t count);
	void calc_filter(void);

	BSubscriber *the_sub;			// Pointer to subscriber
	BDACStream *the_stream;			// Pointer to DAC stream
	bool ready;						// Flag: Subscriber has subscribed
	bool in_stream;					// Flag: Subscriber has entered stream
	uint8 volume;					// Master volume
	bool v3_mute;					// Voice 3 muted

	static uint16 TriTable[0x1000*2];	// Tables for certain waveforms
	static const uint16 TriSawTable[0x100];
	static const uint16 TriRectTable[0x100];
	static const uint16 SawRectTable[0x100];
	static const uint16 TriSawRectTable[0x100];
	static const uint32 EGTable[16];	// Increment/decrement values for all A/D/R settings
	static const uint8 EGDRShift[256]; // For exponential approximation of D/R
	static const int16 SampleTab[3][16]; // Table for samples at volumes 0..2

	uint32 replay_freq;				// Frequency at which the replay routine is called in Hz
	uint32 replay_count;				// Counter for timing replay routine

	DRVoice voice[3];				// Data for 3 voices

	bool f_enabled;					// Flag: Filters enabled
	uint8 f_type;					// Filter type
	uint8 f_freq;					// SID filter frequency (upper 8 bits)
	uint8 f_res;					// Filter resonance (0..15)
	float f_ampl;					// IIR filter input attenuation
	float d1, d2, g1, g2;			// IIR filter coefficients
	float xn1[4], xn2[4], yn1[4], yn2[4];	// IIR filter previous input/output signal

	int v4_state;					// State of voice 4 (Galway noise/samples)
	uint32 v4_count;					// Counter for voice 4
	uint32 v4_add;					// Added to counter in every frame

	uint16 gn_adr;					// C64 address of tone list
	uint16 gn_tone_length;			// Length of each tone in samples
	uint32 gn_volume_add;			// Added to SID volume reg. for every sample
	int	gn_tone_counter;			// Number of tones in list
	uint16 gn_base_cycles;			// Cycles for before sample
	uint16 gn_loop_cycles;			// Cycles between samples

	uint32 sm_adr;					// C64 nybble address of sample
	uint32 sm_end_adr;				// C64 nybble address of end of sample
	uint32 sm_rep_adr;				// C64 nybble address of sample repeat point
	uint16 sm_volume;				// Sample volume (0..2, 0=loudest)
	uint8 sm_rep_count;				// Sample repeat counter (0xff=continous)
	bool sm_big_endian;				// Flag: Sample is big-endian

	uint8 regs[0x80];				// Copy of SID registers
	uint8 last_sid_byte;			// Last value written to SID
};

#endif
