/*
 * Graph.cpp
 * Copyright Jonathan Booth, 1998
 *
 * Graph.cpp contains the definitions for the classes in Load.h
 */
 
#include <Application.h>
#include <Message.h>
#include <Messenger.h>

#include <Alert.h>
#include <MenuItem.h>
#include <Menu.h>
#include <PopUpMenu.h>
 
#include "Graph.h"
#include "Bitmap.h"
#include "Messages.h"

GraphView::GraphView(DataGatherer *data)
	: BView(ViewDefault, "GraphView",
			B_FOLLOW_ALL_SIDES, 
			B_WILL_DRAW | B_FRAME_EVENTS | B_PULSE_NEEDED) {
	// Set up the locking semaphore
	Lock = create_sem(1,"GraphView Locker");
	
	// Save the data
	Data = data;
}	

void GraphView::AttachedToWindow() {
	// Compute the size the graph to draw in should be
	BRect BitSize(0,0,Maximum.x + 1,Maximum.y);
	
	// Now construct the graph we're drawing in
	Graph = new BBitmap(BitSize,B_COLOR_8_BIT,true);
	Pen = new BView(BitSize,"Pen",0,0);

	// And put the pen on the graph so we can use it to draw
	if (Graph->Lock()) {
		Graph->AddChild(Pen);
		Pen->SetHighColor(background);
		Pen->FillRect(BitSize, B_SOLID_HIGH);

		BRect BitSize(Bounds());
		BitSize.right = BitSize.right + 1;
		const float height = BitSize.bottom - BitSize.top;

		Pen->SetHighColor(foreground);
		Pen->StrokeLine(BPoint(BitSize.left,height*1/2),
						BPoint(BitSize.right,height*1/2));
		Pen->StrokeLine(BPoint(BitSize.left,height*3/4),
						BPoint(BitSize.right,height*3/4));
		Pen->StrokeLine(BPoint(BitSize.left,height*1/4),
						BPoint(BitSize.right,height*1/4));
		
		
		Pen->Sync();
		Graph->Unlock();
	}
}

void GraphView::DetachedFromWindow() {
	BMessage *death = new BMessage(B_QUIT_REQUESTED);
	BMessenger *reaper = new BMessenger(be_app);
	reaper->SendMessage(death);
}

void GraphView::Pulse() {
	// Can't run concurrently with a resize
	if (acquire_sem(Lock) == B_NO_ERROR) {
		// Get and graph the data
		GraphData();
		
		// release the semaphore
		release_sem(Lock);
	}
}

void GraphView::Draw(BRect Update) {
	// Lock it
	if (Window()->Lock()) {
		// Only draw in the intersection of what we want to
		// and our graph's bounds
		Update = Update & Graph->Bounds();
		
		// Draw the older bitmap (not the one being worked on)
		if (Graph) { DrawBitmap(Graph,Update,Update); }
		
		// Unlock things
		Window()->Unlock();
	}
}

void GraphView::MessageReceived(BMessage *msg) {
	float delay;
	
	switch (msg->what)
	{
		case WIN_CH_TIME:
			delay = msg->FindFloat((const char *)"delay",(int32)0);
			if (delay > 0) {
				// Set the new delay
				Window()->SetPulseRate(delay);
			}
			break;
		default:
			BView::MessageReceived(msg);
			break;
	}
}

void GraphView::GraphData() {
	// Local vars
	int Length = ((CPUGatherer *)Data)->DataWidth();
	float *Current = ((CPUGatherer *)Data)->AcquireData();

	// Local var
	BRect BitSize(Bounds());
	BitSize.right = BitSize.right + 1;
	float tmp = 0;
	float Scale = BitSize.bottom / Length + 1;
	const float height = BitSize.bottom - BitSize.top;
	
	// Lock the image
	if (Graph->Lock()) {
		// First, blank the line we're about to draw on
		Pen->BeginLineArray(Length + 1 + 3);

		Pen->SetHighColor(background);
		Pen->StrokeLine(BPoint(BitSize.right,BitSize.top),
						BPoint(BitSize.right,BitSize.bottom));
						
		Pen->SetHighColor(foreground);
		Pen->StrokeLine(BPoint(BitSize.right,height*1/2),
						BPoint(BitSize.right,height*1/2));
		Pen->StrokeLine(BPoint(BitSize.right,height*3/4),
						BPoint(BitSize.right,height*3/4));
		Pen->StrokeLine(BPoint(BitSize.right,height*1/4),
						BPoint(BitSize.right,height*1/4));

		// Now use the Loads to draw the colors
		for (int i=0; i < Length; i++) {
			// Figure the height of this bar
			tmp = Current[i] * Scale;
			
			// Don't bother graphing if it's = 0
			if (tmp != 0) {
				// Now graph it on
				Pen->SetHighColor(cpu_colors[i]);
				Pen->StrokeLine(
					BPoint(BitSize.right,BitSize.bottom),
					BPoint(BitSize.right,BitSize.bottom - tmp));
			
				// And add that much to the offset
				BitSize.bottom = BitSize.bottom - tmp;
			}
		}
		Pen->EndLineArray();
		
		// Flush it out
		Pen->Sync();
		
		// Now copy the bitmap one to the left
		BRect Source(Graph->Bounds());
		BRect Dest(Graph->Bounds());
		Source.left = Source.left + 1;
		Dest.right = Dest.right - 1;
		Pen->DrawBitmap(Graph,Source,Dest);
		Pen->Sync();
						
		Graph->Unlock();
	}
	
	// Draw it all
	Draw(Window()->Bounds());
}

void GraphView::MouseDown(BPoint point) {
	thread_id id = spawn_thread(DoMenu, "Menu Thread",
								B_NORMAL_PRIORITY, this);
	resume_thread(id);
}

int32 GraphView::DoMenu(void *what) {
	((GraphView *)what)->DoMenu2();
	return 0;
}

void GraphView::DoMenu2() {
	BPoint point;
	uint32 buttons;
	BPopUpMenu *what = MakeMenu();
	if (Window()->Lock()) {
		GetMouse(&point,&buttons);
		point = ConvertToScreen(point);
		Window()->Unlock();
		what->Go(point,true);
	} else {
		(new BAlert("","Couldn't lock window!","No Menu"))->Go();
	}
}

BPopUpMenu *GraphView::MakeMenu() {
	BPopUpMenu *menu = new BPopUpMenu("Window Menu", false, false);
	BMenu *sub = NULL;
	BMenuItem *item = NULL;
	BMessage *time = NULL;
	
	item = new BMenuItem("About",new BMessage(B_ABOUT_REQUESTED));
	item->SetTarget(be_app);
	menu->AddItem(item);
	
	sub = new BMenu("Update Rate");
	sub->SetRadioMode(true);
	
	int i=0;
	while (speed[i].desc != NULL) {
		time = new BMessage(WIN_CH_TIME);
		time->AddFloat("delay", 1000000.0 * speed[i].mul);
		item = new BMenuItem(speed[i].desc, time);
		item->SetTarget(this);
		if (Window()->PulseRate() == 1000000.0* speed[i].mul) {
			item->SetMarked(true);
		}
		sub->AddItem(item);
		
		i++;
	}

	menu->AddItem(sub);
		
	item = new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED));
	item->SetTarget(be_app);
	menu->AddItem(item);
	
	return menu;
}