// Copyright  1997  Ian Russell Ollmann.
// All rights reserved. 


// Note: At the moment, this code "works" (except for the bugs, of course), but no effort was 
// made, past the original architectural design of the objects, to make sure that it is "fast."
// In tests on the MacOS (using the MacOS native StringToExtended call instead of strtod), PPC native 
// code on a 604/120 runs sufficiently quickly to do real time curve fitting in one window while 
// the user is typing in data in another window on data sets of more than 100 data points using 
// Chi^2 minimiztion with the Powell minimization algorythm, which calls Evaluate() 
// N*(# of datapoints) per iteration. (There are often 2-300 iterations.) Since the MacOS 
// does not preemptively multitask and the test program was not multithreaded, I imagine
// this code will be even more useful in a multithreaded context such as the BeOS. I eventually 
// plan to release that same graphing/curvefitting/plotting program for the BeOS, so those of 
// you who are my would-be competitors for the BeOS spreadsheet, graphing, or function plotting 
// market are requested to play fair and send me any improvements you make to the code so that 
// I can add them to the public version for everyone to use. This is community-ware.

// LICENSING:
// License to use this code in whole or in part in BeOS programs is free, provided that any 
// improvements made to it are sent to me within 30 days so that I may update the code for 
// use by the rest of the parsing world. License to use this code on non-BeOS systems may only 
// be obtained directly from the author, Ian Ollmann, on a case by case basis. Individuals are 
// free to redistribute this code so long as this Licensing message and the accompanying README 
// file are distributed with it. 


#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "Parser.h"
#include "ShortDebug.h"


//Make a global error message so that we are sure that there is one in out of memory situations.
//This has the potential of resulting in the wrong error being reported in multithreaded applications
//in the rare situation that there is more than one simultaneous parsing operation going on, throwing 
// more than one error.  
ParseErrorMsg		gParseError;

//Parser Object methods
//Converting numerical strings to doubles is often a machine dependent operation, so pool the 
//macine dependent code in one place. 
Boolean Parser::ConvertStringToValue(ParseInfo *parseInfo,  double *value) // returns true if conversion was successful
{
	char		*suffix;
	char		*string;
	long int	length = (long) parseInfo->segmentEnd - (long) parseInfo->segmentStart + 1;
	long int	i;
	
	// We have to use a C stdlib function here which requires \0 terminated
	// strings, which we dont have. Since we are not at liberty to change our
	// string, we have to copy it to another one and add the \0 the copy. 
	string = (char *) malloc( (length + 1) * sizeof( char ) );		
	for( i = 0; i < length; i++)
		string[i] = parseInfo->segmentStart[i];
	string[length] = '\0';	
	
	*value = strtod( string, &suffix );
	
	if( strlen( suffix) )
		return false;
	else
		return true;
} 

// Initializes a new Parser object which applies a function (such as abs() ) to the 
// expression coded in parseInfo. functionType informs the routine what kind of object
// to make and **newFunction codes for the address in which to store the pointer to the
// new functionParser object.
void Parser::CreateNewFunction( ParseInfo *parseInfo, FunctionType functionType, Parser **newFunction )
{
	char 		*firstParen = NULL, *lastParen = NULL;
	char 		*currentChar = parseInfo->segmentStart;
	ParseInfo 	newParseInfo;
	
	ASSERT( parseInfo );
	ASSERT( parseInfo->segmentStart <= parseInfo->segmentEnd + 1);

	// Identify the first open parenthesis in the text segment	
	while( currentChar <= parseInfo->segmentEnd )
	{
		if( *currentChar == '(' )
		{
			firstParen = currentChar;
			break;	
		}
		currentChar++;
	}

	// Identify the last clost parenthesis in the text segment	
	currentChar = parseInfo->segmentEnd;
	while( currentChar >= parseInfo->segmentStart )
	{
		if( *currentChar == ')' )
		{
			lastParen = currentChar;
			break;	
		}
		currentChar--;
	}
	
	// Check the obvious error condition -- the "(" and ")" face each other.
	ASSERT( firstParen < lastParen );
	
	//Set the new segment bounds to be the parenthesis appearing after the 
	//Function name
	newParseInfo.string = parseInfo->string;
	newParseInfo.segmentStart = firstParen;
	newParseInfo.segmentEnd = lastParen;
	newParseInfo.root = parseInfo->root;
	
	// make a new function object			
	switch( functionType )
	{
		case sinFunction:
			*newFunction = (Parser *) new SinFunction( &newParseInfo );
			break;
		case cosFunction:
			*newFunction = (Parser *) new CosFunction( &newParseInfo);
			break;
		case tanFunction:
			*newFunction = (Parser *) new TanFunction(&newParseInfo);
			break;
		case expFunction:
			*newFunction = (Parser *) new ExpFunction( &newParseInfo );
			break;
		case lnFunction:
			*newFunction = (Parser *) new LnFunction( &newParseInfo );
			break;
		case logFunction:
			*newFunction = (Parser *) new LogFunction( &newParseInfo );
			break;
		case absFunction:
			*newFunction = (Parser *) new AbsFunction( &newParseInfo );
			break;
		case sqrtFunction:
			*newFunction = (Parser *) new SqrtFunction( &newParseInfo );
			break;
		case negFunction:
			*newFunction = (Parser *) new NegFunction( &newParseInfo );
			break;
		case arcSinFunction:
			*newFunction = (Parser *) new ArcSinFunction( &newParseInfo );
			break;
		case arcCosFunction:
			*newFunction = (Parser *) new ArcCosFunction( &newParseInfo );
			break;
		case arcTanFunction:
			*newFunction = (Parser *) new ArcTanFunction( &newParseInfo );
			break;
		case hyperbolicSinFunction:
			*newFunction = (Parser *) new HSinFunction( &newParseInfo );
			break;
		case hyperbolicCosFunction:
			*newFunction = (Parser *) new HCosFunction( &newParseInfo );
			break;
		case hyperbolicTanFunction:
			*newFunction = (Parser *) new HTanFunction( &newParseInfo );
			break;
		case randomFunction:
			*newFunction = (Parser *) new RandomFunction( &newParseInfo );
			break;
		default:
			ThrowError( pUnknownFunction, firstParen - parseInfo->string, parseInfo->segmentStart - parseInfo->string);
			break;
	}
}
// Initializes a new Parser object which applies an operator (such as +) to the 
// expression coded in leftInfo and rightInfo, which define the portions of the parseInfo
// string segment to the left and right of the operator. theOperator informs the routine 
// what kind of object to make (and its position) and **newParser codes for the address 
// in which to store the pointer to the new operatorParser object. Usage is used to 
// specify the difference between + and - operators which are used for addition and subtraction
// and those used to specify sign
void Parser::CreateNewOperator( ParseInfo *parseInfo, char *theOperator, OperatorUsage usage, Parser **newParser )
{
	ParseInfo	leftInfo, rightInfo;

	//Identify the portions of the parseInfo string segment to the left and to the right of theOperator.
	//Define new parseInfo structs to describe these segments 		
	leftInfo.string = rightInfo.string = parseInfo->string;
	leftInfo.root = rightInfo.root = parseInfo->root;
	leftInfo.segmentStart = parseInfo->segmentStart;
	leftInfo.segmentEnd = theOperator - 1;
	rightInfo.segmentStart = theOperator + 1;
	rightInfo.segmentEnd = parseInfo->segmentEnd;
	
	//Init a new operator
	switch( *theOperator )
	{
		case '+':
			ASSERT( usage != useAsAChar );
			if( usage == useAsAnOperator )
				*newParser = (Parser *) new PlusOperator( &leftInfo, &rightInfo ); 
			else
				*newParser = (Parser *) new FunctionParser( &rightInfo );//Add a do nothing object to separate out non-operator '+'
			break;
		case '-':
			ASSERT( usage != useAsAChar );
			if( usage == useAsAnOperator )
				*newParser = (Parser *) new MinusOperator( &leftInfo, &rightInfo ); 
			else
				*newParser = (Parser *) new NegFunction( &rightInfo );			
			break;
		case '*':
			ASSERT( usage == useAsAnOperator );
			*newParser = (Parser *) new MultiplyOperator( &leftInfo, &rightInfo ); 
			break;
		case '/':
			ASSERT( usage == useAsAnOperator );
			*newParser = (Parser *) new DivideOperator( &leftInfo, &rightInfo ); 
			break;
		case '%':
			ASSERT( usage == useAsAnOperator );
			*newParser = (Parser *) new ModulusOperator( &leftInfo, &rightInfo ); 
			break;
		case '^':
			ASSERT( usage == useAsAnOperator );
			*newParser = (Parser *) new PowerOperator( &leftInfo, &rightInfo ); 
			break;
		default:
			ThrowError( pUnknownOperator, theOperator - parseInfo->string, theOperator - parseInfo->string);
			break;
	}
	//New is overloaded to throw a pOutOfMemory error if malloc falis, so no error checking necessary here. 
}

//	A method place holder to be used at evaluation time. This method must be redefined by a child method
// of the same name to do the mathematical operation that the child codes for. 
double Parser::Evaluate( void )
{
	return 0;
}

// This method takes a string segment and tries to parse it as a numerical constant (e.g. 3 or 2.1e-5)
// If that fails then it parses it as a variable. If alphabetic constants (e.g. pi or π) are to be added 
// to the capabilities of the Parser, that should occur here. 
void Parser::EvaluateConstant( ParseInfo *parseInfo, double *value, Parser **newParser )
{
	long int	i = 0;
	Boolean		isConstant;
	char 		*start, *currentChar;

	//Make sure this string segment is all one word.
	//Scan for spaces in the string. If there are any, report a syntax error 
	currentChar = parseInfo->segmentStart;
	while( currentChar <= parseInfo->segmentEnd )
	{
		if( isspace( *currentChar ) )
		{	
			start = currentChar;
			while( currentChar < parseInfo->segmentEnd && isspace( *(currentChar+1) ) )
				currentChar++;
			ThrowError( pMissingOperator,  start - parseInfo->string, currentChar - parseInfo->string);
		}
		currentChar++;
	}

	//Since string->Double conversions are a bit platform dependent, abstract
	//the conversion routine to a function for easier portability
	isConstant = ConvertStringToValue( parseInfo, value);
	
	if( isConstant )
		*newParser = NULL;	//Set the newParser field to a null pointer to say that the string is a constant
	else
		*newParser = (Parser *) new Variable( parseInfo);	//Init a new Variable class to handle this string segment as a variable

}

//This is the Parser's only error Handling tool. It assigns the method arguments to 
//the appropriate fields in the global struct gParseError. gParseError is a global to
//make sure that in the event that pOutOfMemory errors start to appear, the data space
//to hold them is already allocated. In a multithreaded situation, in the rare situation
// that more than one error is thrown at once, the returned error type may be incorrect. 
void Parser::ThrowError( ParseErrorType errorType, short startOffset, short endOffset )
{
	gParseError.errorType = errorType;
	gParseError.startOffset = startOffset;
	gParseError.endOffset = endOffset;
	
	throw gParseError;	
}

// This function scans the string segment for parentheses, checking along the way to
// make sure that they are balanced. If there is a single outer set of parentheses, then
// it looks for text to the left of it. If that text matches one of the predefined function
// types from Parser.h, then that function type is returned, otherwise a unknown function 
// error is thrown. If there are no paren, then the function returns noFunction.
FunctionType Parser::FindFunction( ParseInfo *parseInfo )
{
	long int	length = (long) parseInfo->segmentEnd - (long) parseInfo->segmentStart + 1;
	char		*firstParen = NULL;
	char 		*lastParen = NULL;
	int			i;
	int			numZeroLevelParen = 0, parenLevel = 0;
	char		*currentChar = parseInfo->segmentStart;
	FunctionType 	functionType = syntaxErr;
	
	//Doing some error checking on the parentheses here. Also setting pointers to the first and last paren
	while( currentChar <= parseInfo->segmentEnd )
	{
		switch( *currentChar )
		{
			case '(':
	//At this point, there should be only one outer set of parenthesis on this level. If there are more, 
	//then there is probably a missing oper
				if( numZeroLevelParen && parenLevel == 0 )
				{
					short			 lowOffset, hiOffset;
					
					//Set the high error offset to be the current paren
					hiOffset = currentChar - parseInfo->string;
					
					//Scan backward to the previous close paren
					while( *currentChar != ')' )
						currentChar--;
					
					//Set the low error offset to be this paren	
					lowOffset = currentChar - parseInfo->string;
					
					ThrowError( pMissingOperator, lowOffset, hiOffset );
				}
				parenLevel++;
				if( ! firstParen )
					firstParen = currentChar;

				break;
			case ')':
				parenLevel--;
				if( parenLevel == 0 )
				{
					numZeroLevelParen++;
					lastParen = currentChar;
				}	
	//If there are more close paren at any point than open paren, then there is a paren imbalance syntax error				
				if( parenLevel < 0)
					ThrowError( pUnbalancedParen, currentChar - parseInfo->string, currentChar - parseInfo->string);
				break;
			default:
				break;
		}
		currentChar++;
	}

	//if paren level is still > 0, then there are not enough close parens	
	if( parenLevel > 0)
		ThrowError( pUnbalancedParen,firstParen - parseInfo->string,firstParen - parseInfo->string);

	//In theory, if the current string holds a function, the last character
	//should now be a close paren, since we already stripped off all of the 
	//extraneous white space. If it is not, then check to see if there 
	//are any parentheses. If there are, then we have something after the last 
	//paren with no operator (which we already looked for) between, so return 
	//a syntax error. If there are no parentheses at all then we treat this 
	//like a constant or a variable and return noFunction.
	
	if( lastParen == parseInfo->segmentEnd && firstParen ) //If the last paren is at the end of the segment and there is a first paren
	{
		char			typeString[4];
		
		strncpy( typeString, parseInfo->segmentStart, 4 );//Copy the first four characters of the segment into typeString
		
		//scan through functionType to locate any spaces or '(' and fill in \0 from then on.
		for( i = 0; i < 4; i++ )
		{
			typeString[i] = toupper( typeString[i] );	//Make sure everything is uppercase so function recognition is case insensitive
			if( typeString[i] == '(' || isspace( typeString[i] ) )	//If you run into a ( or " " chacter pad the rest of typeString with '\0''s
			{
				typeString[i] = '\0';
				i++;
				while( i < 4 )
					typeString[i++] = '\0';
				break;
			}
		}
		
		//The below list looks a bit redundant. It actually a way of screening out unknown function types
		//(which are handled by the default case) so that the proper error can be returned if the function
		//is not recognized. There may be a way to get rid of this check by redefining the functionType list as 
		//a const array or something. That would make adding new function types a little bit easier.
		switch( *((FunctionType *) typeString ) )//a little bit of fancy type casting to the typestring to a FunctionType variable
		{	//New function types should be listed here
			case sinFunction:
				functionType = sinFunction;
				break;
			case cosFunction:
				functionType = cosFunction;
				break;
			case tanFunction:
				functionType = tanFunction;
				break;
			case expFunction:
				functionType = expFunction;
				break;
			case lnFunction:
				functionType = lnFunction;
				break;
			case logFunction:
				functionType = logFunction;
				break;
			case absFunction:
				functionType = absFunction;
				break;
			case sqrtFunction:
				functionType = sqrtFunction;
				break;
			case arcSinFunction:
				functionType = arcSinFunction;
				break;
			case arcCosFunction:
				functionType = arcCosFunction;
				break;
			case arcTanFunction:
				functionType = arcTanFunction;
				break;
			case hyperbolicSinFunction:
				functionType = hyperbolicSinFunction;
				break;
			case hyperbolicCosFunction:
				functionType = hyperbolicCosFunction;
				break;
			case hyperbolicTanFunction:
				functionType = hyperbolicTanFunction;
				break;
			case randomFunction:
				functionType = randomFunction;
				break;
			default:
				ThrowError( pUnknownFunction, parseInfo->segmentStart - parseInfo->string, firstParen - parseInfo->string -1);
				break;
		}				
	}
	else
	{
		if( lastParen ) //There is some garbage between the last paren and the end of the segment
			ThrowError( pSyntaxError, lastParen - parseInfo->string, parseInfo->segmentEnd - parseInfo->string );
		else	//No garbage and no paren, so there is no function here. This is probably a constant or a variable.
			functionType = noFunction;
	}
	
	return functionType;
}

//This method scans the parseInfo string segment for characters associated with operator types.
//It returns the lowest priority (and rightmost) operator it finds. 
char *Parser::FindOperator( ParseInfo *parseInfo, OperatorUsage *usage )
{	
	char 			*lowestOperator = NULL, *temp = NULL;
	char 			*currentChar = parseInfo->segmentStart;
	int				parenLevel = 0;
	short			priority = 0;
	short			lowestPriority = kMaxPriority;
	OperatorUsage	thisOperatorUsage;
	
	
	while( currentChar <= parseInfo->segmentEnd && currentChar )
	{
		switch ( *currentChar )
		{
			case '-':
			case '+':
			case '*':
			case '/':
			case '%':
			case '^':
				if( parenLevel == 0 )
				{
					thisOperatorUsage = OperatorUsageByContext( parseInfo, currentChar );
					priority = Priority( currentChar, thisOperatorUsage ); 
					if(  priority && priority <= lowestPriority  )	
					{
						lowestOperator = currentChar;
						lowestPriority = priority;
						*usage = thisOperatorUsage;
					}
				}
				break;
			case '(':
					parenLevel++;
				break;
			case ')':
					parenLevel--;
					// Return an error if there are ever more clost parens than open parens				
					if( parenLevel < 0 )
						ThrowError( pUnbalancedParen, currentChar - parseInfo->string, currentChar - parseInfo->string);
				break;
			default:
				break;
			
		}
		currentChar++;
	}

	if( parenLevel != 0 )
	{
		// Since we already caught the too many close paren case, this must be too many open paren
		currentChar = parseInfo->segmentStart;
		while(*currentChar != '(' && currentChar <= parseInfo->segmentEnd ) 
			currentChar++;
			
		ThrowError( pUnbalancedParen, currentChar - parseInfo->string, currentChar - parseInfo->string);
	}

	return lowestOperator;

}

//This is another method which should be overloaded by Parser child classes. 
Boolean Parser::IsConstant( void )
{
	return true;
}

//The overloads the new operator of Parser and its children such that it throws
//an error if malloc fails. This saves a lot of error checking elsewhere in the 
//code. 
void *Parser::operator new( size_t size )
{
	void	*ptr;
	ptr = (void *) malloc( size);
	
	if( !ptr )
	{
		ParseErrorMsg	error;
		
		error.errorType = pOutOfMemory;
		error.startOffset = 0;
		error.endOffset = 0;
		
		throw error; 
	}
	
	return ptr;
}

//We overload the delete operator to make sure new and delete are using compatable
//malloc and free calls. We also do a bit of optional error checking.
void Parser::operator delete( void *parserPtr, size_t size)
{
	ASSERT( parserPtr );
	ASSERT( size );
	free( (void *) parserPtr);
}

//Given a pointer to an operator, this function returns whether it is in fact an operator
//or its environt dictates that it is actually a sign specifier. 
OperatorUsage Parser::OperatorUsageByContext(  ParseInfo *parseInfo, char *operatorPtr  )
{
	char			*tempChar = operatorPtr - 1;
	char 			*scanChar;
	OperatorUsage 	operatorUsage = useAsAnOperator;

	//Start from the operator and scan backwards until you find the first non-space character.
	//Don't let the scan go past the beginning of the string segment.		
	while( isspace( *tempChar ) && tempChar > parseInfo->segmentStart )
		tempChar--;
	
	//Check to make sure we didn't start off before the string segment in the first place
	if(  tempChar >= parseInfo->segmentStart )
	{
		switch( *tempChar ) //If the first char (or start of the string segment ) is a space or an operator, and
		{					//the operator is + or -, then it is a sign. This is handles cases like "1 - -2" or sin(-1)
			case ' ':
			case '\t':
			case '\r':
			case '\n':
			case '(':
			case '+':
			case '-':
			case '*':
			case '/':
			case '%':
			case '^':
				if( *operatorPtr == '-' || *operatorPtr == '+' )
					operatorUsage = useAsASign;
				else
					ThrowError( pSyntaxError, operatorPtr - parseInfo->segmentStart, operatorPtr - parseInfo->segmentStart);
				break;
			case 'e':	//If the first char is an e or E, then this is scientific notation perhaps. (1.2e-23)
			case 'E':	//keep scanning to see if the character previous to the e is a number. If so, it is 
						//scientific notation and the operator should be treated as part of the string segment
						//( as a char). If the operator was not + or - then we have a syntax error ( 1.2e*23)
				operatorUsage = useAsAnOperator;
				if( *operatorPtr == '-' || *operatorPtr == '+')
				{
					scanChar = tempChar -1;
					while( isspace( *scanChar) && scanChar >= parseInfo->segmentStart )
						scanChar--;
					if( isdigit( *scanChar ) )
						operatorUsage = useAsAChar;	
				}
				else
					ThrowError( pSyntaxError, operatorPtr - parseInfo->string, operatorPtr - parseInfo->string );
				break;
			default:
				operatorUsage = useAsAnOperator; // the first non space char or start of the string segment was not an error or a space or an E, so this is a valid operator
		}
	}
	else
	{	//operator is at the beginning of the string, so it may be a sign
		if( *operatorPtr == '-' || *operatorPtr == '+' )	//These are reasonable sign characters, so return that it is a sign
			operatorUsage = useAsASign;
		else												//Otherwise, throw a syntax error.
			ThrowError( pSyntaxError, operatorPtr - parseInfo->string, operatorPtr - parseInfo->string );
	}

	return operatorUsage; 					
}

//This is the top level method, which is the first called when a string segment needs to be parsed
//It should be called in the constructor
void Parser::Parse( ParseInfo *parseInfo, Parser **newParser, double *value )
{
	char 			*lowestOperator = NULL;
	OperatorUsage	usage = useAsAnOperator;
	short			start = parseInfo->segmentStart- parseInfo->string;
	short			end = parseInfo->segmentEnd - parseInfo->string;

	//Remove white space and extra bounding parentheses that no longer serve a purpose
	TrimString( parseInfo );

	if( parseInfo->segmentStart <= parseInfo->segmentEnd )		//If after trimming, there is still a string left to parse...
	{			
		lowestOperator = FindOperator( parseInfo, &usage );		//Look for an operator
		
		if( lowestOperator )									//If there is one, make a new operator object
			CreateNewOperator( parseInfo, lowestOperator, usage, newParser );
		else													
		{
			FunctionType	functionType;
			
			functionType = FindFunction( parseInfo );				//No operator, so look for a function
			
			switch ( functionType )
			{
				case syntaxErr:									//If a syntax error comes back, throw an error exception
					ThrowError( pSyntaxError, parseInfo->segmentStart - parseInfo->string, parseInfo->segmentEnd - parseInfo->string);
					break;
				case noFunction:
					EvaluateConstant( parseInfo, value, newParser );	//If no Function is present, evaluate the string as a constant (or variable)
					break;
				default:
					CreateNewFunction( parseInfo, functionType, newParser ); //A function was present, so init a new function object
					break;
			
			}
		}
	}
	else	//There is nothing here to parse
		ThrowError( pEmptyExpression, start, end); 	
}


//Constructor for Parser. There are no member data so nothing to init. 
Parser::Parser( void  )
{
}

//Destructor for Parser. There are no member data so nothing to destruct
Parser::~Parser()
{
}

//Given an operator and its usage, determinet the priority level of the operator
short Parser::Priority( char *c, OperatorUsage usage )
{
	OperatorType 	thisOperator;
	short			priority;
	
	ASSERT( c );

	switch( usage)
	{
		case useAsAnOperator:
			switch( *c )
			{	
				case '+':
					thisOperator = plusOperator;
					break;
				case '-':
					thisOperator = minusOperator;
					break;
				case '*':
					thisOperator = multiplyOperator;
					break;
				case '/':
					thisOperator = divideOperator;
					break;
				case '%':
					thisOperator = modulusOperator;
					break;
				case '^':
					thisOperator = powerOperator;
					break;
				default:
					thisOperator = noOperator;
					break;
			}
			priority = kPriorityArray[ thisOperator ];
			break;
			
		case useAsASign:
			priority = kMaxPriority;
			break;
			
		case useAsAChar:
			priority = 0;
			break;
		default:
			ASSERT( false ); //Unknown usage type came here
	}
	
	return priority;		
}

//Scan the edges of the string segment for white space and move in the edges of the 
//string segment to get rid of it. 
void Parser::StripWhiteSpaceFromEnds( ParseInfo *parseInfo)
{
	
	ASSERT( parseInfo );
	ASSERT( parseInfo->segmentStart && parseInfo->segmentEnd);
	
	while( isspace( *parseInfo->segmentStart ) && parseInfo->segmentStart <= parseInfo->segmentEnd )
		parseInfo->segmentStart++;
			
	while( parseInfo->segmentEnd >= parseInfo->segmentStart && isspace( *(parseInfo->segmentEnd) ) )
		parseInfo->segmentEnd--;
	
}

//This method is used to trim extraneous spaces and parentheses from the ends of the 
//parseInfo string segment.
void Parser::TrimString( ParseInfo *parseInfo )
{
	char  		*firstParen = NULL, *lastParen = NULL;
	char		*currentChar;
	long int	parenLevel = 0;
	Boolean 	doneTrimming = false; 
	

	while( ! doneTrimming )
	{
		doneTrimming = true;
		StripWhiteSpaceFromEnds( parseInfo );

		//The white space is gone, so now trim off any bracketing paren that can be safely removed.
		//However, as we just trimmed off the white space and the string may have all been white space, 
		//check to make sure there still is a string		
		if( parseInfo->segmentStart <= parseInfo->segmentEnd )
		{
			//There is still a string, so search through it for the first open paren and last close paren
			currentChar = parseInfo->segmentStart;
			while( currentChar <= parseInfo->segmentEnd )
			{
				switch( *currentChar )
				{	
					case '(':
						parenLevel++;			
						if( ! firstParen )
							firstParen = currentChar;
						break;
					case ')':
						parenLevel--;
						if( parenLevel == 0 && ! lastParen ) //If we are currently at the lowest paren level and have not yet found the last paren
						{
							if( firstParen == parseInfo->segmentStart && currentChar == parseInfo->segmentEnd )
							{	//If the firstParen is at the start of the string and the last paren is at the end of the string, exclude both from the string
								//Move up null termination
								parseInfo->segmentEnd--;
								
								//Advance string pointer
								parseInfo->segmentStart++;
								
								//Go around again
								doneTrimming = false;
							}
							else
								lastParen = currentChar;	//we found the last paren, and it wasn't at the end of the string, so stop trimming
						}
						break;
					default:
						break;
				}
				currentChar++;
			}
		}
		else
			doneTrimming = true;
	}

}


//Function
// The FunctionParser Object is a child of Parser which should serve as a base class
// for most simple functions. Its constructor does the work of parsing the string 
// for you and assigning the function argument either to value if it is a constant 
// or to the next pointer if it is a more complicated expression which requires further
// processing by another Parser object. 
FunctionParser::FunctionParser( ParseInfo *parseInfo ) : Parser()
{
	ASSERT( parseInfo );	//Check to make sure we were fed a valid (non-NULL, at least) pointer
		
	next = NULL;	//This line may be unnecessary, but best to be safe.
	
	Parse( parseInfo, &next, &value );
	
	//If the Parse routine produced a Parse object limb which contains all constant
	//valued objects, then evaluate it and prune the limb.
	if( next )
	{
		if( next->IsConstant() )
		{
			value = next->Evaluate();
			delete next;
			next = NULL;
		}
	}	
}

//The destructor. Deletes the next object on the tree, if there is one. 
FunctionParser::~FunctionParser()
{
	if( next ) 
		delete next;
}

//This is the default Evaluate function which does nothing but pass the function
//argument on as its result -- an identity function, if you will. Child classes 
//overload this function to apply there own special operation on the return line 
// below.
double FunctionParser::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return value;
}

//This routine checks to see if this object and those it depends on hold a constant value.
//If they do, then return true. 
Boolean FunctionParser::IsConstant( void )
{
	Boolean isConstant = true;
	
	if( next )
		isConstant &= next->IsConstant();
		
	return isConstant;
}

//Operation
// The OperationParser Object is a child of Parser which should serve as a base class
// for most simple operators. Its constructor does the work of parsing the left and right string segments
// for you and assigning the results either to value if they turn out to be a constant 
// or to the next pointer if it is a more complicated expression which requires further
// processing by another Parser object. 
OperationParser::OperationParser( ParseInfo *leftInfo, ParseInfo *rightInfo ) : Parser()
{
	ASSERT( leftInfo );
	ASSERT( rightInfo );
	
	left = right = NULL;	//This line may be unnecessary, but better safe than sorry. 

	Parse( leftInfo, &left, &leftValue );		//Parse the left string segment
	
	Parse( rightInfo, &right, &rightValue );	//Parse the right string segment
	
	//If the left string segment is constant, evaluate it and the prune the limb so we don't
	//waste time evaluating it again.	
	if( left )
	{
		if( left->IsConstant() )
		{
			leftValue = left->Evaluate();
			delete left;
			left = NULL;
		}
	}
	
	//If the right string segment is constant, evaluate it and the prune the limb so we don't
	//waste time evaluating it again.	
	if( right )
	{
		if( right->IsConstant() )
		{
			rightValue = right->Evaluate();
			delete right;
			right = NULL;
		}
	}
}

//The destructor kills off any limbs farther out on the tree if there are any. 
OperationParser::~OperationParser( )
{
	if( left )
		delete left;
		
	if( right )
		delete right;
}

//A default Evaluate() method definition that should be replaced by a child class definition
double OperationParser::Evaluate( void )
{
	return 0;	//Override this function in child classes to perform the designated operation
}

//Returns true if both of its two limbs are constant
Boolean OperationParser::IsConstant( void )
{
	Boolean isConstant = true;
	
	if( left )
		isConstant &= left->IsConstant();
		
	if( right )
		isConstant &= right->IsConstant();
		
	return isConstant;
}

//FunctionInterpreter
//FunctionInterpreter is a Parser child with two missions. It interfaces with the outside 
//application, and it stores and manages the variable names and values. In theory, it 
//probably could be split into an interface object of type FunctionParser (an identity function) 
//and an object which manages a variable list. However, as it is not all that complex in its
//current incarnation and splitting it into two classes would mean keeping track of an extra
//series of pointers, it didn't seem worth it. 

//Add variable is called by a Variable object to check to see if a string segment described by
//parseInfo is already listed as a variable. If it is then AddVariable returns the index of that
//variable. If it isn't already defined, then AddVariableToList is called and the index of the
//new variable is returned. 
int	FunctionInterpreter::AddVariable( ParseInfo *parseInfo )			// Returns the index of the variable					
{
	int		i, j, index = variableCount;
	long 	length = parseInfo->segmentEnd - parseInfo->segmentStart + 1;
	char	*newString = parseInfo->segmentStart;
	char	*varString;
	Boolean	isIdentical;

	ASSERT( parseInfo );	
	
	//Determine if the variable is already used
	if( variableList )
	{
		//Index through the variable list, looking for a match
		for( i = 0; i < variableCount; i++ )
		{
			isIdentical = true;
			varString = variableList[i].name;
			//Compare character by character to see if the new variable and the old variable's names are the same
			for( j = 0; j < length && isIdentical; j++ )
				if( newString[j] != varString[j] )
					isIdentical = false;
			//If they are, check to make sure the old one is null terminated on the next char, so we can be sure they
			//are of the same length. If everything checks out as the same, set the returned index value to this entry
			//in the variable list.
			if( isIdentical && varString[length] == '\0' )
			{
				index = i;	
				break;	
			}
		}
	}

	//If the variable is not already used, index should still be equal to variable count.
	//If that is true, then init a new variable
	if( index == variableCount )
		AddVariableToList( parseInfo ); //Note increments variable count
	
	return index;	//return the index of the variable
}

//This method adds a variable to the list of variables.
void FunctionInterpreter::AddVariableToList( ParseInfo *parseInfo )	
{
	long 			length = parseInfo->segmentEnd - parseInfo->segmentStart + 2;
	int				i;
	char			*newString;
	
	//Make sure there is an initialized variable list pointer
	if( ! variableList )
	{
		variableList = (ParserVariable *) malloc(0);
		if( ! variableList )
			ThrowError( pOutOfMemory );
	}

	//There should definitely be a variable list of some sort at this point, but
	//check just to make sure. 
	ASSERT( variableList );

	// increment the variable count to reflect the fact that we just added a variable
	variableCount++;
		
	//Append the new entry on to the variable list
	variableList = (ParserVariable *) realloc( (void *) variableList, variableCount * sizeof( ParserVariable)  ); 	
	newString = (char *) malloc( length * sizeof( char ) );

	if( ! variableList || ! newString )
		ThrowError( pOutOfMemory );

	//Init the values in the new entry
	variableList[ variableCount-1 ].value = 0;
	for( i = 0; i < length -1; i++ )
		newString[i] = parseInfo->segmentStart[i];			
	newString[ length - 1 ] = '\0';
	variableList[ variableCount-1 ].name = newString;
}

//This Evaluate call passes along the Evaluate call to the next object up the tree, if
//there is one  and  then returns the value it holds (it being ambiguous here).
double FunctionInterpreter::Evaluate( void )
{
	if( calculationStart )
		value = calculationStart->Evaluate();
		
	return value;
}

//You call this from your own program to get the value of a variable
double FunctionInterpreter::GetVariableValue( int index )					// Returns the value of the variable at index index
{
	ASSERT( index >= 0 && index < variableCount );
	
	return variableList[ index ].value;
}

//Returns the number of (different) variables held by FunctionInterpreter
int	FunctionInterpreter::GetNumberOfVariables( void )					// Returns the number of variables
{
	return variableCount;
}

//Returns a pointer to a NULL terminated char array which contains the name of the variable
char *FunctionInterpreter::GetVariableName( int index ) 	// Returns a pointer of type ParserVariable to the varaible of index index
{
	if( index >= 0 && index < variableCount )
		return variableList[index].name;
	else
		return NULL;
}

// This takes a double provided by the outside program and copies to the variable of number index
Boolean	FunctionInterpreter::SetVariableValue( double newValue, int index )	// Returns false if index is out of bounds
{
	if( index >= 0 && index < variableCount )
	{
		variableList[ index ].value = newValue;
		return true;
	}
	else
		return false;
}

//The constructor 
FunctionInterpreter::FunctionInterpreter( ParseInfo *parseInfo ) : Parser()
{
	//Initialize the variable list and variable count to indicate that there are no variables
	variableList = NULL;
	variableCount = 0;

	ASSERT( parseInfo );

	//The root field is used by the Variable objects to figure out who to 
	//register new variables with and whom to ask for the variable values from.	
	parseInfo->root = this;	
			
	calculationStart = NULL; //May be unnecessary, but best to be safe. 
	
	//Skip over '=' if it is the first char in string
	if( *parseInfo->segmentStart == '=')
		parseInfo->segmentStart++;
	
	//Parse the expression
	Parse( parseInfo, &calculationStart, &value );
	
	//If the expression turned out to be constant, evaluate it and prune off the rest of the tree
	if( calculationStart )
	{
		if( calculationStart->IsConstant() )
		{
			value = calculationStart->Evaluate();
			delete calculationStart;
			calculationStart = NULL;
		}
	}
}

FunctionInterpreter::~FunctionInterpreter()
{
	if( variableList )
	{
		int 	i;
		
		//delete the variable name strings		
		for( i = 0; i < variableCount; i++ )
			if( variableList[i].name )
				free( (char *) variableList[i].name );

		//delete the variable list
		free( (void *) variableList );
	}
}


//PlusOperator
double PlusOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	return leftValue + rightValue;
}

PlusOperator::PlusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo ) 
{

}

//MinusOperator
double MinusOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	return leftValue - rightValue;
}

MinusOperator::MinusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo ) 
{

}

//MultiplyOperator
double MultiplyOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	return leftValue * rightValue;
}

MultiplyOperator::MultiplyOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo )  
{

}

//DivideOperator
double DivideOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	if( rightValue == 0 )
		ThrowError( pDivisionByZero, (long) this );

	return leftValue / rightValue;
}

DivideOperator::DivideOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo ) 
{

}

//ModulusOperator
double ModulusOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	return (int) leftValue % (int) rightValue;
}

ModulusOperator::ModulusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo ) 
{

}

//PowerOperator
double PowerOperator::Evaluate( void )
{
	if( left )
		leftValue = left->Evaluate();
		
	if( right )
		rightValue = right->Evaluate();

	if( leftValue > 0 )
		return pow( leftValue, rightValue );
	else
	{
		if( leftValue == (double) 0 )
			return 0;
		else	
			return pow( leftValue, (int) rightValue );
	}
}

PowerOperator::PowerOperator( ParseInfo *leftInfo, ParseInfo *rightInfo ) : OperationParser( leftInfo, rightInfo ) 
{

}

//FUNCTIONS
//Sin
double SinFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return sin( value );
}

SinFunction::SinFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Cos
double CosFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return cos( value );
}

CosFunction::CosFunction(  ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Tan
double TanFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return tan( value );
}

TanFunction::TanFunction(  ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Exp
double ExpFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return exp( value );
}

ExpFunction::ExpFunction(  ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{
}

//Ln
double LnFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	if( value <= 0 )	
		ThrowError( pLnNegNumber );

	return log( value );
}

LnFunction::LnFunction(  ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Log
double LogFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
	if( value <= 0 )	
		ThrowError( pLogNegNumber );
	return log10( value );
}

LogFunction::LogFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Sqrt
double SqrtFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
	if( value < 0 )
		ThrowError( pSqrtNegNumber, (long) this );
	
	return sqrt( value );
}

SqrtFunction::SqrtFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Abs
double AbsFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return fabs( value );
}

AbsFunction::AbsFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//Neg:  reverses the sign of its expression
//This function is called when a "-" sign operator is found which is not part of 
//the exponent portion of a number in scientific notation
double NegFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
		
	return -value;
}

NegFunction::NegFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//ArcSin
double ArcSinFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
	if( value > 1 || value < -1)
		ThrowError( pArcSinOutOfBounds );

	return asin(value);
}

ArcSinFunction::ArcSinFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//ArcCos
double ArcCosFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
	if( value > 1 || value < -1)
		ThrowError( pArcCosOutOfBounds );

	return acos(value);
}

ArcCosFunction::ArcCosFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{
}

//ArcTan
double ArcTanFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
		return atan(value);
}

ArcTanFunction::ArcTanFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{
}

//HSin
double HSinFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
		return sinh(value);
}

HSinFunction::HSinFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//HCos
double HCosFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
		return cosh(value);
}

HCosFunction::HCosFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//HTan
double HTanFunction::Evaluate( void )
{
	if( next )
		value = next->Evaluate();
	
		return cosh(value);
}

HTanFunction::HTanFunction( ParseInfo *parseInfo ) : FunctionParser( parseInfo )
{

}

//RandomFunction
//RAND(expr) returns a random number between 0 and expr. If expr is undefined (e.g. "()")
//then RAND() returns a random number between 0 and 1. Because it can take undefined 
//arguments, we can't use the core routines already defined by FunctionParser, as they assume 
//expr is defined and will throw errors if it is not. As a result, RandomFunction is not a 
//child of FunctionParser, but a direct descendent of Parser. (The problem lies in the Function
//Parser constructor, which you can't overload.) Apart from the constructor, which does added
//error checking and the Evaluate() function which contains the random number generation code, 
//we essentially duplicate the function of FunctionParser here. 
double RandomFunction::Evaluate( void )
{
	//Got to do a lot of type coercion here or else it will define this as a float and a lot
	//of the randomness will go away. 
	const double inverseRandMax = (double) 1 / (double) RAND_MAX;

	if( next )
		value = next->Evaluate();
	
	return value * (double) rand() * inverseRandMax;
}


Boolean RandomFunction::IsConstant( void )
{
	return false; //It returns random results. How could it be constant?
}

// This function may legally take no arguments, so we have to skip over Function Parser 
// so that our own constructor gets called instead to check for that possibility
RandomFunction::RandomFunction( ParseInfo *parseInfo ) : Parser( )
{
	ASSERT( parseInfo );

	//Trim the paren and extra space off of the argument for RandomFunction
	TrimString( parseInfo );
	
	//If there is anything left,
	if( parseInfo->segmentStart <= parseInfo->segmentEnd )		
		Parse( parseInfo, &next, &value );	//parse the string normally using the inherited function
	else	
	{//nothing left, so set next to NULL and value to 1 as our default argument
		next = NULL;
		value = 1;
	}

	//If the argument is constant, evaluate it and prune the object limb so we don't have to
	//evaluate it any more. 
	if( next )
	{
		if( next->IsConstant() )
		{
			value = next->Evaluate();
			delete next;
			next = NULL;
		}
	}	
}

//Variable -- used to hold a variable
//Again because we don't actually want to parse this string, variable is not a 
//FunctionParser child. It is a direct child of Parser. It operates as a place
//holder and wrapper for the FunctionInterpreter object which really stores the 
//variables. 
double Variable::Evaluate( void )
{
	ASSERT( root );
	return root->GetVariableValue( index );
}

Boolean	 Variable::IsConstant( void )
{
	return false;
}

Variable::Variable( ParseInfo *parseInfo ) : Parser()
{
	root = parseInfo->root;
	ASSERT( root );
	index = root->AddVariable( parseInfo );
}
