//=======================================================================
// RfsvServer.cpp
// © Michael Pieper 14. Feb. 1999
//	 last change: 09.10.99 Problem in CHANGEDIR, when folder dosn't exist
//=======================================================================
//
//  This class is used to manage the LINK-Connection from the PSION
//
//=======================================================================
#include "debugMsg.h"

#define DPRINTF_2(a, b) { char text[128]; sprintf(text, a, b); dprint->String(text); }
#define DPRINTF_3(a, b, c) { char text[128]; sprintf(text, a, b, c); dprint->String(text); }
#define DPRINTF_4(a, b, c, d) { char text[128]; sprintf(text, a, b, c, d); dprint->String(text); }

#include <StorageKit.h>		// needed for alle BPath, BEntry stuff
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "RfsvServer.h"
#include "mc_rfile.h"
#include "p_file.h"
#include "../messages.h"

/*===========================================================================
/	to walk to our mime-declarations, we need a mime-structure
/============================================================================*/

struct ext_mime {
	char *extension;
	char *mime;
};

/*===========================================================================
/	We have to declare each file with an MIME-Type. 0,0 declares the end of the list
/============================================================================*/

static struct ext_mime mimes[] = {
	{ ".agn", "application/x-vnd.mp.psion-agn" },	/* The AGENDA File		*/

	{ ".app", "application/x-vnd.mp.psion-app" },	/* executable files		*/
	{ ".img", "application/x-vnd.mp.psion-img" },
	{ ".opa", "application/x-vnd.mp.psion-opa" },
	{ ".opl", "application/x-vnd.mp.psion-opl" },
	{ ".opo", "application/x-vnd.mp.psion-opo" },

	{ ".dbf", "application/x-vnd.mp.psion-dbf" },	/* Database file		*/
	{ ".pic", "application/x-vnd.mp.psion-pic" },	/* picture file			*/
	{ ".spr", "application/x-vnd.mp.psion-spr" },	/* Spreadsheet			*/

	{ ".jot", "application/x-vnd.mp.psion-jot" },	/* Jotter Database file	*/
	{ ".wld", "application/x-vnd.mp.psion-wld" },	/* World File			*/

	{ ".wrd", "text/x-vnd.mp.psion-wrd" },			/* Word File			*/
	{ ".txt", "text/x-vnd.mp.psion-txt" },			/* Text File			*/

	{ ".wve", "audio/x-vnd.mp.psion-wve" },			/* Sound File			*/

	{ ".grp", "application/x-vnd.mp.psion-grp" },	/* Groupdefinition		*/
	{ "",     "application/x-vnd.mp.psion"     },	/* every other file!	*/

	{ 0, 0 }
};

//===========================================================================
// The constructor of the RFSV-Server searches the Ports and get the correct 
// interneal CID (channel ID)
//
RfsvServer::RfsvServer( CONFIG_VAR *config ) {
	int32	message;
	int 	i;
	char	text[50];
	
	if (config->debug) dprint = new DebugMsg( config->debugFile, "RFSV-Server");
	else dprint = new DebugMsg( NULL, NULL );

	pid_i = find_port("SYS$RFSV");		// This port receives all incoming messages
	pid_o = find_port("WritePsion");	// This port is used to send all messages to the PSION

	for (i = 0; i < MAX_OPEN_FILES; i++) {	// Setup all open_files
		open_files[i].aktiv = false;
		open_files[i].fd = i+1;
	}

	if (pid_o < B_NO_ERROR || pid_i < B_NO_ERROR) exit_thread(pid_o);	// ports not found, then leave the thread. "break" should also be possible!

	for (i = 0; i < 5; i++) {						// we have to try it five times, if there are problems.
		read_port(pid_i, &message, NULL, 0);
		if ((message & CMD_MASK) == PSION_START) {	// The first message must be a PSION_START-Message
			cid = message & CID_MASK;
			break;									// for-schleife verlassen
		} else {
			sprintf(text, "Wrong message %lx", message);
			dprint->String(text);
		}
	}

	if (i < 5) RunServer();		// Intiialising done, then start the Server
	else dprint->String( "5 unidentified messages arrived! Can't start Server!" );
}


//=====================================================================
// Starting the RFSV-Server.
//
void RfsvServer::RunServer( void ) {

	unsigned char	*buffer, *b;
	unsigned char	*send_buffer = NULL;	// Der Buffer zum senden kann immer unterschiedlich lang sein!
	unsigned char   outb[256];
	char			*c, *c2, *c3;
	int				ergebnis, typ;
	int32			message;
	ssize_t 		p_len, i, send_len;
	BPath			*bp;
	BEntry			*bent;

	int				word1, word2;
	long			long1, long2;
	char			*str1, *str2;
	
	dprint->String( "Server is running and waiting for messages!" );
	
	do {
		p_len = port_buffer_size(pid_i);			// Wie groß ist die Nachricht, die ankommt?
		if (p_len < B_NO_ERROR) {
			dprint->String( "Port was deleted! Starting Servershutdown!" );
			break;	// Port wurde gelöscht?
}
		buffer = (unsigned char *)malloc(p_len);	// Speicher für die Nachricht reservieren.
		if (buffer != NULL) {
			read_port(pid_i, &message, buffer, p_len);

			dprint->Buffer( buffer, p_len, "==>> Bytes read:" );

			send_len = -1;
			outb[0] = 1;				// Anzahl der folgenden Frames?
			b = &outb[7];				// Bei Byte 7 beginnt der Inhalt der Daten
			
			typ = BytesToWord(buffer);
			word1 = BytesToWord(&buffer[4]);	// normally buffer[4] is used as word!
			
			switch ( typ ) {
				case RF_FOPEN:
					send_buffer = (unsigned char *) malloc (2 + MIN_SEND_BUFFER); // fd wird geliefert
					send_len = 2;
					
					ergebnis = rf_fopen((char *)&buffer[6], word1);	// filename, mode
					
					if (ergebnis > 0) {
						WordToBytes(ergebnis, &send_buffer[MIN_SEND_BUFFER]); // Filedeskriptor kopieren.
						ergebnis = 0;
					} else send_len = 0;
					break;
				case RF_FCLOSE:
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;
					
					ergebnis = rf_fclose( word1 );	// Welchen Kanal schließen?
					break;
				case RF_FREAD:
					word2 = BytesToWord(&buffer[6]);
					
					send_buffer = (unsigned char *)malloc(word2 + MIN_SEND_BUFFER);		// Speicher für den angeforderte Größe + Steuerdaten reservieren!
					ergebnis = rf_fread(word1, word2, &send_buffer[MIN_SEND_BUFFER]);	// Welchen Kanal lesen und wieviel?
					if (ergebnis > 0) {
						send_len = ergebnis;
						ergebnis = 0;
					} else send_len = 0;
					break;
				case RF_FDIRREAD:
					send_buffer = (unsigned char *)malloc(400 + MIN_SEND_BUFFER);			// Speicher für den angeforderte Größe + Steuerdaten reservieren!
					ergebnis = rf_dirread(word1, &send_buffer[MIN_SEND_BUFFER], 400);	// 590 Byte mit Directoryinfos füllen
					if (ergebnis > 0) {
						send_len = ergebnis;
						ergebnis = 0;
					} else send_len = 0;
					break;
				case RF_FDEVICEREAD:		// Info über die verschiedenen Devices erwünscht
					ergebnis = rf_deviceread(word1, b);
					if (ergebnis > 0) {
						send_len = ergebnis;
						ergebnis = 0;
					} else {
						send_len = 0;
					}
					break;
				case RF_FWRITE:		// Schreiben der Daten in ein File
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;
					ergebnis = rf_write(word1, BytesToWord(&buffer[2])-2, &buffer[6]);

					break;
				case RF_FSEEK:		// Positionieren des Filepointers
					long1 = BytesToLong(&buffer[6]);
					word2 = BytesToWord(&buffer[10]);

					send_buffer = (unsigned char *) malloc (4 + MIN_SEND_BUFFER);
					send_len = 4;
					
					long2 = rf_fseek(word1, long1, word2);
					if (long2 >= 0) {
						LongToBytes(long2, &send_buffer[MIN_SEND_BUFFER]);
						ergebnis = 0;
					} else {
						ergebnis = long2;
						send_len = 0;
					}
					break;
				case RF_FFLUSH:		// Schreiben abschliessen und speicher auf Diskette schreiben!
#ifdef DPRINTF_2
DPRINTF_2("RF_FFLUSH 0x%04x", word1);
#endif
					
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;
					ergebnis = 0;	// Wir machen hierbei nichts!

					break;
				case RF_FSETEOF:		// Ende Der Datei auf den übergebenen Wert setzen
					long1 = BytesToLong(&buffer[6]);
					
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;
					ergebnis = rf_fseteof(word1, long1);
					
					break;
				case RF_RENAME:
					str1 = (char *)&buffer[4];
					str2 = (char *)&buffer[5+strlen(str1)];

					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;

					ergebnis = rf_rename(str1, str2);
					break;
				case RF_DELETE:
					str1 = (char *)&buffer[4];
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;

					ergebnis = rf_delete(str1);
					break;
				case RF_FINFO:
					str1 = (char *)&buffer[4];
#ifdef DPRINTF_2
DPRINTF_2("RF_FINFO >%s<", str1);
#endif

					send_buffer = (unsigned char *) malloc (16 + MIN_SEND_BUFFER);
					send_len = 16;

					if (str1[strlen(str1)-1] == '/') {
						ergebnis = E_FILE_NAME;	// Kein erlaubter Filename, wenn am Ende "/" !
					} else {
						str2 = find_filename(str1);
						bent = new BEntry(str2);
						ergebnis = get_status(bent, &send_buffer[MIN_SEND_BUFFER]);
						delete bent;
						free(str2);
					}
					if (ergebnis >= 0) {
						ergebnis = 0;
					} else {
						send_len = 0;
					}
					break;
				case RF_SFSTAT:
					word2 = BytesToWord(&buffer[6]);
					str1 = (char *)&buffer[8];

					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;

					ergebnis = rf_sfstat(str1, word1, word2);
					break;
				case RF_MKDIR:
					str1 = (char *)&buffer[4];
					send_buffer = (unsigned char *) malloc (0 + MIN_SEND_BUFFER);
					send_len = 0;

					ergebnis = rf_mkdir(str1);
					break;
				case RF_PARSE:		// Überprüft lediglich eine Pfad auf sinnvoller Struktur
									// und füllt die Längenwerte. macht das aber sehr kompliziert!

					c = (char *)&buffer[4];
					c2 = c+(strlen(c)+1);
					c3 = c2+(strlen(c2)+1);

					ergebnis = rf_parse(c, c2, c3, b);
					send_len = strlen((char *)&b[6]) + 7;
					break;
				case RF_STATUSDEVICE:
					ergebnis = rf_statusdevice((char *)&buffer[5], b);	// ohne fuehrenden "/"
					if (ergebnis > 0) {
						send_len = ergebnis;
						ergebnis = 0;
					} else {
						send_len = 0;
					}
					break;
				case RF_PATHTEST:
#ifdef DPRINTF_2
DPRINTF_2("RF_PATHTEST >%s<", &buffer[4]);
#endif

					bp = new BPath((char *)&buffer[4], NULL, true);
					if (bp->Path() == NULL) ergebnis = -1;
					else ergebnis = 0;
					send_len = 0;
					delete bp;
					break; 
				case RF_STATUSSYSTEM:
					dprint->String( "RF_STATUSSYSTEM" );

					// Der PSION möchte Informationen über das Filesystem erhalten
					send_buffer = (unsigned char *) malloc (32 + MIN_SEND_BUFFER);
					send_len = 32;
					b = &send_buffer[MIN_SEND_BUFFER];
					
					WordToBytes(RFSV_VERSION, b);		// Versionsnummer für Erweiterungen
					WordToBytes(P_FSYSTYPE_HIER, &b[2]);	// 0 = flaches Filesystem; 1 = Hierarchisches FS
					WordToBytes(false, &b[4]);	// Filesystem erlaubt keine Format-Befehle!
					memset(&b[6], 0xff, 26);	// Reserve auf ff setzen!
					for (i=0; i < 13; WordToBytes(0, &b[6+((i++) *2)])) ;
					ergebnis = 0;					// Kein Fehler
        	        break;
				case RF_CHANGEDIR:
					str1 = (char *)&buffer[6];
					str2 = (char *)&buffer[7+strlen(str1)];
					if (word1 != P_CD_SUBDIR) str2--;	// Auf das 0-Byte von str1 setzen!

					// Filelänge + 6, weil REM:: noch davor kommt und das 0-Byte danach!
					send_buffer = (unsigned char *) malloc (B_FILE_NAME_LENGTH + 6 + MIN_SEND_BUFFER);
					
					ergebnis = rf_changedir(word1, str1, str2, (char *)&send_buffer[MIN_SEND_BUFFER]);
					if (ergebnis >= 0) {
						send_len = strlen((char *)&send_buffer[MIN_SEND_BUFFER])+1;
						ergebnis = 0;
					} else send_len = 0;

					break;
				case RF_SFDATE:
					long1 = BytesToLong(&buffer[4]);
					str1 = (char *) &buffer[8];

					ergebnis = rf_sfdate(str1, long1);
					send_len = 0;
					break;
        	    default:
        	    	ergebnis = -1;		// error
					dprint->Buffer ( buffer, p_len, "unknown message received!" );
					break;
			}

			if (send_buffer == NULL) {		// Zwischenlösung, bis alles auf send_buffer umgestellt ist
				send_buffer = (unsigned char *) malloc (send_len + 7);
				memcpy(send_buffer, outb, send_len + 7);
			}
			if (send_len >= 0) {
	            WordToBytes(ergebnis, &send_buffer[5]);				// Ergebnis 0 = OK
    	        WordToBytes((unsigned int )send_len+2, &send_buffer[3]);	// Länge der Nachricht
        	    WordToBytes(RF_RESPONSE, &send_buffer[1]);
            	write_port(pid_o, PSION_CMD + cid, &send_buffer[1], send_len+6);

				dprint->Buffer( send_buffer, send_len+7, "<<== Bytes sent:" );
			}
			
			if (send_buffer != NULL) {	// Wenn Sendepuffer reserviert wurde, dann auch wieder freigeben!
				free(send_buffer);
				send_buffer = NULL;
			}
			free(buffer);						// Empfangspuffer wird immer reserviert!
		}
	} while (true);
}


//=====================================================================
// Open a file, a folder or all devices. The answer is a filehandle
//
int RfsvServer::rf_fopen(const char *name, int status) {
	BPath		*bp;
	int 		i, erg;
	uint32		mode;
	char		*str;

#ifdef DPRINTF_3
DPRINTF_3("RF_FOPEN >%s< 0x%04x", name, status);
#endif
	
	erg = E_FILE_ALLOC;
	for (i = 0; i < MAX_OPEN_FILES; i++) {
		if (!open_files[i].aktiv) {		// Freien Eintrag gefunden!
			switch (status & P_FMT_MASK) {
				case P_FDEVICE:
					open_files[i].aktiv = true;
					open_files[i].type = OP_BVOL;
					open_files[i].pointer.bvolume = new BVolumeRoster();
					erg = open_files[i].fd;
					break;
				case P_FDIR:
					bp = new BPath(name);
					bp->GetParent(bp);
					open_files[i].aktiv = true;
					open_files[i].type = OP_BDIR;
					open_files[i].pointer.bdir = new BDirectory(bp->Path());
					erg = open_files[i].fd;
					break;
				case P_FTEXT:			// Die Unterschiede sind mir derzeit nicht bekannt!
				case P_FSTREAM:
				case P_FSTREAM_TEXT:
					switch(status & P_FSER_MASK) {
						case P_FOPEN:		mode = B_READ_WRITE; break;
						case P_FCREATE:		mode = B_READ_WRITE | B_CREATE_FILE; break;
						case P_FREPLACE:	mode = B_READ_WRITE | B_CREATE_FILE | B_ERASE_FILE; break;
						case P_FAPPEND:		mode = B_READ_WRITE | B_OPEN_AT_END; break;
				//		case P_FUNIQUE 0x0004 /* Unique file open */ ????
						default: mode = B_READ_ONLY; break;
					}
					open_files[i].aktiv = true;
					open_files[i].type = OP_BFILE;
					
					str = find_filename(name);
					open_files[i].pointer.bfile = new BFile(str, mode);
					free(str);
					
					switch (open_files[i].pointer.bfile->InitCheck()) {
						case B_NO_ERROR:
								if (mode & B_CREATE_FILE) {

									struct ext_mime *p;
									size_t namelen, ext_len;

									namelen = strlen(name);

									for (p = mimes; p->extension; p++) {
										ext_len = strlen(p->extension);

										if (namelen <= ext_len) continue;
										if (strcasecmp(&name[namelen - ext_len], p->extension) == 0) break;
									}

									if (p->mime != NULL) {
										BNodeInfo *bnode = new BNodeInfo(open_files[i].pointer.bfile);
										bnode->SetType(p->mime);
										delete bnode;
									}
								}
								erg = open_files[i].fd; 
								break;
						default: erg = E_FILE_ACCESS; break;
					}
					break;
				default:
					printf("Unbekanntes Öffnen %04x\n", (status & P_FMT_MASK));
					break;
			}
			break;		// for-Schleife verlassen!
		}
	}
	return(erg);
}

//=====================================================================
// Closing the filedeskriptor
//
int RfsvServer::rf_fclose( int fd ) {
	int erg = E_FILE_ALLOC;

#ifdef DPRINTF_2
DPRINTF_2("RF_FCLOSE 0x%04x", fd);
#endif

	if (open_files[fd-1].aktiv) {
		open_files[fd-1].aktiv = false;
		erg = 0;
		switch(open_files[fd-1].type) {
			case OP_BVOL:	delete open_files[fd-1].pointer.bvolume;	break;
			case OP_BDIR:	delete open_files[fd-1].pointer.bdir;		break;
			case OP_BFILE:	delete open_files[fd-1].pointer.bfile;		break;
		}
	}
	return(erg);
}

//=====================================================================
// Reading Data from a opened file
//
int RfsvServer::rf_fread(int fd, int anz, unsigned char *buffer) {

	ssize_t erg = E_FILE_WRITE;

#ifdef DPRINTF_3
DPRINTF_3("RF_FREAD 0x%04x Anz:%d", fd, anz);
#endif

	if (buffer == NULL) return -1;
	
	if (open_files[fd-1].aktiv) {
		if (open_files[fd-1].type == OP_BFILE) {
			erg = open_files[fd-1].pointer.bfile->Read(buffer, anz);
		}
	}
	if (erg == 0) return (E_FILE_EOF);
	else return erg;
}

/* Füllt einen Buffer mit den Directoryeinträgen!		*/
/* len gibt die Größe des Buffers an.					*/
/* Liefert die Länge des gefüllten Buffers				*/

int RfsvServer::rf_dirread(int fd, unsigned char * buffer, size_t len) {
	BEntry				bent;
	BPath				bpath;
	unsigned char *		b;

#ifdef DPRINTF_2
DPRINTF_2("RF_FDIRREAD 0x%04x", fd);
#endif
	
	if (!open_files[fd-1].aktiv) return(-1);
	if (open_files[fd-1].type != OP_BDIR) return(-1);
	
	b = &buffer[2];
	while (open_files[fd-1].pointer.bdir->GetNextEntry(&bent) >= B_NO_ERROR) {
		bent.GetPath(&bpath);
		get_status(&bent, b);
		bent.GetName((char *)&b[16]);

		b = &b[17] + strlen((char *)&b[16]);	// 0-Byte muß stehen bleiben!
		if ((size_t )(b-buffer) >= len) break;
	}
	if (b == &buffer[2]) return(E_FILE_EOF);	// Kein Eintrag mehr!

	WordToBytes((b-buffer)-2, buffer);
	return (b-buffer);
}

/* Holt die nächste Information aus BVolume	*/

int RfsvServer::rf_deviceread(int fd, unsigned char * buffer) {
	BVolume			bvol;
	status_t		erg;
	char			device[128];

#ifdef DPRINTF_2
DPRINTF_2("RF_FDEVICEREAD 0x%04x", fd);
#endif

	memset((char *)buffer, 0xff, 64);
	if (!open_files[fd-1].aktiv) return(-1);
	if (open_files[fd-1].type != OP_BVOL) return(-1);
	
	while ((erg = open_files[fd-1].pointer.bvolume->GetNextVolume(&bvol)) == B_NO_ERROR) {
		bvol.GetName(device);
		if (strlen(device) != 0) {
			get_dev_stat(&bvol, buffer, (char *)&buffer[64]);			// Statusbuffer füllen
			return (64 + 1 + strlen((char *)&buffer[64])); // 0-Byte nicht vergessen zum mitzaehlen!
		}
	}
	return (E_FILE_EOF);
}

// Schreibt Daten in den übergebenen FD

int RfsvServer::rf_write(int fd, int anz, const unsigned char *buffer) {

	ssize_t erg = E_FILE_WRITE;

#ifdef DPRINTF_3
DPRINTF_3("RF_FWRITE 0x%02x 0x%04x", fd, anz);
#endif
	
	if (open_files[fd-1].aktiv) {
		if (open_files[fd-1].type == OP_BFILE) {
			erg = open_files[fd-1].pointer.bfile->Write(buffer, anz);
		}
	}
	if (erg < 0) {
		switch (erg) {
			case B_DEVICE_FULL: return (E_FILE_FULL); break;
			case B_NOT_ALLOWED: return (E_FILE_RDONLY); break;
			default:			return erg;
		}
	} else return(0);
}

// Macht den fseek für eine Datei

int32 RfsvServer::rf_fseek(int fd, long anz, int mode) {
	int m = SEEK_SET;

#ifdef DPRINTF_4
DPRINTF_4("RF_FSEEK 0x%04x 0x%08lx 0x%04x", fd, anz, mode);
#endif
	
	switch(mode) {
		case P_FABS:	m = SEEK_SET; break;
		case P_FEND:	m = SEEK_END; break;
		case P_FCUR:	m = SEEK_CUR; break;
		case P_FRSENSE: /* Sense the record position */
		case P_FRSET: /* Set the record position */
		case P_FREWIND: /* Rewind a text file */
			printf("Dont Know, what to do with mode: %d\n", mode);
			return -1;
	}
	if (open_files[fd-1].aktiv) {
		if (open_files[fd-1].type == OP_BFILE) {
			open_files[fd-1].pointer.bfile->Seek(anz, m);
			return (open_files[fd-1].pointer.bfile->Position());
		}
	}
	return -1;
}

//====================================================================================
// Is used to set the end of the file. If it is greater then the file, it will be
// expanded, if it is smaller, then the file is shortened!
//
int RfsvServer::rf_fseteof(int fd, long laenge) {

	ssize_t erg = E_FILE_WRITE;

#ifdef DPRINTF_3
DPRINTF_3("RF_FSETEOF 0x%04x 0x%08lx", fd, laenge);
#endif

	if (open_files[fd-1].aktiv) {
		if (open_files[fd-1].type == OP_BFILE) {
			erg = open_files[fd-1].pointer.bfile->SetSize(laenge);
		}
	}
	return 0;
}

// verschiebt eine Datei/Ordner

int RfsvServer::rf_rename(const char *filename_alt, const char *filename_neu) {
	BEntry		bent_alt, bent_neu;
	BDirectory	dir;
	char		leaf[128];
	status_t	erg;
	char		*fa, *fn;

#ifdef DPRINTF_3
DPRINTF_3("RF_RENAME >%s< >%s<", filename_alt, filename_neu);
#endif
	
	fa = find_filename(filename_alt);
	fn = find_filename(filename_neu);

	bent_alt.SetTo(fa);
	bent_neu.SetTo(fn);
	if (bent_alt.Exists()) printf("ALT Existiert!\n");
	if (bent_neu.Exists()) printf("NEU existiert!\n");
	if (bent_alt.Exists() && !bent_neu.Exists()) {
		bent_neu.GetParent(&dir);
		bent_neu.GetName(leaf);
		erg = bent_alt.MoveTo(&dir, leaf);
		switch (erg) {
			case B_NO_ERROR: return 0;
			case B_NO_INIT:printf("Ergebnis = B_NO_INIT\n");break;
			case B_ENTRY_NOT_FOUND:printf("Ergebnis = B_ENTRY_NOT_FOUND\n");break;
			case B_FILE_EXISTS:printf("Ergebnis = B_FILE_EXISTS\n");break;
			case B_BUSY:printf("Ergebnis = B_BUSY\n");break;
			default: printf("Ergebnis = %lx\n", erg);break;
		}
	}
	return (E_FILE_NXIST);
}

// verschiebt eine Datei/Ordner in den Mülleimer

int RfsvServer::rf_delete(const char *filename) {
	BEntry		bent;
	BPath		path;
	BDirectory	dir;
	status_t	erg;
	char		*file;

#ifdef DPRINTF_2
DPRINTF_2("RF_DELETE >%s<", filename);
#endif
	
	file = find_filename(filename);
	bent.SetTo(file);
	free(file);
	
	if (bent.Exists()) {
  		find_directory(B_TRASH_DIRECTORY, &path, true);	// Wo ist der Mülleimer
		dir.SetTo(path.Path());
		erg = bent.MoveTo(&dir, NULL, true);
		switch (erg) {
			case B_NO_ERROR: return 0;
			case B_NO_INIT:printf("Ergebnis = B_NO_INIT\n");break;
			case B_ENTRY_NOT_FOUND:printf("Ergebnis = B_ENTRY_NOT_FOUND\n");break;
			case B_FILE_EXISTS:printf("Ergebnis = B_FILE_EXISTS\n");break;
			case B_BUSY:printf("Ergebnis = B_BUSY\n");break;
		}
	}
	return (E_FILE_NXIST);
}

// Diese Funktion muß die entsprechenden Flags setzen!
// Es werden alle Bits bearbeitet, die in MASK gesetzt wurden!
// Problem ergibt sich wieder mit groß/kleinschreibung des Filenamens!

int RfsvServer::rf_sfstat(const char *filename, int bit, int mask) {
	BNode node;
	mode_t permission;
	char	*file;
	int erg = E_FILE_NXIST;

#ifdef DPRINTF_4
DPRINTF_4("RF_SFSTAT %04x %04x >%s<", bit, mask, filename);
#endif

	file = find_filename(filename);
	if (node.SetTo(file) >= B_NO_ERROR) {
		permission = S_IRUSR;
		if ((mask & P_FAWRITE) && (bit & P_FAWRITE)) permission |= S_IWUSR;
		if ((mask & P_FAMOD) && (bit & P_FAMOD)) node.SetModificationTime(real_time_clock());
		erg = 0;
	}
	free(file);
	return erg;	
}

// erstellt ein Verzeichnis

int RfsvServer::rf_mkdir(const char *dirname) {
	BDirectory bdir, neues_bdir;
	status_t	erg;
	char		*c;

#ifdef DPRINTF_2
DPRINTF_2("RF_MKDIR >%s<", dirname);
#endif
	
	c = strrchr(dirname, '/');					// Am Ende des Pfades darf kein "/" stehen, sonst kann der Ordner nicht angelegt werden!
	if (c != NULL && c[1] == '\0') *c = '\0';	// Daher entfernen wir einen evtl. "/" vom Ende. Ich wunder mich, daß das trotz
												// const compiliert werden kann!
	erg = bdir.CreateDirectory(dirname, &neues_bdir);
	if (erg < B_NO_ERROR) return (E_FILE_ACCESS);
	else return 0;
}

/* Überprüft ob der Dateiname ein gültiger Dateiname ist	*/
/* REM::/BeDisk/ordner1/ordner2/datei*.ex*"                 */
/* --5-!---8---!------16-------!--6--!-4-!					*/
/* Filesystem und Device können weggelassen werden, dann ist das Standardlaufwerk einzusetzen */

int RfsvServer::rf_parse(char *file1, char *file2, char *file3, unsigned char *werte) {
	char	*f[3];
	char	*ergebnis;
	BEntry		bent;
	int i, len;

#ifdef DPRINTF_4
DPRINTF_4("RF_PARSE >%s< >%s< >%s<", file1, file2, file3);
#endif

	f[0] = file1;
	f[1] = file2;
	f[2] = file3;
	memset(werte, 0x00, 6);
	ergebnis = (char *)&werte[6];
	for (i = 0; i < 3; i++) if ((len = copy_filesystem(f[i], ergebnis)) > 0) break;	// Filesystem suchen
	if (i >= 3) {	// Kein Filesystem übergeben, also Standard eintragen!
		strcpy(ergebnis, "REM::");
		ergebnis += 5;
		werte[0] = 5;
	} else {
		ergebnis += len;
		werte[0] = len;
		f[i] += len;
		for (i++ ; i < 3; i++) {						// aus allen anderen Strings das Filesystem entfernen!
			len = copy_filesystem(f[i], NULL);
			if (len > 0) f[i] += len;
		}
	}

	for (i = 0; i < 3; i++) if ((len = copy_device(f[i], ergebnis)) > 0) break;	// Device suchen
	if (i >= 3) {	// Kein Device übergeben, also Standard eintragen!

		strcpy(ergebnis, "/boot");
		ergebnis += 5;
		werte[1] = 5;
	} else {	// Hier ist noch ein fehler enthalten!
		ergebnis += len;
		werte[1] = len;
		f[i] += len;
		for (i++ ; i < 3; i++) {						// aus allen anderen Strings das Filesystem entfernen!
			len = copy_device(f[i], NULL);
			if (len > 0) f[i] += len;
		}
	}

	for (i = 0; i < 3; i++) if ((len = copy_path(f[i], ergebnis)) > 0) break;	// Pfad suchen
	if (i >= 3) {	// Kein Pfad übegeben, also Standard eintragen, d.h. kein Pfad
		strcpy(ergebnis, "/");
		ergebnis++;
		werte[2] = 1;
	} else {
		ergebnis += len;
		werte[2] = len;
		f[i] += len;
		for (i++ ; i < 3; i++) {						// aus allen anderen Strings das Filesystem entfernen!
			len = copy_path(f[i], NULL);
			if (len > 0) f[i] += len;
		}
	}

	for (i = 0; i < 3; i++) if ((len = copy_filename(f[i], ergebnis)) > 0) break;	// Filename suchen
	if (i >= 3) {	// Kein Filename übegeben, also nix eintragen und wert3 auf 0 setzen!!
		werte[3] = 0;
		werte[5] = 0; // dann auch kein Wildcard enthalten!
	} else {
		if (strchr(ergebnis, '*') != NULL)  werte[5] |= (P_PWILD_ANY | P_PWILD_NAME); 
		ergebnis += len;
		werte[3] = len;
		f[i] += len;
		for ( ; i < 3; i++) {						// aus allen anderen Strings den Filenamen entfernen!
			len = copy_filename(f[i], NULL);
			if (len > 0) f[i] += len;
		}
	}
	
	for (i = 0; i < 3; i++) {
		if (*f[i] == '\0') continue;
		strcpy(ergebnis, f[i]);
		if (strcmp(f[i], ".*") != 0) break;
	}
	werte[4] = strlen(ergebnis);
	if (strchr(ergebnis, '*') != NULL)  werte[5] |= (P_PWILD_ANY | P_PWILD_EXT);
	return 0;
}

// Überprüft, ob im String "in" ein Filesystem enthalten ist. 
// Wenn ja, dann wird dieser String nach filesys kopiert (wenn filesys != NULL)
// Der Rückgabewert ist die Anzahl der Zeichen die das Filesystem hat.
// Wenn kein FS gefunden, dann wird -1 zurückgegeben!

int copy_filesystem(const char *in, char *filesys) {
	if (strncmp(in, "REM::", 5) == 0) {
		if (filesys != NULL) strcpy(filesys, "REM::");
		return 5;
	} else return -1;
}

// Überprüft, ob im String "in" ein Device enthalten ist. 
// Wenn ja, dann wird dieser String nach device kopiert (wenn device != NULL)
// Der Rückgabewert ist die Anzahl der Zeichen die das Filesystem hat.
// Wenn kein FS gefunden, dann wird -1 zurückgegeben!

// Probleme, wenn Folder nicht existiert!

int copy_device(const char *in, char *device) {
	entry_ref		ref;
	BVolume			bvol;
	BDirectory		bdir;
	BEntry			bent;
	BPath			bpath;
	char			device_name[B_FILE_NAME_LENGTH], *d, *c;
	
	strcpy(device_name, in);
	d = strchr(device_name, '*');
	if (d != NULL) *d = '\0';

	d = (device == NULL) ? device_name : device;	// where sould i copy the filename?
	
	if (*device_name == '/') {						// Continue only, if the string starts with a "/"

		while (bent.SetTo(device_name) == B_ENTRY_NOT_FOUND) {	// dows the file exist?
			c = strrchr(device_name, '/');		// if not, then find the last slash
			if (c != NULL) *c = '\0';			// delete this folder from our searchingpath
			else return -1;						// if the last path was deleted, then there was no device found
		} 										// try to set the Entry with one folder shorter.
		
		bdir.SetTo(device_name);				// And setup the directory to "device_name"
		if (!bdir.IsRootDirectory()) {			// if it is no RootDir, then find the rootDir!
			bent.GetRef(&ref);					// get the entry_ref to find the dev_t
			bvol.SetTo(ref.device);				// Find the coresponding Volume
			if (bvol.InitCheck() == B_OK) {		// Volume found
				bvol.GetRootDirectory(&bdir);	// ask for the Root-Directory
				bdir.GetEntry(&bent);			// create an Entry representing the Root-Dir
			} else return -1;
		}
		bent.GetPath(&bpath);			// find the complete Path
		strcpy(d, bpath.Path());		// and copy it into the correct device-String
		return(strlen(d));				// return the length of the string
	} 
	return -1;
}

// Überprüft, ob im String "in" ein path enthalten ist. 
// Wenn ja, dann wird dieser String nach path kopiert (wenn path != NULL)
// Der Rückgabewert ist die Anzahl der Zeichen die der Path hat.
// Wenn kein path gefunden, dann wird -1 zurückgegeben!

int copy_path(const char *in, char *path) {
	char *c;
	size_t len;
	
	c = strrchr(in, '/');
	if (c != NULL) {
		len = (c-in)+1;
		if (path != NULL) {
			*path = '\0';
			strncat(path, in, len);
		}
		return len;
	}
	return -1;
}

// Überprüft, ob im String "in" ein filename enthalten ist. 
// Wenn ja, dann wird dieser String nach filename kopiert (wenn filename != NULL)
// Der Rückgabewert ist die Anzahl der Zeichen die der filename hat.

int copy_filename(const char *in, char *filename) {
	char *c;
	size_t len;
	
	c = strchr(in, '.');
	if (c != NULL) len = (c-in);
	else len = strlen(in);
	
	if (filename != NULL) {
		*filename = '\0';
		strncat(filename, in, len);
	}
	return len;
}



/* Holt die Information zu einem bestimmten Device */

int RfsvServer::rf_statusdevice(char *device, unsigned char *buffer) {
	BVolumeRoster	brost;
	BVolume			bvol;
	char			device_name[128];
	char 			*c;
	status_t		erg;

#ifdef DPRINTF_2
DPRINTF_2("RF_STATUSDEVICE >%s<", device);
#endif

	if ((c = strchr(device, '/')) != NULL) *c = '\0';
	
	while ((erg = brost.GetNextVolume(&bvol)) == B_NO_ERROR) {
		bvol.GetName(device_name);
		if (strcmp(device_name, device) == 0) {
			get_dev_stat(&bvol, buffer, NULL);		// Statusbuffer füllen aber Pfad uninteressant
			return (64); 							// Struktur ist 64 Byte groß
		}
	}
	if (strcmp(device, "boot") == 0) {				// Der PSION will was vom Bootdevice wissen
		brost.GetBootVolume(&bvol);
		get_dev_stat(&bvol, buffer, NULL);
		return(64);
	}
	
	return (E_FILE_EOF);
}

// gibt als Information zurück, wie ein bestimmter Pfad als Ergebnis aussehen würde.

int RfsvServer::rf_changedir(int art, const char *start, const char *zusatz, char *ergebnis) {
	char		*filename, *c;
	char		f[B_FILE_NAME_LENGTH];
	entry_ref	ref;					// entry_ref needed for P_CD_ROOT
	BEntry 		bent;					// 
	BVolume		bvol;					// We need the Volume.
	BDirectory	bdir;
	BPath		bpath;

#ifdef DPRINTF_4
DPRINTF_4("RF_CHANGEDIR: 0x%04x >%s< >%s<", art, start, zusatz);
#endif

	*f = '\0';
	filename = strrchr(start, '/');
	if (filename != NULL ) {
		filename++;								// last part is the filename, we have to remember it
		strncat(f, start, filename-start-1);	// but we also need the path
	} else strcpy(f, start);					// there was only a path available

	bdir.SetTo(f);

	switch(art) {
		case P_CD_ROOT:							// CD to the corresponding Rootdirectory
			bdir.GetEntry(&bent);				// Create a entry
			if (!bdir.IsRootDirectory()) {		// We have to find the Rootdirectory!
				bent.GetRef(&ref);					// get the entry_ref to find the dev_t
				bvol.SetTo(ref.device);				// Find the coresponding Volume
				if (bvol.InitCheck() == B_OK) {		// Volume found
					bvol.GetRootDirectory(&bdir);	// ask for the Root-Directory
					bdir.GetEntry(&bent);			// create an Entry representing the Root-Dir
				}
			} 
			bent.GetPath(&bpath);					// change it to an Path
			break;
		case P_CD_PARENT:
			if (!bdir.IsRootDirectory()) {			// lets look if our folder is the rootdirectory
				c = strrchr(f, '/');				// otherwise go up one level!
				if (c != NULL) {
					*c = '\0';
					bpath.SetTo(f);
				}
			} else return E_FILE_NXIST;				// we are on the top!
			break;
		case P_CD_SUBDIR:
			strcat(f, "/");
			strcat(f, zusatz);
			bpath.SetTo(f);
			break;
	}
	strcpy(ergebnis, "REM::");				// alway start with "REM::"
	strcpy(&ergebnis[5], bpath.Path());		// copy the complete Pathname.
	strcat(&ergebnis[5], "/");
	if (filename != NULL) strcat(&ergebnis[5], filename);
	return 0;
}

// Diese Funktion setzt das Erstellungsdatum!
// Problem ergibt sich wieder mit groß/kleinschreibung des Filenamens!

int RfsvServer::rf_sfdate(const char *filename, long secs) {
	BNode node;
	char	*file;
	int 	erg = E_FILE_NXIST;

#ifdef DPRINTF_3
DPRINTF_3("RF_SFDATE: %ld, >%s<", secs, filename);
#endif
	
	file = find_filename(filename);
	if (node.SetTo(filename) >= B_NO_ERROR) {
		node.SetCreationTime(secs);
		erg = 0;
	}
	free(file);
	return erg;
}


/* Füllt den Status des Devices in den Buffer	*/

void get_dev_stat(BVolume *bvol, unsigned char *buffer, char *dir) {
	BDirectory		bdir;
	BEntry			bent;
	BPath			bpath;
	int stat;
	char			volume[128];
	
	WordToBytes(RFSV_VERSION, buffer);	
	stat = P_FMEDIA_HARDDISK;
	if (bvol->IsReadOnly()) stat = P_FMEDIA_WRITEPROTECTED;
	else if (bvol->IsRemovable()) stat = P_FMEDIA_FLOPPY;
	else if (bvol->IsPersistent()) stat = P_FMEDIA_HARDDISK;
	WordToBytes(stat, &buffer[2]);	// Mediatyp
	WordToBytes(bvol->IsRemovable(), &buffer[4]);	// Kann das Medium entfernt werden?
	LongToBytes(bvol->Capacity(), &buffer[6]);	// Größe in Bytes
	LongToBytes(bvol->FreeBytes(), &buffer[10]);	// Freier Speicherplatz

	bvol->GetName(volume);
	buffer[14] = '\0';
	strncat((char *)&buffer[14], volume, P_VOLUMENAME-1);	// Devicenamen verwenden

	if (dir != NULL) {										// Pfad feststellen!
		if (bvol->GetRootDirectory(&bdir) >= B_NO_ERROR) {
			if (bdir.GetEntry(&bent) >= B_NO_ERROR) {
				if (bent.GetPath(&bpath) >= B_NO_ERROR) {
					strcpy(dir, bpath.Path());				// Laufwerksnamen kopieren
				}
			}
		}
	}
}

/* holt den Status der Datei/Directory */

int get_status(BEntry *bent, unsigned char *buffer) {
	struct stat			st;
	int					status;
	BEntry				parent;
	char name[B_FILE_NAME_LENGTH]; 
	
	WordToBytes(RFSV_VERSION, buffer);

	bent->GetParent(&parent);
	if (!parent.Exists()) return(E_FILE_DIR);	// Directory existiert nicht!
	if (!bent->Exists()) return(E_FILE_NXIST);	// File gibt es nicht
	bent->GetStat(&st);
	
	parent.GetName(name);	
	if (strcmp(name, ".") == 0) return(E_FILE_NAME);
	
	status=P_FAREAD|P_FAWRITE;
	if (S_ISDIR(st.st_mode)) status |= P_FADIR;
	else status |= P_FASTREAM;
	
	WordToBytes(status, &buffer[2]);
		
	if (status & P_FADIR) LongToBytes(0, &buffer[4]);
	else LongToBytes(st.st_size, &buffer[4]);		// Größe feststellen
	LongToBytes(st.st_mtime, &buffer[8]);		// Modificationtime
		
	memset(&buffer[12], 0xff, 4);	// Spare
	return(0);
}

// Da das Filesystem des PSION nicht casesensitive arbeitet, muss zuerst der richtige Dateinamen
// gesucht werden. Dazu wird in folgenden Schritten vorgegangen:
// 1. Name vorhanden, so wie angegeben?		ja-> Name zurückgeben
// 2. uebergebener Name mit strcasecmp vergleichen.
// wird der Name nicht gefunden, so wird ein kleingeschriebener Name zurückgeben!

char *find_filename(const char *filename) {
	BDirectory	bdir;
	BEntry		bent;
	BPath		path;
	status_t	erg;
	char		*file_erg;
	char		file_pc[B_FILE_NAME_LENGTH], file_psion[B_FILE_NAME_LENGTH], *c;
	
	file_erg = (char *) malloc(B_FILE_NAME_LENGTH);
	strcpy(file_erg, filename);		// Filenamen kopieren um damit arbeiten zu können.

	bent.SetTo(file_erg);
	bent.GetName(file_psion);
	bent.GetParent(&bdir);
	while ((erg = bdir.GetNextEntry(&bent)) >= B_NO_ERROR) {
		bent.GetName(file_pc);
		if (strcasecmp(file_pc, file_psion) == 0) {
//			printf("Datei gefunden! Angefordert: %s gefunden %s\n", file_psion, file_pc);
			bent.GetPath(&path);
			strcpy(file_erg, path.Path());
//			printf("Path:org:%s\n     erg:%s\n", filename, path.Path());
			return(file_erg);
		}
	}
//	printf("\n");
	c = strrchr(file_erg, '/');
	if (c == NULL) c = file_erg;	// wenn kein "/" im Dateiname, dann den ganzen Dateinamen bearbeiten!
	for (; *c != '\0'; c++) if ((*c < 0x80) && (*c >= 0x40))  *c |= 0x20;
	return (file_erg);
}

void WordToBytes(unsigned int i, unsigned char *p) {
    *p++ = i;
    *p = (i >> 8);
}

void LongToBytes(unsigned long l, unsigned char *p) {
    *p++ = l;
    *p++ = (l >> 8);
    *p++ = (l >> 16);
    *p = (l >> 24);
}

unsigned int BytesToWord(const unsigned char *c) {
	return ((unsigned int )*c | ((unsigned int )c[1] << 8)); 
}

unsigned long BytesToLong(const unsigned char *c) {
	return ((unsigned int )*c | ((unsigned int )c[1] << 8) | ((unsigned int )c[2] << 16) | ((unsigned int )c[3] << 24)); 
}

