#define DEBUG_FILE "/var/log/rfsv-client.log"

typedef long long dr9_off_t;
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <KernelExport.h>
#include <time.h>
#include <malloc.h>
#include "fsproto.h"
#include "lock.h"
#include "cache.h"

#include <fs_info.h>		// Diese Datei enthält die Strukturdefinition von fs_info (Filesysteminfo)
//#include "dmalloc.h"
#include <TypeConstants.h>	// Wird für die MIME-Typen benötigt
#include <GraphicsDefs.h>	// for Bitmaps-file
#include "../Icons/PsionBitmaps.h"
#include "attr.h"			// Including all the Attribute-Filesystem-Functions

//zu debugzwecken!
#include "../../DebugWin/debug.h"
#ifdef DEBUG_FILE
void dbg_print(char *head, size_t len, unsigned char *buffer);
#endif
//=================================

#include "psionfs.h"
#include "../messages.h"	// Die definierten Nachrichtentypen zwischen NCP und Client
#include "mc_rfile.h"		// Das RFSV-Protokoll
#include "p_file.h"			// dito.

//=================================

void WordToBytes(unsigned int i, unsigned char *p);
void LongToBytes(unsigned long l, unsigned char *p);
unsigned int BytesToWord(const unsigned char *c);
unsigned long BytesToLong(const unsigned char *c);

vnode *search_vnode(nspace *ns, vnode_id *vnid, vnode_id parentID, const char *file, unsigned char *fileinfo);
void setup_vnode(vnode *node, unsigned char *finfo);

void find_path(nspace *ns, vnode *vn, char *pfad, size_t max_len);
int	open_node(nspace *ns, vnode *vn, int omode);
unsigned char *send_command(nspace *ns, unsigned char *buffer, size_t *receive_len);


/*  Start of fundamental (read-only) required functions */
static int		fs_mount(nspace_id nsid, const char *device, ulong flags,
						void *parms, size_t len, void **data, vnode_id *vnid);
static int		fs_unmount(void *_ns);

static int		fs_walk(void *_ns, void *_base, const char *file,
						char **newpath, vnode_id *vnid);

static int		fs_read_vnode(void *_ns, vnode_id vnid, char r, void **node);
static int		fs_write_vnode(void *_ns, void *_node, char r);
static int		fs_rstat(void *_ns, void *_node, struct stat *st);
static int		fs_open(void *_ns, void *_node, int omode, void **cookie);
static int		fs_read(void *_ns, void *_node, void *cookie, off_t pos,
						void *buf, size_t *len);
/// fs_free_cookie - free cookie for file created in open.
static int		fs_free_cookie(void *ns, void *node, void *cookie);
static int		fs_close(void *ns, void *node, void *cookie);

// fs_access - checks permissions for access.
static int		fs_access(void *_ns, void *_node, int mode);

// fs_opendir - creates fs-specific "cookie" struct that can tell where
//					we are at in the directory list.
static int		fs_opendir(void* _ns, void* _node, void** cookie);
// fs_readdir - read 1 or more dirents, keep state in cookie, return
//					0 when no more entries.
static int		fs_readdir(void *_ns, void *_node, void *cookie,
					long *num, struct dirent *buf, size_t bufsize);
// fs_rewinddir - set cookie to represent beginning of directory, so
//					later fs_readdir calls start at beginning.
static int		fs_rewinddir(void *_ns, void *_node, void *cookie);
// fs_closedir - Do whatever you need to to close a directory (sometimes
//					nothing), but DON'T free the cookie!
static int		fs_closedir(void *_ns, void *_node, void *cookie);
// fs_fee_dircookie - Free the fs-specific cookie struct
static int		fs_free_dircookie(void *_ns, void *_node, void *cookie);

// fs_rfsstat - Fill in fs_info struct for device.
static int		fs_rfsstat(void *_ns, struct fs_info *);

// fs_readlink - Read in the name of a symbolic link.
static int 		fs_readlink(void *_ns, void *_node, char *buf, size_t *bufsize);
/* End of fundamental (read-only) required functions. */


#if 0
static int		fs_remove_vnode(void *ns, void *node, char r);
static int		fs_secure_vnode(void *ns, void *node);
static int		fs_create(void *ns, void *dir, const char *name,
					int perms, int omode, vnode_id *vnid, void **cookie);
static int		fs_mkdir(void *ns, void *dir, const char *name, int perms);
static int		fs_unlink(void *ns, void *dir, const char *name);
static int		fs_rmdir(void *ns, void *dir, const char *name);
static int		fs_wstat(void *ns, void *node, struct stat *st, long mask);
static int		fs_write(void *ns, void *node, void *cookie, off_t pos,
						const void *buf, size_t *len);
static int		fs_ioctl(void *ns, void *node, void *cookie, int cmd,
						void *buf, size_t len);
static int		fs_wfsstat(void *ns, struct fs_info *);
static int		fs_sync(void *ns);
static int     	fs_initialize(const char *devname, void *parms, size_t len);
#endif 



/* vnode_ops struct. Fill this in to tell the kernel how to call
	functions in your driver.
*/

vnode_ops fs_entry =  {
	&fs_read_vnode,						// read_vnode func ptr
	&fs_write_vnode,					// write_vnode func ptr
	NULL, 								// remove_vnode func ptr
	NULL,								// secure_vnode func ptr
	&fs_walk,							// walk func ptr
	&fs_access,							// access func ptr
	NULL, 								// create func ptr
	NULL, 								// mkdir func ptr
	NULL,
	NULL,
	NULL,
	NULL, 								// unlink func ptr
	NULL, 								// rmdir func ptr
	&fs_readlink,						// readlink func ptr
	&fs_opendir,						// opendir func ptr
	&fs_closedir,						// closedir func ptr
	&fs_free_dircookie,					// free_dircookie func ptr
	&fs_rewinddir,						// rewinddir func ptr
	&fs_readdir,						// readdir func ptr
	&fs_open,							// open file func ptr
	&fs_close,							// close file func ptr
	&fs_free_cookie,					// free cookie func ptr
	&fs_read,							// read file func ptr
	NULL, 								// write file func ptr
	NULL, /* readv */
	NULL, /* writev */
	NULL,								// ioctl func ptr
	NULL,								// setflags file func ptr
	&fs_rstat,							// rstat func ptr
	NULL, 								// wstat func ptr
	NULL,								// fsync
	NULL, 								// initialize func ptr
	&fs_mount,							// mount func ptr
	&fs_unmount,						// unmount func ptr
	NULL,								// sync func ptr
	&fs_rfsstat,						// rfsstat func ptr
	NULL,								// wfsstat func ptr
	NULL,								// select
	NULL,								// deselect
	NULL,								// open_indexdir
	NULL,								// close_indexdir
	NULL,								// free_indexdircookie
	NULL,								// rewind_indexdir
	NULL,								// read_indexdir
	NULL,								// create_index
	NULL,								// remove_index
	NULL,								// rename_index
	NULL,								// stat_index

	/* All the Attribute Functions are defined int the attr.c - File */
	&psionfs_open_attrdir,				// open_attrdir
	&psionfs_close_attrdir,				// close_attrdir
	&psionfs_free_attrdircookie,		// free_attrdircookie
	&psionfs_rewind_attrdir,			// rewind_attrdir
	&psionfs_read_attrdir,				// read_attrdir
	&psionfs_write_attr,				// write_attr
	&psionfs_read_attr,					// read_attr
	&psionfs_remove_attr,				// remove_attr
	&psionfs_rename_attr,				// rename_attr
	&psionfs_stat_attr,					// stat_attr

	NULL,								// open_query
	NULL,								// close_query
	NULL,								// free_querycookie
	NULL								// read_query
};

/*=======================================================================================
	We need some informations in global variables. These variables are defined here!
*/

int32			api_version = B_CUR_FS_API_VERSION;
static char		*gFSName = "psionfs";					/* My FilesystemName	*/
thread_id		gSerializeThread;						/* ?? 					*/
static char		debugFile[128];							/* in this file we have to write the debuginfo */

static int 
fs_mount(nspace_id nsid, const char *device, ulong flags, void *parms,
		size_t len, void **data, vnode_id *vnid)
{
	/*
	Kernel passes in nspace_id, (representing a disk or partition?)
	and a string representing the device (eg, "/dev/scsi/disk/030/raw)
	Flags will be used for things like specifying read-only mounting.
	parms is parameters passed in as switches from the mount command, 
	and len is the length of the otions. data is a pointer to a 
	driver-specific struct that should be allocated in this routine. 
	It will then be passed back in by the kernel to a number of the other 
	fs driver functions. vnid should also be passed back to the kernel, 
	representing the vnode id of the root vnode.
*/ 
	vnode *vnod;
	char text[128];
	int	result = EINVAL;
	nspace *nsp;
	size_t	l;
	MNT_INFO *mnt_info = (MNT_INFO *)parms;	// Es wird der Port übergeben unter dem der NCP erreichbar ist.;
	
#ifdef DEBUG_FILE
	int fd_debug;
	acquire_sem(mnt_info->sem);
//	if ((fd_debug = open(DEBUG_FILE, O_RDWR|O_CREAT, 7770)) > 0) {
//		write(fd_debug, "Mount gestartet!\n", 17);
//		close(fd_debug);
//	}
	release_sem(mnt_info->sem);
#endif

	sprintf(text, "fs_mount: dev=>%s< nsid=%ld\n", device, nsid);
	debug_print(text);

	if (mnt_info->pid_read < B_NO_ERROR || mnt_info->pid_write < B_NO_ERROR) {
		debug_print("Falsche PortID's übergeben!\n");
		return result;
	}
	
	nsp = (nspace *) malloc(sizeof(nspace));
	if (nsp == NULL) return ENOMEM;
	
	nsp->id = nsid;
	nsp->pid_read = mnt_info->pid_read;
	nsp->pid_write = mnt_info->pid_write;
	nsp->cid = mnt_info->cid;
	strcpy(nsp->device, mnt_info->device); 
	nsp->port_sem = mnt_info->sem;	// This semaphore is set from the mounter. it is unique for the serial port!
									// gIf the mounter dies, it can be recognized through this semaphore!
	nsp->dev_sem = create_sem(1, "PSION Device");
	sprintf(text, "-- fs_mount: semaphoreId: %ld\n", nsp->dev_sem);
	debug_print(text);

	if (mnt_info->debugFile != NULL) strcpy(debugFile, mnt_info->debugFile);	/* We have to remember the debugFilename!	*/
	else *debugFile = '\0';
	
	*data = nsp;

	vnod = search_vnode(nsp, vnid, 0, "\\", NULL);	// Der Root-Node wird gesetzt!
	nsp->rootDir = vnod;				// Den Root-Node merken!
	
	size_t largeIconLen = sizeof(kLargePsionIconBits);
	size_t smallIconLen  = sizeof(kSmallPsionIconBits);

	psionfs_write_attr(nsp, nsp->rootDir, LARGE_ICON_NAME, LARGE_ICON_TYPE, kLargePsionIconBits, &largeIconLen, (off_t)0);
	psionfs_write_attr(nsp, nsp->rootDir, SMALL_ICON_NAME, SMALL_ICON_TYPE, kSmallPsionIconBits, &smallIconLen, (off_t)0);

	l = sizeof(PSION_VOLUME);
	psionfs_write_attr(nsp, nsp->rootDir, BEOS_MIME_NAME, BEOS_MIME_TYPE, PSION_VOLUME, &l, (off_t)0);

	// You MUST do this. Create the vnode for the root.
	// Dieser Funktionsaufruf stellt einen Speicherbereich für den Vnode zur Verfügung!

	result = new_vnode(nsid, *vnid, (void*)vnod);
	if (result != B_NO_ERROR)
	{
		free(vnod);
		result = EINVAL;
	}
	else 
	{
		result = B_NO_ERROR;	
	}		

	sprintf(text, "fs_mount: *vnid=%Ld\n", *vnid);
	debug_print(text);

	return result;
}

static int		
fs_unmount(void *_ns)
{
	nspace* ns = (nspace*)_ns;
	int result = B_NO_ERROR;
	
	debug_print("## fs_unmount\n");
	
	delete_sem(ns->dev_sem);	/* Delete the Device_Semaphore	*/
	free (ns);					/* free all the needed Space	*/
	
	return result;
}

// fs_rfsstat - Fill in fs_info struct for device.
static int		
fs_rfsstat(void *_ns, struct fs_info * fss)
{
	nspace* ns = (nspace*)_ns;
	unsigned char buffer[10], *inc_buf;
	int		result = ENOMEM;
	size_t	len;
	char text[128];
	
	sprintf(text, "## fs_rfsstat: %s\n", ns->device);
	debug_print(text);
	
	WordToBytes(RF_STATUSDEVICE, buffer);	// Der Befehl RF_STATUSDEVICE
	WordToBytes(strlen(ns->device)+1, &buffer[2]);	// Länge abhängig vom Devicenamen + Nullbyte
	strcpy((char *)&buffer[4], ns->device);		// Infos zum internen Laufwerk

	inc_buf = send_command(ns, buffer, &len);
	if (inc_buf != NULL && len >= 20) {
		if (BytesToWord(&inc_buf[4]) == 0) result = 0;
	
		// Fill in device id.
		fss->dev = ns->id;
		
		// Root vnode ID
		fss->root = ns->rootDir->id;
	
		// File system flags.
		fss->flags =	B_FS_IS_PERSISTENT | B_FS_IS_REMOVABLE | B_FS_IS_READONLY |
						B_FS_HAS_MIME | B_FS_HAS_ATTR;
	
		// FS block size.
		fss->block_size = 1;
	
		// IO size - specifies buffer size for file copying
		fss->io_size = MAX_BLOCK;
	
		// Total blocks?
		fss->total_blocks = BytesToLong(&inc_buf[12]);
	
		// Free blocks = 0, read only
		fss->free_blocks = BytesToLong(&inc_buf[16]);
	
		// Device name.
		strncpy(fss->device_name, ns->device, sizeof(fss->device_name));

		strncpy(fss->volume_name, (char *)&inc_buf[20], sizeof(fss->volume_name));
	
		// File system name
		strcpy(fss->fsh_name,"psionfs");

		free(inc_buf);
	}
	debug_print("fs_rfsstat beendet\n");
	return 0;
}

/* fs_walk - the walk function just "walks" through a directory looking for
	the specified file. When you find it, call get_vnode on its vnid to init
	it for the kernel.
*/

static int		
fs_walk(void *_ns, void *_base, const char *file, char **newpath, 
			vnode_id *vnid)
{
	char text[128];
	nspace*		ns = (nspace*)_ns;
	int			result = ENOENT;
	vnode*		base_vnod = (vnode *)_base;
	vnode*		newNode = NULL;

	/* Starting at the base, find file in the subdir, and return path
		string and vnode id of file. */
	sprintf(text, "## fs_walk: file:>%s< vnodbase:%Ld\n", file, base_vnod->id);
	debug_print(text);

	if (strcmp(file,".") == 0) 					*vnid = base_vnod->id;
	else if (strcmp(file, "..") == 0) 			*vnid = base_vnod->parID;
	else {
		newNode = search_vnode(ns, vnid, base_vnod->id, file, NULL);	// der node wird gesetzt
		if (newNode == NULL) return ENOENT;								// Datei nicht vorhanden!
	}

	sprintf(text, "-- fs_walk: vnid:%Ld\n", *vnid);
	debug_print(text);

	if (get_vnode(ns->id, *vnid, (void**)&newNode) != 0) result = EINVAL;
	else result = B_NO_ERROR;		


	return result;
}

// fs_read_vnode - Using vnode id, read in vnode information into fs-specific struct,
//					and return it in node. the reenter flag tells you if this function
//					is being called via some other fs routine, so that things like 
//					double-locking can be avoided.
static int		
fs_read_vnode(void *_ns, vnode_id vnid, char reenter, void **node)
{
	char text[128];
	int			result = B_NO_ERROR;
	vnode* 		newNode = (vnode *)vnid;	// vnid = Adresse des node!

	sprintf(text, "## fs_read_vnode vnode_id:%Ld, reenter:%d\n", vnid, reenter);
	debug_print(text);
	

	if (newNode != NULL) {
		newNode->read_vnode = true;			// Durch read_vnode angefordert!
		*node = (void *)newNode;
	} else result = ENOMEM;
	
	return result;
}

// gibt den node wieder frei!
static int		
fs_write_vnode(void *ns, void *_node, char reenter)
{
	char	text[128];
	vnode	*node = (vnode*)_node;
	
	sprintf(text, "## fs_write_vnode vnode_id:%Ld, reenter:%d\n", node->id, reenter);
	debug_print(text);
	
	if (node != NULL) {  
		// hier muß noch einiges getan werden, wenn die VNODES wieder freigegeben werden sollen.
		// vor allem dann, wenn ein ROOT freigegeben wird, und die Childs noch nicht....

		if (node->parID != 0) {
			if (node->prev->child == node) node->prev->child = node->next;	// es war ein Kind
			else node->prev->next = node->next;

			if (node->next != NULL) node->next->prev = node->prev;		// der Letzte der Kette!
			
		}
		if (node->filename != NULL) free(node->filename);
		if (node->attr != NULL) free_all_attr(node->attr);
		free(node);
	}
	return B_NO_ERROR;
}

// fs_rstat - fill in stat struct
static int		
fs_rstat(void *_ns, void *_node, struct stat *st)
{
	vnode*	node = (vnode*)_node;
	nspace* ns = (nspace*)_ns;
	char text[128];
	
	sprintf(text, "## fs_rstat: vnod_id:%Ld, parid:%Ld filename:%s\n", node->id, node->parID, node->filename);
	debug_print(text);

	st->st_dev = ns->id;	// Das ist wichtig, da sonst das Verzeichnis nicht als Device anerkannt wird
	st->st_ino = node->id;
	st->st_mode = node->mode;
	st->st_nlink = 1;
	st->st_uid = 0;
	st->st_gid = 0;
	st->st_size = node->size;
	st->st_rdev = 0;
	st->st_blksize = MAX_BLOCK;
	st->st_ctime = st->st_mtime = st->st_atime = st->st_crtime = node->modtime;
	
	return B_NO_ERROR;
}

// fs_open - Create a vnode cookie, if necessary, to use when
// 				reading/writing a file
static int		
fs_open(void *_ns, void *_node, int omode, void **cookie)
{
	dircookie* dirCookie;
	nspace* ns = (nspace*)_ns;
	vnode* node = (vnode *)_node;
	size_t	result = ENOMEM;
	char text[128];
	
	sprintf(text, "## fs_open: vnod_id:%Ld, omode:%d %s\n", node->id, omode, node->filename);
	debug_print(text);
	
	dirCookie = (dircookie *)malloc(sizeof(dircookie));
	if (dirCookie != NULL) {
		result = dirCookie->fd = open_node(ns, node, omode);
		dirCookie->buff = NULL;
		dirCookie->pos = dirCookie->read = 0;
			
		*cookie = (void*)dirCookie;
		sprintf(text, "-- fs_open: filehandler %04x accepted\n", *dirCookie);
		debug_print(text);
		if (result > 0) result = B_NO_ERROR;
	}
	sprintf(text, "-- fs_open beendet mit: %d\n", result);
	debug_print(text);
	
	return result;
}

// fs_read
// Read a file specified by node, using information in cookie
// and at offset specified by pos. read len bytes into buffer buf.
static int		
fs_read(void *_ns, void *_node, void *_cookie, off_t pos, void *buf, 
			size_t *len)
{
	dircookie *dirc = (dircookie*)_cookie;
	nspace* ns = (nspace*)_ns;	
	vnode* node = (vnode *)_node;
	size_t read_len;
	unsigned int	laenge;
	size_t			laenge_org;
	char text[128];
	unsigned char buffer[10], *inc_buf;
	int result = B_NO_ERROR;

	sprintf(text, "## fs_read: vnod_id:%Ld, pos:%Ld, len:%lu %s\n", node->id, pos, *len, node->filename);
	debug_print(text);

	if (pos != dirc->read) {	// FSEEK machen
		WordToBytes(RF_FSEEK, buffer);		// Der Befehl RF_FCLOSE
		WordToBytes(8, &buffer[2]);			// Länge immer 8
		WordToBytes(dirc->fd, &buffer[4]);	// Den Filedeskriptor lesen
		LongToBytes(pos, &buffer[6]);		// Die Position im File von Beginn an!
		WordToBytes(P_FABS, &buffer[10]);	// Immer im Absoluten Modus arbeiten!
		inc_buf = send_command(ns, buffer, &read_len);
		if (inc_buf != NULL && read_len >= 6 && BytesToWord(&inc_buf[4]) == 0) {
			dirc->read = BytesToLong(&inc_buf[6]);
		}
		sprintf(text, "-- fs_read: FSEEK-auf %ld durchgeführt!\n", dirc->read);
		debug_print(text);
	}
	
	WordToBytes(RF_FREAD, buffer);		// Der Befehl RF_FREAD
	WordToBytes(4, &buffer[2]);			// Länge immer 4
	WordToBytes(dirc->fd, &buffer[4]);	// Den Filedeskriptor lesen
	
	laenge_org = *len;
	laenge = (laenge_org > MAX_BLOCK) ? MAX_BLOCK : (unsigned int ) laenge_org;		// Mehr sollte nicht übertragem werden!
	
	WordToBytes((int )laenge, &buffer[6]);		// Anzahl der zu lesenden Bytes

	inc_buf = send_command(ns, buffer, &read_len);
	if (inc_buf != NULL && read_len >= 4 && BytesToWord(&inc_buf[4]) == 0) {
		laenge = BytesToWord(&inc_buf[2])-2;
		dirc->read += laenge;
		laenge_org = laenge;
		*len = laenge_org;
		memcpy(buf, &inc_buf[6], *len);
	} else {
		*len = 0;
	}
	sprintf(text, "-- fs_read: Tatsächlich gelesen %lu\n", *len);
	debug_print(text);
	if (inc_buf != NULL) free(inc_buf);	
	return result;

}

// fs_close - Do whatever is necessary to close a file, EXCEPT for freeing
//				the cookie!
static int		
fs_close(void *ns, void *node, void *cookie)
{
	debug_print("## fs_close\n");
	return 0;
}

static int		
fs_free_cookie(void *_ns, void *node, void *_cookie)
{
	dircookie *dirc = (dircookie*)_cookie;
	nspace* ns = (nspace*)_ns;	
	unsigned char buffer[10], *inc_buf;
	size_t	len;

	debug_print("## fs_free_cookie\n");

	WordToBytes(RF_FCLOSE, buffer);		// Der Befehl RF_FCLOSE
	WordToBytes(2, &buffer[2]);			// Länge immer 2
	WordToBytes(dirc->fd, &buffer[4]);	// Den Filedeskriptor schliessen

	inc_buf = send_command(ns, buffer, &len);
	if (inc_buf != NULL && len >= 4) {
		free(inc_buf);
	}

	if (dirc != NULL) {
		if (dirc->buff != NULL) free(dirc->buff);
		free(dirc);
	}
	return B_NO_ERROR;
}

// fs_access - checks permissions for access.
static int		
fs_access(void *ns, void *node, int mode)
{
	debug_print("?? fs_access\n");
	return 0;
}

static int
fs_readlink(void *_ns, void *_node, char *buf, size_t *bufsize)
{
	debug_print("?? fs_readlink\n");
	return B_NO_ERROR;
}

// fs_opendir - creates fs-specific "cookie" struct that keeps track of where
//					you are at in reading through directory entries in fs_readdir.
static int		
fs_opendir(void *_ns, void *_node, void **cookie)
{
	dircookie* dirCookie;
	unsigned char *gen;
	nspace* ns = (nspace*)_ns;
	vnode* node = (vnode *)_node;
	size_t	result = ENOMEM;
	char text[128];
	
	sprintf(text, "## fs_opendir: vnod_id:%Ld, parid:%Ld %s\n", node->id, node->parID, node->filename);
	debug_print(text);
	
	dirCookie = (dircookie *)malloc(sizeof(dircookie));
	if (dirCookie != NULL) {
		result = dirCookie->fd = open_node(ns, node, 0);
		dirCookie->pos = 0;
		gen = (unsigned char * )malloc(37);	// Speicher für "." und ".." reserviert!
		memset(gen, 0, 37);
		gen[16] = '.';
		gen[18+16] = '.';
		gen[18+17] = '.';
		dirCookie->buff = gen;
		dirCookie->read = 37;				// "." und ".." generiert
		*cookie = (void*)dirCookie;
		sprintf(text, "-- fs_opendir: filehandler %04x accepted\n", *dirCookie);
		debug_print(text);
		if (result > 0) result = B_NO_ERROR;
	}
	sprintf(text, "-- fs_opendir beendet mit: %d\n", result);
	debug_print(text);
	
	return result;
}

// fs_readdir - read 1 or more dirents, keep state in cookie, return
//					0 when no more entries.
static int		
fs_readdir(void *_ns, void *_node, void *_cookie, long *num, 
			struct dirent *buf, size_t bufsize)
{
	char text[128];
	unsigned char buffer[10];
	size_t len;
	int result = ENOENT;
	vnode *node = (vnode *)_node, *n;
	dircookie *dirc = (dircookie *)_cookie;
	nspace* ns = (nspace*)_ns;
	vnode_id dummy;
	
	sprintf(text, "## fs_readdir: vnod_id:%Ld, buffsize:%ld\n", node->id, bufsize);
	debug_print(text);

	if (dirc->pos >= dirc->read) {			// alle gelesenen schon ausgegeben!
		if (dirc->buff != NULL) free(dirc->buff); // dann vergessen wir den alten Buffer!
		WordToBytes(RF_FDIRREAD, buffer);	// Der Befehl RF_FDIRREAD
		WordToBytes(2, &buffer[2]);			// Länge immer 2
		WordToBytes(dirc->fd, &buffer[4]);	// Den Filedeskriptor schliessen

		dirc->buff = send_command(ns, buffer, &len);
		if (dirc->buff != NULL && len >= 8) {
			if (BytesToWord(&dirc->buff[4]) == 0) {
				dirc->read = len;
				dirc->pos = 8;				// hier beginnt der erste Datensatz!
				result = B_NO_ERROR;
			}
		} else result = ENOMEM;
	}
	if (dirc->pos >= dirc->read) {		// Keine weiteren Dateien vorhanden!
		node->read_all = true;			// Verzeichnis wurde komplett eingelesen!
		result = B_NO_ERROR;			// Kein Fehler aufgetreten
		*num = 0;						// Aber kein File mehr eingelesen
	} else {
		result = B_NO_ERROR;
		strcpy(buf->d_name, (char *)&dirc->buff[dirc->pos+16]);
		buf->d_reclen = strlen(buf->d_name);
		n = search_vnode(ns, &dummy, node->id, buf->d_name, &dirc->buff[dirc->pos]);
		buf->d_dev = buf->d_pdev = ns->id;
		buf->d_ino = dummy;	
		buf->d_pino = node->id;

		dirc->pos += (16 + 1 + buf->d_reclen);		// 0-Byte mitzählen
		*num = 1;
		sprintf(text, "-- fs_readdir: File >%s< d_ino:%Ld d_pino:%Ld\n", buf->d_name, buf->d_ino, buf->d_pino);
		debug_print(text);
	}
	sprintf(text, "-- fs_readdir beendet mit: %d\n", result);
	debug_print(text);
	
	return result;
}
			
// fs_rewinddir - set cookie to represent beginning of directory, so
//					later fs_readdir calls start at beginning.
static int		
fs_rewinddir(void *ns, void *node, void* cookie)
{
	debug_print("## fs_rewinddir\n");
	fs_closedir(ns, node, cookie);	// Beim PSION gibts kein rewind, daher schliessen
	fs_opendir(ns, node, &cookie);	// und wieder neu öffnen
	return B_NO_ERROR;
}

// fs_closedir - Do whatever you need to to close a directory (sometimes
//					nothing), but DON'T free the cookie!
static int		
fs_closedir(void *ns, void *node, void *cookie)
{
	debug_print("## fs_closedir\n");		// Wir machen hier garnix! Der Kanal wird im free_dircookie geschlossen
	
	return B_NO_ERROR;
}

// fs_free_dircookie - Free the fs-specific cookie struct
// Dieser cookie wurde bei opendir angelegt!
static int		
fs_free_dircookie(void *_ns, void *node, void *_cookie)
{
	dircookie *dirc = (dircookie*)_cookie;
	nspace* ns = (nspace*)_ns;	
	unsigned char buffer[10], *inc_buf;
	size_t	len;
	char text[128];

	debug_print("## fs_free_dircookie\n");

	WordToBytes(RF_FCLOSE, buffer);		// Der Befehl RF_FLCLOSE
	WordToBytes(2, &buffer[2]);			// Länge immer 2
	WordToBytes(dirc->fd, &buffer[4]);	// Den Filedeskriptor schliessen

	inc_buf = send_command(ns, buffer, &len);
	if (inc_buf != NULL && len >= 4) {
		free(inc_buf);
	}

	if (dirc != NULL) {
		if (dirc->buff != NULL) free(dirc->buff);
		free(dirc);
	}
	return B_NO_ERROR;
}

// Ab hier sinds spezielle Funktionen, die für das Protokoll benötigt werden!

void WordToBytes(unsigned int i, unsigned char *p) {
    *p++ = i;
    *p = (i >> 8);
}

void LongToBytes(unsigned long l, unsigned char *p) {
    *p++ = l;
    *p++ = (l >> 8);
    *p++ = (l >> 16);
    *p = (l >> 24);
}

unsigned int BytesToWord(const unsigned char *c) {
	return ((unsigned int )*c | ((unsigned int )c[1] << 8)); 
}

unsigned long BytesToLong(const unsigned char *c) {
	return ((unsigned int )*c | ((unsigned int )c[1] << 8) | ((unsigned int )c[2] << 16) | ((unsigned int )c[3] << 24)); 
}

// Die folgende Prozedur setzt einen Speicherbereich "vnode" mit den benötigten Daten
// die zugehörige vnode_id sind die unteren 64-Bit (daher eh die komplette Adresse)
// von dem Speicherbereich in dem der vnode abgelegt ist!
//
// parentID ist die vnode-ID des Verzeichnises, in dem die Datei steht!
// myID entspricht dem zurückgegebenen vnode
//
// Um diese allerdings wiederzufinden werden zwei Ketten aufgebaut, die einmal die nodes
// in der gleichen Ebene verbinden und einmal die Nodes darunter.
// ist der VNODE bereits geladen, dann wird kein Speicher reserviert!
//
// ist fileinfo = NULL, wird die Info über das Kommando RF_FINFO angefordert, ansonsten von
// fileinfo übernommen. fileinfo hat das gleiche Format, wie RF_DIRREAD oder RF_FINFO

vnode *search_vnode(nspace *ns, vnode_id *myID, vnode_id parentID, const char *file, unsigned char *fileinfo)

{
	char path[128], *c;
	unsigned char buffer[200], *inc_buf;
	vnode *node, *n = NULL;
	size_t len;

	if (strcmp(file, ".") == 0) n = ((vnode *)parentID);
	else if (strcmp(file, "..") == 0) {
		n = (vnode *)((vnode *)parentID)->parID;
		if (n == 0) *myID = ns->id;
		return n;
	} else if (parentID != 0) {
		n = ((vnode *)parentID)->child;		// schau mer mal, ob wir den VNODE schon geladen haben...
		while (n != NULL) {
			if (strcmp(file, n->filename) == 0) {
				if (fileinfo != NULL) setup_vnode(n, fileinfo);
				break;					// Schleife beenden, wir haben den VNODE gefunden!
			} else n = n->next;
		}
	}
	if (n != NULL) {
		*myID = n->id;
		return n;
	}
										// Nichts gefunden, also reservieren und belegen!
	node = (vnode *)malloc (sizeof(vnode));
	if (node != NULL) {
		*myID = (vnode_id )node;
		node->id = *myID;
		node->parID = parentID;
		node->filename = strdup(file);
		node->read_all = false;			// Initialisieren
		node->read_vnode = false;
		node->attr = NULL;				// No Attribute connected!

		find_path(ns, node, path, sizeof(path));
		c = &path[strlen(path)-1];
		if (*c == '\\') *c = '\0';
		
		node->next = node->child = node->prev = NULL;
		if (parentID == 0) {		// Das ist das Rootverzeichnis!
			node->mode = S_IRWXU | S_IRWXG | S_IRWXO | S_IFDIR;
			node->size = 0;
			node->modtime = 0;
		} else {
			if (fileinfo == NULL) {
				WordToBytes(RF_FINFO, buffer);		// Der Befehl RF_FINFO
				strcpy((char *)&buffer[4], path);	// Filenamen zu dem die Info gewünscht wird.
				WordToBytes(strlen(path)+1, &buffer[2]);	// Länge des Strings plus dem letzten 0-Byte

				inc_buf = send_command(ns, buffer, &len);
				if (inc_buf != NULL && len >= 18 && BytesToWord(&inc_buf[4]) == 0) {
					setup_vnode(node, &inc_buf[6]);
					free(inc_buf);
				} else {			// Datei nicht gefunden!
					if (node->filename != NULL) free(node->filename);
					free(node);
					if (inc_buf != NULL) free(inc_buf);
					return NULL;
				}
			} else {
				setup_vnode(node, fileinfo);
			}
			set_mime_type(ns, node);

			// Knoten in den übergeordneten vnode einhängen!
		
			n = (vnode *)parentID;
			node->next = n->child;	// Knoten als ersten einhängen!
			node->prev = n;
			n->child = node;
		}
	}
	return node;
}

// Setzt die Werte des vnode anhand der finfo-Struktur vom RFSV-Protokoll

void setup_vnode(vnode *node, unsigned char *finfo) {
	mode_t mode;
	
	node->modtime = BytesToLong(&finfo[8]);
	node->size = BytesToLong(&finfo[4]);

	mode = BytesToWord(&finfo[2]);
	
	if (mode & P_FADIR) node->mode		 = S_IFDIR|S_IXUSR|S_IXGRP|S_IXOTH;
	else {
		node->mode = S_IFREG;
		if (mode & P_FAEXEC) node->mode		|= S_IXUSR|S_IXGRP|S_IXOTH;
	}
	
	if (mode & P_FAREAD) node->mode		|= S_IRUSR|S_IRGRP|S_IROTH;
	if (mode & P_FAWRITE) node->mode	|= S_IWUSR|S_IWGRP|S_IWOTH;
}

// Durchsucht die Pfadstruktur rekursiv und füllt dan pfad-String

void find_path(nspace *ns, vnode *vn, char *pfad, size_t max_len) {
	if (vn->parID != 0) {
		find_path(ns, (vnode *)vn->parID, pfad, max_len);
		strcat(pfad, vn->filename);
		if (vn->mode & S_IFDIR) strcat(pfad, "\\");
	} else {
		strcpy(pfad, ns->device);
		strcat(pfad, vn->filename);
	}
}

// Diese Routine öffnet einen vnode im PSION.
// Übergeben wird der VNODE und zurückgegeben, wird der Handler, oder ein Fehlercode!

int	open_node(nspace *ns, vnode *vn, int omode) {
	int result;
	unsigned char buffer[128], *inc_buf;
	char *c;
	size_t len;
	
	c = (char *)&buffer[6];
	find_path(ns, vn, c, sizeof(buffer)-6);	// Größe des Buffers!

	WordToBytes(RF_FOPEN, buffer);				// Der Befehl P_FOPEN
	if (vn->mode & S_IFDIR) {
		WordToBytes(P_FDIR, &buffer[4]);		// Wir wollen ein Verzeichnis öffnen
		strcat(c, "*.*");						// Der PSION will, daß beim Verzeichnisöffnen immer ein *.* angehängt wird!
	} else {
		WordToBytes(P_FOPEN|P_FRANDOM, &buffer[4]);		// Wir wollen eine normale Datei öffnen (derzeit nur lesemodus)
	}
	
	WordToBytes(strlen(c)+3, &buffer[2]);		// Länge berechnen! + 0-Byte + Open-Status

	inc_buf = send_command(ns, buffer, &len);	// Also senden wir das Öffnen kommando an den PSION
	if (inc_buf != NULL) {						// Wir haben eine gültige Antwort erhalten
		if (len >= 6) {	
			switch (BytesToWord(&inc_buf[4])) {	// Hier steht das Ergebnis der Datei öffnen
				case 0:				result = BytesToWord(&inc_buf[6]);	// Filehandle übernehmen!
									break;
				case E_FILE_NAME:	result = ENAMETOOLONG; break;
				case E_FILE_NXIST:	
				case E_FILE_DEVICE:	
				case E_FILE_DIR:	result = ENOENT; break;
			}
		} else result = EIO;					// Wir haben nicht genügen Bytes erhalten, daher IO-Error
		free (inc_buf);
	} else result = ENOMEM;

	return result;
}

// Diese Prozedur sendet einen Befehl and den WriteThread und wartet auf eine Antwort.
// Sie wird durch eine Semaphore geschützt, sodaß die richtige Antwort auch zum richtigen
// Aufrufer zurückkommt!
//
// Im buffer an Byte 2+3 muß die korrekte Länge vorhanden sein!

unsigned char *send_command(nspace *ns, unsigned char *buffer, size_t *msg_len) {
	size_t result, buf_len;
	unsigned char *inc_buf;
	int32	msg;
	
	buf_len = BytesToWord(&buffer[2]) + 4; //Die Zwei Befehls und zwei Längenbytes dazuzählen!
	if (acquire_sem(ns->port_sem) != B_NO_ERROR) return (NULL);	// es ist ein Fehler aufgetreten, die Semaphore existiert nicht mehr!

	write_port(ns->pid_write, PSION_CMD + ns->cid, buffer, buf_len);

#ifdef DEBUG_FILE
dbg_print("\nPC->PSION ", buf_len, buffer);
#endif


	if ((*msg_len = port_buffer_size_etc(ns->pid_read, B_TIMEOUT, 10000000L)) >= B_NO_ERROR) {	// 10 Sekunden auf Nachrichten warten!

		inc_buf = (unsigned char *)malloc(*msg_len);	// Speicher für die Nachricht reservieren.
		if (inc_buf != NULL) {
			result = read_port_etc(ns->pid_read, &msg, (void *)inc_buf, *msg_len, B_TIMEOUT, 10000000L);	// 10 Sekunden warten
			if (result >= B_NO_ERROR) {				// Nachricht wurde korrekt eingelesen!

#ifdef DEBUG_FILE
dbg_print("\nPSION->PC ", *msg_len, inc_buf);
#endif

				release_sem(ns->port_sem);				// Semaphore wieder freigeben!
				return(inc_buf);					// Buffer übergeben. Das Freigeben muß der Aufrufer übernehmen!
			}
		}
	}
	release_sem(ns->port_sem);			// Hierher kommen wir nur im Fehlerfall! Semaphore wieder freigeben!
	if (inc_buf != NULL) free(inc_buf);	// den reserveriten Speicher wieder freigeben
	return(NULL);						// Und Fehlercode zurückschicken!
}

/* This function is used to send the Debuginformation to a file */

void dbg_print(char *head, size_t len, unsigned char *buffer)
{
	int fd_debug;
	size_t i;
	char dbg[10];
	
	if (*debugFile != '\0') {
		if ((fd_debug = open(debugFile, O_APPEND|O_RDWR|O_CREAT)) > 0) {
			write(fd_debug, head, strlen(head));
			for (i = 0; i < len; i++) {
				sprintf(dbg, "%02x ", buffer[i]);
				write(fd_debug, dbg, 3);
			}
			close(fd_debug);
		}
	}
}

