/*
	Copyright 1997 - 1999, Peter Urbanec. All Rights Reserved.

	DumpTrack.c

	Dumps a CDDA track to stdout

	Version 1.1

	This code can be freely used for non-profit or for commercial purposes.

	If you wish to use any portion of this code you must include the following
	disclaimer in documentation associated with the product as well as any
	"about boxes" or other places where acknowledgements are normally listed,
	in a typeface which is no less prominent than the typeface used for any
	other credits.

	"Contains software components developed by Peter Urbanec"

 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <device/scsi.h>
#include "CDDA.h"

/* Prototypes for utility functions */
void dump_one_frame(CDDA_BLOCK *buffer);
void bump_msf(CDDA_MSF *a);
bool msf_equal(CDDA_MSF a, CDDA_MSF b);
void Usage(void);

/* Enty point */
int main(int argc, char *argv[])
{
	int TrackNumber;	/* Which track to rip */
	int source_file;	/* File descriptor for source device */
	int result;
	CDDA_TOC *toc = NULL;				/* Table Of Contents */
	scsi_read_cd *read_cd_cmd = NULL;	/* The READ_CD ioctl() command */
	CDDA_BLOCK *frame_buffer = NULL;	/* Buffer for one CDDA frame */
	CDDA_MSF current, end;				/* MSF positions */

	/* Bare bones argument handling */
	if(argc != 3) Usage();

	TrackNumber = atoi(argv[2]);	/* Desired track is second argument */
	if(TrackNumber < 1) Usage();

	source_file = open(argv[1], O_RDONLY);	/* Open device for reading */
	if(source_file == -1)
	{
		fprintf(stderr, "DumpTrack: Can't open \"%s\" for reading\n", argv[1]);
		exit(1);
	}

	/* Allocate all memory for CDDA operations */
	if((toc = (CDDA_TOC *)malloc(sizeof(scsi_toc))) == NULL) goto cleanup;
	if((read_cd_cmd = (scsi_read_cd *)malloc(sizeof(scsi_read_cd))) == NULL) goto cleanup;
	if((frame_buffer = (CDDA_BLOCK *)malloc(sizeof(CDDA_BLOCK))) == NULL) goto cleanup;

	if((result = ioctl(source_file, B_SCSI_GET_TOC, toc)) != 0)
	{
		fprintf(stderr, "DumpTrack: Can't read table of contents\n");
		goto cleanup;
	}

	/* Now we have the TOC for the CD in standard SCSI-2 format, even for */
	/* ATAPI CD-ROMs, since they just do SCSI-2 and a bit over IDE bus anyway */
	if((TrackNumber > toc->last_track) ||
	   (TrackNumber < toc->first_track) ||
		/* Track numbers normally start at 1, but they do not have to... */
	   (toc->track[(TrackNumber-1)+(toc->first_track-1)].track_number != TrackNumber))
	{
		fprintf(stderr, "DumpTrack: Can't find track %d in the table of contents\n",
		TrackNumber);
		goto cleanup;
	}

	/* Be a good citizen and respect copyright */
	if((toc->track[(TrackNumber-1)+(toc->first_track-1)].flags &
	    CDDA_TOC_FLAG_DIGITAL_COPY_OK) == 0)
	{
		fprintf(stderr, "DumpTrack: Digital copy prohibited on track %d\n"
		"Make sure you are not infringing on the artist's copyright!\n"
		"Continuing with the dump of raw data...\n",
		TrackNumber);
	}

	/* Perform CDDA extraction on audio only */
	if((toc->track[(TrackNumber-1)+(toc->first_track-1)].flags &
	    CDDA_TOC_FLAG_DATA_TRACK) != 0)
	{
		fprintf(stderr, "DumpTrack: Track %d contains data, not audio\n",
		TrackNumber);
		goto cleanup;
	}

	current.m = toc->track[(TrackNumber-1)+(toc->first_track-1)].address.m;
	current.s = toc->track[(TrackNumber-1)+(toc->first_track-1)].address.s;
	current.f = toc->track[(TrackNumber-1)+(toc->first_track-1)].address.f;

	end.m = toc->track[(TrackNumber-1)+(toc->first_track-1)+1].address.m;
	end.s = toc->track[(TrackNumber-1)+(toc->first_track-1)+1].address.s;
	end.f = toc->track[(TrackNumber-1)+(toc->first_track-1)+1].address.f;

	/* Build the READ_CD command */
	read_cd_cmd->start_m = current.m;
	read_cd_cmd->start_s = current.s;
	read_cd_cmd->start_f = current.f;

	read_cd_cmd->length_m = 0;
	read_cd_cmd->length_s = 0;
	read_cd_cmd->length_f = 1;
	
	read_cd_cmd->buffer_length = sizeof(CDDA_BLOCK);
	read_cd_cmd->buffer = (char *)frame_buffer;
	read_cd_cmd->play = false;	/* Obsolete flag */

	/* Read one CDDA block at a time */
	for(;;)
	{
		fprintf(stderr, "%02d:%02d:%02d", current.m, current.s, current.f);
		fflush(stderr);
		if((result = ioctl(source_file, B_SCSI_READ_CD, read_cd_cmd)) != 0)
		{
			fprintf(stderr, "\nDumpTrack: ioctl(B_SCSI_READ_CD) returned with %d and errno 0x%08x\n",
			        result, errno);
			goto cleanup;
		}
		dump_one_frame(frame_buffer);	/* Dump block to destination (stdout in this case) */
		bump_msf(&current);	/* Don't read the last block, it belongs to next track */
		if(msf_equal(current, end) == true) break;

		read_cd_cmd->start_m = current.m;
		read_cd_cmd->start_s = current.s;
		read_cd_cmd->start_f = current.f;
		fprintf(stderr, "\b\b\b\b\b\b\b\b");
		fflush(stderr);
	}
	fprintf(stderr, "\n");

cleanup:
	if(frame_buffer != NULL) free(frame_buffer);
	if(read_cd_cmd  != NULL) free(read_cd_cmd);
	if(toc != NULL) free(toc);
	return 0;
}

/* Utility function to increment an MSF struct by one F */
void bump_msf(CDDA_MSF *a)
{
	a->f++;
	if(a->f > 74)
	{
		a->f = 0;
		a->s++;
		if(a->s > 59)
		{
			a->s = 0;
			a->m++;
		}
	}
}

/* See if two MSF structures are equal */
bool msf_equal(CDDA_MSF a, CDDA_MSF b)
{
	if(a.f != b.f) return false;
	if(a.s != b.s) return false;
	if(a.m != b.m) return false;
	return true;
}

/* Print a brief usage string */
void Usage(void)
{
	fprintf(stderr, "DumpTrack: </dev/CDROM/name/and/unit> <TrackNumber>\n");
	exit(1);
}

/* Dump one frame of CDDA to stdout */
void dump_one_frame(CDDA_BLOCK *buffer)
{
	write(1, buffer, sizeof(CDDA_BLOCK));
}
