//
// CalcView.cpp
//

#include "CalcView.h"
#include "CalcButton.h"
#include "DvBeUtils.h"
#include <assert.h>
#include <ctype.h>

const float kDigitBtnWidth = 40;
const float kDigitBtnHeight = 20;

CalcView::CalcView(BWindow* window)
	: BView(window->Bounds(), "CalcView", B_FOLLOW_ALL, B_WILL_DRAW | B_NAVIGABLE_JUMP)
{
	window->AddChild(this);
	Construct();
}

CalcView::CalcView(BView* view)
	: BView(view->Bounds(), "CalcView", B_FOLLOW_ALL, B_WILL_DRAW | B_NAVIGABLE_JUMP)
{
	view->AddChild(this);
	Construct();
}

CalcView::CalcView(BWindow* window, BRect frame)
	: BView(frame, "CalcView", B_FOLLOW_ALL, B_WILL_DRAW | B_NAVIGABLE_JUMP)
{
	window->AddChild(this);
	Construct();
}

CalcView::CalcView(BView* view, BRect frame)
	: BView(frame, "CalcView", B_FOLLOW_ALL, B_WILL_DRAW | B_NAVIGABLE_JUMP)
{
	view->AddChild(this);
	Construct();
}

void CalcView::Construct()
{
	SetViewColor(225, 225, 225);
	CreateMenus();
	CreateRadixGroup();
	CreateDisplay();
	CreateButtons();
	EnableButtons(calc.Radix());
	MakeFocus(true);
}

void CalcView::EnableButtons(Calculator::RADIX radix)
{
	BButton* b;
	char name[2] = " ";
	
	for (int x = 0; x <= 9; x++)
	{
		name[0] = x + '0';
		b = (BButton*) FindView(name);
		if (b != NULL)
			b->SetEnabled(x < radix);
	}
	
	for (name[0] = 'A'; name[0] <= 'F'; ++name[0])
	{
		b = (BButton*) FindView(name);
		if (b != NULL)
			b->SetEnabled(radix == Calculator::RADIX_HEX);
	}
	
	b = (BButton*) FindView(".");
	assert(b != NULL);
	b->SetEnabled(radix == Calculator::RADIX_DECIMAL);
}

void CalcView::HandleDigit(BMessage* msg)
{
	// Get source to figure out which digit was pressed
	//
	BControl** btn;
	ssize_t numBytes;
	if (msg->FindData("source", B_POINTER_TYPE, &btn, &numBytes) == B_OK)
	{
		calc.EnterDigit((*btn)->Name());
		UpdateDisplay();
	}
	else
	{
		BAlert* alert = new BAlert("Error", 
			"HandleDigit: Unable to get source from message.", "OK");
		alert->Go();
	}
}

void CalcView::HandleOperator(BMessage* msg)
{
	// Get source to figure out which digit was pressed
	//
	BControl** btn;
	ssize_t numBytes;
	if (msg->FindData("source", B_POINTER_TYPE, &btn, &numBytes) == B_OK)
	{
		const char* name = (*btn)->Name();
		if (!strcmp(name, "Add"))
			calc.EnterOp(Calculator::OP_PLUS);
		else if (!strcmp(name, "Minus"))
			calc.EnterOp(Calculator::OP_SUBTRACT);
		else if (!strcmp(name, "Multiply"))
			calc.EnterOp(Calculator::OP_MULTIPLY);
		else if (!strcmp(name, "Divide"))
			calc.EnterOp(Calculator::OP_DIVIDE);
		else if (!strcmp(name, "Equals"))
			calc.EnterOp(Calculator::OP_EQUALS);
		else if (!strcmp(name, "And"))
			calc.EnterOp(Calculator::OP_AND);
		else if (!strcmp(name, "Or"))
			calc.EnterOp(Calculator::OP_OR);
		else if (!strcmp(name, "Xor"))
			calc.EnterOp(Calculator::OP_XOR);

		UpdateDisplay();
	}
	else
	{
		BAlert* alert = new BAlert("Error", 
			"HandleOperator: Unable to get source from message.", "OK");
		alert->Go();
	}
}

void CalcView::KeyDown(const char* bytes, int32 numBytes)
{
	BButton* button = NULL;
	
	// See if it is a decimal digit
	//
	char digit = bytes[0] - '0';
	if (isdigit(bytes[0]))
	{
		if (digit < calc.Radix())
			button = (BButton*) FindView(bytes);
	}
	else if (isxdigit(bytes[0]))
	{
		char buf[2];
		buf[0] = toupper(bytes[0]);
		buf[1] = 0;
		if (calc.Radix() == Calculator::RADIX_HEX)
			button = (BButton*) FindView(buf);
	}
	else
		// Check other possibilities
		//
		switch (bytes[0])
		{
			case '.':
				button = (BButton*) FindView(".");
				break;
				
			case '*':
				button = (BButton*) FindView("Multiply");
				break;
				
			case '-':
				button = (BButton*) FindView("Minus");
				break;
				
			case '+':
				button = (BButton*) FindView("Add");
				break;
				
			case '/':
				button = (BButton*) FindView("Divide");
				break;
				
			case B_ENTER:
				button = (BButton*) FindView("Equals");
				break;
				
			case B_ESCAPE:
				button = (BButton*) FindView("Clear");
				break;
				
			case B_DELETE:
				button = (BButton*) FindView("CA");
				break;
				
			case 'n':
			case 'N':
				button = (BButton*) FindView("Sign");
				break;
				
			case '&':
				button = (BButton*) FindView("And");
				break;
				
			case '|':
			case 'o':
			case 'O':
				button = (BButton*) FindView("Or");
				break;
				
			case 'x':
			case 'X':
			case '^':
				button = (BButton*) FindView("Xor");
				break;
				
			case '~':
			case '!':
				button = (BButton*) FindView("Not");
				break;
				
			default:
				BView::KeyDown(bytes, numBytes);
				break;
		}
	
	// Simulate pressing a button if needed
	//
	if (button != NULL)
	{
		button->SetValue(B_CONTROL_ON);
		button->Invoke();
	}
}

void CalcView::MessageReceived(BMessage* message)
{
	switch (message->what)
	{
		case DV_CLEAR_ALL:
			calc.ClearAll();
			UpdateDisplay();
			break;
			
		case DV_CLEAR:
			calc.Clear();
			UpdateDisplay();
			break;
			
		case DV_DIGIT_BTN:
			HandleDigit(message);
			break;
			
		case DV_OPERATOR:
			HandleOperator(message);
			break;
			
		case DV_UNARY_MINUS:
			calc.UnaryMinus();
			UpdateDisplay();
			break;
			
		case DV_OP_NOT:
			calc.Not();
			UpdateDisplay();
			break;
			
		case DV_DEC:
		case DV_HEX:
		case DV_OCT:
		case DV_BIN:
			SetRadix(message);
			break;
			
		case B_COPY:
			if (be_clipboard->Lock())
			{
				be_clipboard->Clear();
				
				BMessage* clipper = be_clipboard->Data();
				char buf[81];
				calc.GetResults(buf, 81);
				clipper->AddData("text/plain", B_MIME_TYPE, buf, strlen(buf) + 1);
				
				be_clipboard->Commit();
				be_clipboard->Unlock();
			}
			break;
			
		default:
			BView::MessageReceived(message);
			break;
	};
}

void CalcView::CreateDisplay()
{
	BRect r = Bounds();
	r.InsetBy(5, 5);
	r.top = 25;
	r.bottom = r.top + 25;
	BBox* box = new BBox(r, NULL, B_FOLLOW_LEFT, B_WILL_DRAW | B_FRAME_EVENTS, 
		B_FANCY_BORDER);
	AddChild(box);
		
	r = box->Bounds();
	r.InsetBy(1, 1);
	BRect textRect(1, 1, r.Width() - 1, r.Height() -1);
	BTextView* display = new BTextView(r, "Display", textRect, B_FOLLOW_LEFT, 
		B_WILL_DRAW);
	display->MakeSelectable(false);
	display->MakeEditable(false);
	display->SetAlignment(B_ALIGN_RIGHT);
	
	BFont font;
	display->GetFontAndColor(&font, NULL);
	font.SetSize(16);
	display->SetFontAndColor(&font, B_FONT_SIZE);
	
	box->AddChild(display);
}

BButton* CalcView::CreateDigitButton(float left, float top, const char* digit)
{
	BRect r(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	BMessage *msg = new BMessage(DV_DIGIT_BTN);
	BButton *btn = new BButton(r, digit, digit, msg, B_FOLLOW_LEFT | B_FOLLOW_TOP, 
		B_WILL_DRAW);
	AddChild(btn);
	if (btn->SetTarget(this) != B_OK) 
	{
		BAlert* alert = new BAlert("", "Bad Target", "OK");
		alert->Go();
	}
	btn->SetFontSize(14);
	return btn;
}

void CalcView::CreateMenus()
{
	BMenuBar* mb;
	BRect r;
	
	mb = new BMenuBar(r, "MenuBar");
	AddChild(mb);
	
	// Create the File menu
	//
	BMenu* fileMenu = new BMenu("Calculator");
	BMenuItem* item = new BMenuItem("About DevaCalc…", new BMessage(B_ABOUT_REQUESTED));
	item->SetTarget(be_app);
	fileMenu->AddItem(item);
	fileMenu->AddSeparatorItem();
	fileMenu->AddItem(new BMenuItem("Exit", new BMessage(B_QUIT_REQUESTED), 'Q'));
	
	// Create the edit menu
	//
	BMenu* editMenu = new BMenu("Edit");
	editMenu->AddItem(new BMenuItem("Copy", new BMessage(B_COPY), 'C'));
	editMenu->SetTargetForItems(this);
	
	// Now insert the menus into the menu bar
	//
	mb->AddItem(fileMenu);
	mb->AddItem(editMenu);
}

BButton* CalcView::CreateOperatorButton(BRect r, const char* name, const char* label, 
	int32 command)
{
	BButton* btn = new BButton(r, name, label, new BMessage(command), 
		B_FOLLOW_LEFT | B_FOLLOW_TOP, B_WILL_DRAW);
	AddChild(btn);
	btn->SetTarget(this);
	btn->SetFontSize(14);
	return btn;
}

void CalcView::CreateButtons()
{
	const float kLeftMargin = 5;
	float left = 5;
	float top = 60;
	BRect r;
	
	// Create digit buttons
	//
	top += kDigitBtnHeight + 10;
	CreateDigitButton(left, top, "7");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "8");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "9");
	
	left += kDigitBtnWidth + 5;	
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);	
	BButton* b = CreateOperatorButton(r, "Clear", "C", DV_CLEAR);
	b->SetViewColor(255, 0, 0);

	left += kDigitBtnWidth + 5;	
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	b = CreateOperatorButton(r, "CA", "CA", DV_CLEAR_ALL);
	b->SetViewColor(255, 0, 0);

	left = kLeftMargin;
	top += kDigitBtnHeight + 5;
	CreateDigitButton(left, top, "4");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "5");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "6");
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Multiply", "x", DV_OPERATOR);

	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Divide", "÷", DV_OPERATOR);
	
	left = kLeftMargin;
	top += kDigitBtnHeight + 5;
	CreateDigitButton(left, top, "1");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "2");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "3");
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Add", "+", DV_OPERATOR);
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Minus", "-", DV_OPERATOR);
	
	left = kLeftMargin;
	top += kDigitBtnHeight + 5;
	CreateDigitButton(left, top, "0");

	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateDigitButton(left, top, ".");
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Equals", "=", DV_OPERATOR);
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	CreateOperatorButton(r, "Sign", "+/-", DV_UNARY_MINUS);	
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	rgb_color teal = {0, 168, 168};
	b = CreateOperatorButton(r, "Not", "Not", DV_OP_NOT);
	b->SetViewColor(teal);
		
	left = kLeftMargin;
	top += kDigitBtnHeight + 5;
	CreateDigitButton(left, top, "A");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "B");

	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "C");
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	b = CreateOperatorButton(r, "And", "And", DV_OPERATOR);
	b->SetViewColor(teal);
	
	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	b = CreateOperatorButton(r, "Or", "Or", DV_OPERATOR);
	b->SetViewColor(teal);
	
	left = kLeftMargin;
	top += kDigitBtnHeight + 5;
	CreateDigitButton(left, top, "D");

	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "E");
	
	left += kDigitBtnWidth + 5;
	CreateDigitButton(left, top, "F");	

	left += kDigitBtnWidth + 5;
	r.Set(left, top, left + kDigitBtnWidth, top + kDigitBtnHeight);
	b = CreateOperatorButton(r, "Xor", "Xor", DV_OPERATOR);	
	b->SetViewColor(teal);
}

void CalcView::CreateRadixGroup()
{
	const float kRadixWidth = 37;
	
	BRect r = Bounds();
	r.InsetBy(6, 6);
	r.top = 60;
	r.bottom = r.top + 25;
	r.right = r.left + kRadixWidth * 4 + 40;
	BBox* box = new BBox(r, NULL, B_FOLLOW_LEFT, B_WILL_DRAW | B_FRAME_EVENTS);
	AddChild(box);	
	
	r = box->Bounds();
	r.InsetBy(4, 4);
	r.right = r.left + kRadixWidth;
	uint32 resize = B_FOLLOW_LEFT | B_FOLLOW_TOP;
	BRadioButton* btn = new BRadioButton(r, "Dec", "Dec", new BMessage(DV_DEC), resize, 
		B_WILL_DRAW);
	btn->SetValue(1);
	box->AddChild(btn);	
	btn->SetTarget(this);
	
	r.OffsetBy(kRadixWidth + 10, 0);
	btn = new BRadioButton(r, "Hex", "Hex", new BMessage(DV_HEX), resize, B_WILL_DRAW);
	box->AddChild(btn);
	btn->SetTarget(this);
	
	r.OffsetBy(kRadixWidth + 10, 0);
	btn = new BRadioButton(r, "Oct", "Oct", new BMessage(DV_OCT), resize, B_WILL_DRAW);
	box->AddChild(btn);
	btn->SetTarget(this);

	r.OffsetBy(kRadixWidth + 10, 0);	
	btn = new BRadioButton(r, "Bin", "Bin", new BMessage(DV_BIN), resize, B_WILL_DRAW);
	box->AddChild(btn);
	btn->SetTarget(this);
}

void CalcView::SetRadix(BMessage* msg)
{
	Calculator::RADIX radix;	
	switch (msg->what)
	{
		case DV_BIN:	
			radix = Calculator::RADIX_BINARY;
			break;
		case DV_DEC:	
			radix = Calculator::RADIX_DECIMAL;
			break;
		case DV_HEX:	
			radix = Calculator::RADIX_HEX;
			break;
		case DV_OCT:	
			radix = Calculator::RADIX_OCTAL;
			break;
		default:
			DoAlert("Bad message passed to CalcView::SetRadix");
			break;
	}
	
	calc.SetRadix(radix);
	EnableButtons(calc.Radix());
	UpdateDisplay();
}

void CalcView::UpdateDisplay()
{
	char txt[81];
	calc.GetResults(txt, 81);
	
	BTextView* display = (BTextView*) FindView("Display");
	assert(display != NULL);
	display->SetText(txt);
}
