// gnuplot_be
//
// An outboard graphical terminal driver for gnuplot
// Thomas Ryan Hays   1997
// trh@lns598.lns.cornell.edu

#include <stdio.h>
#include "gnuplot_be.h"

double xscale, yscale;
#define X(x) (int) (x * xscale)
#define Y(y) (int) ((4095-y) * yscale)

int  nc = 0, ncalloc = 0;

char buf[Nbuf], **commands = (char **)0;



/*-----------------------------------------------------------------------------
 The plan is to generate a window driver that can take commands from
stdin.  These command line commands are going to be sent in
from the terminal driver via a pipe.  This program will be launched by Be_init()
and terminated with Be_reset().
 *---------------------------------------------------------------------------*/

// This is the unique instance of the graphics information.
// Only one graphics window will be open at a time.
globalBeOSterm gBeOSterm;


/*-----------------------------------------------------------------------------
 *   main 
 *---------------------------------------------------------------------------*/

int main(void) {
    extern globalBeOSterm gBeOSterm;
	GnuplotBe *the_app;
 
    gBeOSterm.window_open = false;
    
	the_app = new GnuplotBe();

	if (the_app != NULL) {
		the_app->Run();
		delete the_app;
	}
	return 0;
}

/*
 *  Application constructor
 */

GnuplotBe::GnuplotBe() : BApplication(APP_SIGNATURE) {
}

/*
 *  Open window
 */

void GnuplotBe::ReadyToRun(void)
{
   thread_id record_thread = spawn_thread(record,"record", B_NORMAL_PRIORITY, this);
   resume_thread(record_thread);}

/*
 *  About requested
 */

void GnuplotBe::AboutRequested(void)
{
	BAlert *the_alert = new BAlert("", "gnuplot_be by Tom Hays\n", "Neat");
	the_alert->Go();
}

/*
 *  Window constructor 
 */

GnuplotWindow::GnuplotWindow(): BWindow(BRect(0, 0, DISPLAY_X-1, DISPLAY_Y-1), "Gnuplot", B_TITLED_WINDOW, B_NOT_ZOOMABLE )
{
    extern globalBeOSterm gBeOSterm;

//    BMenu		*menu;
//    BMenuBar	*menu_bar;
//	BRect		r;
//	float		height;
//	BMenuItem  *anItem;
	
//    r.Set(0, 0, 32767, 15);
//    menu_bar = new BMenuBar(r, "");
//    menu = new BMenu("Try Me");
//    menu->AddItem(anItem = new BMenuItem("Send Now",
//								new BMessage(MSG_TRY_ME)));
//    menu_bar->AddItem(menu);

//    Lock();
//	AddChild(menu_bar);
//	height = menu_bar->Bounds().bottom + 1;
//	Unlock();
	
	// Move window to right position
	MoveTo(40, 40);
    the_picture = new BPicture;
	Lock();
    main_view = new PlotView(BRect(0, 0, DISPLAY_X-1, DISPLAY_Y-1),the_picture);    
	AddChild(main_view);
	Unlock();

    // Put some empty graphics into the picture since it doesn't like
    // it when you DrawPicture something that hasn't gone through
    // BeginEndPicture().  I suppose it is because the drawing environ
    // is copied to the Picture data.
    Lock();
    main_view->BeginPicture(the_picture);
    /* run through the plotting commands here */
    the_picture = main_view->EndPicture();
    Unlock(); 
       
	// Show the window
	Show();    
}

/*
 *  Closing the window doesn't quits the program
 */

bool GnuplotWindow::QuitRequested(void)
{
    extern globalBeOSterm gBeOSterm;

	main_view->Sync();
    gBeOSterm.window_open = false;
	
	return TRUE;
}

/*
 *  Handles all messages
 */

void GnuplotWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
//       case MSG_TRY_ME:
//          printf("You selected TRY ME.\n");
//       
//			break;

		default:
			inherited::MessageReceived(msg);
	}
}

/*
 *  Plot view constructor
 */
PlotView::PlotView(BRect frame, BPicture *newpicture)
   : BView(frame, "plot", B_FOLLOW_ALL_SIDES, B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE|B_FRAME_EVENTS)
{
    the_picture = newpicture;
    pointsize = 2;
}

void PlotView::FrameResized(float new_width, float new_height)
/* We get called whenever the window gets resized, provided the option
   B_FRAME_EVENTS is passed to the BView constuctor. We should probably 
   at some stage make sure, that at least the fonts get properly rescaled.
   sd 25.9.97
*/
{
	display();
}

/*
 *  Draw stuff.
 */

void PlotView::Draw(BRect update)
{
      DrawPicture(the_picture,BPoint(0,0));
}


/*-----------------------------------------------------------------------------
 *   record - record new plot from gnuplot inboard Be driver 
 *---------------------------------------------------------------------------*/

long record(void *arg)
{
   FILE *Be_ipc;
   Be_ipc = stdin;
   while(1){
      while (fgets(buf, Nbuf, Be_ipc)) {
        if (*buf == 'G') {                           /* enter graphics mode */
	       if (commands) {
	          int n; for (n=0; n<nc; n++) free(commands[n]);
	          free(commands);
	       }
	       commands = (char **)0; nc = ncalloc = 0;
        } else if (*buf == 'E'){
           display();
           break;            
        } else if (*buf == 'R'){ /* Reset.  This term no longer needed. */
           be_app->PostMessage(B_QUIT_REQUESTED);
        } else {   /* record command      */
	       char *p;
	       if (nc >= ncalloc) {
	          ncalloc = ncalloc*2 + 1;
	          commands = (commands)
	                   ? (char **)realloc(commands, ncalloc * sizeof(char *))
	                   : (char **)malloc(sizeof(char *));
	       }
	       p = (char *)malloc((unsigned)strlen(buf)+1);
	       if (!commands || !p) {
	          fprintf(stderr, "gnuplot: can't get memory. gnuplot_Be aborted.\n");
              be_app->PostMessage(B_QUIT_REQUESTED);
	       }
	       commands[nc++] = strcpy(p, buf);
	    } /* end if's over 1st character of input line */
      } /* end inside while */
      if (feof(Be_ipc) || ferror(Be_ipc)){
         be_app->PostMessage(B_QUIT_REQUESTED);
      } 
   } /* End infinite while */
}


/*-----------------------------------------------------------------------------
 *   display - display last plot from gnuplot inboard Be driver
 *---------------------------------------------------------------------------*/

void display(void)
{
   int x,y;
   int n;
   char *buf;
   BRect windowSize;
   /* set scaling factor between internal driver & window geometry */
   //   xscale = (double)DISPLAY_X / 4096.;  yscale = (double)DISPLAY_Y / 4096.;  
    
   // printf("I am in display()\n");
   
   /* if window is not open, open the window */
   if (gBeOSterm.window_open == false){
     gBeOSterm.window_open = true;
     gBeOSterm.theWindow= new GnuplotWindow;
   }
 
   windowSize = gBeOSterm.theWindow->Frame();
   xscale = windowSize.Width() / 4096.;  yscale = windowSize.Height() / 4096.;  

   /* tell the window to plot the graphics commands in commands[0..nc-1] */
//   gBeOSterm.theWindow->PlotGnuplotCommands();
   gBeOSterm.theWindow->Lock();
   gBeOSterm.theWindow->main_view->PlotGnuplotCommands();
   gBeOSterm.theWindow->Unlock();

}  


void PlotView::PlotGnuplotCommands()
{
/* Here is where the global variable commands[0..nc-1] will be read and interpreted.
   The drawing commands will be recorded in the picture to be rendered by the Draw
   command.
*/
   int n;
   char *buff;
   unsigned int x,y;
   JUSTIFY jmode=LEFT; /* justification mode */
   int sl;  /* the string length */
   char str[Nbuf]; /* To point to the string in the buffer. */
   float sw; /* width of the string in the current font */
   int xoff,yoff; /* offsets used for placing text */
   int lt;  /* line type */
   int point; /* point type */
   int ang=0;  /* text angle */
   int pointsize = 2; 
   int px,py;
   rgb_color penColor;
   BFont font;
   

#define MAX_DEFINED_LINETYPE 7
/* Pen types:
0 = black
1 = red
2 = green
3 = blue
4 = yellow
5 = purple
6 = darker green
7 = orange
*/
uint8 penColorRed[] = {
    0,        // black
    255,      // red
    0,        // green
    0,        // blue
    230,      // yellow
    255,      // purple
    0,        // darker green
    255,      // orange
};
uint8 penColorGreen[] = {
    0,        // black
    0,        // red
    255,      // green
    0,        // blue
    230,      // yellow
    0,        // purple
    170,      // darker green
    180,      // orange
};
uint8 penColorBlue[] = {
    0,        // black
    0,        // red
    0,        // green
    255,      // blue
    0,        // yellow
    255,      // purple
    0,        // darker green
    0,        // orange
};


   BeginPicture(the_picture);
   /* loop over accumulated commands from inboard driver */
   for (n=0; n<nc; n++) {
      buff = commands[n];
   
      if (*buff == 'V') {   /*   Be_vector(x,y) - draw vector  */
         sscanf(buff, "V%4d%4d", &x, &y);  
         //printf("I am drawing a line to %i,%i\n",X(x),Y(y));
         StrokeLine(BPoint(X(x),Y(y)));

      } else if (*buff == 'M') {  /*   Be_move(x,y) - move  */
         sscanf(buff, "M%4d%4d", &x, &y);  
	     //printf("I am moving to %i,%i\n",X(x),Y(y));
         MovePenTo(BPoint(X(x),Y(y)));
         
      } else if (*buff == 'T') { /*   Be_put_text(x,y,str) - draw text   */
	     sscanf(buff, "T%4d%4d", &x, &y);  
	     strcpy(str,buff+9);
	     sl = strlen(str) - 1;
	     /* coerce a properly terminated string (null out the cr)*/
	     str[sl]='\0';
	     //printf("I am printing the string %s of length %i at %i, %i\n",str,sl,X(x),Y(y));
         // Added justification feature */
         sw = StringWidth(str);
         // printf("StringWidth of %s is %f\n",str,sw);
         // For the moment let us assume that the text is horizontal 
         yoff = 0;
         xoff = 0;
         if (ang == 1){  /* Vertical text */
            switch (jmode){
               case LEFT:
                  yoff = 0;
                  break;
               case RIGHT:
                  yoff = sw;
                  break;
               case CENTER:
                  yoff = sw/2.0;
                  break;
            }
         } else {        /* Horizontal text */
            switch (jmode){
               case LEFT:
                  xoff = 0;
                  break;
               case RIGHT:
                  xoff = -sw;
                  break;
               case CENTER:
                  xoff = -sw/2.0;
                  break;
            }
         }
         DrawString(str,BPoint(X(x)+xoff,Y(y)+yoff));

      } else if (*buff == 'L') { /* Be_linetype(type) - set line type  */
         sscanf(buff, "L%4d", &lt);
         if ((lt>0) || (lt <= MAX_DEFINED_LINETYPE)){   
            penColor.red = penColorRed[lt];
            penColor.green = penColorGreen[lt];
            penColor.blue = penColorBlue[lt];
            penColor.alpha = 0;
            SetHighColor(penColor);
         }
         //printf("set line type to %i\n",lt);

      } else if (*buff == 'J') { /* Be_justify_text(type) - set justification */
         sscanf(buff, "J%4d", &jmode);          

      } else if (*buff == 'A'){ /* Be_text_angle(angle) */
        sscanf(buff,"A%4d", &ang);
        GetFont(&font);         
        if (ang == 0){
           font.SetRotation(0); 
        } else if (ang == 1){
           font.SetRotation(90); 
        }
        SetFont(&font, B_FONT_ROTATION);

      } else if (*buff == 'P') { /* Be_point() */ 
 	    sscanf(buff, "P%1d%4d%4d", &point, &x, &y);  
  	    PlotPoint(point,x,y);
  	    
      }  // end if point
   } // end loop over accumulated commands

   the_picture = EndPicture();
   Invalidate();
}  // end PlotView::PlotGnuplotCommands()

void	PlotView::PlotPoint(int which, int x, int y)
{
/* By Serge Droz and Tom Hays */

/* We plot a point here. O is just a dot. 1-6 are variously shaped points
   7 sets the point size scaling in x and y. In addition there is the
   uniform scaling parameter pointsize.
*/

/* the size of a point */
/* The P7xxxxyyyy command expected to be used when init'ing to useful values.*/
static int px=1;
static int py=1;

int XX = X(x);
int YY = Y(y);
   
switch (which) {
    case 0 : // a dot
       MovePenTo(BPoint(XX,YY));
       StrokeLine(BPoint(XX,YY));
       break;
    case 1: /* do diamond */  
       MovePenTo(BPoint(XX-px,YY));
       StrokeLine(BPoint(XX,YY-py));
       StrokeLine(BPoint(XX+px,YY));
       StrokeLine(BPoint(XX,YY+py));
       StrokeLine(BPoint(XX-px,YY));
       break;
    case 2: /* do plus */
      MovePenTo(BPoint(XX-px,YY));
      StrokeLine(BPoint(XX+px,YY));
      MovePenTo(BPoint(XX,YY-py));
      StrokeLine(BPoint(XX,YY+py));
      break;
    case 3: /* do box */ 
      MovePenTo(BPoint(XX-px,YY-py));
      StrokeLine(BPoint(XX+px,YY-py));
      StrokeLine(BPoint(XX+px,YY+py));
      StrokeLine(BPoint(XX-px,YY+py));
      StrokeLine(BPoint(XX-px,YY-py));
      break;
    case 4: /* do X */ 
      MovePenTo(BPoint(XX-px,YY-py));
      StrokeLine(BPoint(XX+px,YY+py));
      MovePenTo(BPoint(XX-px,YY+py));
      StrokeLine(BPoint(XX+px,YY-py));
      break;
    case 5: /* do triangle */ 
      MovePenTo(BPoint(XX-(int)(px*1.15),YY-(int)(py*0.58)));
      StrokeLine(BPoint(XX,YY+(int)(py*1.15)));
      StrokeLine(BPoint(XX+(int)(px*1.15),YY-(int)(py*0.58)));
      StrokeLine(BPoint(XX-(int)(px*1.15),YY-(int)(py*0.58)));
      break;   
    case 6: /* do star */ 
      MovePenTo(BPoint(XX-(int)(px*0.7),YY-(int)(py*0.7)));
      StrokeLine(BPoint(XX+(int)(px*0.7),YY+(int)(py*0.7)));
      MovePenTo(BPoint(XX+(int)(px*0.7),YY-(int)(py*0.7)));
      StrokeLine(BPoint(XX-(int)(px*0.7),YY+(int)(py*0.7)));
      MovePenTo(BPoint(XX-px,YY));
      StrokeLine(BPoint(XX+px,YY));
      MovePenTo(BPoint(XX,YY-py));
      StrokeLine(BPoint(XX,YY+py));
      break;
    case 7:
      px = (int) (x * xscale * pointsize);
 	  py = (int) (y * yscale * pointsize);
      break;
    };
}
           
