/******************************************************************************/
/* MikMod Player Interface.                                                   */
/*                                                                            */
/* Original player by Miodrag Vallet.                                         */
/* Ported to APlayer by Thomas Neumann.                                       */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PString.h"
#include "PFile.h"
#include "PAlert.h"

// APlayer headers
#include "APlayer.h"

// Player headers
#include "MikModTables.h"
#include "MikMod.h"
#include "MikMod_Lang.h"
#include "MikMod_LangStr.h"


/******************************************************************************/
/* Version                                                                    */
/******************************************************************************/
#define PlayerVersion		1.63



/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
MikMod::MikMod(APGlobalData *global) : APAddOnPlayer(global)
{
	// Fill out the version we have compiled under
	addOnVersion  = AP_ADDON_VERSION;
	playerVersion = AP_PLAYER_VERSION;
	agentVersion  = AP_AGENT_VERSION;

	// Initialize member variables
	readyToPlay = NULL;
	module      = NULL;

	// Register string resources
	strings = AddStringResource(mikStrings);
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
MikMod::~MikMod(void)
{
	// Unregister string resources
	RemoveStringResource(strings);
}



/******************************************************************************/
/* GetVersion() returns the version of the add-on.                            */
/*                                                                            */
/* Output: The add-on version.                                                */
/******************************************************************************/
float MikMod::GetVersion(void)
{
	return (PlayerVersion);
}



/******************************************************************************/
/* GetSupportFlags() returns some flags telling what the add-on supports.     */
/*                                                                            */
/* Output: Is the flags.                                                      */
/******************************************************************************/
uint32 MikMod::GetSupportFlags(void)
{
	return (appSetPosition);
}



/******************************************************************************/
/* GetName() returns the name of the current add-on.                          */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on name.                                                   */
/******************************************************************************/
PString MikMod::GetName(uint32 index)
{
	PString name;

	name.LoadString(strings, IDS_MIK_NAME);
	return (name);
}



/******************************************************************************/
/* GetDescription() returns the description of the current add-on.            */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on description.                                            */
/******************************************************************************/
PString MikMod::GetDescription(uint32 index)
{
	PString description;

	description.LoadString(strings, IDS_MIK_DESCRIPTION);
	return (description);
}



/******************************************************************************/
/* GetModTypeString() returns the module type string for the current player.  */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*                                                                            */
/* Output: The module type string.                                            */
/******************************************************************************/
PString MikMod::GetModTypeString(uint32 index)
{
	PString type;

	type.LoadString(strings, IDS_MIK_MIME);
	return (type);
}



/******************************************************************************/
/* ModuleCheck() tests the module to see if it's a MikMod module.             */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result MikMod::ModuleCheck(uint32 index, PFile *file)
{
	// Check the module size
	if (file->GetLength() < 25)
		return (AP_UNKNOWN);

	// Check the mark
	file->SeekToBegin();

	if ((file->Read_B_UINT32() == 'APUN') && (file->Read_B_UINT16() == 0x0104))
		return (AP_OK);

	return (AP_UNKNOWN);
}



/******************************************************************************/
/* LoadModule() will load the module into the memory.                         */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result MikMod::LoadModule(uint32 index, PFile *file)
{
	ap_result retVal;

	// Initialize the UNIMOD structure
	of.tracks      = NULL;
	of.patterns    = NULL;
	of.pattRows    = NULL;
	of.positions   = NULL;
	of.instruments = NULL;
	of.samples     = NULL;
	of.control     = NULL;
	of.voice       = NULL;

	// Parse the uni module and create structures to use
	retVal = CreateUniStructs(file);

	if (retVal != AP_OK)
		FreeAll();

	return (retVal);
}



/******************************************************************************/
/* InitPlayer() initialize the player.                                        */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool MikMod::InitPlayer(void)
{
	readyToPlay = new PMutex();
	if (readyToPlay == NULL)
		return (false);

	// Initialize mixer stuff
	md_sngChn = max(of.numChn, of.numVoices);

	return (true);
}



/******************************************************************************/
/* EndPlayer() ends the use of the player.                                    */
/******************************************************************************/
void MikMod::EndPlayer(void)
{
	// Destroy the mutex
	delete readyToPlay;
	readyToPlay = NULL;

	FreeAll();
}



/******************************************************************************/
/* CreateUniStructs() will create all the structs needed to play the module.  */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result MikMod::CreateUniStructs(PFile *file)
{
	uint32 v, w;
	SAMPLE *s;
	INSTRUMENT *i;

	// Skip the mark and version
	file->Read_B_UINT32();
	file->Read_B_UINT16();

	// Read the header
	of.flags      = file->Read_B_UINT16();
	of.numChn     = file->Read_UINT8();
	of.numVoices  = file->Read_UINT8();
	of.numPos     = file->Read_B_UINT16();
	of.numPat     = file->Read_B_UINT16();
	of.numTrk     = file->Read_B_UINT16();
	of.numIns     = file->Read_B_UINT16();
	of.numSmp     = file->Read_B_UINT16();
	of.repPos     = file->Read_B_UINT16();
	of.initSpeed  = file->Read_UINT8();
	of.initTempo  = file->Read_UINT8();
	of.initVolume = file->Read_UINT8();

	if (file->IsEOF())
	{
		ShowError(IDS_MIK_ERR_LOADING_HEADER);
		return (AP_ERROR);
	}

	of.songName = file->ReadString();
	of.comment  = file->ReadString();

	if (file->IsEOF())
	{
		ShowError(IDS_MIK_ERR_LOADING_HEADER);
		return (AP_ERROR);
	}

	// Allocate memory to hold all the information
	if (!(AllocSamples()))
	{
		ShowError(IDS_MIK_ERR_MEMORY);
		return (AP_ERROR);
	}

	if (!(AllocTracks()))
	{
		ShowError(IDS_MIK_ERR_MEMORY);
		return (AP_ERROR);
	}

	if (!(AllocPatterns()))
	{
		ShowError(IDS_MIK_ERR_MEMORY);
		return (AP_ERROR);
	}

	if (!(AllocPositions(of.numPos)))
	{
		ShowError(IDS_MIK_ERR_MEMORY);
		return (AP_ERROR);
	}

	// Read arrays
	for (v = 0; v < of.numPos; v++)
		of.positions[v] = file->Read_B_UINT16();

	for (v = 0; v < of.numChn; v++)
		of.panning[v] = file->Read_B_UINT16();

	for (v = 0; v < of.numChn; v++)
		of.chanVol[v] = file->Read_UINT8();

	if (file->IsEOF())
	{
		ShowError(IDS_MIK_ERR_LOADING_HEADER);
		return (AP_ERROR);
	}

	// Load sample headers
	s = of.samples;
	for (v = 0; v < of.numSmp; v++, s++)
	{
		s->flags      = file->Read_B_UINT16();
		s->speed      = file->Read_B_UINT32();
		s->volume     = file->Read_UINT8();
		s->panning    = file->Read_B_UINT16();
		s->length     = file->Read_B_UINT32();
		s->loopStart  = file->Read_B_UINT32();
		s->loopEnd    = file->Read_B_UINT32();
		s->susBegin   = file->Read_B_UINT32();
		s->susEnd     = file->Read_B_UINT32();

		s->globVol    = file->Read_UINT8();
		s->vibFlags   = file->Read_UINT8();
		s->vibType    = file->Read_UINT8();
		s->vibSweep   = file->Read_UINT8();
		s->vibDepth   = file->Read_UINT8();
		s->vibRate    = file->Read_UINT8();
		s->sampleName = file->ReadString();

		if (file->IsEOF())
		{
			ShowError(IDS_MIK_ERR_LOADING_SAMPLEINFO);
			return (AP_ERROR);
		}
	}

	// Load instruments
	if (of.flags & UF_INST)
	{
		if (!(AllocInstruments()))
		{
			ShowError(IDS_MIK_ERR_MEMORY);
			return (AP_ERROR);
		}

		i = of.instruments;
		for (v = 0; v < of.numIns; v++, i++)
		{
			i->flags        = file->Read_UINT8();
			i->nnaType      = file->Read_UINT8();
			i->dca          = file->Read_UINT8();
			i->dct          = file->Read_UINT8();
			i->globVol      = file->Read_UINT8();
			i->panning      = file->Read_B_UINT16();
			i->pitPanSep    = file->Read_UINT8();
			i->pitPanCenter = file->Read_UINT8();
			i->rVolVar      = file->Read_UINT8();
			i->rPanVar      = file->Read_UINT8();

			i->volFade      = file->Read_B_UINT16();

			i->volFlg       = file->Read_UINT8();
			i->volPts       = file->Read_UINT8();
			i->volSusBeg    = file->Read_UINT8();
			i->volSusEnd    = file->Read_UINT8();
			i->volBeg       = file->Read_UINT8();
			i->volEnd       = file->Read_UINT8();

			for (w = 0; w < 32; w++)
			{
				i->volEnv[w].pos = file->Read_B_UINT16();
				i->volEnv[w].val = file->Read_B_UINT16();
			}

			i->panFlg       = file->Read_UINT8();
			i->panPts       = file->Read_UINT8();
			i->panSusBeg    = file->Read_UINT8();
			i->panSusEnd    = file->Read_UINT8();
			i->panBeg       = file->Read_UINT8();
			i->panEnd       = file->Read_UINT8();

			for (w = 0; w < 32; w++)
			{
				i->panEnv[w].pos = file->Read_B_UINT16();
				i->panEnv[w].val = file->Read_B_UINT16();
			}

			i->pitFlg       = file->Read_UINT8();
			i->pitPts       = file->Read_UINT8();
			i->pitSusBeg    = file->Read_UINT8();
			i->pitSusEnd    = file->Read_UINT8();
			i->pitBeg       = file->Read_UINT8();
			i->pitEnd       = file->Read_UINT8();

			for (w = 0; w < 32; w++)
			{
				i->pitEnv[w].pos = file->Read_B_UINT16();
				i->pitEnv[w].val = file->Read_B_UINT16();
			}

			for (w = 0; w < 120; w++)
				i->sampleNumber[w] = file->Read_B_UINT16();

			for (w = 0; w < 120; w++)
				i->sampleNote[w] = file->Read_UINT8();

			i->insName = file->ReadString();

			if (file->IsEOF())
			{
				ShowError(IDS_MIK_ERR_LOADING_INSTRUMENTINFO);
				return (AP_ERROR);
			}
		}
	}

	// Read patterns
	for (v = 0; v < of.numPat; v++)
		of.pattRows[v] = file->Read_B_UINT16();

	for (v = 0; v < (uint32)(of.numPat * of.numChn); v++)
		of.patterns[v] = file->Read_B_UINT16();

	// Read tracks
	for (v = 0; v < of.numTrk; v++)
		of.tracks[v] = TrkRead(file);

	if (file->IsEOF())
	{
		ShowError(IDS_MIK_ERR_LOADING_TRACK);
		return (AP_ERROR);
	}

	// Calculate the sample addresses and fix the samples
	return (FindSamples(file));
}



/******************************************************************************/
/* TrkRead() allocates and read one track.                                    */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: A pointer to the tracker loaded.                                   */
/******************************************************************************/
uint8 *MikMod::TrkRead(PFile *file)
{
	uint8 *t;
	uint16 len, i;

	len = file->Read_B_UINT16();
	t   = new uint8[len];
	for (i = 0; i < len; i++)
		t[i] = file->Read_UINT8();

	return (t);
}



/******************************************************************************/
/* FindSamples() will find the sample addresses and fix them so all samples   */
/*      signed.                                                               */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
#define SLBUFSIZE		2048

ap_result MikMod::FindSamples(PFile *file)
{
	int32 v, w, length;
	SAMPLE *s;
	uint8 *samp;

	s = of.samples;
	for (v = 0; v < of.numSmp; v++, s++)
	{
		// Calculate the length of the sample
		length = s->length;

		if (length != 0)
		{
			int16 old = 0;
			int32 toDo;
			ITPACK status;
			uint16 inCnt;
			bool result;
			int32 cBlock = 0;
			uint8 *dest;

			if (s->flags & SF_16BITS)
				length *= 2;

			if (s->flags & SF_STEREO)
				length *= 2;

			// Allocate memory to hold the sample
			s->handle = new uint8[length];
			if (s->handle == NULL)
			{
				ShowError(IDS_MIK_ERR_MEMORY);
				return (AP_ERROR);
			}

			// Get the length again, because the loop
			// is based on samples, not bytes
			length = s->length;
			dest   = (uint8 *)s->handle;

			while (length)
			{
				toDo = min(length, SLBUFSIZE);

				if (s->flags & SF_ITPACKED)
				{
					// Decompress the sample
					if (!cBlock)
					{
						status.bits = (s->flags & SF_16BITS) ? 17 : 9;
						status.last = status.bufBits = 0;

						// Read the compressed length
						inCnt = file->Read_L_UINT16();

						cBlock = (s->flags & SF_16BITS) ? 0x4000 : 0x8000;

						if (s->flags & SF_DELTA)
							old = 0;
					}

					if (s->flags & SF_16BITS)
						result = DecompressIT16(&status, file, (int16 *)dest, toDo, &inCnt);
					else
						result = DecompressIT8(&status, file, (int8 *)dest, toDo, &inCnt);

					if (!result)
					{
						// Well, some error occurred in the decompressing
						ShowError(IDS_MIK_ERR_ITPACKING);
						return (AP_ERROR);
					}

					cBlock -= toDo;
				}
				else
				{
					// Read the sample into the memory
					if (s->flags & SF_16BITS)
					{
						if (s->flags & SF_BIG_ENDIAN)
							file->ReadArray_B_UINT16s((uint16 *)dest, toDo);
						else
							file->ReadArray_L_UINT16s((uint16 *)dest, toDo);
					}
					else
						file->Read(dest, toDo);
				}

				// Check for end of file
				if (file->IsEOF())
				{
					ShowError(IDS_MIK_ERR_LOADING_SAMPLE);
					return (AP_ERROR);
				}

				// Dedelta the sample
				if (s->flags & SF_DELTA)
				{
					samp = dest;

					if (s->flags & SF_16BITS)
					{
						for (w = 0; w < toDo; w++)
						{
							*((int16 *)samp) += old;
							old = *((int16 *)samp);
							samp += 2;
						}
					}
					else
					{
						for (w = 0; w < toDo; w++)
						{
							*samp += old;
							old = *samp++;
						}
					}
				}

				// Convert the sample to signed
				if (!(s->flags & SF_SIGNED))
				{
					samp = dest;

					if (s->flags & SF_16BITS)
					{
						for (w = 0; w < toDo; w++)
						{
							*((int16 *)samp) += 0x8000;
							samp += 2;
						}
					}
					else
					{
						for (w = 0; w < toDo; w++)
							*samp++ += 0x80;
					}
				}

				// Add number of samples to destination buffer
				dest += toDo;

				if (s->flags & SF_16BITS)
					dest += toDo;

				length -= toDo;
			}
		}
	}

	return (AP_OK);
}



/******************************************************************************/
/* DecompressIT8() decompress an 8-bit IT packed sample.                      */
/*                                                                            */
/* Input:  "status" is a pointer to the ITPACK structure.                     */
/*         "file" is a pointer to a file object to the file with the sample.  */
/*         "dest" is a pointer to where to store the unpacked data.           */
/*         "length" is the size of the destination buffer in samples.         */
/*         "inCnt" is a pointer the variable with the compressed length.      */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool MikMod::DecompressIT8(ITPACK *status, PFile *file, int8 *dest, uint32 length, uint16 *inCnt)
{
	int8 *end = dest + length;
	uint16 x, y, needBits, haveBits, newCount = 0;
	uint16 bits = status->bits;
	uint16 bufBits = status->bufBits;
	int8 last = status->last;
	uint8 buf = status->buf;

	while (dest < end)
	{
		needBits = newCount ? 3 : bits;
		x        = haveBits = 0;

		while (needBits)
		{
			// Feed buffer
			if (!bufBits)
			{
				if ((*inCnt)--)
					buf = file->Read_UINT8();
				else
					buf = 0;

				bufBits = 8;
			}

			// Get as many bits as necessary
			y     = needBits < bufBits ? needBits : bufBits;
			x    |= (buf & ((1 << y) - 1)) << haveBits;
			buf >>=y;

			bufBits  -= y;
			needBits -= y;
			haveBits += y;
		}

		if (newCount)
		{
			newCount = 0;

			if (++x >= bits)
				x++;

			bits = x;
			continue;
		}

		if (bits < 7)
		{
			if (x == (1 << (bits - 1)))
			{
				newCount = 1;
				continue;
			}
		}
		else if (bits < 9)
		{
			y = (0xff >> (9 - bits)) - 4;
			if ((x > y) && (x <= y + 8))
			{
				if ((x -= y) >= bits)
					x++;

				bits = x;
				continue;
			}
		}
		else if (bits < 10)
		{
			if (x >= 0x100)
			{
				bits = x - 0x100 + 1;
				continue;
			}
		}
		else
		{
			// Error in compressed data
			return (false);
		}

		if (bits < 8)	// Extend sign
			x = ((int8)(x << (8 - bits))) >> (8 - bits);

		*(dest++) = (last += x);
	}

	status->bits    = bits;
	status->bufBits = bufBits;
	status->last    = last;
	status->buf     = buf;

	return (true);
}



/******************************************************************************/
/* DecompressIT16() decompress an 16-bit IT packed sample.                    */
/*                                                                            */
/* Input:  "status" is a pointer to the ITPACK structure.                     */
/*         "file" is a pointer to a file object to the file with the sample.  */
/*         "dest" is a pointer to where to store the unpacked data.           */
/*         "length" is the size of the destination buffer in samples.         */
/*         "inCnt" is a pointer the variable with the compressed length.      */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool MikMod::DecompressIT16(ITPACK *status, PFile *file, int16 *dest, uint32 length, uint16 *inCnt)
{
	int16 *end = dest + length;
	int32 x, y, needBits, haveBits, newCount = 0;
	uint16 bits = status->bits;
	uint16 bufBits = status->bufBits;
	int16 last = status->last;
	uint8 buf = status->buf;

	while (dest < end)
	{
		needBits = newCount ? 4 : bits;
		x        = haveBits = 0;

		while (needBits)
		{
			// Feed buffer
			if (!bufBits)
			{
				if ((*inCnt)--)
					buf = file->Read_UINT8();
				else
					buf = 0;

				bufBits = 8;
			}

			// Get as many bits as necessary
			y     = needBits < bufBits ? needBits : bufBits;
			x    |= (buf & ((1 << y) - 1)) << haveBits;
			buf >>= y;

			bufBits  -= y;
			needBits -= y;
			haveBits += y;
		}

		if (newCount)
		{
			newCount = 0;

			if (++x >= bits)
				x++;

			bits = x;
			continue;
		}

		if (bits < 7)
		{
			if (x == (1 << (bits - 1)))
			{
				newCount = 1;
				continue;
			}
		}
		else if (bits < 17)
		{
			y = (0xffff >> (17 - bits)) - 8;
			if ((x > y) && (x <= y + 16))
			{
				if ((x -= y) >= bits)
					x++;

				bits = x;
				continue;
			}
		}
		else if (bits < 18)
		{
			if (x >= 0x10000)
			{
				bits = x - 0x10000 + 1;
				continue;
			}
		}
		else
		{
			// Error in compressed data
			return (false);
		}

		if (bits < 16)		// Extend sign
			x = ((int16)(x << (16 - bits))) >> (16 - bits);

		*(dest++) = (last += x);
	}

	status->bits    = bits;
	status->bufBits = bufBits;
	status->last    = last;
	status->buf     = buf;

	return (true);
}



/******************************************************************************/
/* ShowError() opens a message window and show the error.                     */
/*                                                                            */
/* Input:  "id" is the string id to show.                                     */
/******************************************************************************/
void MikMod::ShowError(uint32 id)
{
	PString title, msg;

	title.LoadString(strings, IDS_MIK_WIN_TITLE);
	msg.LoadString(strings, id);

	PAlert alert(title, msg, PAlert::pStop, PAlert::pOk);
	alert.Show();
}



/******************************************************************************/
/* InitSound() initialize the current song.                                   */
/*                                                                            */
/* Input:  "songNum" is the subsong to play.                                  */
/******************************************************************************/
void MikMod::InitSound(uint16 songNum)
{
	int32 t;

	// Start to lock the mutex
	readyToPlay->Lock();

	of.extSpd       = true;
	of.panFlag      = true;
	of.wrap         = true;
	of.loop         = true;
	of.fadeOut      = false;

	of.relSpd       = 0;

	of.sngTime      = 0;
	of.sngRemainder = 0;

	of.pat_repCrazy = 0;
	of.sngPos       = 0;
	of.sngSpd       = of.initSpeed ? (of.initSpeed < 32 ? of.initSpeed : 32) : 6;
	of.volume       = of.initVolume > 128 ? 128 : of.initVolume;

	of.vbTick       = of.sngSpd;
	of.patDly       = 0;
	of.patDly2      = 0;
	of.bpm          = of.initTempo < 32 ? 32 : of.initTempo;
	of.realChn      = 0;

	of.patPos       = 0;
	of.posJmp       = 2;			// Make sure the player fetches the first note
	of.numRow       = 0xffff;
 	of.patBrk       = 0;

	// Make sure the player doesn't start with garbage
	if ((of.control = new MP_CONTROL[of.numChn]) == NULL)
		return;

	if ((of.voice = new MP_VOICE[md_sngChn]) == NULL)
		return;

	// Clear the structures
	memset(of.control, 0, of.numChn * sizeof(MP_CONTROL));
	memset(of.voice, 0, md_sngChn * sizeof(MP_VOICE));

	for (t = 0; t < of.numChn; t++)
	{
		of.control[t].chanVol = of.chanVol[t];
		of.control[t].panning = of.panning[t];
	}

	// TN: Added some extra initialization. This is done elsewhere in
	// the original MikMod player
	mp_Channel = 0;
	a          = NULL;

	SetTempo(of.bpm);

	// Unlock the mutex again
	readyToPlay->Unlock();
}



/******************************************************************************/
/* EndSound() ends the current song.                                          */
/******************************************************************************/
void MikMod::EndSound(void)
{
	delete[] of.control;
	delete[] of.voice;
	of.control = NULL;
	of.voice   = NULL;
}



/******************************************************************************/
/* SetTempo() sets APlayer to the right BPM tempo.                            */
/*                                                                            */
/* Input:  "tempo" is the new tempo.                                          */
/******************************************************************************/
void MikMod::SetTempo(uint16 tempo)
{
	PString value;

	SetBPMTempo(tempo);

	value.Format("%u", tempo);
	ChangeModuleInfo(3, apValue, value);
}



/******************************************************************************/
/* GetModuleName() returns the name of the module.                            */
/*                                                                            */
/* Output: Is the module name.                                                */
/******************************************************************************/
PString MikMod::GetModuleName(void)
{
	return (of.songName);
}



/******************************************************************************/
/* GetVirtualChannels() returns the number of channels the module use.        */
/*                                                                            */
/* Output: Is the number of channels.                                         */
/******************************************************************************/
uint16 MikMod::GetVirtualChannels(void)
{
	return (of.numVoices);
}



/******************************************************************************/
/* GetModuleChannels() returns the number of channels the module want to use. */
/*                                                                            */
/* Output: Is the number of required channels.                                */
/******************************************************************************/
uint16 MikMod::GetModuleChannels(void)
{
	return (of.numChn);
}



/******************************************************************************/
/* GetSongLength() returns the length of the current song.                    */
/*                                                                            */
/* Output: Is the length of the current song.                                 */
/******************************************************************************/
int16 MikMod::GetSongLength(void)
{
	return (of.numPos);
}



/******************************************************************************/
/* GetSongPosition() returns the current position of the playing song.        */
/*                                                                            */
/* Output: Is the current position.                                           */
/******************************************************************************/
int16 MikMod::GetSongPosition(void)
{
	return (of.sngPos);
}



/******************************************************************************/
/* SetSongPosition() sets the current position of the playing song.           */
/*                                                                            */
/* Input:  "pos" is the new position.                                         */
/******************************************************************************/
void MikMod::SetSongPosition(int16 pos)
{
	int32 t;

	// Start to lock the mutex
	readyToPlay->Lock();

	// Change the position
	if (pos >= of.numPos)
		pos = of.numPos;

	of.posJmp = 2;
	of.patBrk = 0;
	of.sngPos = pos;
	of.vbTick = of.sngSpd;

	for (t = 0; t < md_sngChn; t++)
	{
		VoiceStop(t);
		of.voice[t].i = NULL;
		of.voice[t].s = NULL;
	}

	for (t = 0; t < of.numChn; t++)
	{
		of.control[t].i = NULL;
		of.control[t].s = NULL;
	}

	// Unlock the mutex again
	readyToPlay->Unlock();
}



/******************************************************************************/
/* GetInfoCount() returns the number of module info lines the player have.    */
/*                                                                            */
/* Output: The number of extra module info lines.                             */
/******************************************************************************/
uint32 MikMod::GetInfoCount(void)
{
	return (4);
}



/******************************************************************************/
/* GetInfoString() returns the description or value string on the line given. */
/*                                                                            */
/* Input:  "line" is the line starting from 0.                                */
/*         "type" is the type of string you want.                             */
/*                                                                            */
/* Output: The string you want.                                               */
/******************************************************************************/
PString MikMod::GetInfoString(uint32 line, ap_infoType type)
{
	PString retStr;

	if (type == apDescription)
	{
		// Description strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr.LoadString(strings, IDS_MIK_INFODESCLINE0);
				break;

			// Used Patterns
			case 1:
				retStr.LoadString(strings, IDS_MIK_INFODESCLINE1);
				break;

			// Supported/Used Samples
			case 2:
				retStr.LoadString(strings, IDS_MIK_INFODESCLINE2);
				break;

			// Actual Speed (BPM)
			case 3:
				retStr.LoadString(strings, IDS_MIK_INFODESCLINE3);
				break;
		}
	}
	else
	{
		// Value strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr.Format("%u", of.numPos);
				break;

			// Used Patterns
			case 1:
				retStr.Format("%u", of.numPat);
				break;

			// Supported/Used Samples
			case 2:
				retStr.Format("%u", of.numSmp);
				break;

			// Actual Speed (BPM)
			case 3:
				retStr.Format("%u", of.bpm);
				break;
		}
	}

	return (retStr);
}



/******************************************************************************/
/* GetInstrumentCount() returns the number of instruments the module have.    */
/*                                                                            */
/* Output: The number of instruments.                                         */
/******************************************************************************/
uint32 MikMod::GetInstrumentCount(void)
{
	if (of.flags & UF_INST)
		return (of.numIns);

	return (0);
}



/******************************************************************************/
/* GetInstrumentInfo() fills out the APInstInfo structure given with the      */
/*      instrument information of the instrument number given.                */
/*                                                                            */
/* Input:  "num" is the instrument number starting from 0.                    */
/*         "info" is a pointer to an APInstInfo structure to fill out.        */
/******************************************************************************/
void MikMod::GetInstrumentInfo(uint32 num, APInstInfo *info)
{
	INSTRUMENT *inst;

	// Get pointer to the instrument
	inst = &of.instruments[num];

	// Fill out the instrument structure
	info->name  = inst->insName;
	info->flags = 0;

	// Fill out the note samples
	for (int8 i = 0; i < 10 * 12; i++)
		info->notes[i] = inst->sampleNumber[i];
}



/******************************************************************************/
/* GetSampleCount() returns the number of samples the module have.            */
/*                                                                            */
/* Output: The number of samples.                                             */
/******************************************************************************/
uint32 MikMod::GetSampleCount(void)
{
	return (of.numSmp);
}



/******************************************************************************/
/* GetSampleInfo() fills out the APInstInfo structure given with the sample   */
/*      information of the sample number given.                               */
/*                                                                            */
/* Input:  "num" is the sample number starting from 0.                        */
/*         "info" is a pointer to an APSampleInfo structure to fill out.      */
/******************************************************************************/
void MikMod::GetSampleInfo(uint32 num, APSampleInfo *info)
{
	SAMPLE *samp;

	// Get the sample pointer
	samp = &of.samples[num];

	// Fill out the sample info structure
	info->name       = samp->sampleName;
	info->flags      = 0;
	info->type       = apSample;
	info->bitSize    = (samp->flags & SF_16BITS) ? 16 : 8;
	info->middleC    = (uint32)ceil(GetFrequency(of.flags, GetPeriod(96, samp->speed)));
	info->volume     = samp->volume * 4;
	info->panning    = (samp->panning == PAN_SURROUND) ? APPAN_SURROUND : samp->panning;
	info->address    = samp->handle;
	info->length     = samp->length;
	info->loopStart  = samp->loopStart;
	info->loopLength = samp->loopEnd - samp->loopStart;

	// Add extra loop flags if any
	if (samp->flags & SF_LOOP)
		info->flags |= APSAMP_LOOP;

	if (samp->flags & SF_BIDI)
		info->flags |= APSAMP_PINGPONG;
}



/******************************************************************************/
/* Play() is the main player function.                                        */
/******************************************************************************/
void MikMod::Play(void)
{
	if (readyToPlay->Lock(0) == pSyncOk)
	{
		// Play the module
		Player_HandleTick();

		// Unlock the mutex
		readyToPlay->Unlock();
	}
}



/******************************************************************************/
/* Player_HandleTick() is the main player function.                           */
/******************************************************************************/
void MikMod::Player_HandleTick(void)
{
	int32 max_Volume;

	if (of.sngPos >= of.numPos)
		return;

	// Update time counter (sngTime is in milliseconds (in fact 2^-10))
	of.sngRemainder += (1 << 9) * 5;	// Thus 2.5 * (1 << 10), since fps=0.4 * tempo
	of.sngTime      += of.sngRemainder / of.bpm;
	of.sngRemainder %= of.bpm;

	if (++of.vbTick >= of.sngSpd)
	{
		if (of.pat_repCrazy)
			of.pat_repCrazy = 0;		// Play 2 times row 0
		else
			of.patPos++;

		of.vbTick = 0;

		// Process pattern-delay. of.patDly2 is the counter and of.patDly
		// is the command memory
		if (of.patDly)
		{
			of.patDly2 = of.patDly;
			of.patDly  = 0;
		}

		if (of.patDly2)
		{
			// Patterndelay active
			if (--of.patDly2)
			{
				// So turn back of.patPos by 1
				if (of.patPos)
					of.patPos--;
			}
		}

		// Do we have to get a new patternpointer ?
		// (when of.patPos reaches the pattern size or when
		// a patternbreak is active)
		if (((of.patPos >= of.numRow) && (of.numRow > 0)) && (!of.posJmp))
			of.posJmp = 3;

		if (of.posJmp)
		{
			of.patPos       = of.numRow ? (of.patBrk % of.numRow) : 0;
			of.pat_repCrazy = 0;
			of.sngPos      += (of.posJmp - 2);

			for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
				of.control[mp_Channel].pat_repPos = -1;

			of.patBrk = of.posJmp = 0;

			// Tell APlayer we has changed position
			ChangePosition();

			// Handle the "---" (end of song) pattern since it can occur
			// *inside* the module in .IT and .S3M
			if ((of.sngPos >= of.numPos) || (of.positions[of.sngPos] == 255))
			{
				if (!of.wrap)
					return;

				if (!(of.sngPos = of.repPos))
				{
					of.volume = of.initVolume > 128 ? 128 : of.initVolume;
					of.sngSpd = of.initSpeed ? (of.initSpeed < 32 ? of.initSpeed : 32) : 6;
					of.bpm    = of.initTempo < 32 ? 32 : of.initTempo;
				}

				// Tell APlayer we has restarted
				endReached = true;
			}

			if (of.sngPos < 0)
				of.sngPos = of.numPos - 1;
		}

		if (!of.patDly2)
			PT_Notes();
	}

	if (((of.sngPos == (of.numPos - 1)) || (of.positions[of.sngPos + 1] == 255)) && (of.fadeOut))
		max_Volume = of.numRow ? ((of.numRow - of.patPos) * 128) / of.numRow : 0;
	else
		max_Volume = 128;

	PT_EffectsPass1();

	if (of.flags & UF_NNA)
		PT_NNA();

	PT_SetupVoices();
	PT_EffectsPass2();

	// Now setup the actual hardware channel playback information
	PT_UpdateVoices(max_Volume);
}



/******************************************************************************/
/* PT_UpdateVoices() changes the voices, e.g. create the envelopes.           */
/*                                                                            */
/* Input:  "max_Volume" is the maximum volume.                                */
/******************************************************************************/
void MikMod::PT_UpdateVoices(int32 max_Volume)
{
	int16 envPan, envVol, envPit;
	uint16 playPeriod;
	int32 vibVal, vibDpt;
	uint32 tmpVol;
	MP_VOICE *aout;
	INSTRUMENT *i;
	SAMPLE *s;
	uint16 bpmTempo;

	of.totalChn = of.realChn = 0;

	for (mp_Channel = 0; mp_Channel < md_sngChn; mp_Channel++)
	{
		aout = &of.voice[mp_Channel];
		i    = aout->i;
		s    = aout->s;

		if ((!s) || (!s->length))
			continue;

		if (aout->period < 40)
			aout->period = 40;
		else
			if (aout->period > 50000)
				aout->period = 50000;

		if ((aout->kick == KICK_NOTE) || (aout->kick == KICK_KEYOFF))
		{
			VoicePlay(mp_Channel, s, (aout->start == -1) ? ((s->flags & SF_UST_LOOP) ? s->loopStart : 0) : aout->start);

			aout->fadeVol = 32768;
			aout->aSwpPos = 0;
		}

		if (i && ((aout->kick == KICK_NOTE) || (aout->kick == KICK_ENV)))
		{
			StartEnvelope(&aout->vEnv, aout->volFlg, i->volPts, i->volSusBeg, i->volSusEnd, i->volBeg, i->volEnd, i->volEnv, aout->keyOff);
			StartEnvelope(&aout->pEnv, aout->panFlg, i->panPts, i->panSusBeg, i->panSusEnd, i->panBeg, i->panEnd, i->panEnv, aout->keyOff);
			StartEnvelope(&aout->cEnv, aout->pitFlg, i->pitPts, i->pitSusBeg, i->pitSusEnd, i->pitBeg, i->pitEnd, i->pitEnv, aout->keyOff);

			if (aout->cEnv.flg & EF_ON)
				aout->masterPeriod = GetPeriod((uint16)aout->note << 1, aout->master->speed);
		}

		aout->kick = KICK_ABSENT;

		envVol = (!(aout->volFlg & EF_ON)) ? 256 : ProcessEnvelope(&aout->vEnv, 256, aout->keyOff);
		envPan = (!(aout->panFlg & EF_ON)) ? PAN_CENTER : ProcessEnvelope(&aout->pEnv, PAN_CENTER, aout->keyOff);
		envPit = (!(aout->pitFlg & EF_ON)) ? 32 : ProcessEnvelope(&aout->cEnv, 32, aout->keyOff);

		tmpVol  = aout->fadeVol;		// Max 32768
		tmpVol *= aout->chanVol;		// * max 64
		tmpVol *= aout->volume;			// * max 256
		tmpVol /= 16384L;				// tmpVol is max 32768

		aout->totalVol = tmpVol >> 2;	// TotalVolume used to determine samplevolume

		tmpVol *= envVol;				// * max 256
		tmpVol *= of.volume;			// * max 128
		tmpVol /= 4194304UL;

		// Fade out
		if (of.sngPos >= of.numPos)
			tmpVol = 0;
		else
			tmpVol = (tmpVol * max_Volume) / 128;

		if ((aout->masterChn != -1) && of.control[aout->masterChn].muted)
		{
			// Channel muting line
			VoiceSetVolume(mp_Channel, 0);
		}
		else
		{
			VoiceSetVolume(mp_Channel, tmpVol);
			if ((tmpVol) && (aout->master) && (aout->master->slave == aout))
				of.realChn++;

			of.totalChn++;
		}

		if (aout->panning == PAN_SURROUND)
			VoiceSetPanning(mp_Channel, PAN_SURROUND);
		else
		{
			if ((of.panFlag) && (aout->pEnv.flg & EF_ON))
				VoiceSetPanning(mp_Channel, DoPan(envPan, aout->panning));
			else
				VoiceSetPanning(mp_Channel, aout->panning);
		}

		if (aout->period && s->vibDepth)
		{
			switch (s->vibType)
			{
				case 0:
					vibVal = aVibTab[s->aVibPos & 127];
					if (s->aVibPos & 0x80)
						vibVal = -vibVal;
					break;

				case 1:
					vibVal = 64;
					if (s->aVibPos & 0x80)
						vibVal = -vibVal;
					break;

				case 2:
					vibVal = 63 - (((s->aVibPos + 128) & 255) >> 1);
					break;

				default:
					vibVal = (((s->aVibPos + 128) & 255) >> 1) - 64;
					break;
			}
		}
		else
			vibVal = 0;

		if (s->vibFlags & AV_IT)
		{
			if ((aout->aSwpPos >> 8) < s->vibDepth)
			{
				aout->aSwpPos += s->vibSweep;
				vibDpt         = aout->aSwpPos;
			}
			else
				vibDpt = s->vibDepth << 8;

			vibVal = (vibVal * vibDpt) >> 16;

			if (aout->mFlag)
			{
				if (!(of.flags & UF_LINEAR))
					vibVal >>= 1;

				aout->period -= vibVal;
			}
		}
		else
		{
			// Do XM style auto-vibrato
			if (!(aout->keyOff & KEY_OFF))
			{
				if (aout->aSwpPos < s->vibSweep)
				{
					vibDpt = (aout->aSwpPos * s->vibDepth) / s->vibSweep;
					aout->aSwpPos++;
				}
				else
					vibDpt = s->vibDepth;
			}
			else
			{
				// Key-off -> depth becomes 0 if final depth wasn't reached or
				// stays at final level if depth WAS reached
				if (aout->aSwpPos >= s->vibSweep)
					vibDpt = s->vibDepth;
				else
					vibDpt = 0;
			}

			vibVal        = (vibVal * vibDpt) >> 8;
			aout->period -= vibVal;
		}

		// Update vibrato position
		s->aVibPos = (s->aVibPos + s->vibRate) & 0xff;

		// Process pitch envelope
		playPeriod = aout->period;

		if ((aout->pitFlg & EF_ON) && (envPit != 32))
		{
			int32 p1;

			envPit -= 32;
			if ((aout->note << 1) + envPit <= 0)
				envPit = -(aout->note << 1);

			p1 = GetPeriod(((uint16)aout->note << 1) + envPit, aout->master->speed) - aout->masterPeriod;

			if (p1 > 0)
			{
				if ((uint16)(playPeriod + p1) <= playPeriod)
				{
					p1 = 0;
					aout->keyOff |= KEY_OFF;
				}
			}
			else
			{
				if (p1 < 0)
				{
					if ((uint16)(playPeriod + p1) >= playPeriod)
					{
						p1 = 0;
						aout->keyOff |= KEY_OFF;
					}
				}
			}

			playPeriod += p1;
		}

		if (!aout->fadeVol)		// Check for a dead note (fadevol = 0)
		{
			VoiceStop(mp_Channel);
			of.totalChn--;

			if ((tmpVol) && (aout->master) && (aout->master->slave == aout))
				of.realChn--;
		}
		else
		{
			VoiceSetFrequency(mp_Channel, GetFrequency(of.flags, playPeriod));

			// If keyFade, start subtracting FadeOutSpeed from fadeVol:
			if ((i) && (aout->keyOff & KEY_FADE))
			{
				if (aout->fadeVol >= i->volFade)
					aout->fadeVol -= i->volFade;
				else
					aout->fadeVol = 0;
			}
		}
	}

	bpmTempo = of.bpm + of.relSpd;
	if (bpmTempo < 32)
		bpmTempo = 32;
	else
		if (bpmTempo > 255)
			bpmTempo = 255;

	SetTempo(bpmTempo);
}



/******************************************************************************/
/* PT_Notes() handles new notes or instruments.                               */
/******************************************************************************/
void MikMod::PT_Notes(void)
{
	uint8 c, inst;
	int32 tr, funky;	// Funky is set to indicate note or instrument change

	for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
	{
		a = &of.control[mp_Channel];

		if (of.sngPos >= of.numPos)
		{
			tr        = of.numTrk;
			of.numRow = 0;
		}
		else
		{
			tr        = of.patterns[(of.positions[of.sngPos] * of.numChn) + mp_Channel];
			of.numRow = of.pattRows[of.positions[of.sngPos]];
		}

		a->row = (tr < of.numTrk) ? uniTrk.UniFindRow(of.tracks[tr], of.patPos) : NULL;
		a->newSamp = 0;

		if (!of.vbTick)
			a->noteDelay = 0;

		if (!a->row)
			continue;

		uniTrk.UniSetRow(a->row);
		funky = 0;

		while ((c = uniTrk.UniGetByte()))
		{
			switch (c)
			{
				case UNI_NOTE:
					funky     |= 1;
					a->oldNote = a->aNote;
					a->aNote   = uniTrk.UniGetByte();
					a->kick    = KICK_NOTE;
					a->start   = -1;
					a->sliding = 0;

					// Retrig tremolo and vibrato waves?
					if (!(a->waveControl & 0x80))
						a->trmPos = 0;

					if (!(a->waveControl & 0x08))
						a->vibPos = 0;

					if (!a->panbWave)
						a->panbPos = 0;
					break;

				case UNI_INSTRUMENT:
					inst   = uniTrk.UniGetByte();

					if (inst >= of.numIns)
						break;				// Safety valve

					funky       |= 2;
					a->i         = (of.flags & UF_INST) ? &of.instruments[inst] : NULL;
					a->retrig    = 0;
					a->s3mTremor = 0;
					a->sample    = inst;
					break;

				default:
					uniTrk.UniSkipOpcode(c);
					break;
			}
		}

		if (funky)
		{
			INSTRUMENT *i;
			SAMPLE *s;

			i = a->i;

			if (i)
			{
				if (i->sampleNumber[a->aNote] >= of.numSmp)
					continue;

				s       = &of.samples[i->sampleNumber[a->aNote]];
				a->note = i->sampleNote[a->aNote];
			}
			else
			{
				a->note = a->aNote;
				s       = &of.samples[a->sample];
			}

			if (a->s != s)
			{
				a->s       = s;
				a->newSamp = a->period;
			}

			// Channel or instrument determined panning?
			a->panning = of.panning[mp_Channel];

			if (s->flags & SF_OWNPAN)
				a->panning = s->panning;
			else
				if ((i) && (i->flags & IF_OWNPAN))
					a->panning = i->panning;

			a->handle = s->handle;
			a->speed  = s->speed;

			if (i)
			{
				if ((of.panFlag) && (i->flags & IF_PITCHPAN) && (a->panning != PAN_SURROUND))
				{
					a->panning += ((a->aNote - i->pitPanCenter) * i->pitPanSep) / 8;

					if (a->panning < PAN_LEFT)
						a->panning = PAN_LEFT;
					else
						if (a->panning > PAN_RIGHT)
							a->panning = PAN_RIGHT;
				}

				a->pitFlg = i->pitFlg;
				a->volFlg = i->volFlg;
				a->panFlg = i->panFlg;
				a->nna    = i->nnaType;
				a->dca    = i->dca;
				a->dct    = i->dct;
			}
			else
			{
				a->pitFlg = 0;
				a->volFlg = 0;
				a->panFlg = 0;
				a->nna    = 0;
				a->dca    = 0;
				a->dct    = 0;
			}

			if (funky & 2)	// Instrument change
			{
				// IT random volume variations: 0:8 bit fixed, and one bit for sign
				a->volume = a->tmpVolume = s->volume;

				if ((s) && (i))
				{
					if (i->rVolVar)
					{
						a->volume = a->tmpVolume = s->volume + ((s->volume * ((int32)i->rVolVar * (int32)GetRandom(512))) / 25600);

						if (a->volume < 0)
							a->volume = a->tmpVolume = 0;
						else
							if (a->volume > 64)
								a->volume = a->tmpVolume = 64;
					}

					if ((of.panFlag) && (a->panning != PAN_SURROUND))
					{
						a->panning += ((a->panning * ((int32)i->rPanVar * (int32)GetRandom(512))) / 25600);

						if (a->panning < PAN_LEFT)
							a->panning = PAN_LEFT;
						else
							if (a->panning > PAN_RIGHT)
								a->panning = PAN_RIGHT;
					}
				}
			}

			a->wantedPeriod = a->tmpPeriod = GetPeriod((uint16)a->note << 1, a->speed);
			a->keyOff       = KEY_KICK;
		}
	}
}



/******************************************************************************/
/* PT_EffectsPass1() handles effects.                                         */
/******************************************************************************/
void MikMod::PT_EffectsPass1(void)
{
	MP_VOICE *aout;

	for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
	{
		a = &of.control[mp_Channel];

		if ((aout = a->slave))
		{
			a->fadeVol = aout->fadeVol;
			a->period  = aout->period;

			if (a->kick == KICK_KEYOFF)
				a->keyOff = aout->keyOff;
		}

		if (!a->row)
			continue;

		uniTrk.UniSetRow(a->row);

		a->ownPer = a->ownVol = 0;
		explicitSlides = 0;
		PT_PlayEffects();

		// Continue volume slide if necessary for XM and IT
		if (of.flags & UF_BGSLIDES)
		{
			if (!explicitSlides)
			{
				switch (a->sliding)
				{
					case 1:
						DoS3MVolSlide(0);
						break;

					case 2:
						DoXMVolSlide(0);
						break;
				}
			}
			else
			{
				if (a->tmpVolume)
					a->sliding = explicitSlides;
			}
		}

		if (!a->ownPer)
			a->period = a->tmpPeriod;

		if (!a->ownVol)
			a->volume = a->tmpVolume;

		if (a->s)
		{
			if (a->i)
				a->outVolume = (a->volume * a->s->globVol * a->i->globVol) >> 10;
			else
				a->outVolume = (a->volume * a->s->globVol) >> 4;

			if (a->outVolume > 256)
				a->volume = 256;
			else
				if (a->outVolume < 0)
					a->outVolume = 0;
		}
	}
}



/******************************************************************************/
/* PT_EffectsPass2() second effect pass.                                      */
/******************************************************************************/
void MikMod::PT_EffectsPass2(void)
{
	uint8 c;

	for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
	{
		a = &of.control[mp_Channel];

		if (!a->row)
			continue;

		uniTrk.UniSetRow(a->row);

		while ((c = uniTrk.UniGetByte()))
		{
			if (c == UNI_ITEFFECTS0)
			{
				c = uniTrk.UniGetByte();
				if ((c >> 4) == SS_S7EFFECTS)
					DoNNAEffects(c & 0xf);
			}
			else
				uniTrk.UniSkipOpcode(c);
		}
	}
}



/******************************************************************************/
/* PT_NNA() manages the NNA.                                                  */
/******************************************************************************/
void MikMod::PT_NNA(void)
{
	for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
	{
		a = &of.control[mp_Channel];

		if (a->kick == KICK_NOTE)
		{
			bool k = false;

			if (a->slave)
			{
				MP_VOICE *aout;

				aout = a->slave;
				if (aout->nna & NNA_MASK)
				{
					// Make sure the old MP_VOICE channel knows it has no master now!
					a->slave = NULL;

					// Assume the channel is taken by NNA
					aout->mFlag = 0;

					switch (aout->nna)
					{
						case NNA_CONTINUE:	// Continue note, do nothing
							break;

						case NNA_OFF:		// Note off
							aout->keyOff |= KEY_OFF;

							if ((!(aout->volFlg & EF_ON)) || (aout->volFlg & EF_LOOP))
								aout->keyOff = KEY_KILL;
							break;

						case NNA_FADE:
							aout->keyOff |= KEY_FADE;
							break;
					}
				}
			}

			if (a->dct != DCT_OFF)
			{
				int32 t;

				for (t = 0; t < md_sngChn; t++)
				{
					if ((!VoiceStopped(t)) && (of.voice[t].masterChn == mp_Channel) && (a->sample == of.voice[t].sample))
					{
						k = false;

						switch (a->dct)
						{
							case DCT_NOTE:
								if (a->note == of.voice[t].note)
									k = true;
								break;

							case DCT_SAMPLE:
								if (a->handle == of.voice[t].handle)
									k = true;
								break;

							case DCT_INST:
								k = true;
								break;
						}

						if (k)
						{
							switch (a->dca)
							{
								case DCA_CUT:
									of.voice[t].fadeVol = 0;
									break;

								case DCA_OFF:
									of.voice[t].keyOff |= KEY_OFF;
									if ((!(of.voice[t].volFlg & EF_ON)) || (of.voice[t].volFlg & EF_LOOP))
										of.voice[t].keyOff = KEY_KILL;
									break;

								case DCA_FADE:
									of.voice[t].keyOff |= KEY_FADE;
									break;
							}
						}
					}
				}
			}
		}	// if (a->kick == KICK_NOTE)
	}
}



/******************************************************************************/
/* PT_SetupVoices() setup module and NNA voices.                              */
/******************************************************************************/
void MikMod::PT_SetupVoices(void)
{
	MP_VOICE *aout;

	for (mp_Channel = 0; mp_Channel < of.numChn; mp_Channel++)
	{
		a = &of.control[mp_Channel];

		if (a->noteDelay)
			continue;

		if (a->kick == KICK_NOTE)
		{
			// If no channel was cut above, find an empty or quiet channel here
			if (of.flags & UF_NNA)
			{
				if (!a->slave)
				{
					int32 newChn;

					if ((newChn = FindEmptyChannel()) != -1)
						a->slave = &of.voice[a->slaveChn = newChn];
				}
			}
			else
				a->slave = &of.voice[a->slaveChn = mp_Channel];

			// Assign parts of MP_VOICE only done for a KICK!
			if ((aout = a->slave))
			{
				if (aout->mFlag && aout->master)
					aout->master->slave = NULL;

				a->slave        = aout;
				aout->master    = a;
				aout->masterChn = mp_Channel;
				aout->mFlag     = 1;
			}
		}
		else
			aout = a->slave;

		if (aout)
		{
			aout->kick    = a->kick;
			aout->i       = a->i;
			aout->s       = a->s;
			aout->sample  = a->sample;
			aout->handle  = a->handle;
			aout->period  = a->period;
			aout->panning = a->panning;
			aout->chanVol = a->chanVol;
			aout->fadeVol = a->fadeVol;
			aout->start   = a->start;
			aout->volFlg  = a->volFlg;
			aout->panFlg  = a->panFlg;
			aout->pitFlg  = a->pitFlg;
			aout->volume  = a->outVolume;
			aout->keyOff  = a->keyOff;
			aout->note    = a->note;
			aout->nna     = a->nna;
		}

		a->kick = KICK_ABSENT;
	}
}



/******************************************************************************/
/* GetRandom() returns a random value between 0 and ceil - 1, ceil must be a  */
/*      power of two.                                                         */
/*                                                                            */
/* Input:  "ceil" the maximum number to get + 1.                              */
/*                                                                            */
/* Output: A random number.                                                   */
/******************************************************************************/
int32 MikMod::GetRandom(int32 ceil)
{
	return ((int32)((rand() * ceil) / (RAND_MAX + 1.0)));
}



/******************************************************************************/
/* GetPeriod() calculates the notes period and return it.                     */
/*                                                                            */
/* Input:  "note" is the note to calculate the period on.                     */
/*         "speed" is the middle C speed.                                     */
/*                                                                            */
/* Output: The period.                                                        */
/******************************************************************************/
uint16 MikMod::GetPeriod(uint16 note, uint32 speed)
{
	if (of.flags & UF_XMPERIODS)
		return ((of.flags & UF_LINEAR) ? GetLinearPeriod(note, speed) : GetLogPeriod(note, speed));

	return (GetOldPeriod(note, speed));
}



/******************************************************************************/
/* GetOldPeriod() calculates the notes period and return it.                  */
/*                                                                            */
/* Input:  "note" is the note to calculate the period on.                     */
/*         "speed" is the middle C speed.                                     */
/*                                                                            */
/* Output: The period.                                                        */
/******************************************************************************/
uint16 MikMod::GetOldPeriod(uint16 note, uint32 speed)
{
	uint16 n, o;

	if (!speed)
		return (4242);		// Prevent divide overflow.. (42 eheh)

	n = note % (2 * OCTAVE);
	o = note / (2 * OCTAVE);
	return (((8363L * (uint32)oldPeriods[n]) >> o) / speed);
}



/******************************************************************************/
/* GetLinearPeriod() calculates the notes period and return it.               */
/*                                                                            */
/* Input:  "note" is the note to calculate the period on.                     */
/*         "fine" is the middle C speed.                                      */
/*                                                                            */
/* Output: The period.                                                        */
/******************************************************************************/
uint16 MikMod::GetLinearPeriod(uint16 note, uint32 fine)
{
	uint16 t;

	t = (20L * OCTAVE + 2 - note) * 32L - (fine >> 1);
	return (t);
}



/******************************************************************************/
/* GetLogPeriod() calculates the notes period and return it.                  */
/*                                                                            */
/* Input:  "note" is the note to calculate the period on.                     */
/*         "fine" is the middle C speed.                                      */
/*                                                                            */
/* Output: The period.                                                        */
/******************************************************************************/
uint16 MikMod::GetLogPeriod(uint16 note, uint32 fine)
{
	uint16 n, o;
	uint16 p1, p2;
	uint32 i;

	n = note % (2 * OCTAVE);
	o = note / (2 * OCTAVE);
	i = (n << 2) + (fine >> 4);	// n * 8 + fine / 16

	p1 = logTab[i];
	p2 = logTab[i + 1];

	return (Interpolate(fine >> 4, 0, 15, p1, p2) >> o);
}



/******************************************************************************/
/* GetFrequency() XM linear period to frequency conversion.                   */
/*                                                                            */
/* Input:  "flags" is the instrument flags.                                   */
/*         "period" is the period to convert.                                 */
/*                                                                            */
/* Output: The frequency.                                                     */
/******************************************************************************/
uint32 MikMod::GetFrequency(uint8 flags, uint32 period)
{
	if (flags & UF_LINEAR)
		return (linTab[period % 768] >> (period / 768));
	else
		return ((8363L * 1712L) / (period ? period : 1));
}



/******************************************************************************/
/* Interpolate() interpolates?                                                */
/******************************************************************************/
int16 MikMod::Interpolate(int16 p, int16 p1, int16 p2, int16 v1, int16 v2)
{
	if ((p1 == p2) || (p == p1))
		return (v1);

	return (v1 + ((int32)((p - p1) * (v2 - v1)) / (p2 - p1)));
}



/******************************************************************************/
/* Interpolate() interpolates the envelope.                                   */
/******************************************************************************/
int16 MikMod::InterpolateEnv(int16 p, ENVPT *a, ENVPT *b)
{
	return (Interpolate(p, a->pos, b->pos, a->val, b->val));
}



/******************************************************************************/
/* DoPan() calculates the panning value.                                      */
/*                                                                            */
/* Input:  "envPan" is the envelope panning value.                            */
/*         "pan" is the channels panning.                                     */
/*                                                                            */
/* Output: The new panning.                                                   */
/******************************************************************************/
int16 MikMod::DoPan(int16 envPan, int16 pan)
{
	int32 newPan;

	newPan = pan + (((envPan - PAN_CENTER) * (128 - abs(pan - PAN_CENTER))) / 128);

	return ((newPan < PAN_LEFT) ? PAN_LEFT : ((newPan > PAN_RIGHT) ? PAN_RIGHT : newPan));
}



/******************************************************************************/
/* StartEnvelope() initialize and start the envelope.                         */
/******************************************************************************/
void MikMod::StartEnvelope(ENVPR *t, uint8 flg, uint8 pts, uint8 susBeg, uint8 susEnd, uint8 beg, uint8 end, ENVPT *p, uint8 keyOff)
{
	t->flg    = flg;
	t->pts    = pts;
	t->susBeg = susBeg;
	t->susEnd = susEnd;
	t->beg    = beg;
	t->end    = end;
	t->env    = p;
	t->p      = 0;
	t->a      = 0;
	t->b      = ((t->flg & EF_SUSTAIN) && (!(keyOff & KEY_OFF))) ? 0: 1;

	// Imago Orpheus sometimes stores an extra initial point in the envelope
	if ((t->pts >= 2) && (t->env[0].pos == t->env[1].pos))
	{
		t->a++;
		t->b++;
	}

	if (t->b >= t->pts)
		t->b = t->pts - 1;
}



/******************************************************************************/
/* ProcessEnvelope() calculates the next envelope value.                      */
/*                                                                            */
/* This procedure processes all envelope types, include volume, pitch and     */
/* panning. Envelopes are defined by a set of points, each with a magnitude   */
/* [relating either to volume, panning position or pitch modifier] and a      */
/* tick position.                                                             */
/*                                                                            */
/* Envelopes work in the following manner:                                    */
/*                                                                            */
/* (a) Each tick the envelope is moved a point further in its progression.    */
/*   1. For an accurate progression, magnitudes between two envelope points   */
/*      are interpolated.                                                     */
/*                                                                            */
/* (b) When progression reaches a defined point on the envelope, values are   */
/*     shifted to interpolate between this point and the next, and checks for */
/*     loops or envelope end are done.                                        */
/*                                                                            */
/* Misc:                                                                      */
/*     Sustain loops are loops that are only active as long as the keyoff     */
/*     flag is clear. When a volume envelope terminates, so does the current  */
/*     fadeout.                                                               */
/******************************************************************************/
int16 MikMod::ProcessEnvelope(ENVPR *t, int16 v, uint8 keyOff)
{
	if (t->flg & EF_ON)
	{
		uint8 a, b;			// Actual points in the envelope
		uint16 p;			// The 'tick counter' - real point being played

		a = t->a;
		b = t->b;
		p = t->p;

		// If sustain loop on one point (XM type), don't move and don't
		// interpolate when the point is reached
		if ((t->flg & EF_SUSTAIN) && (t->susBeg == t->susEnd) &&
			(!(keyOff & KEY_OFF)) && (p == t->env[t->susBeg].pos))
		{
			v = t->env[t->susBeg].val;
		}
		else
		{
			// Compute the current envelope value between points a and b
			if (a == b)
				v = t->env[a].val;
			else
				v = InterpolateEnv(p, &t->env[a], &t->env[b]);

			p++;

			// Pointer reached point b?
			if (p >= t->env[b].pos)
			{
				a = b++;		// Shift points a and b

				// Check for loops, sustain loops or end of envelope
				if ((t->flg & EF_SUSTAIN) && (!(keyOff & KEY_OFF)) && (b > t->susEnd))
				{
					a = t->susBeg;
					b = (t->susBeg == t->susEnd) ? a : a + 1;
					p = t->env[a].pos;
				}
				else
				{
					if ((t->flg & EF_LOOP) && (b > t->end))
					{
						a = t->beg;
						b = (t->beg == t->end) ? a : a + 1;
						p = t->env[a].pos;
					}
					else
					{
						if (b >= t->pts)
						{
							if ((t->flg & EF_VOLENV) && (mp_Channel != -1))
							{
								of.voice[mp_Channel].keyOff |= KEY_FADE;

								if (!v)
									of.voice[mp_Channel].fadeVol = 0;
							}

							b--;
							p--;
						}
					}
				}
			}

			t->a = a;
			t->b = b;
			t->p = p;
		}
	}

	return (v);
}



/******************************************************************************/
/* FindEmptyChannel() returns MP_CONTROL index of free channel.               */
/*                                                                            */
/* New note action scoring system:                                            */
/* -------------------------------                                            */
/*  1) Total-volume (fadeVol, chanVol, volume) is the main scorer             */
/*  2) A looping sample is a bonus x2                                         */
/*  3) A forground channel is a bonus x4                                      */
/*  4) An active envelope with keyoff is a handicap -x2                       */
/*                                                                            */
/* Output: The channel index to use.                                          */
/******************************************************************************/
int32 MikMod::FindEmptyChannel(void)
{
	MP_VOICE *a;
	uint32 t, k, tVol, pp;

	for (t = 0; t < md_sngChn; t++)
	{
		if (((of.voice[t].kick == KICK_ABSENT) || (of.voice[t].kick == KICK_ENV)) && (VoiceStopped(t)))
			return (t);
	}

	tVol = 0xffffffUL;
	t    = 0;
	a    = of.voice;

	for (k = 0; k < md_sngChn; k++, a++)
	{
		if ((a->kick == KICK_ABSENT) || (a->kick == KICK_ENV))
		{
			pp = a->totalVol << ((a->s->flags & SF_LOOP) ? 1 : 0);
			if ((a->master) && (a == a->master->slave))
				pp <<= 2;

			if (pp < tVol)
			{
				tVol = pp;
				t    = k;
			}
		}
	}

	if (tVol > 8000 * 7)
		return (-1);

	return (t);
}



/******************************************************************************/
/* PT_PlayEffects() parse the effects.                                        */
/******************************************************************************/
void MikMod::PT_PlayEffects(void)
{
	uint8 dat, c;

	while ((c = uniTrk.UniGetByte()))
	{
		int8 oldSliding = a->sliding;
		a->sliding      = 0;
	
		switch (c)
		{
			case UNI_NOTE:
			case UNI_INSTRUMENT:
				a->sliding = oldSliding;
				uniTrk.UniSkipOpcode(c);
				break;

			// PT Arpeggio
			case UNI_PTEFFECT0:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if ((!dat) && (of.flags & UF_ARPMEM))
						dat = a->arpMem;

					a->arpMem = dat;
				}

				if (a->period)
					DoArpeggio(a->arpMem);
				break;

			// PT Portamento up
			case UNI_PTEFFECT1:
				dat = uniTrk.UniGetByte();

				if ((!of.vbTick) && (dat))
					a->slideSpeed = (uint16)dat << 2;

				if (a->period)
					if (of.vbTick)
						a->tmpPeriod -= a->slideSpeed;
				break;

			// PT Portamento down
			case UNI_PTEFFECT2:
				dat = uniTrk.UniGetByte();

				if ((!of.vbTick) && (dat))
					a->slideSpeed = (uint16)dat << 2;

				if (a->period)
					if (of.vbTick)
						a->tmpPeriod += a->slideSpeed;
				break;

			// PT Tone Portamento
			case UNI_PTEFFECT3:
				dat = uniTrk.UniGetByte();

				if ((!of.vbTick) && (dat))
					a->portSpeed = dat << 2;

				if (a->period)
				{
					if (!a->fadeVol)
						a->kick = (a->kick == KICK_NOTE) ? KICK_NOTE : KICK_KEYOFF;
					else
						a->kick = (a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;

					DoToneSlide();
					a->ownPer = 1;
				}
				break;

			// PT Vibrato & XM Vibrato
			case UNI_PTEFFECT4:
			case UNI_XMEFFECT4:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->vibDepth = dat & 0x0f;

					if (dat & 0xf0)
						a->vibSpd = (dat & 0xf0) >> 2;
				}
				else
				{
					if (a->period)
					{
						DoVibrato();
						a->ownPer = 1;
					}
				}
				break;

			// PT TonePlusVol
			case UNI_PTEFFECT5:
				dat = uniTrk.UniGetByte();

				if (a->period)
				{
					if (!a->fadeVol)
						a->kick = (a->kick == KICK_NOTE) ? KICK_NOTE : KICK_KEYOFF;
					else
						a->kick = (a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;

					DoToneSlide();
					a->ownPer = 1;
				}

				DoVolSlide(dat);
				break;

			// PT VibratoPlusVol
			case UNI_PTEFFECT6:
				dat = uniTrk.UniGetByte();

				if ((a->period) && (of.vbTick))
				{
					DoVibrato();
					a->ownPer = 1;
				}

				DoVolSlide(dat);
				break;

			// PT Tremolo
			case UNI_PTEFFECT7:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->trmDepth = dat & 0x0f;

					if (dat & 0xf0)
						a->trmSpd = (dat & 0xf0) >> 2;
				}

				if (a->period)
				{
					DoTremolo();
					a->ownVol = 1;
				}
				break;

			// Panning
			case UNI_PTEFFECT8:
				dat = uniTrk.UniGetByte();

				if (of.panFlag)
					a->panning = of.panning[mp_Channel] = dat;
				break;

			// PT SampleOffset
			case UNI_PTEFFECT9:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat)
						a->sOffset = (uint16)dat << 8;

					a->start = a->hiOffset | a->sOffset;
					if ((a->s) && ((uint32)a->start > a->s->length))
						a->start = a->s->flags & (SF_LOOP | SF_BIDI) ? a->s->loopStart : a->s->length;
				}
				break;

			// PT VolumeSlide
			case UNI_PTEFFECTA:
				DoVolSlide(uniTrk.UniGetByte());
				break;

			// PT PositionJump
			case UNI_PTEFFECTB:
				dat = uniTrk.UniGetByte();

				if ((of.vbTick) || (of.patDly2))
					break;

				// Vincent Voois uses a nasty trick in "Universal Bolero"
				if ((dat == of.sngPos) && (of.patBrk == of.patPos))
					break;

				if ((!of.loop) && (!of.patBrk) && ((dat < of.sngPos) ||
					((of.sngPos == of.numPos - 1) && (!of.patBrk)) ||
					((dat == of.sngPos) && (of.flags & UF_NOWRAP))))
				{
					// If we don't loop, better not to skip the end of the
					// pattern, after all... so:
					of.posJmp = 3;
				}
				else
				{
					// If we were fading, adjust...
					if (of.sngPos == (of.numPos - 1))
						of.volume = of.initVolume > 128 ? 128 : of.initVolume;

					if (of.sngPos > dat)
					{
						// Tell APlayer the module has ended
						endReached = true;
					}

					of.sngPos = dat;
					of.posJmp = 2;
					of.patPos = 0;
				}
				break;

			// PT VolumeChange
			case UNI_PTEFFECTC:
				dat = uniTrk.UniGetByte();

				if (of.vbTick)
					break;

				if (dat == (uint8)-1)
					a->aNote = dat = 0;		// Note cut
				else
					if (dat > 64)
						dat = 64;

				a->tmpVolume = dat;
				break;

			// PT PatternBreak
			case UNI_PTEFFECTD:
				dat = uniTrk.UniGetByte();

				if ((of.vbTick) || (of.patDly2))
					break;

				if ((of.positions[of.sngPos] != 255) && (dat > of.pattRows[of.positions[of.sngPos]]))
					dat = of.pattRows[of.positions[of.sngPos]];

				of.patBrk = dat;

				if (!of.posJmp)
				{
					// Don't ask me to explain this code - it makes
					// backwards.s3m and children.xm (heretic's version) play
					// correctly, among others. Take that for granted, or write
					// the page of comments yourself... you might need some
					// aspirin - Miod
					if ((of.sngPos == of.numPos - 1) && (dat) && ((of.loop) || (of.positions[of.sngPos] == (of.numPat - 1) && !(of.flags & UF_NOWRAP))))
					{
						of.sngPos = 0;
						of.posJmp = 2;
					}
					else
						of.posJmp = 3;
				}
				break;

			// PT ExtraEffects
			case UNI_PTEFFECTE:
				DoEEffects(uniTrk.UniGetByte());
				break;

			// PT SetSpeed
			case UNI_PTEFFECTF:
				dat = uniTrk.UniGetByte();

				if (of.vbTick || of.patDly2)
					break;

				if (of.extSpd && (dat >= 0x20))
					of.bpm = dat;
				else
				{
					if (dat)
					{
						of.sngSpd = (dat > 32) ? 32 : dat;
						of.vbTick = 0;
					}
				}
				break;

			// S3M SetSpeed
			case UNI_S3MEFFECTA:
				DoS3MSpeed(uniTrk.UniGetByte());
				break;

			// S3M VolumeSlide / Fine VolumeSlide
			case UNI_S3MEFFECTD:
				DoS3MVolSlide(uniTrk.UniGetByte());
				break;

			// S3M Slide / Fine Slide / Extra Fine Slide Down
			case UNI_S3MEFFECTE:
				dat = uniTrk.UniGetByte();

				if (a->period)
					DoS3MSlideDn(dat);
				break;

			// S3M Slide / Fine Slide / Extra Fine Slide Up
			case UNI_S3MEFFECTF:
				dat = uniTrk.UniGetByte();

				if (a->period)
					DoS3MSlideUp(dat);
				break;

			// S3M Tremor
			case UNI_S3MEFFECTI:
				DoS3MTremor(uniTrk.UniGetByte());
				a->ownVol = 1;
				break;

			// S3M Retrig + VolumeSlide
			case UNI_S3MEFFECTQ:
				dat = uniTrk.UniGetByte();

				if (a->period)
					DoS3MRetrig(dat);
				break;

			// S3M Tremolo
			case UNI_S3MEFFECTR:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->trmDepth = dat & 0xf;

					if (dat & 0xf0)
						a->trmSpd = (dat & 0xf0) >> 2;
				}

				DoS3MTremolo();
				a->ownVol = 1;
				break;

			// S3M Tempo
			case UNI_S3MEFFECTT:
				DoS3MTempo(uniTrk.UniGetByte());
				break;

			// S3M Fine Vibrato
			case UNI_S3MEFFECTU:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->vibDepth = dat & 0xf;

					if (dat & 0xf0)
						a->vibSpd = (dat & 0xf0) >> 2;
				}
				else
					if (a->period)
					{
						DoS3MFineVibrato();
						a->ownPer = 1;
					}
				break;

			// XM Volume Slide
			case UNI_XMEFFECTA:
				DoXMVolSlide(uniTrk.UniGetByte());
				break;

			// XM Fine portamento slide up
			case UNI_XMEFFECTE1:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat)
						a->fPortUpSpd = dat;

					if (a->period)
						a->tmpPeriod -= (a->fPortUpSpd << 2);
				}
				break;

			// XM Fine portamento slide down
			case UNI_XMEFFECTE2:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat)
						a->fPortDnSpd = dat;

					if (a->period)
						a->tmpPeriod += (a->fPortDnSpd << 2);
				}
				break;

			// XM Fine volume slide up
			case UNI_XMEFFECTEA:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
					if (dat)
						a->fSlideUpSpd = dat;

				a->tmpVolume += a->fSlideUpSpd;

				if (a->tmpVolume > 64)
					a->tmpVolume = 64;
				break;

			// XM Fine volume slide down
			case UNI_XMEFFECTEB:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
					if (dat)
						a->fSlideDnSpd = dat;

				a->tmpVolume -= a->fSlideDnSpd;

				if (a->tmpVolume < 0)
					a->tmpVolume = 0;
				break;

			// XM Set global volume
			case UNI_XMEFFECTG:
				of.volume = uniTrk.UniGetByte() << 1;

				if (of.volume > 128)
					of.volume = 128;
				break;

			// XM Global volume slide
			case UNI_XMEFFECTH:
				DoXMGlobalSlide(uniTrk.UniGetByte());
				break;

			// XM Set envelope position
			case UNI_XMEFFECTL:
				dat = uniTrk.UniGetByte();

				if ((!of.vbTick) && (a->i))
				{
					uint16 points;
					INSTRUMENT *i = a->i;
					MP_VOICE *aout;

					if ((aout = a->slave))
					{
						points = i->volEnv[i->volPts - 1].pos;
						aout->vEnv.p = aout->vEnv.env[(dat > points) ? points : dat].pos;

						points = i->panEnv[i->panPts - 1].pos;
						aout->pEnv.p = aout->pEnv.env[(dat > points) ? points : dat].pos;
					}
				}
				break;

			// XM Panning slide
			case UNI_XMEFFECTP:
				DoXMPanSlide(uniTrk.UniGetByte());

				dat = uniTrk.UniGetByte();
				if (of.panFlag)
					DoXMPanSlide(dat);
				break;

			// XM Extra fine portamento up
			case UNI_XMEFFECTX1:
				dat = uniTrk.UniGetByte();

				if (dat)
					a->ffPortUpSpd = dat;
				else
					dat = a->ffPortUpSpd;

				if (a->period)
				{
					DoXMExtraFineSlideUp(dat);
					a->ownPer = 1;
				}
				break;

			// XM Extra fine portamento down
			case UNI_XMEFFECTX2:
				dat = uniTrk.UniGetByte();

				if (dat)
					a->ffPortDnSpd = dat;
				else
					dat = a->ffPortDnSpd;

				if (a->period)
				{
					DoXMExtraFineSlideDown(dat);
					a->ownPer = 1;
				}
				break;

			// IT Volume/Panning
			case UNI_VOLEFFECTS:
				DoVolEffects(uniTrk.UniGetByte());
				break;

			// IT Tone portamento
			case UNI_ITEFFECTG:
				dat = uniTrk.UniGetByte();

				if (dat)
					a->portSpeed = dat;

				if (a->period)
				{
					if ((!of.vbTick) && (a->newSamp))
					{
						a->kick  = KICK_NOTE;
						a->start = -1;
					}
					else
						a->kick = (a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;

					DoITToneSlide();
					a->ownPer = 1;
				}
				break;

			// IT Vibrato
			case UNI_ITEFFECTH:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->vibDepth = dat & 0xf;

					if (dat & 0xf0)
						a->vibSpd = (dat & 0xf0) >> 2;
				}

				if (a->period)
				{
					DoITVibrato();
					a->ownPer = 1;
				}
				break;

			// IT Tremor
			case UNI_ITEFFECTI:
				DoITTremor(uniTrk.UniGetByte());
				a->ownVol = 1;
				break;

			// IT Set channel volume
			case UNI_ITEFFECTM:
				a->chanVol = uniTrk.UniGetByte();

				if (a->chanVol > 64)
					a->chanVol = 64;
				else
					if (a->chanVol < 0)
						a->chanVol = 0;
				break;

			// IT Slide / Fine slide channel volume
			case UNI_ITEFFECTN:
				DoITChanVolSlide(uniTrk.UniGetByte());
				break;

			// IT Slide / Fine slide channel panning
			case UNI_ITEFFECTP:
				DoITPanSlide(uniTrk.UniGetByte());

				dat = uniTrk.UniGetByte();
				if (of.panFlag)
					DoITPanSlide(dat);
				break;

			// IT Special
			case UNI_ITEFFECTS0:
				DoSSEffects(uniTrk.UniGetByte());
				break;

			// IT Slide / Fine slide tempo
			case UNI_ITEFFECTT:
				DoITTempo(uniTrk.UniGetByte());
				break;

			// IT Fine vibrato
			case UNI_ITEFFECTU:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->vibDepth = dat & 0xf;

					if (dat & 0xf0)
						a->vibSpd = (dat & 0xf0) >> 2;
				}

				if (a->period)
				{
					DoITFineVibrato();
					a->ownPer = 1;
				}
				break;

			// IT Slide / Fine slide global volume
			case UNI_ITEFFECTW:
				DoITGlobalSlide(uniTrk.UniGetByte());
				break;

			// IT Panbrello
			case UNI_ITEFFECTY:
				dat = uniTrk.UniGetByte();

				if (!of.vbTick)
				{
					if (dat & 0x0f)
						a->panbDepth = (dat & 0xf);

					if (dat & 0xf0)
						a->panbSpd = (dat & 0xf0) >> 4;
				}

				DoITPanbrello();
				if (of.panFlag)
					DoITPanbrello();
				break;

			// IT Resonant filters
			case UNI_ITEFFECTZ:
				// FIXME: Not implemented yet
				uniTrk.UniSkipOpcode(UNI_ITEFFECTZ);
				break;

			// ULT SampleOffset
			case UNI_ULTEFFECT9:		// TN: Added this effect
				DoULTSampleOffset();
				break;

			// UniMod effects
			case UNI_KEYOFF:
				a->keyOff |= KEY_OFF;

				if ((!(a->volFlg & EF_ON)) || (a->volFlg & EF_LOOP))
					a->keyOff = KEY_KILL;
				break;

			case UNI_KEYFADE:
				dat = uniTrk.UniGetByte();

				if ((of.vbTick >= dat) || (of.vbTick == of.sngSpd - 1))
				{
					a->keyOff = KEY_KILL;
					if (!(a->volFlg & EF_ON))
						a->fadeVol = 0;
				}
				break;

			// Unknown effect
			default:
				a->sliding = oldSliding;
				uniTrk.UniSkipOpcode(c);
				break;
		}
	}
}



/******************************************************************************/
/* DoEEffects() parse the ProTracker Extra effects.                           */
/******************************************************************************/
void MikMod::DoEEffects(uint8 dat)
{
	uint8 nib = dat & 0xf;

	switch (dat >> 4)
	{
		// Hardware filter toggle, not supported
		case 0x0:
			break;

		// Fineslide up
		case 0x1:
			if (a->period)
				if (!of.vbTick)
					a->tmpPeriod -= (nib << 2);
			break;

		// Fineslide down
		case 0x2:
			if (a->period)
				if (!of.vbTick)
					a->tmpPeriod += (nib << 2);
			break;

		// Glissando control
		case 0x3:
			a->glissando = nib;
			break;

		// Set vibrato waveform
		case 0x4:
			a->waveControl &= 0xf0;
			a->waveControl |= nib;
			break;

		// Set finetune
		case 0x5:
			if (a->period)
			{
				if (of.flags & UF_XMPERIODS)
					a->speed = nib + 128;
				else
					a->speed = fineTune[nib];

				a->tmpPeriod = GetPeriod((uint16)a->note << 1, a->speed);
			}
			break;

		// Set patternloop
		case 0x6:
			if (of.vbTick)
				break;

			if (nib)		// Set repPos or repCnt?
			{
				// Set repCnt, so check if repCnt already is set,
				// which means we are already looping
				if (a->pat_repCnt)
					a->pat_repCnt--;		// Already looping, decrease counter
				else
				{
#if 0
					// This would make walker.xm, shipped with Xsoundtracker,
					// play correctly, but it's better to remain compatible
					// with FT2
					if ((!(of.flags & UF_NOWRAP)) || (a->pat_repPos != POS_NONE))
#endif
						a->pat_repCnt = nib;	// Not yet looping, so set repCnt
				}

				if (a->pat_repCnt)			// Jump to repPos if repCnt > 0
				{
					if (a->pat_repPos == POS_NONE)
						a->pat_repPos = of.patPos - 1;

					if (a->pat_repPos == -1)
					{
						of.pat_repCrazy = 1;
						of.patPos = 0;
					}
					else
						of.patPos = a->pat_repPos;
				}
				else
					a->pat_repPos = POS_NONE;
			}
			else
				a->pat_repPos = of.patPos - 1;	// Set repPos - cen be (-1)

			break;

		// Set tremolo waveform
		case 0x7:
			a->waveControl &= 0x0f;
			a->waveControl |= nib << 4;
			break;

		// Set panning
		case 0x8:
			if (of.panFlag)
			{
				if (nib <= 8)
					nib <<= 4;
				else
					nib *= 17;

				a->panning = of.panning[mp_Channel] = nib;
			}
			break;

		// Retrig note
		case 0x9:
			// Only retrigger if data nibble > 0
			if (nib)
			{
				if (!a->retrig)
				{
					// When retrig counter reaches 0,
					// reset counter and restart the sample
					if (a->period)
						a->kick = KICK_NOTE;

					a->retrig = nib;
				}

				a->retrig--;	// Countdown
			}
			break;

		// Fine volume slide up
		case 0xa:
			if (of.vbTick)
				break;

			a->tmpVolume += nib;
			if (a->tmpVolume > 64)
				a->tmpVolume = 64;
			break;

		// Fine volume slide down
		case 0xb:
			if (of.vbTick)
				break;

			a->tmpVolume -= nib;
			if (a->tmpVolume < 0)
				a->tmpVolume = 0;
			break;

		// Cut note
		case 0xc:
			// When of.vbTick reaches the cut-note value,
			// turn the volume to zero (Just like on the Amiga)
			if (of.vbTick >= nib)
				a->tmpVolume = 0;	// Just turn the volume down
			break;

		// Note delay
		case 0xd:
			// Delay the start of the sample until of.vbTick == nib
			if (!of.vbTick)
				a->noteDelay = nib;
			else
				if (a->noteDelay)
					a->noteDelay--;
			break;

		// Pattern delay
		case 0xe:
			if (of.vbTick)
				break;

			if (!of.patDly2)
				of.patDly = nib + 1;	// Only once (when of.vbTick = 0)
			break;

		// Invert loop, not supported
		case 0xf:
			break;
	}
}



/******************************************************************************/
/* ProTracker effect 0: Arpeggio or normal note.                              */
/******************************************************************************/
void MikMod::DoArpeggio(uint8 dat)
{
	uint8 note = a->note;

	if (dat)
	{
		switch (of.vbTick % 3)
		{
			case 1:
				note += (dat >> 4);
				break;

			case 2:
				note += (dat & 0x0f);
				break;
		}

		a->period = GetPeriod((uint16)note << 1, a->speed);
		a->ownPer = 1;
	}
}



/******************************************************************************/
/* ProTracker effect 3: Toneportamento.                                       */
/******************************************************************************/
void MikMod::DoToneSlide(void)
{
	if (of.vbTick)
	{
		int32 dist;

		// We have to slide a->period towards a->wantedPeriod, so
		// compute the difference between those two values
		dist = a->period - a->wantedPeriod;

		// If they are equal or if portamentoSpeed is too big...
		if ((!dist) || (a->portSpeed > abs(dist)))
		{
			// ...make tmpPeriod equal tperiod
			a->tmpPeriod = a->period = a->wantedPeriod;
		}
		else
		{
			if (dist > 0)
			{
				a->tmpPeriod -= a->portSpeed;
				a->period -= a->portSpeed;	// dist > 0, slide up
			}
			else
			{
				a->tmpPeriod += a->portSpeed;
				a->period += a->portSpeed;	// dist < 0, slide down
			}
		}
	}
	else
		a->tmpPeriod = a->period;
}



/******************************************************************************/
/* ProTracker effect 4: Vibrato.                                              */
/******************************************************************************/
void MikMod::DoVibrato(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->vibPos >> 2) & 0x1f;

	switch (a->waveControl & 3)
	{
		case 0:		// Sine
			temp = vibratoTable[q];
			break;

		case 1:		// Ramp down
			q <<= 3;

			if (a->vibPos < 0)
				q = 255 - q;

			temp = q;
			break;

		case 2:		// Square wave
			temp = 255;
			break;

		case 3:		// Random wave
			temp = GetRandom(256);
			break;
	}

	temp  *= a->vibDepth;
	temp >>= 7;
	temp <<= 2;

	if (a->vibPos >= 0)
		a->period = a->tmpPeriod + temp;
	else
		a->period = a->tmpPeriod - temp;

	if (of.vbTick)
		a->vibPos += a->vibSpd;
}



/******************************************************************************/
/* ProTracker effect 7: Tremolo.                                              */
/******************************************************************************/
void MikMod::DoTremolo(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->trmPos >> 2) & 0x1f;

	switch ((a->waveControl >> 4) & 3)
	{
		case 0:		// Sine
			temp = vibratoTable[q];
			break;

		case 1:		// Ramp down
			q <<= 3;

			if (a->trmPos < 0)
				q = 255 - q;

			temp = q;
			break;

		case 2:		// Square wave
			temp = 255;
			break;

		case 3:		// Random wave
			temp = GetRandom(256);
			break;
	}

	temp  *= a->trmDepth;
	temp >>= 6;

	if (a->trmPos >= 0)
	{
		a->volume = a->tmpVolume + temp;

		if (a->volume > 64)
			a->volume = 64;
	}
	else
	{
		a->volume = a->tmpVolume - temp;

		if (a->volume < 0)
			a->volume = 0;
	}

	if (of.vbTick)
		a->trmPos += a->trmSpd;
}



/******************************************************************************/
/* ProTracker effect A: Volume slide.                                         */
/******************************************************************************/
void MikMod::DoVolSlide(uint8 dat)
{
	if (!of.vbTick)
		return;

	if (dat & 0xf)
	{
		a->tmpVolume -= (dat & 0x0f);

		if (a->tmpVolume < 0)
			a->tmpVolume = 0;
	}
	else
	{
		a->tmpVolume += (dat >> 4);

		if (a->tmpVolume > 64)
			a->tmpVolume = 64;
	}
}



/******************************************************************************/
/* ScreamTracker 3 Specific Effects                                           */
/******************************************************************************/

/******************************************************************************/
/* S3M Effect A: Set speed                                                    */
/******************************************************************************/
void MikMod::DoS3MSpeed(uint8 speed)
{
	if (of.vbTick || of.patDly2)
		return;

	if (speed > 128)
		speed -= 128;

	if (speed)
	{
		of.sngSpd = speed;
		of.vbTick = 0;
	}
}



/******************************************************************************/
/* S3M Effect D: Volume slide / Fine Volume slide                             */
/******************************************************************************/
void MikMod::DoS3MVolSlide(uint8 inf)
{
	uint8 lo, hi;

	explicitSlides = 1;

	if (inf)
		a->s3mVolSlide = inf;
	else
		inf = a->s3mVolSlide;

	lo  = inf & 0xf;
	hi  = inf >> 4;

	if (!lo)
	{
		if ((of.vbTick) || (of.flags & UF_S3MSLIDES))
			a->tmpVolume += hi;
	}
	else
	{
		if (!hi)
		{
			if ((of.vbTick) || (of.flags & UF_S3MSLIDES))
				a->tmpVolume -= lo;
		}
		else
		{
			if (lo == 0xf)
			{
				if (!of.vbTick)
					a->tmpVolume += (hi ? hi : 0xf);
			}
			else
			{
				if (hi == 0xf)
				{
					if (!of.vbTick)
						a->tmpVolume -= (lo ? lo : 0xf);
				}
				else
					return;
			}
		}
	}

	if (a->tmpVolume < 0)
		a->tmpVolume = 0;
	else
	{
		if (a->tmpVolume > 64)
			a->tmpVolume = 64;
	}
}



/******************************************************************************/
/* S3M Effect E: Slide / Fine Slide / Extra Fine Slide Down                   */
/******************************************************************************/
void MikMod::DoS3MSlideDn(uint8 inf)
{
	uint8 hi, lo;

	if (inf)
		a->slideSpeed = inf;
	else
		inf = a->slideSpeed;

	hi  = inf >> 4;
	lo  = inf & 0xf;

	if (hi == 0xf)
	{
		if (!of.vbTick)
			a->tmpPeriod += (uint16)lo << 2;
	}
	else
	{
		if (hi == 0xe)
		{
			if (!of.vbTick)
				a->tmpPeriod += lo;
		}
		else
		{
			if (of.vbTick)
				a->tmpPeriod += (uint16)inf << 2;
		}
	}
}



/******************************************************************************/
/* S3M Effect F: Slide / Fine Slide / Extra Fine Slide Up                     */
/******************************************************************************/
void MikMod::DoS3MSlideUp(uint8 inf)
{
	uint8 hi, lo;

	if (inf)
		a->slideSpeed = inf;
	else
		inf = a->slideSpeed;

	hi  = inf >> 4;
	lo  = inf & 0xf;

	if (hi == 0xf)
	{
		if (!of.vbTick)
			a->tmpPeriod -= (uint16)lo << 2;
	}
	else
	{
		if (hi == 0xe)
		{
			if (!of.vbTick)
				a->tmpPeriod -= lo;
		}
		else
		{
			if (of.vbTick)
				a->tmpPeriod -= (uint16)inf << 2;
		}
	}
}



/******************************************************************************/
/* S3M Effect I: Tremor                                                       */
/******************************************************************************/
void MikMod::DoS3MTremor(uint8 inf)
{
	uint8 on, off;

	if (inf)
		a->s3mTrOnOf = inf;
	else
	{
		inf = a->s3mTrOnOf;
		if (!inf)
			return;
	}

	if (!of.vbTick)
		return;

	on  = (inf >> 4) + 1;
	off = (inf & 0xf) + 1;

	a->s3mTremor %= (on + off);
	a->volume = (a->s3mTremor < on) ? a->tmpVolume : 0;
	a->s3mTremor++;
}



/******************************************************************************/
/* S3M Effect Q: Retrig + VolumeSlide                                         */
/******************************************************************************/
void MikMod::DoS3MRetrig(uint8 inf)
{
	if (inf)
	{
		a->s3mRtgSlide = inf >> 4;
		a->s3mRtgSpeed = inf & 0xf;
	}

	// Only retrigger if low nibble > 0
	if (a->s3mRtgSpeed > 0)
	{
		if (!a->retrig)
		{
			// When retrig counter reaches 0,
			// reset counter and restart the sample
			if (a->kick != KICK_NOTE)
				a->kick = KICK_KEYOFF;

			a->retrig = a->s3mRtgSpeed;

			if ((of.vbTick) || (of.flags & UF_S3MSLIDES))
			{
				switch (a->s3mRtgSlide)
				{
					case 1:
					case 2:
					case 3:
					case 4:
					case 5:
						a->tmpVolume -= (1 << (a->s3mRtgSlide - 1));
						break;

					case 6:
						a->tmpVolume = (2 * a->tmpVolume) / 3;
						break;

					case 7:
						a->tmpVolume >>= 1;
						break;

					case 9:
					case 0xa:
					case 0xb:
					case 0xc:
					case 0xd:
						a->tmpVolume += (1 << (a->s3mRtgSlide - 9));
						break;

					case 0xe:
						a->tmpVolume = (3 * a->tmpVolume) >> 1;
						break;

					case 0xf:
						a->tmpVolume = a->tmpVolume << 1;
						break;
				}

				if (a->tmpVolume < 0)
					a->tmpVolume = 0;
				else
				{
					if (a->tmpVolume > 64)
						a->tmpVolume = 64;
				}
			}
		}

		a->retrig--;	// Countdown
	}
}



/******************************************************************************/
/* S3M Effect R: Tremolo                                                      */
/******************************************************************************/
void MikMod::DoS3MTremolo(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->trmPos >> 2) & 0x1f;

	switch ((a->waveControl >> 4) & 3)
	{
		case 0:		// Sine
			temp = vibratoTable[q];
			break;

		case 1:		// Ramp down
			q <<= 3;

			if (a->trmPos < 0)
				q = 255 - q;

			temp = q;
			break;

		case 2:		// Square wave
			temp = 255;
			break;

		case 3:		// Random
			temp = GetRandom(256);
			break;
	}

	temp *= a->trmDepth;
	temp >>= 7;

	if (a->trmPos >= 0)
	{
		a->volume = a->tmpVolume + temp;
		if (a->volume > 64)
			a->volume = 64;
	}
	else
	{
		a->volume = a->tmpVolume - temp;
		if (a->volume < 0)
			a->volume = 0;
	}

	if (of.vbTick)
		a->trmPos += a->trmSpd;
}



/******************************************************************************/
/* S3M Effect T: Tempo                                                        */
/******************************************************************************/
void MikMod::DoS3MTempo(uint8 tempo)
{
	if (of.vbTick || of.patDly2)
		return;

	of.bpm = (tempo < 32) ? 32 : tempo;
}



/******************************************************************************/
/* S3M Effect U: Fine vibrato                                                 */
/******************************************************************************/
void MikMod::DoS3MFineVibrato(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->vibPos >> 2) & 0x1f;

	switch (a->waveControl & 3)
	{
		case 0:		// Sine
			temp = vibratoTable[q];
			break;

		case 1:		// Ramp down
			q <<= 3;
			if (a->vibPos < 0)
				q = 255 - q;
			temp = q;
			break;

		case 2:		// Square wave
			temp = 255;
			break;

		case 3:		// Random
			temp = GetRandom(256);
			break;
	}

	temp *= a->vibDepth;
	temp >>= 8;

	if (a->vibPos >= 0)
		a->period = a->tmpPeriod + temp;
	else
		a->period = a->tmpPeriod - temp;

	a->vibPos += a->vibSpd;
}



/******************************************************************************/
/* FastTracker II Specific Effects                                            */
/******************************************************************************/

/******************************************************************************/
/* XM Effect G: Volume Slide                                                  */
/******************************************************************************/
void MikMod::DoXMVolSlide(uint8 inf)
{
	uint8 lo, hi;

	explicitSlides = 2;

	if (inf)
		a->s3mVolSlide = inf;
	else
		inf = a->s3mVolSlide;

	if (!of.vbTick)
		return;

	lo = inf & 0xf;
	hi = inf >> 4;

	if (!hi)
	{
		a->tmpVolume -= lo;

		if (a->tmpVolume < 0)
			a->tmpVolume = 0;
	}
	else
	{
		a->tmpVolume += hi;

		if (a->tmpVolume > 64)
			a->tmpVolume = 64;
	}
}



/******************************************************************************/
/* XM Effect H: Global Slide                                                  */
/******************************************************************************/
void MikMod::DoXMGlobalSlide(uint8 inf)
{
	if (of.vbTick)
	{
		if (inf)
			of.globalSlide = inf;
		else
			inf = of.globalSlide;

		if (inf & 0xf0)
			inf &= 0xf0;

		of.volume = of.volume + ((inf >> 4) - (inf & 0xf)) * 2;

		if (of.volume < 0)
			of.volume = 0;
		else
		{
			if (of.volume > 128)
				of.volume = 128;
		}
	}
}



/******************************************************************************/
/* XM Effect P: Panning Slide                                                 */
/******************************************************************************/
void MikMod::DoXMPanSlide(uint8 inf)
{
	uint8 lo, hi;
	int16 pan;

	if (inf)
		a->pansSpd = inf;
	else
		inf = a->pansSpd;

	if (!of.vbTick)
		return;

	lo = inf & 0xf;
	hi = inf >> 4;

	// Slide right has absolute priority
	if (hi)
		lo = 0;

	pan = ((a->panning == PAN_SURROUND) ? PAN_CENTER : a->panning) + hi - lo;

	a->panning = (pan < PAN_LEFT) ? PAN_LEFT : (pan > PAN_RIGHT ? PAN_RIGHT : pan);
}



/******************************************************************************/
/* XM Effect X1: Extra fine portamento up                                     */
/******************************************************************************/
void MikMod::DoXMExtraFineSlideUp(uint8 inf)
{
	if (!of.vbTick)
	{
		a->period    -= inf;
		a->tmpPeriod -= inf;
	}
}



/******************************************************************************/
/* XM Effect X2: Extra fine portamento down                                   */
/******************************************************************************/
void MikMod::DoXMExtraFineSlideDown(uint8 inf)
{
	if (!of.vbTick)
	{
		a->period    += inf;
		a->tmpPeriod += inf;
	}
}



/******************************************************************************/
/* ImpulseTracker Specific Effects                                            */
/******************************************************************************/

/******************************************************************************/
/* IT Volume/Panning column effects                                           */
/******************************************************************************/
void MikMod::DoVolEffects(uint8 c)
{
	uint8 inf = uniTrk.UniGetByte();

	if ((!c) && (!inf))
	{
		c   = a->volEffect;
		inf = a->volData;
	}
	else
	{
		a->volEffect = c;
		a->volData   = inf;
	}

	if (c)
	{
		switch (c)
		{
			// Volume
			case VOL_VOLUME:
				if (of.vbTick)
					break;

				if (inf > 64)
					inf = 64;

				a->tmpVolume = inf;
				break;

			// Panning
			case VOL_PANNING:
				if (of.panFlag)
					a->panning = inf;
				break;

			// Volume slide
			case VOL_VOLSLIDE:
				DoS3MVolSlide(inf);
				break;

			// Pitch slide down
			case VOL_PITCHSLIDEDN:
				if (a->period)
					DoS3MSlideDn(inf);
				break;

			// Pitch slide up
			case VOL_PITCHSLIDEUP:
				if (a->period)
					DoS3MSlideUp(inf);
				break;

			// Tone portamento
			case VOL_PORTAMENTO:
				if (inf)
					a->slideSpeed = inf;

				if (a->period)
				{
					if ((!of.vbTick) || (a->newSamp))
					{
						a->kick  = KICK_NOTE;
						a->start = -1;
					}
					else
						a->kick = (a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;

					DoITToneSlide();
					a->ownPer = 1;
				}
				break;

			// Vibrato
			case VOL_VIBRATO:
				if (!of.vbTick)
				{
					if (inf & 0x0f)
						a->vibDepth = inf & 0xf;

					if (inf & 0xf0)
						a->vibSpd = (inf & 0xf0) >> 2;
				}

				if (a->period)
				{
					DoITVibrato();
					a->ownPer = 1;
				}
				break;
		}
	}
}



/******************************************************************************/
/* IT Effect G: Tone portamento                                               */
/******************************************************************************/
void MikMod::DoITToneSlide(void)
{
	// If we don't come from another note, ignore the slide and play the note
	// as is
	if (!a->oldNote)
		return;

	if (of.vbTick)
	{
		int32 dist;

		// We have to slide a->period towards a->wantedPeriod,
		// compute the difference between those two values
		dist = a->period - a->wantedPeriod;

		// If they are equal or if portamentospeed is too big...
		if ((!dist) || ((a->portSpeed << 2) > abs(dist)))
		{
			// ...make tmpPeriod equal tPeriod
			a->tmpPeriod = a->period = a->wantedPeriod;
		}
		else
		{
			if (dist > 0)
			{
				a->tmpPeriod -= a->portSpeed << 2;
				a->period    -= a->portSpeed << 2;	// Dist > 0, slide up
			}
			else
			{
				a->tmpPeriod += a->portSpeed << 2;
				a->period    += a->portSpeed << 2;	// Dist < 0, slide down
			}
		}
	}
	else
		a->tmpPeriod = a->period;
}



/******************************************************************************/
/* IT Effect H: Vibrato                                                       */
/******************************************************************************/
void MikMod::DoITVibrato(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->vibPos >> 2) & 0x1f;

	switch (a->waveControl & 3)
	{
		// Sine
		case 0:
			temp = vibratoTable[q];
			break;

		// Square wave
		case 1:
			temp = 255;
			break;

		// Ramp down
		case 2:
			q <<= 3;
			if (a->vibPos < 0)
				q = 255 - q;
			temp = q;
			break;

		// Random
		case 3:
			temp = GetRandom(256);
			break;
	}

	temp *= a->vibDepth;
	temp >>= 8;
	temp <<= 2;

	if (a->vibPos >= 0)
		a->period = a->tmpPeriod + temp;
	else
		a->period = a->tmpPeriod - temp;

	a->vibPos += a->vibSpd;
}



/******************************************************************************/
/* IT Effect I: Tremor                                                        */
/******************************************************************************/
void MikMod::DoITTremor(uint8 inf)
{
	uint8 on, off;

	if (inf)
		a->s3mTrOnOf = inf;
	else
	{
		inf = a->s3mTrOnOf;
		if (!inf)
			return;
	}

	if (!of.vbTick)
		return;

	on  = (inf >> 4);
	off = (inf & 0xf);

	a->s3mTremor %= (on + off);
	a->volume = (a->s3mTremor < on) ? a->tmpVolume : 0;
	a->s3mTremor++;
}



/******************************************************************************/
/* IT Effect N: Slide / Fine slide channel volume                             */
/******************************************************************************/
void MikMod::DoITChanVolSlide(uint8 inf)
{
	uint8 lo, hi;

	if (inf)
		a->chanVolSlide = inf;

	inf = a->chanVolSlide;

	lo = inf & 0xf;
	hi = inf >> 4;

	if (!hi)
		a->chanVol -= lo;
	else
	{
		if (!lo)
			a->chanVol += hi;
		else
		{
			if (hi == 0xf)
			{
				if (!of.vbTick)
					a->chanVol -= lo;
			}
			else
			{
				if (lo == 0xf)
					if (!of.vbTick)
						a->chanVol += hi;
			}
		}
	}

	if (a->chanVol < 0)
		a->chanVol = 0;

	if (a->chanVol > 64)
		a->chanVol = 64;
}



/******************************************************************************/
/* IT Effect P: Slide / Fine slide channel panning                            */
/******************************************************************************/
void MikMod::DoITPanSlide(uint8 inf)
{
	uint8 lo, hi;
	int16 pan;

	if (inf)
		a->pansSpd = inf;
	else
		inf = a->pansSpd;

	lo = inf & 0xf;
	hi = inf >> 4;

	pan = (a->panning == PAN_SURROUND) ? PAN_CENTER : a->panning;

	if (!hi)
		pan += lo << 2;
	else
	{
		if (!lo)
			pan -= hi << 2;
		else
		{
			if (hi == 0xf)
			{
				if (!of.vbTick)
					pan += lo << 2;
			}
			else
			{
				if (lo == 0xf)
				{
					if (!of.vbTick)
						pan -= hi << 2;
				}
			}
		}
	}

	a->panning = (pan < PAN_LEFT) ? PAN_LEFT : (pan > PAN_RIGHT ? PAN_RIGHT : pan);
}



/******************************************************************************/
/* IT Effect S: Special                                                       */
/******************************************************************************/
void MikMod::DoSSEffects(uint8 dat)
{
	uint8 inf, c;

	inf = dat & 0xf;
	c   = dat >> 4;

	if (!dat)
	{
		c   = a->ssEffect;
		inf = a->ssData;
	}
	else
	{
		a->ssEffect = c;
		a->ssData   = inf;
	}

	switch (c)
	{
		// S1x: Set glissando voice
		case SS_GLISSANDO:
			DoEEffects(0x30 | inf);
			break;

		// S2x: Set finetune
		case SS_FINETUNE:
			DoEEffects(0x50 | inf);
			break;

		// S3x: Set vibrato waveform
		case SS_VIBWAVE:
			DoEEffects(0x40 | inf);
			break;

		// S4x: Set tremolo waveform
		case SS_TREMWAVE:
			DoEEffects(0x70 | inf);
			break;

		// S5x: Set panbrello waveform
		case SS_PANWAVE:
			a->panbWave = inf;
			break;

		// S6x: Delay x number of frames (patDly)
		case SS_FRAMEDELAY:
			DoEEffects(0xe0 | inf);
			break;

		// S7x: Instrument / NNA commands
		case SS_S7EFFECTS:
			DoNNAEffects(inf);
			break;

		// S8x: Set panning position
		case SS_PANNING:
			DoEEffects(0x80 | inf);
			break;

		// S9x: Set surround sound
		case SS_SURROUND:
			if (of.panFlag)
				a->panning = of.panning[mp_Channel] = PAN_SURROUND;
			break;

		// SAy: Set high order sample offset yxx00h
		case SS_HIOFFSET:
			if (!of.vbTick)
			{
				a->hiOffset = inf << 16;
				a->start    = a->hiOffset | a->sOffset;

				if ((a->s) && (a->start > (int32)a->s->length))
					a->start = a->s->flags & (SF_LOOP | SF_BIDI) ? a->s->loopStart : a->s->length;
			}
			break;

		// SBx: Pattern loop
		case SS_PATLOOP:
			DoEEffects(0x60 | inf);
			break;

		// SCx: Notecut
		case SS_NOTECUT:
			DoEEffects(0xc0 | inf);
			break;

		// SDx: Notedelay
		case SS_NOTEDELAY:
			DoEEffects(0xd0 | inf);
			break;

		// SEx: Pattern delay
		case SS_PATDELAY:
			DoEEffects(0xe0 | inf);
			break;
	}
}



/******************************************************************************/
/* IT Effect S7: NNA Effects                                                  */
/******************************************************************************/
void MikMod::DoNNAEffects(uint8 dat)
{
	int32 t;
	MP_VOICE *aout;

	dat &= 0xf;
	aout = (a->slave) ? a->slave : NULL;

	switch (dat)
	{
		// Past note cut
		case 0x0:
			for (t = 0; t < md_sngChn; t++)
				if (of.voice[t].master == a)
					of.voice[t].fadeVol = 0;
			break;

		// Past note off
		case 0x1:
			for (t = 0; t < md_sngChn; t++)
			{
				if (of.voice[t].master == a)
				{
					of.voice[t].keyOff |= KEY_OFF;
					if ((!(of.voice[t].vEnv.flg & EF_ON)) || (of.voice[t].vEnv.flg & EF_LOOP))
						of.voice[t].keyOff = KEY_KILL;
				}
			}
			break;

		// Past note fade
		case 0x2:
			for (t = 0; t < md_sngChn; t++)
				if (of.voice[t].master == a)
					of.voice[t].keyOff |= KEY_FADE;
			break;

		// Set NNA note cut
		case 0x3:
			a->nna = (a->nna & ~NNA_MASK) | NNA_CUT;
			break;

		// Set NNA note continue
		case 0x4:
			a->nna = (a->nna & ~NNA_MASK) | NNA_CONTINUE;
			break;

		// Set NNA note off
		case 0x5:
			a->nna = (a->nna & ~NNA_MASK) | NNA_OFF;
			break;

		// Set NNA note fade
		case 0x6:
			a->nna = (a->nna & ~NNA_MASK) | NNA_FADE;
			break;

		// Disable volume envelope
		case 0x7:
			if (aout)
				aout->volFlg &= ~EF_ON;
			break;

		// Enable volume envelope
		case 0x8:
			if (aout)
				aout->volFlg |= EF_ON;
			break;

		// Disable panning envelope
		case 0x9:
			if (aout)
				aout->panFlg &= ~EF_ON;
			break;

		// Enable panning envelope
		case 0xa:
			if (aout)
				aout->panFlg |= EF_ON;
			break;

		// Disable pitch envelope
		case 0xb:
			if (aout)
				aout->pitFlg &= ~EF_ON;
			break;

		// Enable pitch envelope
		case 0xc:
			if (aout)
				aout->pitFlg |= EF_ON;
			break;
	}
}



/******************************************************************************/
/* IT Effect T: Set tempo                                                     */
/******************************************************************************/
void MikMod::DoITTempo(uint8 tempo)
{
	int16 temp = of.bpm;

	if (of.vbTick || of.patDly2)
		return;

	if (tempo & 0x10)
		temp += (tempo & 0x0f);
	else
		temp -= tempo;

	of.bpm = (temp > 255) ? 255 : (temp < 1 ? 1 : temp);
}



/******************************************************************************/
/* IT Effect U: Fine vibrato                                                  */
/******************************************************************************/
void MikMod::DoITFineVibrato(void)
{
	uint8 q;
	uint16 temp = 0;

	q = (a->vibPos >> 2) & 0x1f;

	switch (a->waveControl & 3)
	{
		// Sine
		case 0:
			temp = vibratoTable[q];
			break;

		// Square wave
		case 1:
			temp = 255;
			break;

		// Ramp down
		case 2:
			q <<= 3;

			if (a->vibPos < 0)
				q = 255 - q;

			temp = q;
			break;

		// Random
		case 3:
			temp = GetRandom(256);
			break;
	}

	temp *= a->vibDepth;
	temp >>= 8;

	if (a->vibPos >= 0)
		a->period = a->tmpPeriod + temp;
	else
		a->period = a->tmpPeriod - temp;

	a->vibPos += a->vibSpd;
}



/******************************************************************************/
/* IT Effect W: Slide / Fine slide global volume                              */
/******************************************************************************/
void MikMod::DoITGlobalSlide(uint8 inf)
{
	uint8 lo, hi;

	if (inf)
		of.globalSlide = inf;

	inf = of.globalSlide;

	lo = inf & 0xf;
	hi = inf >> 4;

	if (!lo)
	{
		if (of.vbTick)
			of.volume += hi;
	}
	else
	{
		if (!hi)
		{
			if (of.vbTick)
				of.volume -= lo;
		}
		else
		{
			if (lo == 0xf)
			{
				if (!of.vbTick)
					of.volume += hi;
			}
			else
			{
				if (hi == 0xf)
					if (!of.vbTick)
						of.volume -= lo;
			}
		}
	}

	if (of.volume < 0)
		of.volume = 0;

	if (of.volume > 128)
		of.volume = 128;
}



/******************************************************************************/
/* IT Effect Y: Panbrello                                                     */
/******************************************************************************/
void MikMod::DoITPanbrello(void)
{
	uint8 q;
	int32 temp = 0;

	q = a->panbPos;

	switch (a->panbWave)
	{
		// Sine
		case 0:
			temp = panbrelloTable[q];
			break;

		// Square wave
		case 1:
			temp = (q < 0x80) ? 64 : 0;
			break;

		// Ramp down
		case 2:
			q <<= 3;
			temp = q;
			break;

		// Random
		case 3:
			if (a->panbPos >= a->panbSpd)
			{
				a->panbPos = 0;
				temp = GetRandom(256);
			}
			break;
	}

	temp *= a->panbDepth;
	temp  = (temp / 8) + of.panning[mp_Channel];

	a->panning  = (temp < PAN_LEFT) ? PAN_LEFT : (temp > PAN_RIGHT) ? PAN_RIGHT : temp;
	a->panbPos += a->panbSpd;
}



/******************************************************************************/
/* UltraTracker Specific Effects                                              */
/******************************************************************************/

/******************************************************************************/
/* ULT Effect 9: Sample offset                                                */
/******************************************************************************/
void MikMod::DoULTSampleOffset(void)
{
	uint16 offset;

	offset = uniTrk.UniGetWord();

	if (offset)
		a->ultOffset = offset;

	a->start = a->ultOffset << 2;
	if ((a->s) && ((uint32)a->start > a->s->length))
		a->start = a->s->flags & (SF_LOOP | SF_BIDI) ? a->s->loopStart : a->s->length;
}



/******************************************************************************/
/* VoicePlay() starts to play the sample.                                     */
/*                                                                            */
/* Input:  "voice" is the voice to play the sample in.                        */
/*         "s" is a pointer to the sample information.                        */
/*         "start" is where in the sample to start at.                        */
/******************************************************************************/
void MikMod::VoicePlay(int8 voice, SAMPLE *s, uint32 start)
{
	uint32 repEnd;
	uint16 bits;
	AP_LoopType type;

	if ((voice < 0) || (voice >= md_sngChn) || (start >= s->length))
		return;

	// Play the sample
	if (s->flags & SF_16BITS)
		bits = 16;
	else
		bits = 8;

	virtChannels[voice]->PlaySample(s->handle, start, s->length, bits);

	// Setup the loop if any
	if (s->flags & SF_LOOP)
	{
		repEnd = s->loopEnd;

		if (repEnd > s->length)
			repEnd = s->length;		// RepEnd can't be bigger than size

		if (s->flags & SF_BIDI)
			type = APLOOP_PingPong;
		else
			type = APLOOP_Normal;

		virtChannels[voice]->SetLoop(s->loopStart, repEnd - s->loopStart, type);
	}
}



/******************************************************************************/
/* VoiceSetVolume() changes the volume in the channel.                        */
/*                                                                            */
/* Input:  "voice" is the voice to change.                                    */
/*         "vol" is the new volume.                                           */
/******************************************************************************/
void MikMod::VoiceSetVolume(int8 voice, uint16 vol)
{
	if ((voice < 0) || (voice >= md_sngChn))
		return;

	virtChannels[voice]->SetVolume(vol);
}



/******************************************************************************/
/* VoiceSetPanning() changes the panning in the channel.                      */
/*                                                                            */
/* Input:  "voice" is the voice to change.                                    */
/*         "pan" is the new panning.                                          */
/******************************************************************************/
void MikMod::VoiceSetPanning(int8 voice, uint32 pan)
{
	if ((voice < 0) || (voice >= md_sngChn))
		return;

	if (pan == PAN_SURROUND)
		pan = APPAN_SURROUND;

	virtChannels[voice]->SetPanning(pan);
}



/******************************************************************************/
/* VoiceSetFrequency() changes the frequency in the channel.                  */
/*                                                                            */
/* Input:  "voice" is the voice to change.                                    */
/*         "frq" is the new frequency.                                        */
/******************************************************************************/
void MikMod::VoiceSetFrequency(int8 voice, uint32 frq)
{
	if ((voice < 0) || (voice >= md_sngChn))
		return;

	virtChannels[voice]->SetFrequency(frq);
}



/******************************************************************************/
/* VoiceStop() stops the channel.                                             */
/*                                                                            */
/* Input:  "voice" is the voice to change.                                    */
/******************************************************************************/
void MikMod::VoiceStop(int8 voice)
{
	if ((voice < 0) || (voice >= md_sngChn))
		return;

	virtChannels[voice]->Mute();
}



/******************************************************************************/
/* VoiceStopped() returns true if the voice doesn't play anymore, else false. */
/*                                                                            */
/* Input:  "voice" is the voice to check on.                                  */
/*                                                                            */
/* Output: True if it's stopped, else false.                                  */
/******************************************************************************/
bool MikMod::VoiceStopped(int8 voice)
{
	if ((voice < 0) || (voice > md_sngChn))
		return (false);

	return (!virtChannels[voice]->IsActive());
}
