/******************************************************************************/
/* APlayer Module Info agent class.                                           */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PException.h"
#include "PAlert.h"
#include "GlobalFuncs.h"

// APlayer headers
#include "APlayer.h"

// Needed headers
#include "MIAgent.h"
#include "ModuleInfo_Lang.h"
#include "ModuleInfo_LangStr.h"


/******************************************************************************/
/* Version                                                                    */
/******************************************************************************/
#define AgentVersion		1.42



/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
MIAgent::MIAgent(APGlobalData *global) : APAddOnAgent(global)
{
	// Fill out the version we have compiled under
	addOnVersion  = AP_ADDON_VERSION;
	playerVersion = AP_PLAYER_VERSION;
	agentVersion  = AP_AGENT_VERSION;

	// Register string resources
	strings = AddStringResource(miStrings);
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
MIAgent::~MIAgent(void)
{
	RemoveStringResource(strings);
}



/******************************************************************************/
/* GetVersion() returns the version of the add-on.                            */
/*                                                                            */
/* Output: The add-on version.                                                */
/******************************************************************************/
float MIAgent::GetVersion(void)
{
	return (AgentVersion);
}



/******************************************************************************/
/* GetName() returns the name of the current add-on.                          */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on name.                                                   */
/******************************************************************************/
PString MIAgent::GetName(uint32 index)
{
	PString name;

	name.LoadString(strings, IDS_MI_NAME);
	return (name);
}



/******************************************************************************/
/* GetDescription() returns the description of the current add-on.            */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on description.                                            */
/******************************************************************************/
PString MIAgent::GetDescription(uint32 index)
{
	PString description;

	description.LoadString(strings, IDS_MI_DESCRIPTION);
	return (description);
}



/******************************************************************************/
/* GetSupportFlags() returns some flags telling what the add-on supports.     */
/*                                                                            */
/* Output: Is the flags.                                                      */
/******************************************************************************/
uint32 MIAgent::GetSupportFlags(void)
{
	return (apaInfoButton | apaModuleInfoChange | apaAfterInitSong | apaAfterFree);
}



/******************************************************************************/
/* InitAgent() will initialize the agent.                                     */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool MIAgent::InitAgent(void)
{
	try
	{
		PString title;
		int32 x, y, w, h;
		void *regHd;

		// Fix the registry
		FixRegistry();

		// Open the registry
		regHd = globalData->useReg->OpenKey("/Agents/ModuleInfo/");

		try
		{
			// Get the window position and size
			x = globalData->useReg->GetNumber(regHd, "WinX");
			y = globalData->useReg->GetNumber(regHd, "WinY");
			w = globalData->useReg->GetNumber(regHd, "WinW");
			h = globalData->useReg->GetNumber(regHd, "WinH");

			openIt = globalData->useReg->GetNumber(regHd, "OpenWindow");
		}
		catch(...)
		{
			globalData->useReg->CloseKey(regHd);
			throw;
		}

		// Close the registry again
		globalData->useReg->CloseKey(regHd);

		// Allocate the window object
		title.LoadString(strings, IDS_MI_WIN_TITLE);
		window = new MIWindow(BRect(x, y, x + w, y + h), title, globalData, strings);

		// This make sure that the window is opened in hidden mode
		window->Hide();
		window->onScreen = false;
		window->Show();
	}
	catch(PException e)
	{
		try
		{
			PString title, msg;

			title.LoadString(strings, IDS_MI_WIN_TITLE);
			msg.LoadString(strings, IDS_MI_ERR_INIT);

			PAlert alert(title, msg, PAlert::pStop, PAlert::pOk);
			alert.Show();
		}
		catch(...)
		{
			PlayBeep();
		}

		return (false);
	}

	return (true);
}



/******************************************************************************/
/* EndAgent() will clean up the agent.                                        */
/******************************************************************************/
void MIAgent::EndAgent(void)
{
	window->winLock->Lock();
	window->shutDown = true;
	window->winLock->Unlock();

	// Quit and clean up the window
	window->Lock();
	window->QuitRequested();
	window->Quit();
}



/******************************************************************************/
/* OpenWindows() will open the agent windows if any.                          */
/******************************************************************************/
void MIAgent::OpenWindows(void)
{
	if (openIt)
	{
		window->Show();
		window->onScreen = true;
	}
}



/******************************************************************************/
/* Run() is the main function. It do what the agent have to do.               */
/******************************************************************************/
void MIAgent::Run(void)
{
	window->winLock->Lock();

	if (!window->onScreen)
	{
		window->Show();
		window->onScreen = true;
	}

	window->winLock->Unlock();

	// Activate the window
	window->Activate();
}



/******************************************************************************/
/* Plugin_AfterInitSong() will be called after the module subsong             */
/*      initializing.                                                         */
/******************************************************************************/
void MIAgent::Plugin_AfterInitSong(void)
{
	window->RefreshWindow();
}



/******************************************************************************/
/* Plugin_AfterFree() will be called after the module has been freed.         */
/******************************************************************************/
void MIAgent::Plugin_AfterFree(void)
{
	window->RefreshWindow();
}



/******************************************************************************/
/* PlugIn_ModuleInfoChanged() will be called everytime a new value or         */
/*      description has changed.                                              */
/*                                                                            */
/* Input:  "line" is the line starting from 0.                                */
/*         "type" is the type of string changed.                              */
/*         "newString" is the new string.                                     */
/******************************************************************************/
void MIAgent::Plugin_ModuleInfoChanged(uint32 line, ap_infoType type, PString newString)
{
	window->UpdateWindow(line, type, newString);
}



/******************************************************************************/
/* FixRegistry() puts all the default registry entries into the registry.     */
/*                                                                            */
/* Except: PFileException.                                                    */
/******************************************************************************/
void MIAgent::FixRegistry(void)
{
	void *regHd;

	regHd = globalData->useReg->OpenKey("/Agents/");

	try
	{
		// Create config key
		globalData->useReg->CreateKey(regHd, "ModuleInfo");
		globalData->useReg->CloseKey(regHd);
		regHd = NULL;

		regHd = globalData->saveReg->OpenKey("/Agents/");

		try
		{
			globalData->saveReg->CreateKey(regHd, "ModuleInfo");
			globalData->saveReg->CloseKey(regHd);
			regHd = NULL;
		}
		catch(...)
		{
			globalData->saveReg->CloseKey(regHd);
			throw;
		}

		regHd = globalData->useReg->OpenKey("/Agents/ModuleInfo/");

		// Set default window position and size
		if (!(globalData->useReg->Exists(regHd, "WinX")))
			globalData->useReg->CreateNumber(regHd, "WinX", 460);

		if (!(globalData->useReg->Exists(regHd, "WinY")))
			globalData->useReg->CreateNumber(regHd, "WinY", 25);

		if (!(globalData->useReg->Exists(regHd, "WinW")))
			globalData->useReg->CreateNumber(regHd, "WinW", 290);

		if (!(globalData->useReg->Exists(regHd, "WinH")))
			globalData->useReg->CreateNumber(regHd, "WinH", 190);

		if (!(globalData->useReg->Exists(regHd, "OpenWindow")))
			globalData->useReg->CreateNumber(regHd, "OpenWindow", 0);

		// Set default column size
		if (!(globalData->useReg->Exists(regHd, "Col1W")))
			globalData->useReg->CreateNumber(regHd, "Col1W", 137);

		if (!(globalData->useReg->Exists(regHd, "Col2W")))
			globalData->useReg->CreateNumber(regHd, "Col2W", 130);
	}
	catch(...)
	{
		globalData->useReg->CloseKey(regHd);
		throw;
	}

	globalData->useReg->CloseKey(regHd);
}
