/******************************************************************************/
/* APlayer MikMod converter agent class.                                      */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PString.h"
#include "PFile.h"

// Needed headers
#include "MikAgent.h"
#include "MikConverter.h"
#include "MikModConverter_Lang.h"
#include "MikModConverter_LangStr.h"


/******************************************************************************/
/* Version                                                                    */
/******************************************************************************/
#define AgentVersion		1.34



/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
MikAgent::MikAgent(APGlobalData *global) : APAddOnAgent(global)
{
	// Fill out the version we have compiled under
	addOnVersion  = AP_ADDON_VERSION;
	playerVersion = AP_PLAYER_VERSION;
	agentVersion  = AP_AGENT_VERSION;

	// Register string resources
	strings = AddStringResource(mikcStrings);
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
MikAgent::~MikAgent(void)
{
	RemoveStringResource(strings);
}



/******************************************************************************/
/* GetVersion() returns the version of the add-on.                            */
/*                                                                            */
/* Output: The add-on version.                                                */
/******************************************************************************/
float MikAgent::GetVersion(void)
{
	return (AgentVersion);
}



/******************************************************************************/
/* GetName() returns the name of the current add-on.                          */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on name.                                                   */
/******************************************************************************/
PString MikAgent::GetName(uint32 index)
{
	PString name;

	name.LoadString(strings, IDS_MIKC_NAME);
	return (name);
}



/******************************************************************************/
/* GetDescription() returns the description of the current add-on.            */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on description.                                            */
/******************************************************************************/
PString MikAgent::GetDescription(uint32 index)
{
	PString description;

	description.LoadString(strings, IDS_MIKC_DESCRIPTION);
	return (description);
}



/******************************************************************************/
/* GetSupportFlags() returns some flags telling what the add-on supports.     */
/*                                                                            */
/* Output: Is the flags.                                                      */
/******************************************************************************/
uint32 MikAgent::GetSupportFlags(void)
{
	return (apaAfterLoadingBeforeCheck);
}



/******************************************************************************/
/* InitAgent() will initialize the agent.                                     */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool MikAgent::InitAgent(void)
{
	// Add all the converters supported
	converters.AddTail(new Mik669(strings));
	converters.AddTail(new MikAMF(strings));
	converters.AddTail(new MikDSM(strings));
	converters.AddTail(new MikFAR(strings));
	converters.AddTail(new MikGDM(strings));
	converters.AddTail(new MikIMF(strings));
	converters.AddTail(new MikIT(strings));
	converters.AddTail(new MikS3M(strings));
	converters.AddTail(new MikSTM(strings));
	converters.AddTail(new MikSTX(strings));
	converters.AddTail(new MikULT(strings));
	converters.AddTail(new MikUNI(strings));
	converters.AddTail(new MikXM(strings));

	return (true);
}



/******************************************************************************/
/* EndAgent() will clean up the agent.                                        */
/******************************************************************************/
void MikAgent::EndAgent(void)
{
	uint32 i, count;
	MikConverter *convItem;

	// Remove all the converters again
	count = converters.CountItems();

	for (i = 0; i < count; i++)
	{
		convItem = converters.GetAndRemoveItem(0);
		delete convItem;
	}
}



/******************************************************************************/
/* Plugin_AfterLoadingBeforeCheck() will be called after a module has been    */
/*      loaded into the memory, but before the module check begins.           */
/*                                                                            */
/* Input:  "info" is a pointer to a info structure to read and write too.     */
/*                                                                            */
/* Output: An APlayer return code.                                            */
/******************************************************************************/
ap_result MikAgent::Plugin_AfterLoadingBeforeCheck(AfterLoadingBeforeCheckInfo *info)
{
	uint32 i, count;
	MikConverter *convItem;
	PMemFile *memFile;
//	PFile *memFile;
	uint8 *memBuffer;
	uint32 size;
	ap_result retVal = AP_UNKNOWN;

	// Call the test function in the converters
	count = converters.CountItems();

	for (i = 0; i < count; i++)
	{
		convItem = (MikConverter *)converters.GetItem(i);
		if (convItem->CheckModule(info))
		{
			// Found the converter, so now convert it to UniMod structures
			info->moduleFile->SeekToBegin();
			if (convItem->ConvertModule(info) == AP_OK)
			{
				// Allocate a buffer to store the memory file in.
				// The only reason we do this, is to save some time. The
				// buffer won't be reallocated all the time.
				//
				// Create the buffer with the same size as the original
				// file. There is a big chance that the new module won't
				// use the whole buffer, infact it can cut of 10-20 kb!
				size      = info->moduleFile->GetLength();
				memBuffer = new uint8[size];

				// Create a APMemFile object to store the converted module in
				memFile = new PMemFile(memBuffer, size, 4 * 1024);
//				memFile = new PFile("/boot/home/module", PFile::pModeReadWrite | PFile::pModeCreate);

				// Save the UniMod structures into the new module
				if (convItem->ConvertToUniMod(info->moduleFile, memFile) == AP_OK)
				{
					// Initialize the info structure
					info->newModuleFile = memFile;
					retVal = AP_OK;
				}
				else
				{
					delete memFile;
					retVal = AP_ERROR;
				}
			}
			else
				retVal = AP_ERROR;

			// Free all memory used
			convItem->FreeAll();
			break;
		}
	}

	return (retVal);
}
