#!/bin/sh
# make-keymap104 
# create and load 104-key keymap based on current keymap
# Copyright © 1999 Be, Inc; All Rights Reserved.

FILENAME="$HOME/104-enabled-keymap"
TMPFILE="/tmp/$0.$$"
VERSION="1.0"

FORCE=0
LOADMAP=1

function help() {
	cat << FEOF
make-keymap104, Version $VERSION
Copyright (c) 1999, Be, Inc. All rights Reserved.
Create and load 104-key keymap based on current keymap
Args:
  -f|--force : Overwrite any existing saved keymap file
               stored in ($FILENAME)
  -n|--no-load : don't load the resulting keymap
  -h|--help : Show this help text
FEOF
}

while [ $# -gt 0 ]
do
	case "$1" in
		-n|--no-load)
			LOADMAP=0
			;;
		-f|--force)
			FORCE=1
			;;
		-h|--help)
			help
			exit 0
			;;
	esac
	shift
done

keymap -d | sed -e 's:^#keymap$:#!/bin/keymap -l:' -e 's:^RControl = 0x00$:RControl = 0x60:' -e 's:^ROption = 0x60$:ROption = 0x67:' > "$TMPFILE"

if [ $LOADMAP -eq 1 ]
then
	keymap -l < "$TMPFILE"
else
	echo "Not loading 104-key keymap file."
fi

function writekeymap() {
	echo "Writing new keymap file to $FILENAME"
	/bin/mv -f "$TMPFILE" "$FILENAME"
	chmod +x "$FILENAME"
	echo
	echo "To load this keymap in the future, execute the file: $FILENAME"
	echo
	echo "To have it load at boot time add that line to your user boot script,"
	echo " or load it and use the Keymap preference to save the current keymap"
	echo " as a file."
}

if [ -e "$FILENAME" ]
then
	if [ $FORCE -eq 0 ]
	then
		echo "Not overwriting saved keymap location: $FILENAME"
		if ! diff -q "$TMPFILE" "$FILENAME" > /dev/null
		then
			if [ $LOADMAP -eq 1 ]
			then
				echo "NOTE: Existing file is different than current keymap."
			else
				echo "NOTE: Existing file is not what this script would produce right now."
			fi
		else
			echo "You have this keymap already stored in: $FILENAME."
			echo "Executing that file will be faster than running this script."
			echo
			echo "To have it load at boot time add that line to your user boot script,"
			echo " or load it and use the Keymap preference to save the current keymap"
			echo " as a file."
		fi
		rm -f "$TMPFILE"
	else
		writekeymap
	fi
else
	writekeymap
fi

