#pragma once

#include <Bitmap.h>
#include <GraphicsDefs.h>

#include <cstdint>
#include <cstring>
#include <cstdio>
#include <cmath>
#include <string>
#include <map>
#include <sstream>
#include <vector>
#include <algorithm>
#include <zlib.h>

struct SkinRect { int x, y, w, h; };
static constexpr SkinRect kCBtn_Prev_Up  = {  0,  0, 23, 18 };
static constexpr SkinRect kCBtn_Play_Up  = { 23,  0, 23, 18 };
static constexpr SkinRect kCBtn_Pause_Up = { 46,  0, 23, 18 };
static constexpr SkinRect kCBtn_Stop_Up  = { 69,  0, 23, 18 };
static constexpr SkinRect kCBtn_Next_Up  = { 92,  0, 22, 18 };
static constexpr SkinRect kCBtn_Prev_Dn  = {  0, 18, 23, 18 };
static constexpr SkinRect kCBtn_Play_Dn  = { 23, 18, 23, 18 };
static constexpr SkinRect kCBtn_Pause_Dn = { 46, 18, 23, 18 };
static constexpr SkinRect kCBtn_Stop_Dn  = { 69, 18, 23, 18 };
static constexpr SkinRect kCBtn_Next_Dn  = { 92, 18, 22, 18 };
static constexpr SkinRect kPosBar_Bar = {  0, 0, 248, 10 };
static constexpr SkinRect kPosBar_Btn = {248, 0,  29, 10 };
static constexpr int kVolStripStep = 15;   // row interval between strips
static constexpr int kVolStripW    = 68;   // qmmp: qMin(pixmap->width(), 68)
static constexpr int kVolStripH    = 13;   // strip render height
static constexpr int kVolKnobY     = 422;  // knob sprite y in VOLUME.BMP
static constexpr int kVolKnobW     = 14;
static constexpr int kDigitW = 9;
static constexpr int kDigitH = 13;
static constexpr SkinRect kPP_Play  = { 1, 0, 8, 9 };
static constexpr SkinRect kPP_Pause = { 9, 0, 9, 9 };
static constexpr SkinRect kPP_Stop  = {18, 0, 9, 9 };
static constexpr SkinRect kTB_MenuN    = {  0,  0,   9,  9 };
static constexpr SkinRect kTB_MenuP    = {  0,  9,   9,  9 };
static constexpr SkinRect kTB_MinN     = {  9,  0,   9,  9 };
static constexpr SkinRect kTB_MinP     = {  9,  9,   9,  9 };
static constexpr SkinRect kTB_CloseN   = { 18,  0,   9,  9 };
static constexpr SkinRect kTB_CloseP   = { 18,  9,   9,  9 };
static constexpr SkinRect kTB_BgActive = { 27,  0, 275, 14 }; // TITLEBAR_A
static constexpr SkinRect kTB_BgInact  = { 27, 15, 275, 14 }; // TITLEBAR_I
static constexpr SkinRect kPL_CornerUL_A = {   0,  0,  25, 20 };
static constexpr SkinRect kPL_CornerUL_I = {   0, 21,  25, 20 };
static constexpr SkinRect kPL_CornerUR_A = { 153,  0,  25, 20 };
static constexpr SkinRect kPL_CornerUR_I = { 153, 21,  25, 20 };
static constexpr SkinRect kPL_TitleA     = {  26,  0, 100, 20 };  // center piece
static constexpr SkinRect kPL_TitleI     = {  26, 21, 100, 20 };
static constexpr SkinRect kPL_TFill1A    = { 127,  0,  25, 20 };  // fill tile
static constexpr SkinRect kPL_TFill1I    = { 127, 21,  25, 20 };
static constexpr SkinRect kPL_LFill      = {   0, 42,  12, 29 };  // left edge, tiled
static constexpr SkinRect kPL_RFill      = {  31, 42,  20, 29 };
// Bottom bar:
static constexpr SkinRect kPL_LSBar      = {   0, 72, 125, 38 };
static constexpr SkinRect kPL_SFill1     = { 179,  0,  25, 38 };  // tiled center
static constexpr SkinRect kPL_RSBar      = { 126, 72, 150, 38 };
// Mini-transport control bar:
static constexpr SkinRect kPL_Control    = { 129, 94,  60,  8 };
// Buttons:
static constexpr SkinRect kPL_CloseN     = { 167,  3,   9,  9 };
static constexpr SkinRect kPL_CloseP     = {  52, 42,   9,  9 };
static constexpr SkinRect kPL_AddBtn     = {  11, 80,  25, 18 };
static constexpr SkinRect kPL_AddBtnP    = {  11, 80,  25, 18 };  // same as N (qmmp: no separate pressed)
static constexpr SkinRect kPL_SubBtn     = {  40, 80,  25, 18 };
static constexpr SkinRect kPL_ScrollN    = {  52, 53,   8, 18 };
static constexpr SkinRect kPL_ScrollP    = {  61, 53,   8, 18 };
struct VisColors {
    // Packed RGB for each of the 24 slots — defaults are qmmp "glare" skin values
    uint8_t r[24], g[24], b[24];
    VisColors() {
        static const uint8_t dr[24] = {13,13,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,64};
        static const uint8_t dg[24] = {13,13,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,125};
        static const uint8_t db[24] = {13,13,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236};
        memcpy(r, dr, 24); memcpy(g, dg, 24); memcpy(b, db, 24);
    }
};

struct PlEditColors {
    uint8_t normalR=0,  normalG=200,normalB=0;   // item text
    uint8_t normBgR=0,  normBgG=0,  normBgB=0;   // background
    uint8_t selR=255,   selG=255,   selB=255;     // selected text
    uint8_t selBgR=0,   selBgG=0,   selBgB=150;  // selected bg
    uint8_t curR=255,   curG=255,   curB=0;       // current track
};

// ─────────────────────────────────────────────────────────────────────────────
class WinampSkin {
public:
    WinampSkin()  = default;
    ~WinampSkin() { Unload(); }

    bool Load(const char* path);
    void Unload();
    bool IsLoaded() const { return fLoaded; }

    void SetScale(int scale, bool smooth = false);
    int  CurrentScale()  const { return fScale; }
    bool CurrentSmooth() const { return fSmooth; }

    const std::vector<uint8_t>* RawFile(const char* name) const {
        auto it = fFiles.find(Lower(name));
        return it != fFiles.end() ? &it->second : nullptr;
    }

    // Full decoded+scaled sheet at fScale -- cached, caller must NOT delete
    BBitmap* Sheet(const char* name) const;

    // Crop from a sheet -- caller owns result
    BBitmap* Crop(const char* sheet, SkinRect r) const { return Crop(Sheet(sheet), r); }
    BBitmap* Crop(BBitmap* src, SkinRect r) const;

    // Integer nearest-neighbour upscale -- caller owns result
    BBitmap* Scale(BBitmap* src, int factor) const;

    // Bilinear upscale -- caller owns result
    BBitmap* ScaleSmooth(BBitmap* src, int factor) const;

    // Volume strip for 0-100% -- already at fScale, caller owns
    BBitmap* VolumeStrip(int pct) const;

    // Volume knob -- already at fScale, caller owns
    BBitmap* VolumeKnob(bool pressed) const;

    // Digit glyph from NUMBERS.BMP -- already at fScale, caller owns
    BBitmap* Digit(char c) const;

    const PlEditColors& PlColors()   const { return fPlColors; }
    const VisColors&   VisCol()     const { return fVisColors; }

private:
    bool         fLoaded = false;
    int          fScale  = 1;
    bool         fSmooth = false;
    PlEditColors fPlColors;
    VisColors    fVisColors;

    void ParsePleditTxt();
    void ParseViscolorTxt();
    std::map<std::string, std::vector<uint8_t>> fFiles;
    mutable std::map<std::string, BBitmap*>     fSheets;

    static std::string Lower(const char* s) {
        std::string o(s);
        for (char& c : o) if (c>='A'&&c<='Z') c+=32;
        return o;
    }

    bool     ExtractZip(const char* path);
    BBitmap* DecodeBMP(const std::vector<uint8_t>& data) const;

    static rgb_color ParseHexColor(const char* s) {
        rgb_color c={0,0,0,255};
        if (*s=='#') s++;
        if (strlen(s)<6) return c;
        auto h=[](char c)->int{
            if(c>='0'&&c<='9') return c-'0';
            if(c>='a'&&c<='f') return c-'a'+10;
            if(c>='A'&&c<='F') return c-'A'+10;
            return 0;};
        c.red  =(h(s[0])<<4)|h(s[1]);
        c.green=(h(s[2])<<4)|h(s[3]);
        c.blue =(h(s[4])<<4)|h(s[5]);
        return c;
    }
};

inline bool WinampSkin::ExtractZip(const char* path) {
    FILE* f=fopen(path,"rb"); if(!f) return false;
    fseek(f,0,SEEK_END); long fsz=ftell(f);
    if(fsz<=0){fclose(f);return false;}
    fseek(f,0,SEEK_SET);
    std::vector<uint8_t> zip((size_t)fsz);
    if((long)fread(zip.data(),1,(size_t)fsz,f)!=fsz){fclose(f);return false;}
    fclose(f);

    size_t pos=0;
    while(pos+30<=zip.size()) {
        if(zip[pos]!=0x50||zip[pos+1]!=0x4b||zip[pos+2]!=0x03||zip[pos+3]!=0x04)
            {pos++;continue;}
        uint16_t method  =(uint16_t)(zip[pos+8] |(zip[pos+9]<<8));
        uint32_t compSz  =(uint32_t)(zip[pos+18]|(zip[pos+19]<<8)|(zip[pos+20]<<16)|(zip[pos+21]<<24));
        uint32_t uncompSz=(uint32_t)(zip[pos+22]|(zip[pos+23]<<8)|(zip[pos+24]<<16)|(zip[pos+25]<<24));
        uint16_t fnLen   =(uint16_t)(zip[pos+26]|(zip[pos+27]<<8));
        uint16_t exLen   =(uint16_t)(zip[pos+28]|(zip[pos+29]<<8));
        size_t   dataOff =pos+30+fnLen+exLen;
        if(dataOff>zip.size()||compSz>zip.size()-dataOff) break;

        std::string raw(zip.begin()+pos+30, zip.begin()+pos+30+fnLen);
        size_t sl=raw.rfind('/'); if(sl==std::string::npos) sl=raw.rfind('\\');
        std::string key=Lower((sl!=std::string::npos?raw.substr(sl+1):raw).c_str());

        if(!key.empty()&&uncompSz>0) {
            std::vector<uint8_t> out(uncompSz,0);
            if(method==0) {
                memcpy(out.data(),zip.data()+dataOff,
                       std::min((size_t)compSz,(size_t)uncompSz));
            } else if(method==8) {
                z_stream zs{}; zs.next_in=zip.data()+dataOff; zs.avail_in=compSz;
                zs.next_out=out.data(); zs.avail_out=uncompSz;
                if(inflateInit2(&zs,-MAX_WBITS)==Z_OK){inflate(&zs,Z_FINISH);inflateEnd(&zs);}
            }
            fFiles[key]=std::move(out);
        }
        pos=dataOff+compSz;
    }
    return !fFiles.empty();
}

inline BBitmap* WinampSkin::DecodeBMP(const std::vector<uint8_t>& d) const {
    if(d.size()<54||d[0]!='B'||d[1]!='M') return nullptr;
    uint32_t dataOff=(uint32_t)(d[10]|(d[11]<<8)|(d[12]<<16)|(d[13]<<24));
    uint32_t dibSz  =(uint32_t)(d[14]|(d[15]<<8)|(d[16]<<16)|(d[17]<<24));
    if(dibSz<40||dataOff>=d.size()) return nullptr;
    int32_t  bmpW =(int32_t)(d[18]|(d[19]<<8)|(d[20]<<16)|(d[21]<<24));
    int32_t  bmpH =(int32_t)(d[22]|(d[23]<<8)|(d[24]<<16)|(d[25]<<24));
    uint16_t bpp  =(uint16_t)(d[28]|(d[29]<<8));
    uint32_t compr=(uint32_t)(d[30]|(d[31]<<8)|(d[32]<<16)|(d[33]<<24));
    if(bmpW<=0||bmpH==0||compr>3) return nullptr;
    if(compr==1||compr==2) return nullptr; // RLE not supported

    bool    flipY=(bmpH>0); int32_t absH=bmpH<0?-bmpH:bmpH;
    uint32_t palOff=14+dibSz, palCnt=(bpp<=8)?(1u<<bpp):0;
    std::vector<uint32_t> pal(palCnt,0);
    for(uint32_t i=0;i<palCnt;i++){
        size_t p=palOff+i*4; if(p+2>=d.size()) break;
        pal[i]=((uint32_t)d[p+2]<<16)|((uint32_t)d[p+1]<<8)|(uint32_t)d[p];
    }

    BBitmap* bm=new BBitmap(BRect(0,0,(float)(bmpW-1),(float)(absH-1)),B_RGB32);
    if(!bm||bm->InitCheck()!=B_OK){delete bm;return nullptr;}
    uint8_t* dst=(uint8_t*)bm->Bits(); int dBPR=(int)bm->BytesPerRow();
    int srcStr=((bmpW*bpp+31)/32)*4;

    for(int32_t row=0;row<absH;row++) {
        int32_t srcRow=flipY?(absH-1-row):row;
        size_t srcOff=(size_t)dataOff+(size_t)srcRow*(size_t)srcStr;
        if(srcOff+(size_t)srcStr>d.size()) break;
        const uint8_t* src=d.data()+srcOff;
        uint8_t* dRow=dst+row*dBPR;
        if(bpp==24) {
            for(int32_t x=0;x<bmpW;x++){
                dRow[x*4+0]=src[x*3+0]; dRow[x*4+1]=src[x*3+1];
                dRow[x*4+2]=src[x*3+2]; dRow[x*4+3]=255;}
        } else if(bpp==32) {
            for(int32_t x=0;x<bmpW;x++){
                dRow[x*4+0]=src[x*4+0]; dRow[x*4+1]=src[x*4+1];
                dRow[x*4+2]=src[x*4+2]; dRow[x*4+3]=255;}
        } else if(bpp==8) {
            for(int32_t x=0;x<bmpW;x++){
                uint32_t c=src[x]<palCnt?pal[src[x]]:0;
                dRow[x*4+0]=c&0xff; dRow[x*4+1]=(c>>8)&0xff;
                dRow[x*4+2]=(c>>16)&0xff; dRow[x*4+3]=255;}
        } else if(bpp==4) {
            for(int32_t x=0;x<bmpW;x++){
                uint8_t idx=(x&1)?(src[x/2]&0x0f):(src[x/2]>>4);
                uint32_t c=idx<palCnt?pal[idx]:0;
                dRow[x*4+0]=c&0xff; dRow[x*4+1]=(c>>8)&0xff;
                dRow[x*4+2]=(c>>16)&0xff; dRow[x*4+3]=255;}
        } else { memset(dRow,0,(size_t)dBPR); }
    }
    return bm;
}

inline void WinampSkin::ParsePleditTxt() {
    const auto* raw=RawFile("pledit.txt"); if(!raw) return;
    std::string txt(raw->begin(), raw->end());

    // Returns processed color string for a given key (case-insensitive)
    auto readKey=[&](const char* key)->std::string {
        std::string lk(key); for(char& c:lk) if(c>='A'&&c<='Z') c+=32;
        std::string lt=txt;  for(char& c:lt) if(c>='A'&&c<='Z') c+=32;
        size_t p=lt.find(lk+"="); if(p==std::string::npos) return "";
        p+=lk.size()+1;
        size_t e=txt.find_first_of("\r\n",p);
        std::string v=txt.substr(p, e==std::string::npos?std::string::npos:e-p);
        // strip quotes
        v.erase(std::remove(v.begin(),v.end(),'"'),v.end());
        // strip comments
        size_t cm=v.find("//"); if(cm!=std::string::npos) v=v.substr(0,cm);
        // trim
        while(!v.empty()&&(v.front()==' '||v.front()=='\t')) v=v.substr(1);
        while(!v.empty()&&(v.back()==' '||v.back()=='\t'||v.back()=='\r')) v.pop_back();
        // add # prefix if missing (qmmp: if(!value.startsWith("#")&&key!="font"))
        if(!v.empty()&&v[0]!='#') v="#"+v;
        // remove alpha channel from ARGB (qmmp: if size>7 remove(1, size-7))
        if(v.size()>7) v="#"+v.substr(v.size()-6);
        return v;
    };

    auto apply=[&](const char* key, uint8_t& r, uint8_t& g, uint8_t& b){
        std::string v=readKey(key); if(v.empty()) return;
        rgb_color c=ParseHexColor(v.c_str()); r=c.red; g=c.green; b=c.blue;
    };
    apply("Normal",     fPlColors.normalR, fPlColors.normalG, fPlColors.normalB);
    apply("NormalBg",   fPlColors.normBgR, fPlColors.normBgG, fPlColors.normBgB);
    apply("Selected",   fPlColors.selR,    fPlColors.selG,    fPlColors.selB);
    apply("SelectedBg", fPlColors.selBgR,  fPlColors.selBgG,  fPlColors.selBgB);
    apply("Current",    fPlColors.curR,    fPlColors.curG,    fPlColors.curB);
}

// Mirrors qmmp Skin::loadVisColor(). VISCOLOR.TXT has 24 lines, each r,g,b
inline void WinampSkin::ParseViscolorTxt() {
    fVisColors = VisColors{};  // reset to defaults
    const auto* raw = RawFile("viscolor.txt");
    if (!raw) return;
    std::string txt(raw->begin(), raw->end());
    std::istringstream ss(txt);
    std::string line;
    int slot = 0;
    while (std::getline(ss, line) && slot < 24) {
        // strip quotes
        line.erase(std::remove(line.begin(), line.end(), '"'), line.end());
        // strip // comments
        size_t cm = line.find("//");
        if (cm != std::string::npos) line = line.substr(0, cm);
        // trim
        while (!line.empty() && (line.front()==' '||line.front()=='\t'||line.front()=='\r')) line=line.substr(1);
        while (!line.empty() && (line.back()==' '||line.back()=='\t'||line.back()=='\r'||line.back()=='\n')) line.pop_back();
        if (line.empty()) continue;
        // parse "r, g, b"
        int r2=0, g2=0, b2=0;
        if (sscanf(line.c_str(), "%d,%d,%d", &r2, &g2, &b2) == 3) {
            fVisColors.r[slot] = (uint8_t)std::max(0,std::min(255,r2));
            fVisColors.g[slot] = (uint8_t)std::max(0,std::min(255,g2));
            fVisColors.b[slot] = (uint8_t)std::max(0,std::min(255,b2));
            slot++;
        }
    }
}

inline bool WinampSkin::Load(const char* path) {
    // Clear sheets but keep fFiles if reloading from same zip
    for(auto& p:fSheets) delete p.second;
    fSheets.clear();
    fFiles.clear();
    fLoaded=false;
    fPlColors=PlEditColors{};
    fVisColors=VisColors{};

    if(!ExtractZip(path)) return false;
    fLoaded=true;
    ParsePleditTxt();
    ParseViscolorTxt();

    // decode all BMPs at 1x into fSheets
    for(auto& kv:fFiles) {
        const std::string& k=kv.first;
        if(k.size()>4 && k.substr(k.size()-4)==".bmp") {
            BBitmap* bm=DecodeBMP(kv.second);
            fSheets[k]=bm;
        }
    }

    // if scale>1, replace every sheet in-place (qmmp double_size block)
    if(fScale > 1) {
        for(auto& kv:fSheets) {
            if(!kv.second) continue;
            BBitmap* scaled = fSmooth ? ScaleSmooth(kv.second, fScale)
                                       : Scale(kv.second, fScale);
            delete kv.second;
            kv.second = scaled;
        }
    }
    return true;
}

inline void WinampSkin::SetScale(int scale, bool smooth) {
    fScale  = (scale >= 2) ? 2 : 1;
    fSmooth = smooth;
    if(!fLoaded) return;

    // Re-decode all sheets at 1x
    for(auto& kv:fSheets) delete kv.second;
    fSheets.clear();
    for(auto& kv:fFiles) {
        const std::string& k=kv.first;
        if(k.size()>4 && k.substr(k.size()-4)==".bmp") {
            BBitmap* bm=DecodeBMP(kv.second);
            fSheets[k]=bm;
        }
    }
    // Scale in-place if needed
    if(fScale > 1) {
        for(auto& kv:fSheets) {
            if(!kv.second) continue;
            BBitmap* scaled = fSmooth ? ScaleSmooth(kv.second, fScale)
                                       : Scale(kv.second, fScale);
            delete kv.second;
            kv.second = scaled;
        }
    }
}

inline void WinampSkin::Unload() {
    for(auto& p:fSheets) delete p.second;
    fSheets.clear(); fFiles.clear(); fLoaded=false;
    fPlColors=PlEditColors{};
    fScale=1; fSmooth=false;
}

inline BBitmap* WinampSkin::Sheet(const char* name) const {
    std::string key=Lower(name);
    auto it=fSheets.find(key); if(it!=fSheets.end()) return it->second;
    BBitmap* bm=nullptr;
    auto fit=fFiles.find(key); if(fit!=fFiles.end()) bm=DecodeBMP(fit->second);
    fSheets[key]=bm; return bm;
}

inline BBitmap* WinampSkin::Crop(BBitmap* src, SkinRect r) const {
    if(!src||r.w<=0||r.h<=0) return nullptr;
    int srcW=(int)src->Bounds().Width()+1;
    int srcH=(int)src->Bounds().Height()+1;
    if(r.x>=srcW||r.y>=srcH) return nullptr;
    int w=std::min(r.w,srcW-r.x);
    int h=std::min(r.h,srcH-r.y);
    BBitmap* out=new BBitmap(BRect(0,0,(float)(w-1),(float)(h-1)),B_RGB32);
    if(!out||out->InitCheck()!=B_OK){delete out;return nullptr;}
    const uint8_t* s=(const uint8_t*)src->Bits();
    uint8_t*       d=(uint8_t*)out->Bits();
    int sBPR=(int)src->BytesPerRow(), dBPR=(int)out->BytesPerRow();
    for(int row=0;row<h;row++)
        memcpy(d+row*dBPR, s+(r.y+row)*sBPR+r.x*4, (size_t)w*4);
    return out;
}

// qmmp: pix.scaled(w*r, h*r, KeepAspectRatio, FastTransformation)
inline BBitmap* WinampSkin::Scale(BBitmap* src, int factor) const {
    if(!src) return nullptr;
    int sw=(int)src->Bounds().Width()+1, sh=(int)src->Bounds().Height()+1;
    if(factor<=1) {
        BBitmap* c=new BBitmap(BRect(0,0,sw-1,sh-1),B_RGB32);
        if(c&&c->InitCheck()==B_OK){memcpy(c->Bits(),src->Bits(),(size_t)src->BitsLength());return c;}
        delete c; return nullptr;
    }
    BBitmap* out=new BBitmap(BRect(0,0,sw*factor-1,sh*factor-1),B_RGB32);
    if(!out||out->InitCheck()!=B_OK){delete out;return nullptr;}
    const uint8_t* sp=(const uint8_t*)src->Bits();
    uint8_t*       dp=(uint8_t*)out->Bits();
    int sBPR=(int)src->BytesPerRow(), dBPR=(int)out->BytesPerRow();
    for(int sy2=0;sy2<sh;sy2++) {
        const uint8_t* sRow=sp+sy2*sBPR;
        for(int sx2=0;sx2<sw;sx2++) {
            const uint8_t* px=sRow+sx2*4;
            for(int fy=0;fy<factor;fy++) {
                uint8_t* dRow=dp+(sy2*factor+fy)*dBPR+sx2*factor*4;
                for(int fx=0;fx<factor;fx++){
                    dRow[fx*4+0]=px[0];dRow[fx*4+1]=px[1];
                    dRow[fx*4+2]=px[2];dRow[fx*4+3]=px[3];}
            }
        }
    }
    return out;
}

inline BBitmap* WinampSkin::ScaleSmooth(BBitmap* src, int factor) const {
    if(!src) return nullptr;
    if(factor != 2) return Scale(src, factor); // only implement 2x bilinear
    int sw=(int)src->Bounds().Width()+1, sh=(int)src->Bounds().Height()+1;
    int dw=sw*2, dh=sh*2;
    BBitmap* out=new BBitmap(BRect(0,0,dw-1,dh-1),B_RGB32);
    if(!out||out->InitCheck()!=B_OK){delete out;return nullptr;}
    const uint8_t* sp=(const uint8_t*)src->Bits();
    uint8_t*       dp=(uint8_t*)out->Bits();
    int sBPR=(int)src->BytesPerRow(), dBPR=(int)out->BytesPerRow();
    // Bilinear 2x: for each source pixel (sx,sy), output 4 pixels using
    // weighted blends with right/below/diagonal neighbours.
    for(int sy2=0;sy2<sh;sy2++) {
        for(int sx2=0;sx2<sw;sx2++) {
            int nx=std::min(sx2+1,sw-1), ny=std::min(sy2+1,sh-1);
            const uint8_t* p00=sp+sy2*sBPR+sx2*4;
            const uint8_t* p10=sp+sy2*sBPR+nx*4;
            const uint8_t* p01=sp+ny*sBPR+sx2*4;
            const uint8_t* p11=sp+ny*sBPR+nx*4;
            // 4 output pixels at (2*sx,2*sy), (2*sx+1,2*sy), (2*sx,2*sy+1), (2*sx+1,2*sy+1)
            auto blend=[](const uint8_t* a, const uint8_t* b,
                          const uint8_t* c2, const uint8_t* d,
                          int wa, int wb, int wc, int wd,
                          uint8_t* out2) {
                int total=wa+wb+wc+wd;
                for(int ch=0;ch<3;ch++)
                    out2[ch]=(uint8_t)((a[ch]*wa+b[ch]*wb+c2[ch]*wc+d[ch]*wd)/total);
                out2[3]=255;
            };
            blend(p00,p10,p01,p11, 9,3,3,1, dp+(2*sy2)*dBPR+(2*sx2)*4);
            blend(p00,p10,p01,p11, 3,9,1,3, dp+(2*sy2)*dBPR+(2*sx2+1)*4);
            blend(p00,p10,p01,p11, 3,1,9,3, dp+(2*sy2+1)*dBPR+(2*sx2)*4);
            blend(p00,p10,p01,p11, 1,3,3,9, dp+(2*sy2+1)*dBPR+(2*sx2+1)*4);
        }
    }
    return out;
}

inline BBitmap* WinampSkin::VolumeStrip(int pct) const {
    if (pct < 0)   pct = 0;
    if (pct > 100) pct = 100;
    int n=pct*27/100; // strip index 0-27
    int s=fScale;
    return Crop("volume.bmp", {0, n*kVolStripStep*s, kVolStripW*s, kVolStripH*s});
}

inline BBitmap* WinampSkin::VolumeKnob(bool pressed) const {
    BBitmap* vol=Sheet("volume.bmp"); if(!vol) return nullptr;
    int h=(int)vol->Bounds().Height()+1;
    int s=fScale;
    // At 1x: knob at y=422. At 2x: knob at y=422*2=844.
    if(h <= 425*s) return nullptr; // no knob sprite in this skin
    int kx=pressed?0:15*s;
    return Crop("volume.bmp", {kx, kVolKnobY*s, kVolKnobW*s, h-kVolKnobY*s});
}

inline BBitmap* WinampSkin::Digit(char c) const {
    BBitmap* sh=Sheet("nums_ex.bmp"); int cols=12;
    if(!sh){sh=Sheet("numbers.bmp"); cols=11;}
    if(!sh) return nullptr;
    int idx=(c>='0'&&c<='9')?(c-'0'):(c=='-'&&cols==12)?11:10;
    int s=fScale;
    return Crop(sh,{idx*kDigitW*s, 0, kDigitW*s, kDigitH*s});
}
