#pragma once

#include <Window.h>
#include <View.h>
#include <Button.h>
#include <TextControl.h>
#include <ListView.h>
#include <ScrollView.h>
#include <StringView.h>
#include <String.h>
#include <StringItem.h>
#include <LayoutBuilder.h>
#include <Messenger.h>
#include <Message.h>

#include <thread>
#include <vector>
#include <string>
#include <functional>

#include "PandoraAPI.h"

static constexpr uint32 MSG_STATION_CREATED  = 'StCr';  // AddString("token"), AddString("name")
static constexpr uint32 MSG_ADD_STATION      = 'StAd';  // open search window

static constexpr uint32 MSG_SSW_SEARCH       = 'SrSr';
static constexpr uint32 MSG_SSW_RESULTS      = 'SrRs';  // results ready
static constexpr uint32 MSG_SSW_ADD          = 'SrAd';
static constexpr uint32 MSG_SSW_DONE         = 'SrDn';  // station created
static constexpr uint32 MSG_SSW_ERROR        = 'SrEr';  // AddString("e")

class StationSearchWindow : public BWindow {
public:
    StationSearchWindow(BMessenger owner, PandoraAPI* api)
        : BWindow(BRect(0, 0, 419, 299), "Add Station",
                  B_FLOATING_WINDOW_LOOK, B_FLOATING_APP_WINDOW_FEEL,
                  B_NOT_RESIZABLE | B_AUTO_UPDATE_SIZE_LIMITS | B_CLOSE_ON_ESCAPE)
        , fOwner(owner)
        , fAPI(api)
    {
        BView* root = new BView(Bounds(), "root", B_FOLLOW_ALL, B_WILL_DRAW);
        root->SetViewUIColor(B_PANEL_BACKGROUND_COLOR);
        AddChild(root);

        fQueryField = new BTextControl(BRect(10, 12, 300, 32), "q", "", "",
                                       new BMessage(MSG_SSW_SEARCH));
        fQueryField->SetModificationMessage(nullptr);
        fQueryField->SetDivider(0);
        fQueryField->TextView()->SetMaxBytes(128);
        root->AddChild(fQueryField);

        fSearchBtn = new BButton(BRect(308, 10, 408, 34), "search", "Search",
                                 new BMessage(MSG_SSW_SEARCH));
        root->AddChild(fSearchBtn);

        BView* sep1 = new BView(BRect(0, 42, 419, 43), "sep1", B_FOLLOW_LEFT_RIGHT, 0);
        sep1->SetViewUIColor(B_SHADOW_COLOR);
        root->AddChild(sep1);

        fResultsList = new BListView(BRect(10, 50, 398, 218), "results",
                                     B_SINGLE_SELECTION_LIST, B_FOLLOW_ALL);
        fResultsList->SetSelectionMessage(new BMessage('selc'));
        BScrollView* sv = new BScrollView("rsv", fResultsList,
                                          B_FOLLOW_ALL, 0, false, true);
        sv->MoveTo(10, 50);
        sv->ResizeTo(400, 170);
        root->AddChild(sv);

        // Status
        fStatus = new BStringView(BRect(10, 228, 409, 244), "status", "Type a name and press Search.");
        root->AddChild(fStatus);

        // Bottom separator
        BView* sep2 = new BView(BRect(0, 248, 419, 249), "sep2", B_FOLLOW_LEFT_RIGHT, 0);
        sep2->SetViewUIColor(B_SHADOW_COLOR);
        root->AddChild(sep2);

        // Buttons
        fAddBtn = new BButton(BRect(10, 256, 140, 280), "add", "Add Station",
                              new BMessage(MSG_SSW_ADD));
        fAddBtn->SetEnabled(false);
        root->AddChild(fAddBtn);

        fCancelBtn = new BButton(BRect(310, 256, 408, 280), "cancel", "Cancel",
                                 new BMessage(B_QUIT_REQUESTED));
        root->AddChild(fCancelBtn);

        fQueryField->MakeFocus(true);
        SetDefaultButton(fSearchBtn);
        CenterOnScreen();
    }

    void MessageReceived(BMessage* msg) override {
        switch (msg->what) {

        case MSG_SSW_SEARCH: {
            const char* q = fQueryField->Text();
            if (!q || q[0] == '\0') break;
            fSearchBtn->SetEnabled(false);
            fAddBtn->SetEnabled(false);
            fResultsList->MakeEmpty();
            fResults.clear();
            SetStatus("Searching…");
            std::string query(q);
            std::thread([this, query]() {
                fAPI->SearchMusic(query, [this](bool ok,
                        const std::vector<SearchResult>& res,
                        const std::string& err) {
                    if (ok) {
                        // Pack results into message as parallel string arrays
                        BMessage m(MSG_SSW_RESULTS);
                        for (auto& r : res) {
                            m.AddString("name",  r.name.c_str());
                            m.AddString("token", r.musicToken.c_str());
                            m.AddBool("artist",  r.isArtist);
                        }
                        PostMessage(&m);
                    } else {
                        BMessage m(MSG_SSW_ERROR);
                        m.AddString("e", err.c_str());
                        PostMessage(&m);
                    }
                });
            }).detach();
            break;
        }

        case MSG_SSW_RESULTS: {
            fResults.clear();
            fResultsList->MakeEmpty();

            const char* name  = nullptr;
            const char* token = nullptr;
            bool artist       = false;
            int32 i = 0;
            while (msg->FindString("name",  i, &name)  == B_OK &&
                   msg->FindString("token", i, &token) == B_OK &&
                   msg->FindBool ("artist", i, &artist) == B_OK) {
                SearchResult r;
                r.name       = name;
                r.musicToken = token;
                r.isArtist   = artist;
                fResults.push_back(r);

                // Prefix label: 🎤 for artist, 🎵 for song
                BString label(artist ? "Artist: " : "Song: ");
                label << name;
                fResultsList->AddItem(new BStringItem(label.String()));
                i++;
            }

            if (fResults.empty())
                SetStatus("No results found.");
            else {
                BString s; s << fResults.size() << " result(s). Select one and press Add Station.";
                SetStatus(s.String());
            }
            fSearchBtn->SetEnabled(true);
            break;
        }

        case 'selc':  // list selection changed
            fAddBtn->SetEnabled(fResultsList->CurrentSelection() >= 0);
            break;

        case MSG_SSW_ADD: {
            int32 sel = fResultsList->CurrentSelection();
            if (sel < 0 || sel >= (int32)fResults.size()) break;
            std::string token = fResults[sel].musicToken;
            std::string name  = fResults[sel].name;
            fAddBtn->SetEnabled(false);
            fSearchBtn->SetEnabled(false);
            SetStatus("Creating station…");
            std::thread([this, token, name]() {
                fAPI->CreateStation(token, [this, name](bool ok,
                        const PandoraStation& st,
                        const std::string& err) {
                    if (ok) {
                        BMessage m(MSG_SSW_DONE);
                        m.AddString("token", st.stationToken.c_str());
                        m.AddString("name",  st.stationName.c_str());
                        PostMessage(&m);
                    } else {
                        BMessage m(MSG_SSW_ERROR);
                        m.AddString("e", err.c_str());
                        PostMessage(&m);
                    }
                });
            }).detach();
            break;
        }

        case MSG_SSW_DONE: {
            const char* token = nullptr; const char* name = nullptr;
            msg->FindString("token", &token);
            msg->FindString("name",  &name);
            // Forward to owner
            BMessage fwd(MSG_STATION_CREATED);
            fwd.AddString("token", token ? token : "");
            fwd.AddString("name",  name  ? name  : "");
            fOwner.SendMessage(&fwd);
            PostMessage(B_QUIT_REQUESTED);
            break;
        }

        case MSG_SSW_ERROR: {
            const char* e = nullptr;
            msg->FindString("e", &e);
            BString s("Error: "); s << (e ? e : "unknown");
            SetStatus(s.String());
            fSearchBtn->SetEnabled(true);
            fAddBtn->SetEnabled(fResultsList->CurrentSelection() >= 0);
            break;
        }

        default:
            BWindow::MessageReceived(msg);
        }
    }

    bool QuitRequested() override { return true; }

private:
    void SetStatus(const char* s) {
        if (fStatus) fStatus->SetText(s);
    }

    BMessenger              fOwner;
    PandoraAPI*             fAPI;
    BTextControl*           fQueryField = nullptr;
    BButton*                fSearchBtn  = nullptr;
    BListView*              fResultsList = nullptr;
    BStringView*            fStatus     = nullptr;
    BButton*                fAddBtn     = nullptr;
    BButton*                fCancelBtn  = nullptr;
    std::vector<SearchResult> fResults;
};
